/*
 *  cmd-read.c
 *		The functions to actually execute the for the
 *		client.  Commands to read texts.
 *
 *
 *  Copyright (C) 1990	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Thomas Bellman
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	Bellman@Lysator.LiU.SE
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL opinions,
 *  and does NOT, repeat NOT, represent any official standpoint of Lysator,
 *  even if so stated.
 */

#include <config.h>

#include <stdio.h>
/* #include <time.h> is included from kom-types.h */
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if STDC_HEADERS || HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#ifdef HAVE_ASSERT_H
#include <assert.h>
#endif

#include <libintl.h>

#include <kom-errno.h>
#include <kom-types.h>
#include <services.h>
#ifdef __vax__
#define __SIZE_T included from stddef.h
#define __WCHAR_T included from stddef.h
#endif
#include <zmalloc.h>
#include <s-string.h>
#include <s-collat.h>

#include "conf.h"

#include "xoutput.h"
#include "copy.h"
#include "read-line.h"
#include "commands.h"
#include "parser.h"
#include "edit.h"
#include "misc-parser.h"
#include "error.h"
#include "cache.h"
#include "reading.h"
#include "quit.h"

#include "internal.h"
#include "cmds.h"
#include "offline.h"


/* This is to emphasize that an object is exported. */
#define Export

/* We are keeping track of whats going on here. */
Export Internal_state_t	reading_state;

/* Local function. */
static int recursively_mark_texts_as_read(Text_no);


/*
 *  The parameter 'ARGUMENT' in all the external functions, is a
 *  String containing the rest of the line, following the command.
 *  E g if the user gave the command "Skapa m|te Inl{gg }t mig",
 *  then the String "Inl{gg }t mig" is passed as the parameter
 *  'ARGUMENT' to the function responsible for the command 'skapa
 *  m|te'.  Leading and trailing spaces are skipped.  EMPTY_STRING
 *  is passed if no arguments to the command was given.
 */


/*  Find and display text(s) according to strange search patterns.  */
Export  Success
cmd_find_and_view_text (String   argument)
{
    Text_no		  wanted_text;
    const String	  the_commented	= s_fcrea_str(gettext("(det) kommenterade"));
    String_size		  error_check;	/* First uncoverted char str->num */
    Success		  retval;


#define CLEAN_UP()	do {	s_clear (&argument);	} while (0)

    TOPLOOP_SETJMP();

    newline();

    if (s_empty(argument))
    {
	xprintf(gettext("Vilket inlgg vill du terse?\n= "));
	read_line (&argument);		/* BUG! Check retval! */
	newline();
    }
    argument = s_strip_trailing (argument, command_separators);

    /* A number, and nothing else? */
    wanted_text = (Text_no) s_strtol (argument, &error_check,
				      DEFAULT_NUMBER_BASE);
    if (error_check < s_strlen (argument))
    {
	/* No, it wasn't a number.  Then try "(the) commented" */
	/* BUG: Check this out */
	if (! s_usr_strhead (argument, the_commented, DEFAULT_COLLAT_TAB))
	{
	    xprintf (gettext("Oknt argument: "));
	    s_puts(argument);
	    newline();
	    RETURN (FAILURE);
	}
	cmd_not_yet_implemented (argument);
	RETURN (FAILURE);
    }
    else
    {
	/* OK, it was a number */
	retval = show_text_no (wanted_text);
	RETURN (retval);
    }

    /*NOTREACHED*/
    RETURN (OK);
#undef CLEAN_UP
}




/*  Read the next text in straight order  */
Export  Success
cmd_read_next_text (String   argument)

{
#define FUNCTION "read_next_text()"

    Success		  exit_value;
    Text_no		  global_no_to_read;
    Local_text_no	  ltno;

#define CLEAN_UP()	do {	} while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    if (ccn == 0)
    {
	xprintf (gettext("Du r inte nrvarande in ngot mte.\n"));
	RETURN (FAILURE);
    }

    global_no_to_read = reading_next_text_to_be_read_with_local_no(&ltno);

    if (global_no_to_read == 0)
    {
	xprintf (gettext("Det finns inga fler olsta i detta mte.\n"));
	RETURN (FAILURE);
    }

    exit_value = show_text_no (global_no_to_read);

    /* The text will be marked as read when the next command is given. */
    reading_read_text(global_no_to_read);

    if (exit_value == OK)
    {
	/* BUG: We don't care that the text was not readable. We mark it
	   as read no matter what.
	   This will be a problem in the offline case when a text that was
	   not downloaded will be marked as read.
	*/
    }
    else
    {
	reading_read_text_local_no(ccn, ltno);
	if (offline)
	    mark_as_read_locally(ccn, 1, &ltno);
    }

    RETURN (exit_value);

#undef CLEAN_UP
#undef FUNCTION
}


/*  Read the commented text  */
extern Success
cmd_read_commented_text (String  argument)
{
#define FUNCTION "cmd_read_commented_text()"

    Text_no		  text_to_search_from;
    volatile Text_no	  text_to_read = 0;
    Text_stat		* text_stat_commented = NULL;

#define CLEAN_UP()	do {				\
	s_clear (&argument); 				\
	if (text_stat_commented)			\
	{						\
	    release_text_stat(text_stat_commented);	\
	    zfree(text_stat_commented);			\
	}						\
    } while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    text_to_search_from = reading_state.last_viewed;

    if (text_to_search_from
	&& (text_stat_commented = text_stat(text_to_search_from)) != NULL)
    {
	Misc_info     * misc;
	unsigned short	r;

	newline();
	for (misc = text_stat_commented->misc_items, r = 0;
	     r < text_stat_commented->no_of_misc && !text_to_read;
	     r++, misc++)
	{
	    if (misc->type == comm_to)
	    {
		text_to_read = misc->datum.comment_to;
	    }
	}

	if (text_to_read)
	{
	    RETURN (show_text_no (text_to_read));
	}
	else 
	{
	    xprintf(gettext("Det finns inget sdant inlgg.\n"));
	    RETURN (OK);
	}
    }
    
    RETURN (FAILURE);
#undef CLEAN_UP
#undef FUNCTION
}


/*
 * cmd_read_next_comment
 * Read the next text in comment tree.
 *
 * This is easy since we have already calculated what text to read.
 */
Export  Success
cmd_read_next_comment (String   argument)

{
#define FUNCTION "cmd_read_next_comment()"

    Success		  exit_value;
    Text_no		  global_no_to_read;


#define CLEAN_UP()	do {	} while (0)


    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    global_no_to_read = reading_next_comment_to_be_read();
    if (global_no_to_read == 0) 
    {
	xprintf(gettext("Det finns ingen kommentar.\n"));
	RETURN(FAILURE);
    }

    /* BUG: Doesn't check for error */
    exit_value = show_text_no (global_no_to_read);

    /* The text will be marked as read when the next command is given. */
    reading_read_text(global_no_to_read);

    if (exit_value == OK) {
	/* BUG: We don't care that the text was not readable. We mark it
	   as read no matter what.
	   This will be a problem in the offline case when a text that was
	   not downloaded will be marked as read.
	*/
    }

    RETURN (exit_value);

#undef CLEAN_UP
#undef FUNCTION
}


/*
 * cmd_read_next_footnote
 * Read the next text in footnote tree.
 *
 * This is easy since we have already calculated what text to read.
 */
Export  Success
cmd_read_next_footnote (String   argument)

{
#define FUNCTION "cmd_read_next_footnote()"

    Success		  exit_value;
    Text_no		  global_no_to_read;


#define CLEAN_UP()	do {	} while (0)


    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    global_no_to_read = reading_next_footnote_to_be_read();
    if (global_no_to_read == 0) 
    {
	xprintf(gettext("Det finns inga fotnoter.\n"));
	RETURN(FAILURE);
    }

    /* BUG: Doesn't check for error */
    exit_value = show_text_no (global_no_to_read);

    /* The text will be marked as read when the next command is given. */
    reading_read_text(global_no_to_read);

    if (exit_value == OK) {
	/* BUG: We don't care that the text was not readable. We mark it
	   as read no matter what.
	   This will be a problem in the offline case when a text that was
	   not downloaded will be marked as read.
	*/
    }

    RETURN (exit_value);

#undef CLEAN_UP
#undef FUNCTION
}


/*
 * Jump comments to the last read article
 * The search is done recursively.
 * Loops are handled in the way that we only continue the recursive
 * search if the comment is newer.
 */
extern Success
cmd_jump_tree (String	argument)
{
#define FUNCTION "cmd_jump_tree()"
    
#define CLEAN_UP()	do { ; } while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    newline();
    if (reading_state.last_viewed) {
	int removed;

	removed = recursively_mark_texts_as_read(reading_state.last_viewed);
	removed--;
	if (removed == 0)
	    xprintf(gettext("Du hoppade inte ver ngra inlgg.\n"));
	else if (removed == 1)
	    xprintf(gettext("Du hoppade ver ett inlgg.\n"));
	else if (removed > 1)
	    xprintf(gettext("Du hoppade ver %d inlgg.\n"), removed);

	RETURN (OK);
    }

    RETURN (FAILURE);
#undef CLEAN_UP
#undef FUNCTION
}


/*  Goto the next conference with unread news. */
extern Success
cmd_goto_next_conf (String   argument)
{
#define FUNCTION "cmd_goto_next_conf()"

#define CLEAN_UP() do { } while (0)

    Conf_no conf;

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    review_reset();		/* Stop reviewing. */

    while ((conf = reading_next_conf_with_unread()) != 0)
    {
	int unread;

	reading_goto_conf(conf);

	unread = reading_unread_texts_count();

	if (unread == 0)
	{
	    /*
	     * The texts to be read did not exist.
	     */
	    if (!offline)
	    {
		/*
		 * This tries to catch up.
		 */
	        /* Preferred solution:
	        Kom_err err;
		kom_set_last_read(conf, conf_last(conf, &err));
		*/
	        kom_set_unread(conf, 0);
	    }
	    continue;
	}

	print_conf_name(conf);
	xprintf(" - %d %s.\n", unread, 
		(unread == 1 ? gettext("olst inlgg")
		 : gettext("olsta inlgg")));
	RETURN (OK);
    }
    
    RETURN (FAILURE);

#undef CLEAN_UP
#undef FUNCTION
}


/*  Endast, mark text as read quickly!
 * 
 *  If the argument given is smaller than the amount of unread, then we
 *  calculate the amount of articles that are to be marked as read and
 *  move up the last_read-counter with the special request.
 *
 *  If the argument given is bigger than the amount of unread, the we
 *  move down the last_read-counter
 */
Export  Success
cmd_only (String   argument)
{
#define FUNCTION "cmd_only()"

    int			  unread_wanted;
    String_size           error_check;  /* First uncoverted char str->num */
    Success		  retval;


#define CLEAN_UP()	do { s_clear(&argument); } while (0)

    TOPLOOP_SETJMP();

    OFFLINE();

    newline();

    if (ccn == 0)
    {
	xprintf(gettext("Du r inte nrvarande i ngot mte just nu.\n"));
	RETURN (FAILURE);
    }

    if (s_empty(argument))
    {
        xprintf (gettext("Hur mnga texter vill du se? = "));
        read_line (&argument);          /* BUG! Check retval! */
	newline();
    }
    argument = s_strip_trailing (argument, command_separators);

    /* A number, and nothing else? */
    unread_wanted = s_strtol (argument, &error_check,
			      DEFAULT_NUMBER_BASE);
    if (error_check < s_strlen (argument))
    {
        xprintf (gettext("Oknt argument: "));
        s_puts(argument);
        newline();
        RETURN (FAILURE);
    }
    else
    {
	/* Forget all cached information about that conf. */
	Conf_no saved_ccn = ccn;

	cache_change_person(cpn); /* Reset all the cache. */
	reading_forget_about_conf();
	ccn = saved_ccn;
	
	/* Just set the new value */
	retval = kom_set_unread(ccn, unread_wanted);

	if (retval == FAILURE)
	{
	    switch (kom_errno)
	    {
	    case  KOM_UNDEF_CONF:
	    case  KOM_PERM:
		xprintf (gettext("Du fr inte lngre lsa det hr mtet.\n"));
		break;

	    default:
		fatal3 (CLIENT_SHOULDNT_HAPPEN,
			"kom_set_unread(%d, %d) returns bogus",
			ccn, unread_wanted);
		break;
	    }

	    RETURN (FAILURE);
	}

    }

    RETURN (OK);
#undef CLEAN_UP
#undef FUNCTION
}

/* Display all new articles. */
Export  Success
cmd_all (String   argument)
{
#define FUNCTION "cmd_all()"

    Success (* next_command) (String arg);

#define CLEAN_UP()	do { s_clear(&argument); } while (0)

    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_NO_ARGUMENTS();

    newline();

    for(;;) {
	handle_all_asyncs();	/* Read all asyncs. */

	next_command = cmd_show_current_time; /* Default command. */

	if (reading_next_footnote_to_be_read() != 0)
	{
	    next_command = cmd_read_next_footnote;
	}
	else if (reading_next_comment_to_be_read() != 0)
	{
	    next_command = cmd_read_next_comment;
	}
	else if (reading_next_text_to_be_read() != 0)
	{
	    next_command = cmd_read_next_text;
	}
	else if (reading_next_conf_with_unread() != 0)
	{
	    next_command = cmd_goto_next_conf;
	} 

	printed_lines = 0;
	current_column = 0;
	newline();

        if (next_command == cmd_show_current_time)
	    break;

        (next_command)(EMPTY_STRING);
    }

    RETURN (OK);
#undef CLEAN_UP
#undef FUNCTION
}   /* END: command_loop() */

/* ==================================================================== */
/*			Internal functions				*/


/*
 *  Fetch and show the text WANTED if the server is willing to let
 *  the person see it.  Currently only one format for showing the
 *  texts - no options available.
 */
Export  Success
show_text_no (Text_no	wanted)

{
/*    static char		* this_function	= "show_text_no()";*/

    Text_stat	*volatile stat		= NULL;
    String		  text		= EMPTY_STRING;
    Kom_err		  error;


#define CLEAN_UP()	do {				\
	if (stat != NULL)				\
	    zfree (stat->misc_items);			\
	zfree (stat);					\
	s_clear (&text);					} while (0)

    TOPLOOP_SETJMP();

    if ((stat = text_stat(wanted)) == NULL
	|| (s_empty((text = get_text (wanted, 0, END_OF_STRING, &error)))
	    && (error != KOM_NO_ERROR)))
    {
	xprintf (gettext("Det finns inget sdant inlgg.\n"));
	RETURN (FAILURE);
    }

    newline();
    print_text_header (wanted, stat);
    print_misc_list (stat->misc_items, stat->no_of_misc,
		     PRINT_RECPT | PRINT_CC_RECPT);
    print_misc_list (stat->misc_items, stat->no_of_misc,
		     PRINT_COMM_TO | PRINT_FOOTN_TO);
    print_subject_line (gettext("rende"), text);
    xprintf ("------------------------------------------------------------\n");
    print_rest_of_text (text);
    xprintf ("(%lu)------------------------------\n", wanted);
    print_misc_list (stat->misc_items, stat->no_of_misc,
		     PRINT_COMM_IN | PRINT_FOOTN_IN);
    reading_state.last_viewed = wanted;

    RETURN (OK);

#undef CLEAN_UP
}



Export  void
print_text_header (Text_no		  number,
		   const Text_stat	* stat   )
{
    xprintf ("(%lu) ", number);
    print_time (&stat->creation_time);
    xprintf (gettext(" /%d rader/ "), stat->no_of_lines);
    print_conf_name ((Conf_no) stat->author);
    newline();
}



Export  void
print_misc_list (const Misc_info	* misc_list,
		 unsigned short		  no_of_misc,
		 unsigned short		  what_to_print)

{
#define FUNCTION "print_misc_list()"

    const Misc_info	* stop_ptr;	/* Address of element beyond last */
    Misc_info_group	  misc_data;
    Text_stat		* stat2;	/* For info about other texts */

    stop_ptr = misc_list + no_of_misc;
    while ((misc_data = parse_next_misc (&misc_list, stop_ptr)).type
	   != m_end_of_list)
    {
	switch (misc_data.type)
	{
	case m_recpt:
	    if (!(what_to_print & PRINT_RECPT))
		break;
	    /* Else fall through */

	case m_cc_recpt:
	    if (!(what_to_print & (PRINT_RECPT | PRINT_CC_RECPT)))
		break;

	    if (misc_data.type == m_recpt)
	    {
		if (is_sent_by(misc_data))
		    xprintf(gettext("Fr knnedom: "));
		else
		    xprintf(gettext("Mottagare: "));
		print_conf_name(misc_data.recipient);
	    }
	    else
	    {
		xprintf(gettext("Extra kopia: "));
		print_conf_name(misc_data.recipient);
	    }
	    if (misc_data.local_no) /* If we have a local_no */
		xprintf(" <%lu>", misc_data.local_no);

	    if (is_received(misc_data))
	    {
		xprintf(gettext(" -- Mottaget: "));
		print_time (&misc_data.received_at);
	    }

	    newline();

	    if (is_sent_by(misc_data))
	    {
		xprintf(gettext(" Sndare: "));
		print_conf_name((Conf_no) misc_data.sender);
	    }
	    if (is_sent(misc_data))
	    {
		xprintf(gettext(" -- Snt: "));
		print_time(&misc_data.sent_at);
		newline();
	    }

	    break;


	case m_comm_to:
	    if (! (what_to_print & PRINT_COMM_TO))
		break;

	    xprintf(gettext("Kommentar till text %lu"), misc_data.text_ref);
	    if ((stat2 = text_stat(misc_data.text_ref)) != NULL)
	    {
		xprintf(gettext(" av "));
		print_conf_name((Conf_no) stat2->author);
		zfree(stat2->misc_items);
		zfree(stat2);
	    }
	    newline();
	    break;


	case m_footn_to:
	    if (!(what_to_print & PRINT_FOOTN_TO))
		break;

	    xprintf(gettext("Fotnot till text %lu"), misc_data.text_ref);
	    if ((stat2 = text_stat(misc_data.text_ref)) != NULL)
	    {
		xprintf(gettext(" av "));
		print_conf_name((Conf_no) stat2->author);
		zfree(stat2->misc_items);
		zfree(stat2);
	    }
	    newline();
	    break;


	case m_comm_in:
	    if (!(what_to_print & PRINT_COMM_IN))
		break;

	    xprintf(gettext("(Kommentar i text %lu"), misc_data.text_ref);
	    if ((stat2 = text_stat(misc_data.text_ref)) != NULL)
	    {
		xprintf(gettext(" av "));
		print_conf_name((Conf_no) stat2->author);
		zfree(stat2->misc_items);
		zfree(stat2);
	    }
	    xputchar(')');
	    newline();
	    break;


	case m_footn_in:
	    if (!(what_to_print & PRINT_FOOTN_IN))
		break;

	    xprintf(gettext("(Fotnot i text %lu"), misc_data.text_ref);
	    if ((stat2 = text_stat(misc_data.text_ref)) != NULL)
	    {
		xprintf(gettext(" av "));
		print_conf_name((Conf_no) stat2->author);
		zfree(stat2->misc_items);
		zfree(stat2);
	    }
	    xputchar(')');
	    newline();
	    break;


	case m_error:
	    server_bug("Bad misc-items", FUNCTION, (bug_info) misc_list);
	    break;

	case m_end_of_list:
	    abort();	/* This case is taken care of by the while */
	}
    }	/* while (! end-of-misc-list) */
#undef FUNCTION
}



Export  void
print_subject_line (const unsigned char	* subject_text,
		    const String	text		)
{
    String_size		  end_of_line_pos;


    xprintf ("%s:  ", subject_text);
    end_of_line_pos = s_strchr (text, '\n', 0);
    s_xputs (s_fsubstr (text, 0, (end_of_line_pos == -1
				  ? END_OF_STRING
				  : end_of_line_pos)));
}



Export  void
print_rest_of_text (const String	text)
{
    String_size		  end_of_line_pos;


    end_of_line_pos = s_strchr (text, '\n', 0);
    s_xputs (s_fsubstr (text, end_of_line_pos + 1, END_OF_STRING));
    if (text.len == 0 || text.string[s_strlen (text) - 1] != '\n')
	newline();
}



Export  void
print_time (const struct tm	* time_val)
{
    xprintf ("%04d-%02d-%02d %02d.%02d",
	     time_val->tm_year+1900,
	     time_val->tm_mon+1,
	     time_val->tm_mday,
	     time_val->tm_hour,
	     time_val->tm_min);
}



/*
 * Recursively mark all comments to the text TNO read.
 * Returns how many we have marked.
 */
static int
recursively_mark_texts_as_read(Text_no tno)
{
    Text_stat	      * text_stat_root = NULL;
    volatile int	jumped = 0;

#define CLEAN_UP()	do {			\
	if (text_stat_root != NULL)		\
	{					\
	    release_text_stat(text_stat_root);	\
	    zfree(text_stat_root);		\
	}					\
    } while (0)

    TOPLOOP_SETJMP();

    if (tno			/* Check for stupid calls. */
	&& (text_stat_root = text_stat(tno)) != NULL)
    {
	const Misc_info       * misc;
	Misc_info_group 	misc_data;

	/* Do the recursive search */
	for (misc = text_stat_root->misc_items;
	     (misc_data = parse_next_misc(&misc,
					  text_stat_root->misc_items +
					  text_stat_root->no_of_misc)).type
	     != m_end_of_list;
	     )
	{
	    if (misc_data.type == m_comm_in)
	    {
		if (misc_data.text_ref > tno)
		    jumped += recursively_mark_texts_as_read(misc_data.text_ref);
	    }

	    if (misc_data.type == m_recpt
		|| misc_data.type == m_cc_recpt)
	    {
		reading_read_text_local_no(misc_data.recipient, 
					   misc_data.local_no);
	    }
	}
	
	/* Mark this article as read. */
	
	reading_read_text_simple(tno);
	jumped ++;
    }
    RETURN(jumped);
#undef CLEAN_UP
}


/*
 * Check if a text has been read by the user.
 * Returns False if unread.
 */
Export  Bool
has_been_read (Text_no		  text)
{
#define FUNCTION "has_been_read()"
    Text_stat * volatile  stat = NULL;
    Kom_err		  error;
    const Misc_info 	* misc;
    Misc_info_group	  misc_data;
    Membership		  membship = EMPTY_MEMBERSHIP;

#define CLEAN_UP()	do { release_text_stat(stat); \
			     if (stat != NULL) zfree(stat); \
			 } while (0)
/* Dont release the membership here. It is not marked. */

    TOPLOOP_SETJMP();

    stat = text_stat(text);

    if (stat == NULL) {
	switch(error) {
	default:
	    /* Something went wrong. Consider the text as read. */
	    RETURN(TRUE);
	}
	/*NOTREACHED*/
    }

    for (misc = stat->misc_items;
	 (misc_data = parse_next_misc(&misc,
				      stat->misc_items +
				      stat->no_of_misc)).type
	 != m_end_of_list;
	 )
    /* This for-loop is really a joke because we RETURN out of it at the first
       conference. */
    {
	if (misc_data.type == m_recpt
	    || misc_data.type == m_cc_recpt)
	{
	    if (locate_membership (misc_data.recipient, cpn, &membship) >= 0)
	    {
		/* At least we are members. */
		int r;

		if (misc_data.local_no <= membship.last_text_read)
		{
		    RETURN(TRUE); /* Pheuw */
		}
		if (membship.no_of_read == 0)
		{
		    RETURN(FALSE); /* If it hasn't been read here, 
				      it hasn't been read anywhere. */
		}
		for (r = 0; r < membship.no_of_read; r++) {
		    if (membship.read_texts[r] == misc_data.local_no)
		    {
			RETURN(TRUE);
		    }
		}
		RETURN(FALSE);	/* It hasn't been read. */
	    }
	}
    }
    RETURN (TRUE);		/* No recipient. This is read. */
#undef FUNCTION
#undef CLEAN_UP
}



/*
 *  Report the number of unread texts the person WHO has in
 *  conference WHERE.
 */
Export  long
unread_in_conf (Pers_no	    who,
		Conf_no	    where)

{
    Membership		  membship	= EMPTY_MEMBERSHIP;
    long		  no_of_unread;
    Kom_err		  error;

#define CLEAN_UP()	do { } while (0)

    TOPLOOP_SETJMP();

    if (locate_membership (where, who, &membship) < 0)
    {
	RETURN (-1);
    }

    no_of_unread = (conf_last(where, &error)
		    - membship.last_text_read - membship.no_of_read);

    RETURN (no_of_unread);

#undef CLEAN_UP
}   /* END: unread_in_conf() */

