/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  displaymanager.cc - GdkDisplayManager C++ wrapper interface
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "displaymanager.h"
#include "private/displaymanager_p.h"
#include "display.h"

using namespace Inti;

/*  Gdk::DisplayManager
 */

Gdk::DisplayManager::DisplayManager(GdkDisplayManager *display_manager, bool reference)
: G::Object((GObject*)display_manager, reference)
{
}

Gdk::DisplayManager::~DisplayManager()
{
}
	
GdkDisplayManagerClass*
Gdk::DisplayManager::gdk_display_manager_class() const
{
	return get_class<GdkDisplayManagerClass>();
}

Gdk::DisplayManager::operator GdkDisplayManager* () const
{
	return this ? gdk_display_manager() : 0;
}

Gdk::DisplayManager*
Gdk::DisplayManager::get()
{
	return G::Object::wrap<DisplayManager>(gdk_display_manager_get());
}

Gdk::Display*
Gdk::DisplayManager::get_default_display() const
{
	return G::Object::wrap<Display>(gdk_display_manager_get_default_display(gdk_display_manager()));
}

bool 
Gdk::DisplayManager::list_displays(std::vector<Display*>& displays) const
{
	g_return_val_if_fail(displays.empty(), false);
	GSList *first = gdk_display_manager_list_displays(gdk_display_manager());
	GSList *next = first;

	while (next != 0)
	{
		displays.push_back(G::Object::wrap<Display>((GdkDisplay*)next->data));
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !displays.empty();
}

void
Gdk::DisplayManager::set_default_display(const Display& display)
{
	gdk_display_manager_set_default_display(gdk_display_manager(), display.gdk_display());
}

/*  Gdk::DisplayManagerClass
 */

void
Gdk::DisplayManagerClass::init(GdkDisplayManagerClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
	g_class->display_opened = &display_opened_proxy;
}

GType
Gdk::DisplayManagerClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GDK_TYPE_DISPLAY_MANAGER, (GClassInitFunc)init);
	}
	return type;
}

void*
Gdk::DisplayManagerClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gdk::DisplayManagerClass::display_opened_proxy(GdkDisplayManager *display_manager, GdkDisplay *display)
{
	DisplayManager *tmp_display_manager = G::Object::pointer<DisplayManager>(display_manager);
	if (tmp_display_manager)
		tmp_display_manager->on_display_opened(*G::Object::wrap<Display>(display));
	else
	{
		GdkDisplayManagerClass *tmp_class = GDK_DISPLAY_MANAGER_GET_CLASS(display_manager);
		GdkDisplayManagerClass *g_class = G::TypeInstance::class_peek_parent<GdkDisplayManagerClass>(tmp_class);
		if (g_class->display_opened)
			g_class->display_opened(display_manager, display);
	}
}

/*  Gdk::DisplayManager signal handlers
 */

void
Gdk::DisplayManager::on_display_opened(Display& display)
{
	GdkDisplayManagerClass *g_class = class_peek_parent<GdkDisplayManagerClass>(gdk_display_manager_class());
	if (g_class->display_opened)
		g_class->display_opened(gdk_display_manager(), display.gdk_display());
}

/*  Gtk::DisplayManager properties
 */

const Gdk::DisplayManager::DefaultDisplayPropertyType Gdk::DisplayManager::default_display_property("default-display");

/*  Gdk::DisplayManager signals
 */

const Gdk::DisplayManager::DisplayOpenedSignalType Gdk::DisplayManager::display_opened_signal("display_opened");

