/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/checkmenuitem.h
//! @brief A GtkCheckMenuItem C++ wrapper interface.

#ifndef INTI_GTK_MENU_CHECK_ITEM_H
#define INTI_GTK_MENU_CHECK_ITEM_H

#ifndef INTI_GTK_MENU_ITEM_H
#include <inti/gtk/menuitem.h>
#endif

#ifndef __GTK_MENU_CHECK_ITEM_H__
#include <gtk/gtkcheckmenuitem.h>
#endif

namespace Inti {

namespace Gtk {

class CheckMenuItemClass;

//! @class CheckMenuItem checkmenuitem.h inti/gtk/checkmenuitem.h
//! @brief A GtkCheckMenuItem C++ wrapper class.
//!
//! A CheckMenuItem is a menu item that maintains the state of a bool value 
//! in addition to a MenuItem's usual role in activating application code.
//! A check box indicating the state of the bool value is displayed at the
//! left side of the MenuItem. Activating the MenuItem toggles the value.

class CheckMenuItem : public MenuItem
{
	friend class G::Object;
	friend class CheckMenuItemClass;

	CheckMenuItem(const CheckMenuItem&);
	CheckMenuItem& operator=(const CheckMenuItem&);

protected:
//! @name Constructors
//! @{

	explicit CheckMenuItem(GtkCheckMenuItem *check_menu_item, bool reference = false);
	//!< Construct a new CheckMenuItem from an existing GtkCheckMenuItem.
	//!< @param check_menu_item A pointer to a GtkCheckMenuItem.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>check_menu_item</EM> can be a newly created GtkCheckMenuItem or an existing
	//!< GtkCheckMenuItem. (see G::Object::Object).

//! @}
//  Override this do_ method when you want to change the default behaviour of the GtkCheckMenuItem.

	virtual void do_draw_indicator(GdkRectangle *area);

//! @name Signal Handlers
//! @{
	virtual void on_toggled();
	//!< Called when the state of the check box is changed; call get_active() to discover the new state.

//! @}
//  Properties

	typedef G::Property<bool> ActivePropertyType;
	typedef G::PropertyProxy<G::Object, ActivePropertyType> ActivePropertyProxy;
	static const ActivePropertyType active_property;

	typedef G::Property<bool> InconsistentPropertyType;
	typedef G::PropertyProxy<G::Object, InconsistentPropertyType> InconsistentPropertyProxy;
	static const InconsistentPropertyType inconsistent_property;

//  Signals
	
	typedef G::Signal0<void> ToggledSignalType;
	typedef G::SignalProxy<TypeInstance, ToggledSignalType> ToggledSignalProxy;
	static const ToggledSignalType toggled_signal;

public:
//! @name Constructors
//! @{

	CheckMenuItem();
	//!< Construct a new CheckMenuItem.

	explicit CheckMenuItem(const String& label, bool use_underline = false);
	//!< Construct a new CheckMenuItem with a Label to the right of it.
	//!< @param label The text for the menu item.
	//!< @param use_underline <EM>true</EM> if the label contains an underscore, indicating a mnemonic.
	//!<
	//!< <BR>If <EM>use_underline</EM> is true an underscore in label indicates that the next
	//!< character is the mnemonic accelerator for the menu item.

	virtual ~CheckMenuItem();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkCheckMenuItem* gtk_check_menu_item() const { return (GtkCheckMenuItem*)instance; }
	//!< Get a pointer to the GtkCheckMenuItem structure.

	GtkCheckMenuItemClass* gtk_check_menu_item_class() const;
	//!< Get a pointer to the GtkCheckMenuItemClass structure.

	operator GtkCheckMenuItem* () const;
	//!< Conversion operator; safely converts a CheckMenuItem to a GtkCheckMenuItem pointer.

	bool get_active() const;
	//!< Returns true if the check menu item is active (see set_active()).

	bool get_inconsistent() const;
	//!< Returns true if the check menu item is in an "in between" state.

//! @}
//! @name Methods
//! @{

	void set_active(bool is_active);
	//!< Sets the active state of the menu item's check box.
	//! @param is_active A bool value indicating whether the check box is active.
	
	void set_inconsistent(bool setting);
	//!< Display the check in an "in between" state.
	//! @param setting <EM>true</EM> to display an "inconsistent" third state check.
	//!<
	//!< <BR>If the user has selected a range of elements (such as some text or spreadsheet
	//!< cells) that are affected by a boolean setting, and the current values in that
	//!< range are inconsistent, you may want to display the check in an "in between"
	//!< state. This function turns on "in between" display. Normally you would turn off
	//!< the inconsistent state again if the user explicitly selects a setting. This has
	//!< to be done manually, set_inconsistent() only affects visual appearance, it 
	//!< doesn't affect the semantics of the widget.

	void toggled();
	//!< Emits the "toggled" signal.

//! @}
//! @name Property Proxies
//! @{
	
	const ActivePropertyProxy prop_active()
	{
		return ActivePropertyProxy(this, &active_property);
	}
	//!< Whether the menu item is checked (bool : Read / Write).

	const InconsistentPropertyProxy prop_inconsistent()
	{
		return InconsistentPropertyProxy(this, &inconsistent_property);
	}
	//!< Whether to display an "inconsistent" state (bool : Read / Write).

//! @}
//! @name Signal Proxies
//! @{

	const ToggledSignalProxy sig_toggled()
	{
		return ToggledSignalProxy(this, &toggled_signal);
	}
	//!< Connect to the toggled_signal; emitted when the state of the check box is changed. 
	
//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_MENU_CHECK_ITEM_H

