/***************************************************************************
 $RCSfile: bank.h,v $
 -------------------
 cvs         : $Id: bank.h,v 1.18 2003/06/13 21:50:00 aquamaniac Exp $
 begin       : Wed Jun 12 2002
 copyright   : (C) 2002 by Martin Preuss
 email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/** @file bank.h
 * 
 * @short HBCI::Bank, its C wrapper @ref HBCI_Bank. Also
 * HBCI::instituteMessage but no C wrappers.
 *
 * This file contains both the class defition for a C++ application as
 * well as the C wrappers for a C application.  */

#ifndef HBCIBANK_H
#define HBCIBANK_H

#ifdef __cplusplus
#include <string>
#include <list>
using namespace std;
#endif /* __cplusplus */

#include <openhbci/dllimport.h>
#include <openhbci/pointer.h>
#ifdef __cplusplus
namespace HBCI {
  class Bank;
};
/** @short This is the C accessible type of a Bank in %HBCI. 
 * @ingroup HBCI_Bankg 
 */
typedef struct HBCI::Bank HBCI_Bank;
/** @short List of HBCI_Bank's
 * @ingroup HBCI_Bankg */
typedef list<HBCI::Pointer<HBCI::Bank> > list_HBCI_Bank;
/** @short Iterator for List of HBCI_Bank's
 * @ingroup HBCI_Bankg */
typedef list<HBCI::Pointer<HBCI::Bank> >::const_iterator list_HBCI_Bank_iter;
#else /* __cplusplus */
typedef struct HBCI_Bank HBCI_Bank;
typedef struct list_HBCI_Bank list_HBCI_Bank;
typedef struct list_HBCI_Bank_iter list_HBCI_Bank_iter;
#endif /* __cplusplus */

#include <openhbci/hbci.h>
#include <openhbci/date.h>
#include <openhbci/account.h>
#include <openhbci/user.h>
#include <openhbci/listwrappers.h>
#include <openhbci/bpdjob.h>


#ifdef __cplusplus
namespace HBCI {
/**
 * This class is a message sent to us by the credit institute. It
 * occurs when performing the opening of a dialog (JOBDialogInit,
 * JOBFirstInit) and can contain all sorts of information that the
 * bank thinks is important (even advertisements).
 *
 * @short A message sent to us by the bank.
 * @author Martin Preuss <openhbci@aquamaniac.de>
 */
class DLLIMPORT instituteMessage {
private:
    Date _date;
    Time _time;
    string _subject;
    string _text;
    int _country;
    string _instituteCode;
    bool _read;

public:
    instituteMessage();
    ~instituteMessage();

	const bool operator==(const instituteMessage &msg);

    /** Returns the Date of arrival. */
    const Date &date() const { return _date;};
    /** Set the Date of arrival. */
    void setDate(const Date &d) { _date=d;};

    /** Returns the Time of arrival. */
    const Time &time() const { return _time;};
    /** Set the Time of arrival. */
    void setTime(const Time &t) { _time=t;};

    /** Returns the subject of this message. */
    const string &subject() const { return _subject;};
    /** Set the subject of this message. */
    void setSubject(const string &t) { _subject=t;};

    /** Returns the text of the message, just as it was sent by the bank. */
    const string &text() const { return _text;};
    /** Set the text of the message. */
    void setText(const string &t) { _text=t;};

    /** Returns the ISO-3166 country code of the bank that sent this
        message (to identify it) <br>
		<b>Note:</b>
		When creating a Transaction... don't use 
		Account.bank().ref().countryCode() but use
		Account.countryCode().<br>
		The reason is, that the country code stored along with the
		account object may differ from the code stored along with the
		bank object (don't blame openHBCI for this fact)
	*/
    int country() const { return _country;};
    /** Set the ISO-3166 country code of the bank that sent this
        message (to identify it) */
    void setCountry(int i) { _country=i;};

    /** Returns the Bank code ("BLZ") of the bank that sent this
        message (to identify it) <br>
		<b>Note:</b>
		When creating a Transaction... don't use 
		Account.bank().ref().bankCode() but use
		Account.instituteCode().<br>
		The reason is, that the institute code stored along with the
		account object may differ from the code stored along with the
		bank object (don't blame openHBCI for this fact)	   
	*/
    const string &bankCode() const { return _instituteCode;};
    /** Set the Bank code ("BLZ") of the bank that sent this
        message (to identify it) */
    void setBankCode(const string &s) { _instituteCode=s;};

    /** Helper flag: Returns true if message has been read. However,
        this is neither set nor reset by OpenHBCI. The state of the
        flag is stored (FIXME: is this correct), but the application
        has to manage the flag by itself. */
    bool read() const { return _read;};
    /** Set the flag that shows whether this message has been
        read. See @ref read(). */
    void setRead(bool b) { _read=b;};
};
}
#endif /* __cplusplus */

#ifdef __cplusplus
typedef struct HBCI::instituteMessage HBCI_instituteMessage;
typedef list<HBCI::instituteMessage> list_HBCI_instituteMessage;
#else /* __cplusplus */
typedef struct HBCI_instituteMessage HBCI_instituteMessage;
typedef struct list_HBCI_instituteMessage list_HBCI_instituteMessage;
#endif /* __cplusplus */


#ifdef __cplusplus
namespace HBCI {
/**
 * @short The abstract base class for a bank in %HBCI. 
 *
 * This is the abstract base class for a bank in %HBCI.  A bank is
 * uniquely identified by its bank code ("BLZ") and country code. For
 * the network connection this bank has a connection address (IP
 * address). The real-world name of the bank is usually retrieved
 * automatically upon our first connection to the bank.
 *
 * Each bank object maintains a list of its users (which in turn have
 * a list of customers) as well as a list of its accounts. There is
 * also a list of institute messages which this bank has sent to
 * us. Additionally, the bank might support several %HBCI versions
 * and/or dialog languages.
 *
 * You can't create an object of this class directly; instead, use the
 * factory method provided by API::bankFactory. Internally another
 * class is used as implementation of this abstract class.
 *
 * @author Martin Preuss<martin@libchipcard.de> */
class DLLIMPORT Bank {
public:
  Bank();
  virtual ~Bank();

  /**  Returns the central %HBCI object that controls the user
   * interaction and version control. */
  virtual const Hbci *hbci() const =0;

  /** @name Bank Identification
   *
   * Methods for setting or getting the data that uniquely identify
   * a bank in %HBCI.  */
  /*@{*/
  /**
   *  Returns the ISO-3166 countrycode of this bank.
   *
   * In %HBCI, "280" is still used for Germany according to the %HBCI
   * specs -- even though in the year 1990 the code was changed from
   * 280 to 276 in ISO-3166.
   */
  virtual int countryCode() const =0;

  /**
   *  Set the ISO-3166 countrycode of this bank.
   *
   * (In %HBCI: 280 for Germany) */
  virtual void setCountryCode(int i)=0;

  /**  Returns the bank code of this bank.
   *
   * ("Bankleitzahl", BLZ) */
  virtual const string &bankCode() const =0;

  /**  Set the bank code of this bank.
   *
   * ("Bankleitzahl", BLZ) */
  virtual void setBankCode(const string &s)=0;
  /*@}*/

  /** @name Bank Information
   *
   * Methods that return information that OpenHBCI learned about this bank.
   */
  /*@{*/
  /**  Returns the name of this Bank.
   *
   * This name is the one that the bank sends back upon initally
   * connecting. */
  virtual const string &name() const =0;

  /**  Returns a list of all institute messages that we
   * received from the bank so far.
   *
   * This is a text message like an email. Your bank might send such
   * a message containing usefull information concerning you account.
   * They even might contain advertisements. That only depends on your
   * bank.
   *
   * @author Martin Preuss <martin@libchipcard.de> */
  virtual const list<instituteMessage> &messages() const =0;

  /**
   * Remove a given institute message
   */
  virtual void deleteInstituteMessage(const instituteMessage &msg)=0;
  /*@}*/

  /** @name Object retrieval
   *
   * Methods that return the objects that are owned by this bank.
   */
  /*@{*/
  /**
   *  Returns the list of accounts at this bank.
   * @author Martin Preuss <martin@libchipcard.de>
   */
  virtual const list<Pointer<Account> > &accounts() const =0;

  /**
   *  Returns the list of users of this bank.
   * @author Martin Preuss <martin@libchipcard.de>
   */
  virtual const list<Pointer<User> > &users() const =0;

  /**
   *  Searches for a user in the bank's internal list.
   *
   * No wildcards or jokers allowed.
   * @param userid userid to look for
   * @author Martin Preuss <martin@libchipcard.de>
   */
  virtual Pointer<User>
    findUser(const string &userid) const =0;

  /**
   *  Searches for a customer in the bank's internal list of users.
   *
   * No wildcards or jokers allowed.
   * @param id customer id to look for
   * @author Martin Preuss <martin@libchipcard.de>
   */
  virtual Pointer<Customer>
    findCustomer(const string &id) const =0;

  /**
   *  Searches for an account in the banks internal list.
   *
   * No wildcards or jokers allowed.
   * @param nr account number to match
   * @param subid account subid (may be omitted)
   * @author Martin Preuss <martin@libchipcard.de>
   */
  virtual Pointer<Account>
    findAccount(const string &nr,
		const string &subid="") const =0;
  /*@}*/

  /** @name Manage Objects
   *
   * Methods that manage the lists of objects owned by this bank.
   */
  /*@{*/
  /**
   *  Add an account to the account list of this bank.
   *
   * Ideally, you wouldn't need this as the bank is supposed to send
   * you a list of all accounts valid for your user ID. However,
   * some banks simply don't do that, so that's why this method is
   * here.  If there already is an entry for the same account a
   * Error will be thrown.
   *
   * @author Martin Preuss <martin@libchipcard.de> */
  virtual void addAccount(Pointer<Account> a)=0;

  /**
   *  Removes an account from the internal list.
   *
   * FIXME: This method may not totally clean up all references to
   * that account. This still has to be implemented one day.
   *
   * The object is not destroyed (since Pointers have automatic
   * reference-counting). After this method call, the object is in
   * the same state as one returned by
   * HBCIAPI::accountFactory(). I.e. since it doesn't belong to any
   * internal lists it is not allowed to be used in subsequent calls
   * to HBCIAPI etc.
   *
   * @author Martin Preuss <martin@libchipcard.de>
   */
  virtual void removeAccount(Pointer<Account> a)=0;

  /**
   *  Adds a customer to the list of customers of this bank.
   *
   * If there already is an entry for this customer a Error will
   * be thrown.
   * @author Martin Preuss <martin@libchipcard.de>
   */
  virtual void addUser(Pointer<User> c)=0;

  /**
   *  Removes a customer from the internal list.
   *
   * FIXME: This method may not totally clean up all references to
   * that customer. This still has to be implemented one day.
   *
   * The object is not destroyed (since Pointers have automatic
   * reference-counting). After this method call, the object is in
   * the same state as one returned by
   * HBCIAPI::customerFactory(). I.e. since it doesn't belong to any
   * internal lists it is not allowed to be used in subsequent calls
   * to HBCIAPI etc.
   *
   * @author Martin Preuss <martin@libchipcard.de> */
  virtual void removeUser(Pointer<User> c)=0;
  /*@}*/

  /** @name Communication Parameters
   *
   * Methods that get and set the network communication parameters of
   * this bank (i.e. of the internet server of this bank).  */
  /*@{*/
  /**
   *  Returns the communication address.
   *
   * Depends on type of protocol stack:
   * If T-Online, addr is the address of the Gateway as a number.
   * If TCP/IP, addr is the IP address of the %HBCI server in
   * dotted notation, e.g. "123.123.123.123".
   */
  virtual const string &addr() const =0;
  /**
   *  Set the communication address.
   *
   * Depends on type of protocol
   * stack: For TCP/IP, addr is the IP address of the %HBCI server in
   * dotted notation, e.g. "123.123.123.123".
   */
  virtual void setAddr(const string &s)=0;

  /**
   * Returns the supported type of protocolstack:
   * 1 == T-Online (ETSI 300 072); 2 == TCP/IP
   */
  virtual int type() const =0;
  /**
   * Set the type of protocolstack for this bank:
   * 1 == T-Online (ETSI 300 072); 2 == TCP/IP
   */
  virtual void setType(int i)=0;

  /**
   * Returns the suffix of the communication address.
   * Depends on type of protocol stack:
   * If T-Online, suffix is the region code of the service.
   * If TCP/IP, this is ignored.
   */
  virtual const string &suffix() const =0;
  /**
   * Set the suffix of the communication address. For TCP/IP, this
   * is ignored.
   */
  virtual void setSuffix(const string &s)=0;
  /*@}*/

  /** @name Dialog Language
   *
   * Methods to get and set the dialog language to be used with this
   * bank.  */
  /*@{*/
  /**   Returns the language that is currently used in the
   * dialog with this bank.
   *
   * Either this was set by setLanguage() or this is the bank's
   * default language.  */
  virtual int language() const =0;
  /**   Set the language that should be used in the dialog
   * with this bank.  */
  virtual void setLanguage(int l)=0;

  /**  Returns a list of languages supported by this bank. */
  virtual const list<int> &languages() const =0;
  /*@}*/

  /** @name HBCI Version
   *
   * Methods to get and set the %HBCI Version to be used with this
   * bank.  */
  /*@{*/
  /**
   *  Returns the %HBCI protocol version used with this bank.
   *
   * This %HBCI version number is one out of the list<int> returned
   * by supportedVersions(). This has been set by setHbciVersion,
   * or, if it hasn't been set, it is some default value.  */
  virtual int hbciVersion() const =0;

  /**
   *  Set the %HBCI protocol version to be used with this bank.
   */
  virtual void setHbciVersion(int v)=0;

  /**
   * Return the version of the "Bank Parameter Daten" (BPD).
   *
   * The BPD contains information about the institute, such as address of
   * the server, country code, bank code etc. If these data change on the
   * server side then the server will increment this number. Upon next
   * dialog initialisation this library checks the versions and receives
   * the new BPD if necessary.
   */
  virtual int bpdVersion() const = 0;

  /**
   * Set the version of the "Bank Parameter Daten" (BPD).
   *
   * Set it to "0" if you change the Bank's HBCI-Version because you need
   * the up-to-date-BPD. They are automatically returned if BPD-Version=0
   * in the Init-Dialog.
   */
  virtual void setBPDVersion(int i) = 0;

  /**
   *  Returns a list of %HBCI protocol versions that this bank supports.
   */
  virtual list<int> supportedVersions() const =0;

  /**
   * Searches for bank parameter data about a HBCI job which matches the
   * given criteria.
   *
   * @param segname The segment name of the HBCI job (e.g. HKSAL...)
   * @param minv The minimum segment version to look for
   * @param maxv The maximum segment version to look for
   */
  virtual const bpdJob *findJob(const string &segname,
				int minv=0, int maxv=9999) const =0;
  /*@}*/
};
} /* namespace HBCI */
#endif /* __cplusplus */

#ifdef __cplusplus
extern "C" {
#endif

  /** @defgroup HBCI_Bankg HBCI_Bank functions
   *
   * @short These are the functions for the C accessible type of
   * HBCI::Bank.*/
  /*@{*/

  /** @name Casts */
  /*@{*/
  /**  Returns the central %HBCI object that controls the user
   * interaction and version control.
   * @param h The object to perform this action on.
   */
  extern const HBCI_Hbci *HBCI_Bank_hbci(const HBCI_Bank *h);
  /*@}*/

  /** @name Bank identification */
  /*@{*/
  /**
   *  Returns the ISO-3166 countrycode of this bank.
   *
   * In %HBCI, "280" is still used for Germany according to the %HBCI
   * specs -- even though in the year 1990 the code was changed from
   * 280 to 276 in ISO-3166.
   * @param h The object to perform this action on.
   */
  extern int HBCI_Bank_countryCode(const HBCI_Bank *h);
  /**
   *  Set the ISO-3166 countrycode of this bank.
   *
   * (In %HBCI: 280 for Germany)
   * @param h The object to perform this action on.
   * @param i New ISO-3166 country code.
   */
  extern void HBCI_Bank_setCountryCode(HBCI_Bank *h, int i);

  /**  Returns the bank code of this bank.
   *
   * ("Bankleitzahl", BLZ)
   * @param h The object to perform this action on.
   */
  extern const char *HBCI_Bank_bankCode(const HBCI_Bank *h);
  /**  Set the bank code of this bank.
   *
   * ("Bankleitzahl", BLZ)
   * @param h The object to perform this action on.
   * @param s New bank code string.
   */
  extern void HBCI_Bank_setBankCode(HBCI_Bank *h, const char *s);
  /*@}*/

  /** @name Bank Information */
  /*@{*/
  /**  Returns the name of this Bank.
   *
   * This name is the one that the bank sends back upon initally
   * connecting.
   * @param h The object to perform this action on.
   */
  extern const char *HBCI_Bank_name(const HBCI_Bank *h);

  /**  Returns a list of all institute messages that we
   * received from the bank so far.
   *
   * This is a text message like an email. Your bank might send such
   * a message containing usefull information concerning you account.
   * They even might contain advertisements. That only depends on your
   * bank.
   */
  extern const list_HBCI_instituteMessage *
    HBCI_Bank_messages(const HBCI_Bank *h);
  /*@}*/

  /** @name Object retrieval */
  /*@{*/
  /**
   *  Returns the list of accounts at this bank.
   * @param h The object to perform this action on.
   */
  extern const list_HBCI_Account *HBCI_Bank_accounts(const HBCI_Bank *h);
  /**
   *  Returns the list of users of this bank.
   * @param h The object to perform this action on.
   */
  extern const list_HBCI_User *HBCI_Bank_users(const HBCI_Bank *h);
  /**
   *  Searches for a user in the bank's internal list.
   *
   * No wildcards or jokers allowed.
   * @param h The object to perform this action on.
   * @param userid userid to look for
   */
  extern const HBCI_User *HBCI_Bank_findUser(const HBCI_Bank *h,
					     const char *userid);
  /**
   *  Searches for an account in the banks internal list.
   *
   * No wildcards or jokers allowed.
   * @param h The object to perform this action on.
   * @param nr account number to match
   */
  extern const HBCI_Account *HBCI_Bank_findAccount(const HBCI_Bank *h,
						   const char *nr);
  /**
   *  Searches for an account in the banks internal list.
   *
   * This method searches for accounts with account code suffix (new
   * in %HBCI 2.2).
   *
   * No wildcards or jokers allowed.
   * @param h The object to perform this action on.
   * @param nr account number to match
   * @param suf account suffix */
  extern const HBCI_Account *HBCI_Bank_findAccountWSuf(const HBCI_Bank *h,
						       const char *nr,
						       const char *suf);
  /*@}*/

  /** @name Manage Objects*/
  /*@{*/
  /**
   *  Add an account to the account list of this bank.
   *
   * Ideally, you wouldn't need this as the bank is supposed to send
   * you a list of all accounts valid for your user ID. However,
   * some banks simply don't do that, so that's why this method is
   * here.  If there already is an entry for the same account a
   * Error will be returned.
   *
   * @param h The object to perform this action on.
   * @param a Account to add.
   * @param autoDelete If TRUE, then the HBCI_Bank takes ownership of
   * the HBCI_Account object and will delete it if the life time is
   * over. If FALSE, the HBCI_Bank will <i>not</i> delete the
   * HBCI_Account object; instead, the caller is still responsible for
   * its deletion.
   * @return NULL if successful, or pointer to HBCI_Error upon failure.
   */
  extern HBCI_Error *HBCI_Bank_addAccount(HBCI_Bank *h,
					  HBCI_Account *a,
					  int autoDelete);
  /**
   *  Adds a customer to the list of customers of this bank.
   *
   * If there already is an entry for this customer a Error will
   * be returned.
   *
   * @param h The object to perform this action on.
   * @param c User to add.
   * @param autoDelete If TRUE, then the HBCI_Bank takes ownership of
   * the HBCI_User object and will delete it if the life time is
   * over. If FALSE, the HBCI_Bank will <i>not</i> delete the
   * HBCI_User object; instead, the caller is still responsible for
   * its deletion.
   * @return NULL if successful, or pointer to HBCI_Error upon failure.
   */
  extern HBCI_Error *HBCI_Bank_addUser(HBCI_Bank *h,
				       HBCI_User *c,
				       int autoDelete);
  /*@}*/

  /** @name Communication Parameters */
  /*@{*/
  /**
   *  Returns the communication address.
   *
   * Depends on type of protocol stack:
   * If T-Online, addr is the address of the Gateway as a number.
   * If TCP/IP, addr is the IP address of the %HBCI server in
   * dotted notation, e.g. "123.123.123.123".
   * @param h The object to perform this action on.
   */
  extern const char *HBCI_Bank_addr(const HBCI_Bank *h);
  /**
   *  Set the communication address.
   *
   * Depends on type of protocol
   * stack: For TCP/IP, addr is the IP address of the %HBCI server in
   * dotted notation, e.g. "123.123.123.123".
   * @param h The object to perform this action on.
   * @param s New communication address.
   */
  extern void HBCI_Bank_setAddr(HBCI_Bank *h, const char *s);
  /**
   * Returns the supported type of protocolstack:
   * 1 == T-Online (ETSI 300 072); 2 == TCP/IP
   * @param h The object to perform this action on.
   */
  extern int HBCI_Bank_type(const HBCI_Bank *h);
  /**
   * Set the type of protocolstack for this bank:
   * 1 == T-Online (ETSI 300 072); 2 == TCP/IP
   * @param h The object to perform this action on.
   * @param i New type of protocolstack.
   */
  extern void HBCI_Bank_setType(HBCI_Bank *h, int i);
  /*@}*/

  /** @name HBCI Version */
  /*@{*/
  /**
   *  Returns the %HBCI protocol version used with this bank.
   *
   * This %HBCI version number is one out of the list<int> returned
   * by supportedVersions(). This has been set by setHbciVersion,
   * or, if it hasn't been set, it is some default value.  */
  extern int HBCI_Bank_hbciVersion(const HBCI_Bank *h);
  /**
   *  Set the %HBCI protocol version to be used with this bank.
   */
  extern void HBCI_Bank_setHbciVersion(HBCI_Bank *h, int v);

  /**
   * Return the version of the "Bank Parameter Daten" (BPD).
   *
   * The BPD contains information about the institute, such as address of
   * the server, country code, bank code etc. If these data change on the
   * server side then the server will increment this number. Upon next
   * dialog initialisation this library checks the versions and receives
   * the new BPD if necessary.
   */
  extern int HBCI_Bank_bpdVersion(const HBCI_Bank *h);
  /**
   * Set the version of the "Bank Parameter Daten" (BPD).
   *
   * Set it to "0" if you change the Bank's HBCI-Version because you need
   * the up-to-date-BPD. They are automatically returned if BPD-Version=0
   * in the Init-Dialog.
   */
  extern void HBCI_Bank_setBPDVersion(HBCI_Bank *h, int i);
  /**
   *  Returns a list of %HBCI protocol versions that this bank supports.
   */
  extern list_int *HBCI_Bank_supportedVersions(const HBCI_Bank *h);

  /**
   * Searches for bank parameter data about a HBCI job which matches the
   * given criteria.
   *
   * @param h The object to perform this action on.
   * @param segname The segment name of the HBCI job (e.g. HKSAL...)
   */
  extern const HBCI_bpdJob *HBCI_Bank_findJob(const HBCI_Bank *h,
					      const char *segname);
  /**
   * Searches for bank parameter data about a HBCI job which matches the
   * given criteria.
   *
   * @param h The object to perform this action on.
   * @param segname The segment name of the HBCI job (e.g. HKSAL...)
   * @param minv The minimum segment version to look for
   */
  extern const HBCI_bpdJob *HBCI_Bank_findJob_minv(const HBCI_Bank *h,
						   const char *segname,
						   int minv);
  /**
   * Searches for bank parameter data about a HBCI job which matches the
   * given criteria.
   *
   * @param h The object to perform this action on.
   * @param segname The segment name of the HBCI job (e.g. HKSAL...)
   * @param minv The minimum segment version to look for
   * @param maxv The maximum segment version to look for
   */
  extern const HBCI_bpdJob *HBCI_Bank_findJob_maxv(const HBCI_Bank *h,
						   const char *segname,
						   int minv, int maxv);
  /*@}*/

  /** @name C wrapper for list of Banks
   *
   * C wrapper for STL's list<Bank> type. To browse through a list,
   * get an iterator by list_HBCI_Bank_begin, increment it by
   * list_HBCI_Bank_iter_next, access its elements by
   * list_HBCI_Bank_iter_get, and see whether you're at the end of
   * the list by comparing with list_HBCI_Bank_iter_equal against
   * list_HBCI_Bank_end. */
  /*@{*/
  extern void list_HBCI_Bank_delete(list_HBCI_Bank *l);
  extern void list_HBCI_Bank_iter_delete(list_HBCI_Bank_iter *l);
  extern int list_HBCI_Bank_size(const list_HBCI_Bank *l);
  extern list_HBCI_Bank_iter* list_HBCI_Bank_begin(const list_HBCI_Bank *l);
  extern list_HBCI_Bank_iter* list_HBCI_Bank_end(const list_HBCI_Bank *l);
  extern const HBCI_Bank*
    list_HBCI_Bank_iter_get(const list_HBCI_Bank_iter *l);
  extern void list_HBCI_Bank_iter_next(list_HBCI_Bank_iter *l);
  extern int list_HBCI_Bank_iter_equal(const list_HBCI_Bank_iter *l1,
				       const list_HBCI_Bank_iter *l2);
  /** Callback function for @ref list_HBCI_Bank_foreach. */
  typedef void *(*list_HBCI_Bank_cb) (const HBCI_Bank *bank,
				      void *user_data);
  /** Traverses the list, calling the callback function 'func' on
   * each list element.  Traversal will stop when 'func' returns a
   * non-NULL value, and the routine will return with that
   * value. Otherwise the routine will return NULL.
   *
   * @param list The list to traverse.
   * @param func The function to be called with each list element.
   * @param user_data A pointer passed on to the function 'func'.
   * @return The non-NULL pointer returned by 'func' as soon as it
   * returns one. Otherwise (i.e. 'func' always returns NULL)
   * returns NULL.
   * @author Christian Stimming <stimming@tuhh.de> */
  extern void *list_HBCI_Bank_foreach (const list_HBCI_Bank *list,
				       list_HBCI_Bank_cb func,
				       void *user_data);
  /*@}*/
  /*@}*/


  /** @name HBCI_instituteMessage wrappers */
  /*@{*/
  /** Returns the Date of arrival. */
  extern const HBCI_Date *
    HBCI_instituteMessage_date(const HBCI_instituteMessage *m);

  /** Returns the Time of arrival. */
  extern const HBCI_Time *
    HBCI_instituteMessage_time(const HBCI_instituteMessage *m);

  /** Returns the subject of this message. */
  extern const char *
    HBCI_instituteMessage_subject(const HBCI_instituteMessage *m);

  /** Returns the text of the message, just as it was sent by the bank. */
  extern const char *
    HBCI_instituteMessage_text(const HBCI_instituteMessage *m);

  /** Returns the ISO-3166 country code of the bank that sent this
   message (to identify it) */
  extern int HBCI_instituteMessage_country(const HBCI_instituteMessage *m);

  /** Returns the Bank code ("BLZ") of the bank that sent this
   message (to identify it) */
  extern const char *
    HBCI_instituteMessage_bankCode(const HBCI_instituteMessage *m);

  /** Helper flag: Returns TRUE if message has been read. However,
   this is neither set nor reset by OpenHBCI. The state of the
   flag is stored (FIXME: is this correct), but the application
   has to manage the flag by itself. */
  extern int HBCI_instituteMessage_read(const HBCI_instituteMessage *m);
  /** Set the flag that shows whether this message has been
   read. See @ref HBCI_instituteMessage_read(). */
  extern void HBCI_instituteMessage_setRead(HBCI_instituteMessage *m, int b);
  /*@}*/


  /** @name C wrapper for list of HBCI_instituteMessage
   *
   * C wrapper for STL's list<instituteMessage> type. To browse
   * through the list, use list_HBCI_instituteMessage_foreach. Since
   * that list never has to be deleted from the application, we
   * don't provide destructors etc. */
  /*@{*/
  /** Callback function for list_HBCI_instituteMessage_foreach. */
  typedef void *(*list_HBCI_instituteMessage_cb)
    (HBCI_instituteMessage *msg,
     void *user_data);
  /** Traverses the list, calling the callback function 'func' on
   * each list element.  Traversal will stop when 'func' returns a
   * non-NULL value, and the routine will return with that
   * value. Otherwise the routine will return NULL.
   *
   * @param list The list to traverse.
   * @param func The function to be called with each list element.
   * @param user_data A pointer passed on to the function 'func'.
   * @return The non-NULL pointer returned by 'func' as soon as it
   * returns one. Otherwise (i.e. 'func' always returns NULL)
   * returns NULL.
   * @author Christian Stimming <stimming@tuhh.de> */
  extern void *
    list_HBCI_instituteMessage_foreach (const list_HBCI_instituteMessage *list,
					list_HBCI_instituteMessage_cb func,
					void *user_data);
  /*@}*/

#ifdef __cplusplus
}
#endif

#endif /* HBCIBANK_H */
