// Maria structure component reference expression class -*- c++ -*-

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include "StructComponent.h"
#include "StructType.h"
#include "StructValue.h"
#include "Printer.h"

/** @file StructComponent.C
 * Structure component reference
 */

/* Copyright  1999-2002 Marko Mkel (msmakela@tcs.hut.fi).

   This file is part of MARIA, a reachability analyzer and model checker
   for high-level Petri nets.

   MARIA is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   MARIA is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

StructComponent::StructComponent (class Expression& structure,
				  card_t i) :
  myStructure (&structure), myIndex (i)
{
  assert (myStructure && myStructure->getType ()->getKind () == Type::tStruct);
  const class StructType* type =
    static_cast<const class StructType*>(myStructure->getType ());
  assert (myStructure->isBasic ());
  setType ((*type)[myIndex]);
}

StructComponent::~StructComponent ()
{
  myStructure->destroy ();
}

class Value*
StructComponent::do_eval (const class Valuation& valuation) const
{
  class Value* v = myStructure->eval (valuation);
  if (!v)
    return NULL;

  assert (v->getKind () == Value::vStruct);
  class Value* u = NULL;
  if (const class Value* w = (*static_cast<class StructValue*>(v))[myIndex])
    u = w->copy ();
  delete v;
  return u;
}

class Expression*
StructComponent::ground (const class Valuation& valuation,
			 class Transition* transition,
			 bool declare)
{
  class Expression* structure =
    myStructure->ground (valuation, transition, declare);
  if (!structure)
    return NULL;
  assert (valuation.isOK ());
  if (structure == myStructure) {
    structure->destroy ();
    return copy ();
  }
  else
    return static_cast<class Expression*>
      (new class StructComponent (*structure, myIndex))->ground (valuation);
}

class Expression*
StructComponent::substitute (class Substitution& substitution)
{
  class Expression* structure = myStructure->substitute (substitution);
  if (structure == myStructure) {
    structure->destroy ();
    return copy ();
  }
  else
    return (new class StructComponent (*structure, myIndex))->cse ();
}

bool
StructComponent::depends (const class VariableSet& vars,
			  bool complement) const
{
  return myStructure->depends (vars, complement);
}

bool
StructComponent::forVariables (bool (*operation)
			       (const class Expression&,void*),
			       void* data) const
{
  return myStructure->forVariables (operation, data);
}

#ifdef EXPR_COMPILE
# include "CExpression.h"

void
StructComponent::compile (class CExpression& cexpr,
			  unsigned indent,
			  const char* lvalue,
			  const class VariableSet* vars) const
{
  char* rvalue = 0;
  class StringBuffer& out = cexpr.getOut ();
  if (cexpr.getVariable (*myStructure, rvalue))
    myStructure->compile (cexpr, indent, rvalue, vars);
  out.indent (indent);
  out.append (lvalue);
  out.append ("=");
  out.append (rvalue);
  out.append (".s");
  out.append (myIndex);
  out.append (";\n");
  delete[] rvalue;
}

#endif // EXPR_COMPILE

/** Determine whether an expression needs to be enclosed in parentheses
 * @param kind	kind of the expression
 * @return	whether parentheses are necessary
 */
inline static bool
needParentheses (enum Expression::Kind kind)
{
  switch (kind) {
  case Expression::eVariable:
  case Expression::eUndefined:
  case Expression::eStructComponent:
  case Expression::eUnionComponent:
  case Expression::eVectorIndex:
    return false;
  case Expression::eCardinality:
  case Expression::eUnionType:
  case Expression::eUnion:
  case Expression::eVector:
  case Expression::eBinop:
  case Expression::eBooleanBinop:
  case Expression::eNot:
  case Expression::eRelop:
  case Expression::eBuffer:
  case Expression::eBufferRemove:
  case Expression::eBufferWrite:
  case Expression::eBufferIndex:
  case Expression::eSet:
  case Expression::eTemporalBinop:
  case Expression::eTemporalUnop:
  case Expression::eMarking:
  case Expression::eTransitionQualifier:
  case Expression::ePlaceContents:
  case Expression::eSubmarking:
  case Expression::eMapping:
  case Expression::eEmptySet:
  case Expression::eVectorAssign:
  case Expression::eVectorShift:
    assert (false);
  case Expression::eUnop:
  case Expression::eConstant:
  case Expression::eStruct:
  case Expression::eStructAssign:
  case Expression::eIfThenElse:
  case Expression::eBufferUnop:
  case Expression::eTypecast:
    break;
  }

  return true;
}

void
StructComponent::display (const class Printer& printer) const
{
  if (::needParentheses (myStructure->getKind ())) {
    printer.delimiter ('(')++;
    myStructure->display (printer);
    --printer.delimiter (')');
  }
  else
    myStructure->display (printer);

  printer.delimiter ('.');
  printer.print (static_cast<const class StructType*>(myStructure->getType ())
		 ->getComponentName (myIndex));
}
