/*!
  @file           DBMCli_EventListener.cpp
  @author         Bernd Vorsprach - bernd.vorsprach@sap.com
  @brief          Listener for database kernel events

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/

/*
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
 */
#include "SAPDB/RunTime/RTE_ISystem.hpp"
#include "SAPDB/DBM/Cli/DBMCli_DateTime.hpp"
#include "SAPDB/DBM/Cli/DBMCli_EventListener.hpp"

#define DBMCLI_EVENT_SEP_CHAR '/'

/*! @brief Constructor  */
DBMCli_Event::DBMCli_Event
    (DBMCli_Result & oResult )
    : m_sName(""),
      m_oPriority(prio_unknown),
      m_oDate(),
      m_oTime(),
      m_nValue1(0),
      m_nValue2(0),
      m_nCount(0),
      m_sText(""),
      m_sDescription("")
{
  DBMCli_String sLine;
  
  oResult.SetPos(0);

  while (oResult.GetLine(sLine)) {
    SetProperty(sLine);
  } // end while
} // end DBMCli_Event::DBMCli_Event

DBMCli_Event::DBMCli_Event(
    const DBMCli_String& name,
    const SAPDB_Int4 id,
    const Priority prio,
    const DBMCli_String& text,
    const SAPDB_Int4 count,
    const RTE_ISystem::DateTime rteDatetime
    ): m_sName(name),
       m_oPriority(prio),
       m_oDate(DBMCli_Date::Now()),
       m_oTime(DBMCli_Time::Now()),
       m_nValue1(id),
       m_nValue2(0),
       m_nCount(count),
       m_sText(text),
       m_sDescription("")
{
    DBMCli_Date eventDate(rteDatetime.Year, rteDatetime.Month, rteDatetime.Day);
    DBMCli_Time eventTime(rteDatetime.Hour, rteDatetime.Minute, rteDatetime.Second);
    DBMCli_DateTime dbmDateTime(eventDate, eventTime);
    m_sText += "; ";
    m_sText += dbmDateTime.Get(DBMCLI_DT_INT_DATETIME_FMT);
}

DBMCli_Event::DBMCli_Event (
    const DBMCli_String& name,
    const Priority prio,
    const DBMCli_String& text,
    const SAPDB_Int4 count,
    const SAPDB_Int4 value1,
    const SAPDB_Int4 value2)
         : m_sName(name),
           m_oPriority(prio),
           m_oDate(DBMCli_Date::Now()),
           m_oTime(DBMCli_Time::Now()),
           m_nValue1(value1),
           m_nValue2(value2),
           m_nCount(count),
           m_sText(text),
           m_sDescription(""){}


DBMCli_Event::DBMCli_Event (const DBMCli_String& oEvtString )
    : m_sName(""),
      m_oPriority(prio_unknown),
      m_oDate(),
      m_oTime(),
      m_nValue1(0),
      m_nValue2(0),
      m_nCount(0),
      m_sText(""),
      m_sDescription("")
{
    int start(0), len(oEvtString.Find(DBMCLI_EVENT_SEP_CHAR));
    if( len < 0 )
        return;
    m_sName = oEvtString.Mid(start, len);

    start += len+1; len = oEvtString.Find(DBMCLI_EVENT_SEP_CHAR, start) - start;
    if( len < 0 )
        return;
    m_oPriority = (Priority) atoi((const char *)oEvtString.Mid(start, len));

    start += len+1; len = oEvtString.Find(DBMCLI_EVENT_SEP_CHAR, start) - start;
    if( len < 0 )
        return;
    DBMCli_DateTime dateTime(oEvtString.Mid(start, len), DBMCLI_DT_NUM_DATETIME_FMT);
    m_oDate = dateTime.Date();
    m_oTime = dateTime.Time();

    start += len+1; len = oEvtString.Find(DBMCLI_EVENT_SEP_CHAR, start) - start;
    if( len < 0 )
        return;
    m_nValue1 = atoi((const char *)oEvtString.Mid(start, len));

    start += len+1; len = oEvtString.Find(DBMCLI_EVENT_SEP_CHAR, start) - start;
    if( len < 0 )
        return;
    m_nValue2 = atoi((const char *)oEvtString.Mid(start, len));

    start += len+1; len = oEvtString.Find(DBMCLI_EVENT_SEP_CHAR, start) - start;
    if( len < 0 )
        return;
    m_nCount = atoi((const char *)oEvtString.Mid(start, len));

    start += len+1; len = oEvtString.GetLength() - start;
    if( len <= 0 ) {
        // no text in event
        return;
    }
    m_sText = oEvtString.Mid(start, len);
}

    DBMCli_String DBMCli_Event::serialize() const {
    DBMCli_String oEvt;
    
    oEvt += GetName();
    oEvt += DBMCLI_EVENT_SEP_CHAR;

    switch(GetPriority()) {
        case prio_low:
            oEvt += "1";
            break;
        case prio_medium:
            oEvt += "2";
            break;
        case prio_high:
            oEvt += "3";
            break;
        default:
            oEvt += "0";
    }
    
    oEvt += DBMCLI_EVENT_SEP_CHAR;
    DBMCli_DateTime dt(GetDate(),GetTime());
    oEvt += dt.Get(DBMCLI_DT_NUM_DATETIME_FMT);

    char buffer[256];
    oEvt += DBMCLI_EVENT_SEP_CHAR;
    sprintf(buffer,"%d",GetValue1());
    oEvt += buffer;
    
    oEvt += DBMCLI_EVENT_SEP_CHAR;
    sprintf(buffer,"%d",GetValue2());
    oEvt += buffer;

    oEvt += DBMCLI_EVENT_SEP_CHAR;
    sprintf(buffer,"%d",GetCount());
    oEvt += buffer;

    oEvt += DBMCLI_EVENT_SEP_CHAR;
    sprintf(buffer,"%s", (const char*)GetText());
    oEvt += buffer;

    return DBMCli_String(oEvt);
}

SAPDB_Bool DBMCli_Event::isKernelWarning() const {
    static const DBMCli_String myWarning("WARNING");
    return GetName() == myWarning;
}

SAPDB_Bool DBMCli_Event::isKernelError() const {
    static const DBMCli_String myError("ERROR");
    return GetName() == myError;
}


/*! @brief private member  */
void DBMCli_Event::SetProperty
    ( const DBMCli_String & sProperty )
{
  int nAssign = sProperty.Find("=");
  if (nAssign > 0) {
    DBMCli_String sName  = sProperty.Left(nAssign);
    DBMCli_String sValue = sProperty.Mid(nAssign + 1);
    
    sName.Trim();
    sValue.Trim();
    
    if (sName.Compare("NAME")        == 0) {
      m_sName = sValue;
    } else if (sName.Compare("PRIORITY")    == 0) {
      if (sValue.Compare("LOW") == 0) {
        m_oPriority = prio_low;
      } else if (sValue.Compare("MEDIUM") == 0) {
        m_oPriority = prio_medium;
      } else if (sValue.Compare("HIGH") == 0) {
        m_oPriority = prio_high;
      } else  {
        m_oPriority = prio_unknown;
      } // end if
    } else if (sName.Compare("DATE")        == 0) {
      m_oDate.Set(sValue, DT_NUM_DATE_FMT);
    } else if (sName.Compare("TIME")        == 0) {
      m_oTime.Set(sValue, DT_NUM_TIME_FMT);
    } else if (sName.Compare("VALUE1")      == 0) {
      m_nValue1      = atoi(sValue);
    } else if (sName.Compare("VALUE2")      == 0) {
      m_nValue2      = atoi(sValue);
    } else if (sName.Compare("COUNT")       == 0) {
      m_nCount       = atoi(sValue);
    } else if (sName.Compare("TEXT")        == 0) {
      m_sText        = sValue;
    } else if (sName.Compare("DESCRIPTION") == 0) {
      m_sDescription = sValue;
    } // end if
  } // end if

} // end DBMCli_Event::SetProperty

/*! @brief Constructor  */
DBMCli_EventListener::DBMCli_EventListener
    ( const DBMCli_String  & sServer,
      const DBMCli_String  & sDatabase,
      const DBMCli_String  & sUserPwd,
      SAPDBErr_MessageList & oMessageList)
    : DBMCli_Database(sServer, sDatabase, sUserPwd, oMessageList)
{

} // end DBMCli_EventListener::DBMCli_EventListener

/*! @brief public member  */
bool DBMCli_EventListener::Run
     ( )
{
  m_oMessageList.ClearMessageList();

  while (DBMCli_Session::Execute("event_wait", m_oMessageList) ) {
    DBMCli_Result & oResult = DBMCli_Session::GetResult();
    if (!oResult.Check(m_oMessageList)) {
      // generate error and return
      return false;
    } // end if
    DBMCli_Event oEvent(oResult);

    if (!EventProc(oEvent)) {
      return true;
    } // end if

    m_oMessageList.ClearMessageList();
  } // end while

  return m_oMessageList.IsEmpty();
} // end DBMCli_EventListener::Run

/*! @brief public member  */
bool DBMCli_EventListener::EventProc
    ( const DBMCli_Event & oEvent )
{
  return true;
} // end DBMCli_EventListener::EventProc
