#!/usr/bin/perl
# @(#) buildmsg.pl (&gar)
#
# Tool for generation of Mesagefile
#
#    ========== licence begin  GPL
#    Copyright (C) 2004 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end
#

package BuildMsg;

use DBI;
use strict;
use File::Path;
use Getopt::Long;
use Cwd;


#############################
# GLOBALS
#############################
my $verbose = 0;
my $dbh = undef;
my $progname = $0;
$progname=~ s,(.*/)|(.*\\),,;
my %Languages = (); 
my $error_count = 0;

my $all_actions = undef;
my $relative_msgfilepath = "sys/src/SAPDB/SAPDBCommon/Messages";
my $file_prefix = "SDBMsg_";
my $msgfilepath = undef;
my %xml_substitition = (
                          '&' => '&amp;',
                          '<' => '&lt;'
                       );
my $xml_search_string = join "|", (keys %xml_substitition);

#my %c_substitition = (
#                          '%' => '%%',
#                          '\\' => '\\\\'
#                       );
#my $c_search_string = join "|", (keys %c_substitition);                       

my $mac_filename = "XMLMessages.mac";
my @mac_filelist = ();
my %current_date = ();
my $msg_version_number = undef;


#############################
# FUNCTIONS
#############################


######################################################
#
# GetLicenceText
#
# Writes final words into header files and copies them if no parsing error occured.
#
#
sub GetLicenceText {
	my $type = shift;
	dbgdbg("Generate licence text\n", 2);
	my ($year, $mon, $day) = get_date();
	my ($result, $relver);
	$relver = $ENV{'RELVER'};
	if ((substr ($relver, 0, 1) eq 'R') && ($relver ge 'R75')) {
		$result = <<EOF;
        ========== licence begin  GPL
        Copyright (C) $year SAP AG

        This program is free software; you can redistribute it and/or
        modify it under the terms of the GNU General Public License
        as published by the Free Software Foundation; either version 2
        of the License, or (at your option) any later version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program; if not, write to the Free Software
        Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
        ========== licence end
EOF
	}
	else {
		$result = <<EOF;
        ========== licence begin LGPL
        Copyright (C) %s SAP AG

        This library is free software; you can redistribute it and/or
        modify it under the terms of the GNU Lesser General Public
        License as published by the Free Software Foundation; either
        version 2.1 of the License, or (at your option) any later version.

        This library is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
        Lesser General Public License for more details.

        You should have received a copy of the GNU Lesser General Public
        License along with this library; if not, write to the Free Software
        Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
        ========== licence end
EOF
	}
	if ($type =~ /^c$/)
	{
		$result = "/*\n$result\n*/\n";
	}
	elsif ($type =~ /^desc$/)
	{
		$result =~ s/\n(.*)/\n#$1/g;
		$result = "#$result\n";
	}
	return $result;
}

sub get_date
{
	unless (defined $current_date{'year'})
	{
		($current_date{'sec'}, $current_date{'min'}, $current_date{'hour'}, 
		 $current_date{'day'}, $current_date{'month'}, $current_date{'year'})= localtime (time);
		$current_date{'year'} += 1900;	
	}
	return wantarray ? ($current_date{'year'}, $current_date{'month'}, $current_date{'day'}) : (
	          "$current_date{'year'}-" . 
	          ($current_date{'month'} > 9 ? $current_date{'month'} : "0$current_date{'month'}")."-".
	          ($current_date{'day'} > 9 ? $current_date{'day'} : "0$current_date{'day'}"));
}

sub GetXmlIntro
{
	dbgdbg("Generate xml intro\n", 2);
	my $text = "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n".
	          "  <MessageDefinitions>\n".
	          "    <Metadata generated=\"".get_date()."\" releaseGlobal=\"YES\" versionnr=\"".get_version_number()."\"/>\n";
	return $text;
}

sub convert_to_xml
{
	my $text  = shift;
	$text =~ s/($xml_search_string)/$xml_substitition{$1}/g;
	return $text;
}

sub convert_to_c
{
	my $text  = shift;
	$text =~ s/\\/\\\\/g;
	$text =~ s/\"/\\\"/g;
	return $text;
}


sub dbgdbg
{
	my $text = shift;
	my $level = shift;
	my $DBG_PREFIX = $level == 0 ? "" : ( $level == 1 ? "  INFO:" : "  DBG$level:" );
	$level = 0 unless (defined $level);	
	if ($level <= $verbose)
	{
		#my $textlen = length ($text);
		# look for line breaks too
		#if ($textlen > 74)
		#{
		#	my $nexttext = "";
		#	my $lcount = 74;
		#	while ($lcount < $textlen)
		#	{
		#		substr ($text, $lcount, 0) = "\n";
		#		$textlen++;
		#		$lcount += 75;
		#	}
		#}
		$text =~ s/\n(.+)/\n$DBG_PREFIX$1/g;
		print "$DBG_PREFIX$text";	
	}
}

sub connect_qadb
{
	unless ($dbh)
	{
		dbgdbg("connecting to the database ... ");
		$dbh = DBI->connect("DBI:MaxDB:PTS/QADB","MSG_USER","MSG_USER") or die "Can't connect $DBI::err $DBI::errstr\n";
		dbgdbg("successfully\n");
	}
}

# 
# get_component_filename (component)
# 
# 
sub get_component_info 
{
	my $component = shift;
	my %component_info = ();
	
	connect_qadb();
	
	dbgdbg ("look for component info for '$component' in MSG_COMPONENTS\n", 2 );
	my $cmd = "SELECT COMPONENT,DESCRIPTION FROM MSG_COMPONENTS WHERE COMPONENT = '$component'";
	dbgdbg ("Execute: $cmd\n", 3);
	($component_info{'COMPONENT'},$component_info{'DESCRIPTION'}) = $dbh->selectrow_array($cmd);
	if ( ! $component_info{'COMPONENT'} )
	{
		if ( $DBI::err )
		{	die "selectrow_array failed $DBI::err $DBI::errstr"; }
		else
		{  
			dbgdbg ("Warning: no component '$component' found\n", 1);
			return undef;
		}
	}
	else
	{
		return \%component_info;
	}	
}

sub get_messagelist 
{
	my $component = shift;
	my $language = shift;
	$language = 'EN' unless (defined $language);
	my @msg_list = ();
	connect_qadb();
	my $cmd = "SELECT MESSAGEID, DEFINENAME, SHORTTEXT, DESCRIPTION ".
	          "FROM V_MSG_MESSAGES WHERE COMPONENT = '$component' and LANGUAGE = '$language' ORDER BY COMPONENT, MESSAGEID";
	dbgdbg ("collect all message infos for '$component' for language '$language'\n", 1);
	dbgdbg ("Execute: $cmd\n", 3);
	my $ref = $dbh->selectall_arrayref($cmd);
	unless ($ref)
	{
		if ( $DBI::err )
		{	die "selectrow_array failed $DBI::err $DBI::errstr"; }
		else
		{  	dbgdbg ("Warning: no messages for component '$component' found\n"); }
	}
	else
	{
		foreach my $row (@$ref) {
			my %message = ();
			($message{'ID'},$message{'DEFINENAME'},$message{'SHORTTEXT'},$message{'DESCRIPTION'}) = @$row;
	    	dbgdbg ("ADD $message{'DEFINENAME'}($message{'ID'}) to msglist\n",2);
	    	dbgdbg ("SHORTTEXT:$message{'SHORTTEXT'}\n", 3);
	    	dbgdbg ("DESCRIPTION:\n$message{'DESCRIPTION'}\n", 3);
	    	push @msg_list, \%message;
		}
	}	
	return \@msg_list;
}


#
# collect all actions in a hash
#
sub collect_all_actions
{
	if (defined $all_actions)
	{
		return 0;
	}
	my $cmd = "SELECT ACTIONID, DESCRIPTION, ACTIONNAME FROM MSG_ACTIONS";
	connect_qadb();
	dbgdbg ("collect all existing actions from action table\n", 1 );
	dbgdbg ("Execute: $cmd\n", 3);
	my $ref = $dbh->selectall_arrayref($cmd);
	unless ($ref)
	{
		if ( $DBI::err )
		{	die "selectrow_array failed $DBI::err $DBI::errstr"; }
		else
		{  	
			dbgdbg ("Warning: no actions in action table found\n"); 
			%$all_actions = ();
		}
	}
	else
	{
		foreach my $row (@$ref) {
	    	my %action = (); 
	    	($action{'ACTIONID'},$action{'DESCRIPTION'},$action{'ACTIONNAME'}) = @$row;
	    	dbgdbg ("ADD ACTION $action{'ACTIONID'}($action{'ACTIONNAME'}) to complete actionlist\n",2);
	    	dbgdbg ("DESCRIPTION:\n$action{'DESCRIPTION'}\n", 3);
	    	$all_actions->{$action{'ACTIONID'}} = \%action;
	    }
	}
}

#
sub get_action_list_for_message
{	
	my $component = shift;
	my $msgid = shift;
	my $language = shift;
	$language = 'EN' unless (defined $language);
	my @action_list = ();
	connect_qadb();
	my $cmd = "SELECT DBRELEASE, ACTIONNAME, DESCRIPTION FROM V_MSG_ACTION ".
	                  "where COMPONENT = '$component' and MESSAGEID = '$msgid' and LANGUAGE = '$language' ".
	                         "and DBRELEASE = '0'";
	                  #                     "ORDER BY DBRELEASE desc";
	dbgdbg ("collect all actions for message $msgid ($component) for language '$language'\n", 1 );
	dbgdbg ("Execute: $cmd\n", 3);
	my $ref = $dbh->selectall_arrayref($cmd);
	unless ($ref)
	{
		if ( $DBI::err )
		{	die "selectrow_array failed $DBI::err $DBI::errstr"; }
	}
	else
	{
		foreach my $row (@$ref) {
	    	my %action = ();
	    	($action{'DBRELEASE'},$action{'ACTIONNAME'},$action{'DESCRIPTION'} ) = @$row;
	    	
	    	dbgdbg ("ADD ACTION $action{'ACTIONNAME'}(>=$action{'DBRELEASE'}) to message\n",2);
	    	dbgdbg ("DESCRIPTION:\n$action{'HANDLE'}->{'DESCRIPTION'}\n", 3);
	    	push @action_list, \%action;
	    }
	}
	return \@action_list;
}



sub get_action_hash_for_component
{	
	my $component = shift;
	my $language = shift;
	$language = 'EN' unless (defined $language);
	my %action_hash = ();
	my $actioncount = 0;
	connect_qadb();
	my $cmd = "SELECT MESSAGEID, DBRELEASE, ACTIONNAME, DESCRIPTION FROM V_MSG_ACTION " . 
	                  "where COMPONENT = '$component' and LANGUAGE = '$language' ".
	                  #"where COMPONENT = '$component' and LANGUAGE = 'DE' ".
	                  "and DBRELEASE = '0'" .
	                  "ORDER BY MESSAGEID desc";
	dbgdbg ("collect all actions for component '$component' for language '$language'\n", 1);
	dbgdbg ("Execute: $cmd\n", 3);
	my $ref = $dbh->selectall_arrayref($cmd);
	unless ($ref)
	{
		if ( $DBI::err )
		{	die "selectrow_array failed $DBI::err $DBI::errstr"; }
	}
	else
	{
		foreach my $row (@$ref) {
	    	my %action = ();
	    	my $messageid = undef;
	    	$actioncount++;
	    	($messageid, $action{'DBRELEASE'},$action{'ACTIONNAME'},$action{'DESCRIPTION'} ) = @$row;
	    	dbgdbg ("ADD ACTION $action{'ACTIONNAME'}(>=$action{'DBRELEASE'}) to message\n",2);
	    	dbgdbg ("DESCRIPTION:\n$action{'DESCRIPTION'}\n", 3);
	    	$action_hash{$messageid} = \%action;
	    }
	}
	return $actioncount > 0 ? \%action_hash : undef;
}


sub get_version_number
{	
	unless (defined $msg_version_number)
	{
		$msg_version_number = '12345123';
		dbgdbg ("Start request for new message version number \n", 1);
		connect_qadb();
		dbgdbg ("Execute: select S_MSG_VERSION_NR.NEXTVAL INTO ? FROM DUAL\n", 3);
		my $dbs = $dbh->prepare("select S_MSG_VERSION_NR.NEXTVAL INTO ? FROM DUAL") or die "prepare select NEXTVAL failed $DBI::err $DBI::errstr\n";
		$dbs->bind_param_inout(1, \$msg_version_number, 10) or MaxDBTest::logerror(qq{bind_param_inout failed (undef for column 1, INTEGER) $DBI::err $DBI::errstr});
		$dbs->execute() or die "execute failed $DBI::err $DBI::errstr\n";
		dbgdbg ("Message verion number is $msg_version_number\n", 2);
	}
	return $msg_version_number;	
}


sub create_headerfile
{
	my $refto_component_data = shift;
	my $msglist = $refto_component_data->{'EN'}->{'messages'};
	my $count = scalar (@$msglist);
	my $upperComponent = $refto_component_data->{'COMPONENT'};
	$upperComponent =~ tr/[a-z]/[A-Z]/;
	if ($count > 0)
	{
		dbgdbg ("Generate Header file for $refto_component_data->{'COMPONENT'}\n");
		File::Path::mkpath("$msgfilepath", 0, 0777) if ( ! -d "$msgfilepath" );
		my $headerfile = "$msgfilepath/$file_prefix$refto_component_data->{'COMPONENT'}".".h";
		my $UppercaseHeader = "$file_prefix$refto_component_data->{'COMPONENT'}_H";
		$UppercaseHeader =~ tr/[a-z]/[A-Z]/;
		if ( checkout($headerfile) == 0 )
		{				
			dbgdbg ("headerfile is '$headerfile'\n", 1);
			open( HEADER, ">$headerfile" )
						|| die ( "Cant create header file $headerfile\n" );
			dbgdbg("Generating header intro ...\n", 2);			
			print HEADER GetLicenceText ("c");
			print HEADER "\n#ifndef $UppercaseHeader\n";
			print HEADER "#define $UppercaseHeader\n\n";
			print HEADER "/*\n This headerfile was generated by $progname".
			              "\n--- Do not edit....\n*/\n\n";
			dbgdbg("Insert message defines in header ...\n", 2);
			print HEADER "#ifdef __cplusplus\n";
			print HEADER "#ifndef MSG_LIST_HPP\n";
			print HEADER "#include \"SAPDB/Messages/Msg_List.hpp\"\n";
			print HEADER "#endif\n";
			print HEADER "#ifndef MSG_ARG_HPP\n";
			print HEADER "#include \"SAPDB/Messages/Msg_Arg.hpp\"\n";
			print HEADER "#endif\n";
			print HEADER "#endif\n\n";
			
			dbgdbg("Insert message defines in header ...\n", 2);
			my $componentprefix = "$file_prefix$refto_component_data->{'COMPONENT'}_";
			$componentprefix =~ tr/[a-z]/[A-Z]/;
			print HEADER "#define ${componentprefix}COMPONENT \"$refto_component_data->{'COMPONENT'}\"\n\n";
			
			my $TilosEnumerationString = undef;
			my $TilosInitListString = undef;
			foreach my $message (@$msglist)
			{
				my $headerline = "#define ${componentprefix}$message->{'DEFINENAME'} $message->{'ID'},".
				                 " \"$refto_component_data->{'COMPONENT'}\",".
				                 " __FILE__, __LINE__, \"".convert_to_c($message->{'SHORTTEXT'})."\"";	       
				dbgdbg ("  $headerline\n", 3);                  
				print HEADER "/* The next line was generated by $progname ... Do not edit....*/\n";
				print HEADER "$headerline\n";	       
				if (defined $TilosEnumerationString)
				{
					$TilosEnumerationString .= ",\\\n                    $refto_component_data->{'COMPONENT'}_$message->{'DEFINENAME'}=$message->{'ID'}";  
					$TilosInitListString  .=   ",\\\n                    x(SDBMSG_${upperComponent}_$message->{'DEFINENAME'}, \"$message->{'DEFINENAME'}\")";                  
				}
				else
				{
					$TilosEnumerationString = "#define SDBMSG_${upperComponent}_ENUMERATION\\\n".
					                          "                    $refto_component_data->{'COMPONENT'}_$message->{'DEFINENAME'}=$message->{'ID'}"; 
					$TilosInitListString = 	  "#define SDBMSG_${upperComponent}_INITIALIZATIONLIST(x) \\\n".
									          "                    x(SDBMSG_${upperComponent}_$message->{'DEFINENAME'}, \"$message->{'DEFINENAME'}\")";                  
				}					                          	
			}
			print HEADER "\n/* other macro definitions (created by $progname)*/\n$TilosEnumerationString\n\n$TilosInitListString\n\n";
			
			print HEADER "\n#endif /* $UppercaseHeader */\n";
			close HEADER;
		}
		else
		{
			die "Can't check out header $headerfile\n";
		}
	}
	else
	{
		dbgdbg  ("Warning: no messages for component '$refto_component_data->{'COMPONENT'}'\n");
	}
	return $count;
}

sub create_messagefile
{
	my $refto_component_data = shift;
	my $language = shift;
	
	dbgdbg ("Generate message file for $refto_component_data->{'COMPONENT'} for language '$language'\n");
	File::Path::mkpath("$msgfilepath/$language", 0, 0777) if ( ! -d "$msgfilepath/$language" );
	my $msgfile = "$msgfilepath/$language/$file_prefix$refto_component_data->{'COMPONENT'}".".xml";
	my $upperComponent = $refto_component_data->{'COMPONENT'};
	$upperComponent =~ tr/[a-z]/[A-Z]/;
	
	if ( checkout($msgfile) == 0 )
	{				
		dbgdbg ("messagefile is '$msgfile'\n", 1);
		open( MSGFILE, ">$msgfile" )
					|| die ( "Cant create message file $msgfile\n" );
		dbgdbg("Generating xml intro", 2);			
		print MSGFILE GetXmlIntro();
		dbgdbg("OK\n", 2);
		my $msglist = $refto_component_data->{$language}->{'messages'};
		foreach my $message (@$msglist)
		{
			my $msgline = "    <MSG COMP=\"$refto_component_data->{'COMPONENT'}\" ID=\"$message->{'ID'}\" VERSION=\"1\">\n";
			$msgline .= "    <DefineName>\n      ".
			        "${upperComponent}_$message->{'DEFINENAME'}\n".
			        "    </DefineName>\n";
			$msgline .= "    <ShortText>\n      ".
			            convert_to_xml($message->{'SHORTTEXT'})."\n".
			            "    </ShortText>\n";
			            
			if ($message->{'DESCRIPTION'} =~ /\w/ )
			{            
				$msgline .= "    <Description>\n      ".
				            convert_to_xml($message->{'DESCRIPTION'})."\n".
				            "    </Description>\n";			            
			}
			if ( $refto_component_data->{$language}->{'actions'} && defined $refto_component_data->{$language}->{'actions'}->{$message->{'ID'}})
			{
				$msgline .= "    <Action>\n";
				$msgline .= "        ". convert_to_xml($refto_component_data->{$language}->{'actions'}->{$message->{'ID'}}->{'DESCRIPTION'}) ."\n";	
				$msgline .= "    </Action>\n"
				
				#foreach my $action (@{$refto_component_data->{$language}->{'actions'}->{$message->{'ID'}}})
				#{
			#		$msgline .= "      <Action sinceRelease=\"$action->{'DBRELEASE'}\">\n";
			#		$msgline .= "        ". convert_to_xml($action->{'DESCRIPTION'}) ."\n";
			#		$msgline .= "      </Action>\n";
			#	}	
				#$msgline .= "    </Actions>\n"
			}
			$msgline .= "    </MSG>\n";
			dbgdbg ("MSGLINE:\n$msgline\n", 3);                  
			print MSGFILE "$msgline";	       
		}
		print MSGFILE  "  </MessageDefinitions>\n";
		close MSGFILE;
		push @mac_filelist, "./$language/$file_prefix$refto_component_data->{'COMPONENT'}".".xml";
	}
	else
	{
		die "Can't check out header $msgfile\n";
	}
}

sub create_mac_file
{
	my $macfile = "$msgfilepath/$mac_filename";
	dbgdbg ("Check (and recreate) mac description for new message files\n");
	dbgdbg ("mac description is '$macfile'\n", 1);
	my ($rc, $already_edit_head_rev) = checkout($macfile);
	if ( $rc == 0 )
	{	
		my %existing_msgfiles = ();
		
		# open existing mac file and parse for existing files
		open( MACFILE, "<$macfile" ) || die ( "Cant read mac description $macfile\n" );
		{
			my $line = undef;
			my $newfiles_count = 0;
			while ($line = <MACFILE>) 
			{	last if ($line =~ /was generated by $progname/);	}
			if ($line)
			{
				dbgdbg ("Found following (right) lines in mac description", 3);
				while  ($line = <MACFILE>)
				{
					if ($line =~ /^\.\/(.*)\/(.*\.xml)(.*)$/)
					{
						$existing_msgfiles{"./$1/$2"} = defined $3 ? "$3" : "";
						dbgdbg ("Found - $line", 3);
					}
					else
					{
						# ignored lines
					}
				}
				close MACFILE;
				# check for existence of generated xml files
				dbgdbg ("Look for generated files in mac description", 2);
				foreach (@mac_filelist)
				{	
					unless (defined $existing_msgfiles{$_})
					{
						dbgdbg ("File not found: $_\n", 3);
						$existing_msgfiles{$_} = undef;	
						$newfiles_count++;
					}
				}
			}
			else
			{
				# empty mac description (or manipulated)
				# regeneration necessary
				$newfiles_count = -1;
			}
			if ($newfiles_count == 0)
			{
				dbgdbg ("Found all generated files in mac description - recreate not necessary\n", 1);
				P4Lib::p4_revert($macfile) unless ($already_edit_head_rev);
			}
			else
			{
				dbgdbg ("New files found - mac description must be recreated\n", 1);
				open( MACFILE, ">$macfile" )
						|| die ( "Cant create header file $macfile\n" );
				dbgdbg("Generate intro for mac description\n", 2);			
				dbgdbg ("write intro fo mac description\n", 2);
				print MACFILE "# @(#) $relative_msgfilepath/$mac_filename   - ".get_date()."\n";
				print MACFILE GetLicenceText("desc");
				print MACFILE "#\n#\n#     This description was generated by $progname\n".
			                  "#       !!! do not edit !!!\n\n";
				dbgdbg ("write list of message files to mac description\n", 2);			                  
			    foreach (sort keys %existing_msgfiles)
			    {
			    	my $line = "$_";
			    	if (defined $existing_msgfiles{$_})
			    	{
			    		$line .= $existing_msgfiles{$_};
			    		dbgdbg ("OLD:$line\n", 3);
			    	}
			    	else
			    	{
			    	    $line .= " " x ((length($_) > 45 ) ? 1 : ( 45 - length($_)))."distrib     # added by $progname";
			    	    dbgdbg ("NEW:$line\n", 3);
			    	}
			    	print MACFILE "$line\n";
			    }
			    close MACFILE;
			}
		}		
		return 0;
	}
	else
	{
			die "Can't check out mac description $macfile\n";
	}
}


sub checkout 
{
	my $file = shift;
	my $forced_changelist = shift; 
	require P4Lib;
	import P4Lib;
	my $rc = 0;
	my $current_changelist = 0;
	my $already_edit_head_rev = 0;
	
	my ( $fileDoesNotExist, $fileNotSynced, $fileOverwritable, $fileAdded ) = P4Lib::p4fstat($file);
	if ( $fileDoesNotExist )
	{
		unless ( -f $file)
		{
			open( DUMMY, ">$file" )	|| die ( "Cant create file $file\n" );
			print DUMMY "first generation for add";
			close DUMMY;
		}
		$rc = P4Lib::p4_add($file);
	}
	else
	{
		if ($fileNotSynced)
		{
			if ($fileOverwritable)	
			{	
				$current_changelist = P4Lib::p4_opened($file);	
				$rc = P4Lib::p4_revert($file);	
				$fileOverwritable = 0;
			}
			$rc = P4Lib::p4_sync($file);
		}
		elsif ($fileOverwritable || $fileAdded)
		{
			$already_edit_head_rev = 1;
		}
		unless ($fileOverwritable || $fileAdded)
		{
			my $changelist = (defined $forced_changelist) ? $forced_changelist : $current_changelist;
			$rc = ($changelist > 0) ?  P4Lib::p4_edit_cl($file,$changelist) : P4Lib::p4_edit($file);
		}
	}	
	
	return wantarray () ? ($rc,$already_edit_head_rev) : $rc  ;
}

sub get_all_components
{
	my @comp_list = ();
	my $cmd = "SELECT COMPONENT FROM MSG_COMPONENTS";
	connect_qadb();
	dbgdbg ("Look for existing components in components table\n");
	dbgdbg ("Execute: $cmd\n", 3);
	my $ref = $dbh->selectcol_arrayref($cmd);
	unless ($ref)
	{
		if ( $DBI::err )
		{	die "selectrow_array failed $DBI::err $DBI::errstr"; }
		else
		{  	dbgdbg("Warning: no component entries found\n"); }
	}
	else
	{
		@comp_list =  @$ref;
	}
	dbgdbg ("Found:\n  ". (join "\n  ", @comp_list)."\n", 3 );
	return @comp_list;
}


sub get_all_languages
{
	my %lang_hash = ();
	my $cmd = "SELECT LANGUAGE FROM MSG_LANGUAGES";
	connect_qadb();
	dbgdbg ("look existing languages in languages table\n", 1 );
	dbgdbg ("Execute: $cmd\n", 3);
	my $ref = $dbh->selectcol_arrayref($cmd);
	unless ($ref)
	{
		if ( $DBI::err )
		{	die "selectrow_array failed $DBI::err $DBI::errstr"; }
		else
		{   dbgdbg ("Warning: no component entries found\n"); }
	}
	else
	{
		foreach (@$ref)
		{ $lang_hash{$_} = 1; }
	}
	dbgdbg ("Found:\n  ". (join "\n  ", keys %lang_hash)."\n", 3 );
	return \%lang_hash;
}


sub Usage
{
	print <<USAGE_END;
    
Usage: $progname [-c <component>] [-(a|all_components)] 
                   [-(ho|header_only)] [-(mo|msgfile_only)]
                   [-no_p4] [-(cl|changelist) <changelist>]  
                   [-(fs|file_single) <msgfile|headerfile>] 
                   [-(fa|file_all) <msgfile|headerfile>] 
                   [-(lc|list_components)] [-(s|silent)]
                   [-(v|verbose)] [-(d|debug) <level>]
                   [-(h|help|?)]
                   
                                       
Where:
     -c <component>
         List of components (separated by comma without spaces)
     
     -a 
         create files for all existing components    
          
     -ho or -header_only  *
         Generate header files only             
            
     -mo or -msgfile_only *
         Generate xmlfiles for messages only             
        
     -fs <file> or -file_single <file>
         Only this file will be created or regenerated 
         (component option not necessary)
     
     -fa <file> or -file_all <file>
         All related files (header,xml) will be created or regenerated 
         (component option not necessary)
        
     -no_p4
         Switch perforce handling off
            
     -cl <changelist> or -changelist <changelist>
         Assign the changes to a specific changelist (open on local client)
     
     -lc or -list_components
         List all existing components and stop
     
     -v or -verbose  
         A little bit mor output
         
     -s or silent 
         silent mode: only errors will be print
                
     -h or -help -?
         Show this help          

Note: 
     If neither -mo nor -ho is given buildmsg.pl will generate all files.
        
            
USAGE_END
}

##########################
# MAIN
##########################



my @complist = ();
my %Opts;
if (!GetOptions( \%Opts,
                'a|all',
                'c|component=s',
                'cl|changelist=s',
                'd|debug=i',
                'fa|file_all=s',                
                'fs|file_single=s',
                'h|help|?',
                'ho|header_only',
                #'l|language=s',
                'lc|list_comonents',
                'mo|msgfile_only',
                'no_p4',
                's|silent',
                'v|verbose') || $Opts{'h'})
{
    Usage();
    exit(0);
}

$Opts{'l'} = "EN";

###############################
# check for paranoid options
###############################
my $opt_check = (defined $Opts{'c'} ? 1 : 0)  + 
                (defined $Opts{'a'} ? 1 : 0)  + 
                (defined $Opts{'fs'} ? 1 : 0) + 
                (defined $Opts{'fa'} ? 1 : 0) + 
                (defined $Opts{'lc'} ? 1 : 0);

if ( $opt_check == 0 )
{
	die "Error: option -c, -a, -fs  or -fa is mandatory\n". 
	    "       (For more help call with -h)\n";
}
elsif ( $opt_check > 1 )
{
	die "Error: only -c, -a, -fs or -fa is allowed\n". 
	    "       (For more help call with -h)\n";
}

$verbose = (defined $Opts{'s'} ? -1 : (defined $Opts{'d'} ? $Opts{'d'} : ( defined $Opts{'v'} ? 1 : 0 )));

if (defined $Opts{'c'})
{
	@complist = split ',', $Opts{'c'} ;
}
elsif (defined $Opts{'a'} || defined $Opts{'lc'} )
{
	@complist = get_all_components();
}


if (defined $Opts{'lc'})
{
	dbgdbg ("Existing components:\n");
	foreach (@complist)
	{ 
		dbgdbg ("  $_\n");
	}
	exit 0;
}

# if file name is given 
# -> try to find out component from filename 
if (defined $Opts{'fs'} || defined $Opts{'fa'})
{
	my $filepath = defined $Opts{'fs'} ? $Opts{'fs'} : $Opts{'fa'};
	unless (-f $filepath)
	{
		die "Error: can use option -f". 
		      (defined $Opts{'fs'} ? "s" : "a").
		      " only for existing files\n".
		      "(file $filepath not existis)\n";
	} 
	my $namepos = undef;
	# if not complete path append current path
	unless ($filepath =~ /^(\/|\\|\w:)/)
	{
		my $current_dir = cwd();
		$filepath = "$current_dir/$filepath";
	}
	$filepath =~ s/\\/\//g;
	if ($filepath =~ /^(.*)\/$relative_msgfilepath\/(.*)$/)
	{
		$msgfilepath = "$1/$relative_msgfilepath";
		my $filename = $2;
		my $filetype = undef;
		my $comp_name = undef;
		my $lang_name = undef;
		
		if ($filename =~ /$file_prefix(.*)\.(h|xml)$/)
		{
			# a header file
			$comp_name = $1;
			$filetype  = $2;
			if ($filename =~ /^(.*)\/$file_prefix$comp_name\.xml$/)
			{	
				$lang_name = $1;
				$lang_name =~ tr/[a-z]/[A-Z]/;
			}
		}
		die "Error: unknown file type \n" unless (defined $filetype);
		die "Error: wrong language directory\n" if (($filetype =~ /xml/) && ! defined $lang_name);
		@complist = ($comp_name);
		if (defined $Opts{'fs'})
		{
			if ($filetype =~ /xml/)
			{
				$Opts{'mo'} = 1 ;
				$Languages{$lang_name} = 0 ; 
				delete $Opts{'ho'};
			}
			if ($filetype =~ /h/)
			{
				$Opts{'ho'} = 1 ;
				delete $Opts{'mo'};
				$Languages{'EN'} = 0 ; 
			}			
		}
	}
	else
	{
		die "Error: given file in wrong directory \n". 
		      "(it should be located in $relative_msgfilepath)\n";
	}
}
else
{
	$msgfilepath = "$ENV{'OWN'}/$relative_msgfilepath";
}

unless (defined $Opts{'l'} || scalar (keys %Languages) > 0 )
{
	%Languages = %{get_all_languages()};
}
else
{
	if (defined $Opts{'l'})
	{
		foreach (split ',', $Opts{'l'})
		{
			tr/[a-z]/[A-Z]/;
			$Languages{$_} = 0;
		}
	}
	# check languages
	dbgdbg ("Check given languages (". (join ",", keys %Languages) . ")\n", 2);
	my $check_lang = get_all_languages();
	dbgdbg ("database have following languages:".(join ",", keys %$check_lang) . ")\n", 3);
	foreach (keys %Languages)
	{
		unless (defined $check_lang->{$_})
		{	
			die "Error: unknown language '$_'\n".
			    "       existing languages: ".(join ",", keys %$check_lang)."\n";
		}
		# language is ok 
		$Languages{$_} = 1 ;
	}
}

# and now generate the message files
foreach my $component (@complist)
{
	my $refto_component_data = get_component_info ($component);
	if ($refto_component_data)
	{	
		my %component_data = %$refto_component_data; 
		my $msg_list = get_messagelist ($component, 'EN');
		if (scalar (@$msg_list) > 0)
		{
			$component_data{'EN'}->{'messages'} = $msg_list;
			$component_data{'COMPONENT'} = $component;
						
			if ( (defined $Opts{'mo'} && ! defined $Opts{'ho'}) || create_headerfile(\%component_data) > 0)
			{
				unless ( defined $Opts{'ho'} && ! defined $Opts{'mo'})
				{
					
					# generate messagefile
					foreach my $language (keys %Languages)
					{
						unless (defined $component_data{$language}->{'messages'})
						{
							$component_data{$language}->{'messages'} = get_messagelist ($component, $language);
						}
						unless (defined $component_data{$language}->{'actions'})
						{
							$component_data{$language}->{'actions'} = get_action_hash_for_component ($component, $language);
						}
						if (scalar @{$component_data{$language}->{'messages'}} > 0)
						{
							dbgdbg ("create message file for component $component for language '$language'\n");
							create_messagefile(\%component_data, $language);
						}
						else
						{
							dbgdbg ("Warning: No mesages for component '$component' for language '$language' found\n".
							        "         Skip generation of the message file of this component and language\n" );
						}
					}
				}
			}
		}
		else
		{
			dbgdbg ("Warning: No mesages for component '$component' for header generation found\n".
			        "         Skip generation of all header/message files of this component\n" );
			        
		}
	}
	else
	{  
		dbgdbg ("Error: Unknown component definition '$component'\n");
		dbgdbg ("         (it will be ignored)\n");
		$error_count++;
	}
}

if (scalar @mac_filelist > 0)
{
	$error_count += create_mac_file();
}

exit $error_count;




#
# TESTS and some helpfull functions
sub test_xml_convert
{
	my $testtext="Das ist ein Test um zum Beispiel < und & zu testten.\n". 
	"Alles andre sollte noch normal weiter funktionieren, ausser natrlich & und \n".
	"<\n";
	
	$testtext = convert_to_xml($testtext);
	print "$testtext";
	exit;
}

sub test_c_convert
{
	my $testtext="Das ist ein Test um zum Beispiel \" und % und \\ zu testten.\n". 
	"Alles andere sollte noch normal weiter funktionieren, ausser natrlich % und \\\n";
	
	$testtext = convert_to_c($testtext);
	print "$testtext";
	exit;
}


sub named_connect
{
	my $db_handle = shift;
	my $database = shift;
	my $server   = shift;
	
	unless ($$db_handle)
	{
		dbgdbg("connecting to the database ... ");
		my $dbh = DBI->connect("DBI:MaxDB:$server/$database","MSG_USER","MSG_USER") or die "Can't connect $DBI::err $DBI::errstr\n";
		$$db_handle = $dbh;
		dbgdbg("successfully\n");
	}
}

sub copy_tables
{
	my $src_dbh = undef;
	my $trg_dbh = undef;
	named_connect(\$trg_dbh, "QADB", "PTS" );
	named_connect(\$src_dbh, "QADB2", "pwdf2027.wdf.sap.corp" );
	my $src_cmd =  "select * from MSG_ACTIONS"; 
	dbgdbg ("Execute: $src_cmd\n", 3);
	
	my $ref = $src_dbh->selectall_arrayref($src_cmd);
	unless ($ref)
	{
		if ( $DBI::err )
		{	die "selectrow_array failed $DBI::err $DBI::errstr"; }
	}
	else
	{		
		foreach my $row (@$ref) {
			my $insert_cmd = "INSERT INTO MSG_ACTIONS (ACTIONNAME, CREATEDBY, CREATEDATE) VALUES ('@$row[0]', 132, '2001-09-16 21:03:00.548000')";
			my $ret = $trg_dbh->do($insert_cmd);	
			$insert_cmd = "INSERT INTO MSG_ACTIONS_TEXT VALUES ('@$row[0]', '@$row[1]', '@$row[2]', @$row[3], @$row[4], '@$row[5]')" ;
			$ret = $trg_dbh->do($insert_cmd);	
	    }
	}
	
}



# VIEW definitions
my $view_msg = 'create view V_MSG_MESSAGES as select  MM.COMPONENT AS COMPONENT, MM.MESSAGEID AS MESSAGEID, MM.DEFINENAME AS DEFINENAME,  MT.LANGUAGE AS LANGUAGE, MT.SHORTTEXT as SHORTTEXT, MT.DESCRIPTION as DESCRIPTION from MSG_MESSAGES MM, MSG_TEXT MT where MM.COMPONENT = MT.COMPONENT and MM.MESSAGEID = MT.MESSAGEID'; 
my $view_actions = 'create view V_MSG_ACTION as select  MAA.COMPONENT AS COMPONENT, MAA.MESSAGEID AS MESSAGEID, MAA.DBRELEASE AS DBRELEASE, MAA.ACTIONNAME AS ACTIONNAME,  MAT.LANGUAGE as LANGUAGE,  MAT.DESCRIPTION AS DESCRIPTION,  MAT.STATUS AS STATUS from MSG_ASSIGN_MSG_ACTION MAA, MSG_ACTIONS_TEXT MAT where MAA.ACTIONNAME = MAT.ACTIONNAME';

