/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_ARPA_NAMESER_H
#include <arpa/nameser.h>
#endif
#ifdef HAVE_RESOLV_H
#include <resolv.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#include <assert.h>


#include "udm_common.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_sqldbms.h"
#include "udm_utils.h"
#include "udm_url.h"
#include "udm_sdp.h"
#include "udm_vars.h"
#include "udm_mutex.h"
#include "udm_searchtool.h"
#include "udm_result.h"
#include "udm_log.h"

#include "udm_proto.h"
#include "udm_host.h"
#include "udm_hash.h"
#include "udm_doc.h"
#include "udm_services.h"
#include "udm_xmalloc.h"
#include "udm_searchcache.h"

#define UDM_THREADINFO(A,s,m)	if(A->Conf->ThreadInfo)A->Conf->ThreadInfo(A,s,m)


void *UdmDBInit(void *vdb)
{
  UDM_DB *db=vdb;
  size_t  nbytes=sizeof(UDM_DB);
  
  if(!db)
  {
    db=(UDM_DB*)UdmMalloc(nbytes);
    bzero((void*)db, nbytes);
    db->freeme=1;
  }
  else
  {
    bzero((void*)db, nbytes);
  }
  db->numtables=1;

#if HAVE_SQL
  UdmWordCacheInit(&db->WordCache);
#endif
  
#if (HAVE_ODBC)
  db->hDbc=SQL_NULL_HDBC;
  db->hEnv=SQL_NULL_HENV;
  db->hstmt=SQL_NULL_HSTMT;
#endif
  return db;
}


void UdmDBFree(void *vdb)
{
  UDM_DB  *db=vdb;

  UDM_FREE(db->DBName);
  UDM_FREE(db->where);
  UDM_FREE(db->from);
  
  if (db->searchd) UdmSearchdClose(db);
  
#if HAVE_SQL
  if (db->connected) UdmSQLClose(db);
#endif
  
  UdmVarListFree(&db->Vars);
  if(db->freeme)UDM_FREE(vdb);
  return;
}



__C_LINK int __UDMCALL UdmLimit8(UDM_ENV *Conf,
                                 UDM_UINT8URLIDLIST *L,
                                 const char *field,int type, void *vdb)
{
  UDM_DB  *db=vdb;
  int  rc=UDM_OK;
#ifdef HAVE_SQL
  rc=UdmLimit8SQL(Conf, L, field, type, db);
#endif
  strcpy(Conf->errstr, db->errstr);
  return rc;
}

__C_LINK int __UDMCALL UdmLimit4(UDM_ENV *Conf,
                                 UDM_UINT4URLIDLIST *L,
                                 const char *field, int type, void *vdb)
{
  UDM_DB  *db=vdb;
  int  rc=UDM_OK;
  
#ifdef HAVE_SQL
  rc=UdmLimit4SQL(Conf, L, field, type, db);
#endif
  strcpy(Conf->errstr, db->errstr);
  return rc;
}

/*
int UdmURLData(UDM_ENV *Conf, UDM_URLDATALIST *L, UDM_DB *db)
{
  int  res=UDM_OK;
  
  L->nitems=0;
  
#ifdef HAVE_SQL
  res=UdmURLDataSQL(Conf, L, db);
#endif
  return res;
}
*/

__C_LINK int __UDMCALL UdmClearDatabase(UDM_AGENT *A)
{
  int  res=UDM_ERROR;
  UDM_DB  *db;
  size_t i, dbto =  A->Conf->dbl.nitems;

  for (i = 0; i < dbto; i++)
  {
    db = &A->Conf->dbl.db[i];
#ifdef HAVE_SQL
    res = UdmClearDBSQL(A, db);
    UDM_FREE(db->where);          /* clear db->where for next parameters */
#endif
    if (res != UDM_OK) break;
  }
  if(res!=UDM_OK)
  {
    strcpy(A->Conf->errstr,db->errstr);
  }
  return res;
}



static int DocUpdate(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc)
{
  int        result=UDM_OK;
  const char *c;
  int        status=UdmVarListFindInt(&Doc->Sections,"Status",0);
  urlid_t    origin_id = 0;
  urlid_t    url_id = (urlid_t)UdmVarListFindInt(&Doc->Sections, "ID", 0);
  time_t     next_index_time;
  char       dbuf[64];
  int        use_crosswords, use_newsext; 

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  use_crosswords = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"CrossWords","no"),"yes");
  use_newsext    = !strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"NewsExtensions","no"),"yes");
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
  
  /* First of all check that URL must be delated */
  
  if(Doc->method==UDM_METHOD_DISALLOW)
  {
    UdmLog(Indexer,UDM_LOG_ERROR,"Deleting %s", UdmVarListFindStr(&Doc->Sections, "URL", ""));
    result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_DELETE);
    return result;
  }

  next_index_time=time(NULL)+Doc->Spider.period;
  UdmTime_t2HttpStr(next_index_time,dbuf);
  UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
  
  switch(status){
  
  case 0: /* No HTTP code */
    if (Doc->connp.Host != NULL) Doc->connp.Host->net_errors++;
    UdmLog(Indexer,UDM_LOG_ERROR,"No HTTP response status");
    next_index_time=time(NULL)+Doc->Spider.net_error_delay_time;
    UdmTime_t2HttpStr(next_index_time,dbuf);
    UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
    result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
    return result;
  
  case UDM_HTTP_STATUS_OK:                            /* 200 */
  case UDM_HTTP_STATUS_PARTIAL_OK:                    /* 206 */
    if(!UdmVarListFind(&Doc->Sections,"Content-Type"))
    {
      UdmLog(Indexer,UDM_LOG_ERROR,"No Content-type header");
      next_index_time=time(NULL)+Doc->Spider.net_error_delay_time;
      UdmTime_t2HttpStr(next_index_time,dbuf);
      UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
      UdmVarListReplaceInt(&Doc->Sections,"Status",UDM_HTTP_STATUS_INTERNAL_SERVER_ERROR);
      if (Doc->connp.Host != NULL) Doc->connp.Host->net_errors++;
      result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
      return result;
    }
    else
    {
        if (Doc->connp.Host != NULL) Doc->connp.Host->net_errors = 0;
    }
    break;
  
  case UDM_HTTP_STATUS_MULTIPLE_CHOICES:              /* 300 */
  case UDM_HTTP_STATUS_MOVED_PARMANENTLY:             /* 301 */
  case UDM_HTTP_STATUS_MOVED_TEMPORARILY:             /* 302 */
  case UDM_HTTP_STATUS_SEE_OTHER:                     /* 303 */
  case UDM_HTTP_STATUS_NOT_MODIFIED:                  /* 304 */
    /* FIXME: check that status is changed and remove words if necessary */
    result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
    return result;
    break;
  
  case UDM_HTTP_STATUS_USE_PROXY:                     /* 305 */
  case UDM_HTTP_STATUS_BAD_REQUEST:                   /* 400 */
  case UDM_HTTP_STATUS_UNAUTHORIZED:                  /* 401 */
  case UDM_HTTP_STATUS_PAYMENT_REQUIRED:              /* 402 */
  case UDM_HTTP_STATUS_FORBIDDEN:                     /* 403 */
  case UDM_HTTP_STATUS_NOT_FOUND:                     /* 404 */
  case UDM_HTTP_STATUS_METHOD_NOT_ALLOWED:            /* 405 */
  case UDM_HTTP_STATUS_NOT_ACCEPTABLE:                /* 406 */
  case UDM_HTTP_STATUS_PROXY_AUTHORIZATION_REQUIRED:  /* 407 */
  case UDM_HTTP_STATUS_REQUEST_TIMEOUT:               /* 408 */
  case UDM_HTTP_STATUS_CONFLICT:                      /* 409 */
  case UDM_HTTP_STATUS_GONE:                          /* 410 */
  case UDM_HTTP_STATUS_LENGTH_REQUIRED:               /* 411 */
  case UDM_HTTP_STATUS_PRECONDITION_FAILED:           /* 412 */
  case UDM_HTTP_STATUS_REQUEST_ENTITY_TOO_LARGE:      /* 413 */
  case UDM_HTTP_STATUS_REQUEST_URI_TOO_LONG:          /* 414 */  
  case UDM_HTTP_STATUS_UNSUPPORTED_MEDIA_TYPE:        /* 415 */
  case UDM_HTTP_STATUS_NOT_IMPLEMENTED:               /* 501 */
  case UDM_HTTP_STATUS_BAD_GATEWAY:                   /* 502 */
  case UDM_HTTP_STATUS_NOT_SUPPORTED:                 /* 505 */
  
    /*
      FIXME: remove words from database
      Check last reffering time to remove when
      there are no links to this document anymore
      UdmLog(Indexer,UDM_LOG_EXTRA,"Deleting URL");
    */
    result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
    return result;
  
  case UDM_HTTP_STATUS_INTERNAL_SERVER_ERROR:         /* 500 */
  case UDM_HTTP_STATUS_SERVICE_UNAVAILABLE:           /* 503 */
  case UDM_HTTP_STATUS_GATEWAY_TIMEOUT:               /* 504 */
  
    /* Keep words in database                */
    /* We'll retry later, maybe host is down */
    if (Doc->connp.Host != NULL) Doc->connp.Host->net_errors++;
    next_index_time=time(NULL)+Doc->Spider.net_error_delay_time;
    UdmTime_t2HttpStr(next_index_time,dbuf);
    UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
    result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
    return result;
  
  default: /* Unknown status, retry later */
    UdmLog(Indexer,UDM_LOG_WARN,"HTTP %d We don't yet know how to handle it, skipped",status);
    result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
    return result;
  }
  
  
  if(Doc->method==UDM_METHOD_GET && Doc->Spider.use_clones)
  {
    result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_FINDORIG);
    if(result!=UDM_OK)return result;
    origin_id = (urlid_t)UdmVarListFindInt(&Doc->Sections,"Origin-ID",0);
  }
  
  
  /* Check clones */
  if((origin_id)&&(origin_id!=url_id))
  {
    if (UdmNeedLog(UDM_LOG_EXTRA))
      UdmLog(Indexer, UDM_LOG_EXTRA, "Duplicate Document %s with #%d", 
             UdmVarListFindStr(&Doc->Sections, "URL", ""), origin_id);
    result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_DELWORDS);
    if(use_crosswords)
    {
      if(result == UDM_OK) result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_DELCWORDS);
    }
    if(result == UDM_OK) result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_UPDCLONE);
    return result;
  }
  
  /* Check that document wasn't modified since last indexing */
  if ((UdmVarListFindInt(&Doc->Sections,"crc32", 0) != 0) 
      &&  (UdmVarListFindInt(&Doc->Sections,"crc32old",0)==UdmVarListFindInt(&Doc->Sections,"crc32",0)) 
      &&  (!(Indexer->flags&UDM_FLAG_REINDEX)))
  {
    result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_SUPDATE);
    return result;
  }
  
  /* Compose site_id string and calc it's Hash32 */
  /*
  sprintf(site_id_str,"%s://%s/",Doc->CurURL.schema,Doc->CurURL.hostinfo);
  UdmVarListReplaceInt(&Doc->Sections, "Site-ID", UdmStrHash32(site_id_str)); for what this need ?
  */
  
  /* Copy default languages, if not given by server and not guessed */
  if(!(c=UdmVarListFindStr(&Doc->Sections,"Content-Language",NULL)))
  {
    if((c=UdmVarListFindStr(&Doc->Sections,"DefaultLang",NULL)))
      UdmVarListReplaceStr(&Doc->Sections,"Content-Language",c);
  }
  
  /* For NEWS extension: get rec_id from my */
  /* parent out of db (if I have one...)    */
  if(use_newsext)
  {
    UDM_VAR    *Sec;
    const char  *parent=NULL;
    int    parent_id=0;
    
    if((Sec=UdmVarListFind(&Doc->Sections,"Header.References")) && Sec->val)
    {
      /* References contains all message IDs of my */
      /* predecessors, space separated             */
      /* my direct parent is the last in the list  */
      if((parent = strrchr(Sec->val,' ')))
      {
        /* parent now points to the space */
        /* character skip it              */
        ++parent;
      }
      else
      {
        /* there is only one entry in */
        /* references, so this is my parent */
        parent=Sec->val;
      }  
    }
    
    /* get parent from database */
    if(parent && strlen(parent) && strchr(parent,'@'))
    {
      UDM_DOCUMENT Msg;
      
      UdmDocInit(&Msg);
      UdmVarListReplaceStr(&Msg.Sections,"Header.Message-ID",parent);
      result = UdmURLAction(Indexer, &Msg, UDM_URL_ACTION_FINDBYMSG);
      parent_id = UdmVarListFindInt(&Msg.Sections,"ID",0);
      UdmVarListReplaceInt(&Doc->Sections,"Header.Parent-ID",parent_id);
      UdmDocFree(&Msg);
    }
    
    /* Now register me with my parent  */
    if(parent_id) result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_REGCHILD);
    
    if(result!=UDM_OK)return result;
  }
  
  /* Now store words and crosswords */
  if(UDM_OK != (result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_INSWORDS)))
    return result;
  
  if(use_crosswords)
    if(UDM_OK != (result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_INSCWORDS)))
      return result;
  
  result = UdmURLAction(Indexer, Doc, UDM_URL_ACTION_LUPDATE);
  
  return result;
}


static int UdmDocUpdate(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc)
{
  size_t    maxsize;
  size_t    sec;
  int    flush=0;
  int    rc=UDM_OK;
  UDM_RESULT  *I = &Indexer->Indexed;

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  maxsize = UdmVarListFindInt(&Indexer->Conf->Vars,"DocMemCacheSize",0) * 1024 * 1024;
  UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);

  if (maxsize > 0 && I->memused > 0) UdmLog(Indexer, UDM_LOG_EXTRA, "DocCacheSize: %d/%d", I->memused, maxsize);
  if (Doc)
  {
    I->memused += sizeof(UDM_DOCUMENT);
    /* Aproximation for Words memory usage  */
    I->memused += Doc->Words.nwords * (sizeof(UDM_WORD) + 5);
    /* Aproximation for CrossWords memory usage */
    I->memused += Doc->CrossWords.ncrosswords * (sizeof(UDM_CROSSWORD) + 35);
    /* Aproximation for Sections memory usage */
    for(sec = 0; sec < Doc->Sections.nvars; sec++) {
      I->memused += sizeof(UDM_VAR);
      I->memused += Doc->Sections.Var[sec].maxlen * 3 + 10;
    }
    I->memused += (sizeof(UDM_HREF) + 35) * Doc->Hrefs.nhrefs;
    if (I->memused >= maxsize) flush = 1;
    if (I->num_rows >= 1024) flush = 1;
  } else flush = 1;

  if (flush)
  {
    size_t  docnum;

    if (I->num_rows)
      UdmLog(Indexer, UDM_LOG_EXTRA, "Flush %d document(s)", I->num_rows + ((Doc != NULL) ? 1 : 0));
    
    if (Doc)
    {
      UDM_THREADINFO(Indexer, "Updating", UdmVarListFindStr(&Doc->Sections, "URL", ""));
      if(UDM_OK != (rc = DocUpdate(Indexer, Doc))) return rc;
      UdmDocFree(Doc);
    }
    
    for (docnum = 0; docnum < I->num_rows; docnum++)
    {
      /* Flush all hrefs from cache in-memory    */
      /* cache into database. Note, this must    */
      /* be done before call of  StoreCrossWords */
      /* because we need to know all new URL IDs */
      
      UDM_THREADINFO(Indexer, "Updating", UdmVarListFindStr(&I->Doc[docnum].Sections, "URL", ""));
      if(UDM_OK != (rc = DocUpdate(Indexer, &I->Doc[docnum])))
        return rc;
    }
    
    /* this should be only if we use blob mode, otherwise built-in wan't work
    if(UDM_OK != (rc = UdmResAction(Indexer, &Indexer->Indexed, UDM_RES_ACTION_INSWORDS)))
      return rc;
    */
   
    if (Indexer->Indexed.num_rows) UdmResultFree(&Indexer->Indexed);
  }
  else
  {
    /* Add document into cache */
    I->Doc=(UDM_DOCUMENT*)UdmRealloc(I->Doc, (I->num_rows + 1) * sizeof(UDM_DOCUMENT));
    I->Doc[I->num_rows] = Doc[0];
    I->Doc[I->num_rows].freeme = 0;
    I->num_rows++;
  }
  return rc;
}

__C_LINK int __UDMCALL UdmURLAction(UDM_AGENT *A, UDM_DOCUMENT *D, int cmd) {
  int res=UDM_ERROR, execflag = 0;
  size_t i, dbfrom = 0, dbto;
  UDM_DB  *db;
  int dbnum = (cmd == UDM_URL_ACTION_GET_CACHED_COPY ? UdmVarListFindInt(&D->Sections, "dbnum", 0) : -1);

  if(cmd == UDM_URL_ACTION_FLUSH)
    return UdmDocUpdate(A, D);


#ifdef USE_TRACE
  fprintf(A->TR, "[%d] URLAction: %d\n", A->handle, cmd);
#endif
  
  UDM_GETLOCK(A, UDM_LOCK_CONF);
  dbto =  A->Conf->dbl.nitems;
  if (dbnum < 0 && D != NULL)
  {
    udmhash32_t  url_id=UdmVarListFindInt(&D->Sections,"URL_ID", 0);
    dbfrom = dbto = ((url_id) ? url_id : UdmStrHash32(UdmVarListFindStr(&D->Sections, "URL", ""))) % A->Conf->dbl.nitems;
    dbto++;
  }
  UDM_RELEASELOCK(A, UDM_LOCK_CONF);

  for (i = dbfrom; i < dbto; i++)
  {
    if (dbnum >= 0 && dbnum != i) continue;
    db = &A->Conf->dbl.db[i];

    UDM_GETLOCK(A, UDM_LOCK_DB);
    switch(db->DBDriver)
    {
      case UDM_DB_SEARCHD:
        res = UdmSearchdURLAction(A, D, cmd, db);
        execflag = 1;
        break;
      
#ifdef HAVE_SQL
      default:
        res=UdmURLActionSQL(A,D,cmd,db);
        if (cmd == UDM_URL_ACTION_EXPIRE) UDM_FREE(db->where);  /* clear db->where for next parameters */
        execflag = 1;
        break;
#endif
    }
    
    if (res != UDM_OK && execflag)
    {
      UdmLog (A, UDM_LOG_ERROR, db->errstr);
    }
    UDM_RELEASELOCK(A, UDM_LOCK_DB);
    if (res != UDM_OK) break;
  }
  
  if ((res != UDM_OK) && !execflag)
  {
    UdmLog(A, UDM_LOG_ERROR, "no supported DBAddr specified");
  }
  return res;
}


__C_LINK int __UDMCALL UdmTargets(UDM_AGENT *A)
{
  int  res=UDM_ERROR;
  UDM_DB  *db;
  size_t i, dbfrom = 0, dbto;

  UDM_GETLOCK(A, UDM_LOCK_CONF);
  dbto =  A->Conf->dbl.nitems;
  UdmResultFree(&A->Conf->Targets);
  UDM_RELEASELOCK(A, UDM_LOCK_CONF);

  for (i = dbfrom; i < dbto; i++)
  {
    db = &A->Conf->dbl.db[i];
    UDM_GETLOCK(A, UDM_LOCK_DB);
#ifdef HAVE_SQL
    res = UdmTargetsSQL(A, db);
#endif
    if(res != UDM_OK)
    {
      UdmLog(A, UDM_LOG_ERROR, db->errstr);
    }
    UDM_RELEASELOCK(A, UDM_LOCK_DB);
    if (res != UDM_OK) break;
  }
  return res;
}

__C_LINK int __UDMCALL UdmResAction(UDM_AGENT *A, UDM_RESULT *R, int cmd)
{
  int  res=UDM_ERROR;
  UDM_DB  *db;
  size_t i, dbfrom = 0, dbto;
  
  UDM_GETLOCK(A, UDM_LOCK_CONF);
  dbto =  A->Conf->dbl.nitems;
  UDM_RELEASELOCK(A, UDM_LOCK_CONF);

  for (i = dbfrom; i < dbto; i++)
  {
    db = &A->Conf->dbl.db[i];
    UDM_GETLOCK(A, UDM_LOCK_DB);
#ifdef HAVE_SQL
    res = UdmResActionSQL(A, R, cmd, db, i);
#endif
    if(res != UDM_OK)
    {
      UdmLog(A, UDM_LOG_ERROR, db->errstr);
    }
    UDM_RELEASELOCK(A, UDM_LOCK_DB);
    if (res != UDM_OK) break;
  }
  return res;
}


__C_LINK int __UDMCALL UdmCatAction(UDM_AGENT *A, UDM_CATEGORY *C, int cmd)
{
  UDM_DB  *db;
  int  res=UDM_ERROR;
  size_t i, dbfrom = 0, dbto;

  UDM_GETLOCK(A, UDM_LOCK_CONF);
  dbto =  A->Conf->dbl.nitems;
  UDM_RELEASELOCK(A, UDM_LOCK_CONF);

  for (i = dbfrom; i < dbto; i++)
  {
    db = &A->Conf->dbl.db[i];
    UDM_GETLOCK(A, UDM_LOCK_DB);
    switch(db->DBDriver)
    {
      case UDM_DB_SEARCHD:
        res = UdmSearchdCatAction(A, C, cmd, db);
        break;
#ifdef HAVE_SQL
      default:
        res=UdmCatActionSQL(A,C,cmd,db);
#endif
    }
    if(res != UDM_OK)
    {
      UdmLog(A, UDM_LOG_ERROR, db->errstr);
    }
    UDM_RELEASELOCK(A, UDM_LOCK_DB);
    if (res != UDM_OK) break;
  }
  return res;
}

__C_LINK int __UDMCALL UdmSrvAction(UDM_AGENT *A, UDM_SERVERLIST *S, int cmd)
{
  UDM_DB  *db;
  int  res=UDM_ERROR;
  size_t i, dbfrom = 0, dbto;
  
  UDM_GETLOCK(A, UDM_LOCK_CONF);
  dbto =  A->Conf->dbl.nitems;
  UDM_RELEASELOCK(A, UDM_LOCK_CONF);

  strcpy(A->Conf->errstr, "No appropriate storage support compiled");
  for (i = dbfrom; i < dbto; i++)
  {
    db = &A->Conf->dbl.db[i];

    UDM_GETLOCK(A, UDM_LOCK_DB); 
#ifdef HAVE_SQL
    res = UdmSrvActionSQL(A, S, cmd, db);
#endif
    if(res != UDM_OK){
    UdmLog(A, UDM_LOG_ERROR, db->errstr);
    }
    UDM_RELEASELOCK(A, UDM_LOCK_DB);
    if (res != UDM_OK) break;
  }
  return res;
}

static const int search_cache_size=1000;

int UdmFindWords(UDM_AGENT *A, UDM_RESULT *Res)
{
  const char      *cache_mode  = UdmVarListFindStr(&A->Conf->Vars, "Cache", "no");
  size_t i, dbfrom = 0, dbto =  A->Conf->dbl.nitems;
  int res = UDM_OK;
  size_t    nwrd=0;
  size_t    nwrdX[256];
  size_t          *PerSite[256], *persite, *curpersite;
  size_t          ResultsLimit = UdmVarListFindUnsigned(&A->Conf->Vars, "ResultsLimit", 0);
  UDM_URL_CRD  *wrdX[256];
  UDM_URL_CRD  *wrd = NULL, *curwrd = NULL;
  UDM_URLDATA     *udtX[256];
  UDM_URLDATA     *udt = NULL, *curudt = NULL;

  if( strcasecmp(cache_mode, "yes") || UdmSearchCacheFind(A, Res))
  {
    /* If not found in search cache       */
    /* Let's do actual search now:        */
    /* Get groupped by url_id words array */
  
    for (i = dbfrom; i < dbto; i++)
    {
      UDM_DB *db = &A->Conf->dbl.db[i];
      const char *dbaddr= UdmVarListFindStr(&db->Vars,"DBAddr","<noaddr>");
      
      UdmLog(A, UDM_LOG_DEBUG, "UdmFind for %s", dbaddr);
      Res->CoordList.Coords = NULL;
      Res->CoordList.Data = NULL;
      Res->CoordList.ncoords = 0;
      Res->total_found = 0;
      switch(db->DBDriver)
      {
        case UDM_DB_SEARCHD:
          res = UdmFindWordsSearchd(A, Res, db);
          break;
#ifdef HAVE_SQL
        default:
          res = UdmFindWordsSQL(A, Res, db);
          UdmUserCacheStoreSQL(A, Res, db);
          break;
#endif
      }
      wrdX[i] = Res->CoordList.Coords;
      udtX[i] = Res->CoordList.Data;
      nwrdX[i] = Res->total_found;
      nwrd += Res->total_found;
      if ((PerSite[i] = Res->PerSite) == NULL)
      {
        PerSite[i] = (nwrdX[i]) ? (size_t*)UdmXmalloc(sizeof(size_t) * nwrdX[i]) : NULL;
      }
    }

    if (nwrd > 0)
    {
      curwrd=wrd=(UDM_URL_CRD*)UdmMalloc(sizeof(*wrd)*nwrd);
      curudt=udt=(UDM_URLDATA*)UdmMalloc(sizeof(*udt)*nwrd);
      Res->PerSite = persite = curpersite = (size_t*)UdmMalloc(sizeof(size_t) * nwrd);
      for (i = dbfrom; i < dbto; i++)
      {
        if(wrdX[i])
        {
          size_t j;
          
          /* Set machine number */
          for(j=0;j<nwrdX[i];j++)
          {
            /* 
              We use (256-i) to sort a document from the first database
              before the same document from the second database.
            */
            wrdX[i][j].coord = (wrdX[i][j].coord << 8) + (255 - (i & 255));
          }
          
          memcpy(curwrd,wrdX[i],sizeof(*curwrd)*nwrdX[i]);
          curwrd+=nwrdX[i];
          UDM_FREE(wrdX[i]);
          memcpy(curpersite, PerSite[i], sizeof(size_t) * nwrdX[i]);
          curpersite += nwrdX[i];
          UDM_FREE(PerSite[i]);
          if (udtX[i] != NULL)
          {
            memcpy(curudt, udtX[i], sizeof(*curudt) * nwrdX[i]);
          }
          else
          {
            bzero(curudt, sizeof(*curudt) * nwrdX[i]);
          }
          curudt += nwrdX[i];
          UDM_FREE(udtX[i]);
        }
      }
/*    if (Res->offset) UdmSortSearchWordsByWeight(wrd,nwrd);*/
    }

    Res->total_found = Res->CoordList.ncoords = nwrd;
    Res->CoordList.Coords = wrd;
    Res->CoordList.Data = udt;
    Res->num_rows=Res->CoordList.ncoords;
    
    if (dbto - dbfrom > 1)
    {
      int use_site_id = ((!strcasecmp(UdmVarListFindStr(&A->Conf->Vars, "GroupBySite", "no"), "yes"))
                          && (UdmVarListFindInt(&A->Conf->Vars, "site", 0) == 0));
      if (use_site_id)
      {
        UdmSortSearchWordsBySite(&Res->CoordList, Res->CoordList.ncoords);
        UdmGroupBySite(A, Res);
      }
      UdmSortSearchWordsByPattern(Res, &Res->CoordList, Res->CoordList.ncoords, UdmVarListFindStr(&A->Conf->Vars, "s", "RP"));
      Res->total_found = Res->CoordList.ncoords;
    }

    if (ResultsLimit > 0 && ResultsLimit < Res->total_found)
    {
      Res->total_found = Res->CoordList.ncoords = ResultsLimit;
    }

    if((!strcasecmp(cache_mode,"yes"))&&(search_cache_size>-1))
    {
      fflush(stdout);
      fflush(stderr);
      UdmSearchCacheStore(A, Res);
    }
  }
  return res;
}


UDM_RESULT * __UDMCALL UdmFind(UDM_AGENT *A)
{
  UDM_RESULT  *Res;
  int    res=UDM_OK;
  unsigned long  ticks=UdmStartTimer(), ticks_;
  size_t i, dbfrom = 0, dbto =  A->Conf->dbl.nitems;
  int    page_number = UdmVarListFindInt(&A->Conf->Vars, "np", 0);
  int    page_size   = UdmVarListFindInt(&A->Conf->Vars, "ps", 10);
  char    str[128];
  
  UdmLog(A,UDM_LOG_DEBUG,"Start UdmFind");
  
  Res=UdmResultInit(NULL);
  UdmPrepare(A, Res);
  
  UdmVarListAddStr(&A->Conf->Vars, "orig_m", UdmVarListFindStr(&A->Conf->Vars, "m", "all"));

  if (UDM_OK != (res= UdmFindWords(A, Res)))
      goto ret;

  if (!Res->total_found)
  {
    int suggest= UdmVarListFindInt(&A->Conf->Vars, "Suggest", 0);
    if (suggest)
    {
      if(UDM_OK != (res= UdmResAction(A, Res, UDM_RES_ACTION_SUGGEST)))
        goto ret;
    }
  }
  
  UdmVarListReplaceStr(&A->Conf->Vars, "m", UdmVarListFindStr(&A->Conf->Vars, "orig_m", "all"));
  UdmVarListDel(&A->Conf->Vars, "orig_m");

  Res->first = page_number * page_size;  
  if(Res->first >= Res->total_found) Res->first = (Res->total_found)? (Res->total_found - 1) : 0;
  if((Res->first+page_size)>Res->total_found)
  {
    Res->num_rows=Res->total_found-Res->first;
  }
  else
  {
    Res->num_rows=page_size;
  }
  Res->last=Res->first+Res->num_rows-1;

  /* Allocate an array for documents information */
  if (Res->num_rows > 0) Res->Doc = (UDM_DOCUMENT*)UdmMalloc(sizeof(UDM_DOCUMENT) * (Res->num_rows));
  
  /* Copy url_id and coord to result */
  for(i=0;i<Res->num_rows;i++)
  {
    uint4  score = Res->CoordList.Coords[i + Res->first /* * Res->offset*/].coord;
    UdmDocInit(&Res->Doc[i]);
    UdmVarListReplaceInt(&Res->Doc[i].Sections, "ID", Res->CoordList.Coords[i + Res->first /* * Res->offset*/].url_id);
    udm_snprintf(str, 128, "%.3f%%", ((double)(score >> 8)) / 1000);
    UdmVarListReplaceStr(&Res->Doc[i].Sections, "Score", str);
    UdmVarListReplaceInt(&Res->Doc[i].Sections,"Order",(int)(i + Res->first + 1));
    UdmVarListReplaceInt(&Res->Doc[i].Sections, "dbnum", 255 - (int)(score & 0xFF));
    if (Res->PerSite)
    {
      UdmVarListReplaceUnsigned(&Res->Doc[i].Sections, "PerSite", Res->PerSite[i + Res->first]);
    }
  }
  
  for (i = dbfrom; i < dbto; i++)
  {
    UDM_DB *db= &A->Conf->dbl.db[i];
    switch(db->DBDriver){
    case UDM_DB_SEARCHD:
      res = UdmResAddDocInfoSearchd(A, db, Res, i);
      break;
#ifdef HAVE_SQL
    default:
      res = UdmResAddDocInfoSQL(A, db, Res, i);
      break;
#endif
    }
  }
  
  ticks_=UdmStartTimer();
  UdmLog(A, UDM_LOG_DEBUG, "Start Clones");
  
  if(UdmVarListFindInt(&A->Conf->Vars, "DetectClones", 1))
  {
    size_t num=Res->num_rows;
    for(i=0;i<num;i++){
      UDM_RESULT *Cl = UdmCloneList(A, &Res->Doc[i]);
      if(Cl){
        Res->Doc=(UDM_DOCUMENT*)UdmRealloc(Res->Doc,sizeof(UDM_DOCUMENT)*(Res->num_rows+Cl->num_rows));
        memcpy(&Res->Doc[Res->num_rows],Cl->Doc,sizeof(UDM_DOCUMENT)*Cl->num_rows);
        Res->num_rows+=Cl->num_rows;
        UDM_FREE(Cl->Doc);
        UdmResultFree(Cl);
      }
    }
  }
  ticks_ = UdmStartTimer() - ticks_;
  UdmLog(A, UDM_LOG_DEBUG, "Stop  Clones:\t\t%.2f", (float)ticks_/1000);
  
  ticks_=UdmStartTimer();
  UdmLog(A, UDM_LOG_DEBUG, "Start adding Order");
  
  /* first and last begins from 0, make it begin from 1 */
  Res->first++;
  Res->last++;
  for(i=0;i<Res->num_rows;i++)
  {
    UdmVarListReplaceInt(&Res->Doc[i].Sections,"Order",(int)(Res->first+i));
  }
  ticks_ = UdmStartTimer() - ticks_;
  UdmLog(A, UDM_LOG_DEBUG, "Stop  Order:\t\t\t%.2f", (float)ticks_/1000);
  
  ticks_=UdmStartTimer();
  UdmLog(A, UDM_LOG_DEBUG, "Start UdmConvert");
  UdmConvert(A->Conf, Res, A->Conf->lcs, A->Conf->bcs);
  ticks_ = UdmStartTimer() - ticks_;
  UdmLog(A, UDM_LOG_DEBUG, "Stop  UdmConvert:\t\t%.2f", (float)ticks_/1000);
  
  Res->work_time=ticks=UdmStartTimer()-ticks;
  UdmLog(A,UDM_LOG_DEBUG,"Done  UdmFind %.2f",(float)ticks/1000);
  
  UdmTrack(A, Res);

ret:
  if(res!=UDM_OK)
  {
    UdmResultFree(Res);
    Res=NULL;
  }
  return Res;
}


static int UdmStr2DBMode(const char * str1)
{
  int m = -1;
  if(!strncasecmp(str1,"single",6))m=UDM_DBMODE_SINGLE;
  else if(!strncasecmp(str1,"multi",5))m=UDM_DBMODE_MULTI;
  else if(!strncasecmp(str1,"blob",4))m=UDM_DBMODE_BLOB;
  else if(!strncasecmp(str1,"fly",3))m=UDM_DBMODE_FLY;
  return(m);
}

__C_LINK const char* __UDMCALL UdmDBTypeToStr(int dbtype)
{
  switch(dbtype)
  {
    case UDM_DB_MYSQL:   return "mysql";
    case UDM_DB_PGSQL:   return "pgsql";
    case UDM_DB_IBASE:   return "ibase";
    case UDM_DB_MSSQL:   return "mssql";
    case UDM_DB_ORACLE8: return "oracle";
    case UDM_DB_SQLITE:  return "sqlite";
    case UDM_DB_MIMER:   return "mimer";
    case UDM_DB_VIRT:    return "virtuoso";
    case UDM_DB_ACCESS:  return "access";
    case UDM_DB_DB2:     return "db2";
    case UDM_DB_CACHE:   return "cache";
  }
  return "unknown_dbtype";
}


__C_LINK const char* __UDMCALL UdmDBModeToStr(int dbmode)
{
  switch(dbmode) 
  {
    case UDM_DBMODE_SINGLE:  return "single";
    case UDM_DBMODE_MULTI:   return "multi";
    case UDM_DBMODE_BLOB:    return "blob";
    case UDM_DBMODE_FLY:     return "fly";
  }
  return "unknown_dbmode";
}


static int UdmDBSetParam(UDM_DB *db, char *param)
{
  char *tok, *lt;
  
  for(tok = udm_strtok_r(param, "&",&lt) ; tok ; 
      tok = udm_strtok_r(NULL,"&",&lt))
  {
    char * val;
    if((val=strchr(tok,'=')))
    {
      *val++='\0';
      UdmVarListReplaceStr(&db->Vars, tok, val);
    }
    else
    {
      UdmVarListReplaceStr(&db->Vars, tok, "");
    }
  }
  return UDM_OK;
}


typedef struct udm_sqldb_driver_st
{
  const char *name;
  int DBType;
  int DBDriver;
  int DBSQL_IN;
  int flags;
  UDM_SQLDB_HANDLER *handler;
} UDM_SQLDB_DRIVER;


static UDM_SQLDB_DRIVER SQLDriver[]=
{
#if (HAVE_ORACLE8)
  {
    "oracle8", UDM_DB_ORACLE8, UDM_DB_ORACLE8, 1, 
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_TRUNCATE|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_oracle_handler
  },
  {
    "oracle", UDM_DB_ORACLE8, UDM_DB_ORACLE8, 1, 
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_TRUNCATE|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_oracle_handler
  },
#endif
#if (HAVE_CTLIB)
  {
    "mssql", UDM_DB_MSSQL, UDM_DB_MSSQL, 1,
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_TRUNCATE,
    &udm_sqldb_ctlib_handler
  },
#endif
#if (HAVE_MYSQL)
  { 
    "mysql", UDM_DB_MYSQL, UDM_DB_MYSQL, 1,
    UDM_SQL_HAVE_LIMIT|UDM_SQL_HAVE_GROUPBY,
    &udm_sqldb_mysql_handler
  },
#endif
#if (HAVE_PGSQL)
  {
    "pgsql", UDM_DB_PGSQL, UDM_DB_PGSQL, 1,
    UDM_SQL_HAVE_LIMIT|UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_pgsql_handler,
  },
#endif
#if (HAVE_IBASE)
  {
    "ibase", UDM_DB_IBASE, UDM_DB_IBASE, 0,
    /* 
    while indexing large sites and using the SQL in statement 
    interbase will fail when the items in the in IN statements
    are more then 1500. We'd better have to fix code to avoid 
    big INs instead of hidding DBSQL_IN.
    */
    UDM_SQL_HAVE_GROUPBY,
    &udm_sqldb_ibase_handler,
  },
#endif
#if (HAVE_SQLITE)
  {
    "sqlite",
    UDM_DB_SQLITE, UDM_DB_SQLITE, 1,
    UDM_SQL_HAVE_LIMIT|UDM_SQL_HAVE_GROUPBY,
    &udm_sqldb_sqlite_handler,
  },
#endif
#if (HAVE_ODBC)
  {
    "odbc-solid", UDM_DB_SOLID, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-sapdb", UDM_DB_SAPDB, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-db2", UDM_DB_DB2, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-access", UDM_DB_ACCESS, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-mimer", UDM_DB_MIMER, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-cache", UDM_DB_CACHE, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_odbc_handler,
  },
  { 
    "odbc-virtuoso", UDM_DB_VIRT, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-oracle", UDM_DB_ORACLE8, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_TRUNCATE|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-oracle8", UDM_DB_ORACLE8, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_TRUNCATE|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-mssql", UDM_DB_MSSQL, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_TRUNCATE,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-mysql", UDM_DB_MYSQL, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_LIMIT|UDM_SQL_HAVE_GROUPBY,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-pgsql", UDM_DB_PGSQL, UDM_DB_ODBC, 1,
    UDM_SQL_HAVE_LIMIT|UDM_SQL_HAVE_GROUPBY|UDM_SQL_HAVE_SUBSELECT,
    &udm_sqldb_odbc_handler,
  },
  {
    "odbc-ibase", UDM_DB_IBASE, UDM_DB_ODBC, 0,
    /* 
    while indexing large sites and using the SQL in statement 
    interbase will fail when the items in the in IN statements
    are more then 1500. We'd better have to fix code to avoid 
    big INs instead of hidding DBSQL_IN.
    */
    UDM_SQL_HAVE_GROUPBY,
    &udm_sqldb_odbc_handler,
  },
#endif
  {
    NULL, 0, 0, 0, 0, NULL
  }
};


static UDM_SQLDB_DRIVER *UdmSQLDriverByName(const char *name)
{
  UDM_SQLDB_DRIVER *drv;
  for (drv= SQLDriver; drv->name; drv++)
  {
    if (!strcasecmp(name, drv->name))
      return drv;
    if (!strncasecmp(drv->name, "odbc-", 5) &&
        !strcasecmp(drv->name + 5, name))
      return drv;
  }
  return NULL;
}


int UdmDBSetAddr(UDM_DB *db, const char *dbaddr, int mode)
{
  UDM_URL    addr;
  char     *s;
  const char *v;
  int        rc= UDM_OK;
  
  UdmVarListFree(&db->Vars);
  UDM_FREE(db->DBName);
  UDM_FREE(db->where);
  UDM_FREE(db->from);
  
  UdmVarListReplaceStr(&db->Vars, "DBAddr", dbaddr);
  
  UdmURLInit(&addr);
  
  if((!dbaddr) || UdmURLParse(&addr, dbaddr) || (!addr.schema))
  {
    /* TODO: add better error message here */
    rc= UDM_ERROR;
    goto ret;
  }
  
  if (addr.auth)
  {
    /*
      Unescape user and password to allow URL specific
      characters like '"<>@#? to be used as user or password part.
      
      It's safe to spoil addr.auth here, as we don't
      need it anymore after setting DBUser and DBPass
    */
    
    if ((s= strchr(addr.auth,':')))
    {
      *s++= 0;
      UdmUnescapeCGIQuery(s, s);
      UdmVarListReplaceStr(&db->Vars, "DBPass", s);
    }
    UdmUnescapeCGIQuery(addr.auth, addr.auth);
    UdmVarListReplaceStr(&db->Vars, "DBUser", addr.auth);
  }
  
  UdmVarListReplaceStr(&db->Vars, "DBHost", addr.hostname);
  if (addr.port)
    UdmVarListReplaceInt(&db->Vars, "DBPort", addr.port);
  
  if((s = strchr(UDM_NULL2EMPTY(addr.filename), '?')))
  {
    *s++='\0';
    if (UDM_OK != UdmDBSetParam(db, s))
    {
      rc= UDM_ERROR;
      goto ret;
    }
    UdmVarListReplaceStr(&db->Vars, "filename", s);
  }
  else
  {
    UdmVarListReplaceStr(&db->Vars, "filename", addr.filename);
  }
  
  if(!strcasecmp(addr.schema, "searchd"))
  {
    db->DBType=UDM_DB_SEARCHD;
    db->DBDriver=UDM_DB_SEARCHD;
    if (UDM_OK != UdmSearchdConnect(db))
    {
      rc= UDM_ERROR;
      goto ret;
    }
  }
  else
  {
    UDM_SQLDB_DRIVER *drv= UdmSQLDriverByName(addr.schema);
    if (!drv)
    {
    
      rc= UDM_ERROR;
      goto ret;
    }
    
    db->DBType= drv->DBType;
    db->DBDriver= drv->DBDriver;
    db->DBSQL_IN= drv->DBSQL_IN;
    db->flags= drv->flags;
    db->sql= drv->handler;
  }
  
  if((v= UdmVarListFindStr(&db->Vars,"numtables",NULL)))
  {
    db->numtables= atoi(v);
    if(!db->numtables)
      db->numtables=1;
  }

  if((v= UdmVarListFindStr(&db->Vars,"dbmode",NULL)))
  {
    if ((db->DBMode=UdmStr2DBMode(v)) < 0) 
    return UDM_ERROR;
  }
  
  if((v= UdmVarListFindStr(&db->Vars,"dbmodesearch",NULL)))
  {
    int DBMode;
    if ((DBMode=UdmStr2DBMode(v)) < 0) 
      return UDM_ERROR;
    if (DBMode == UDM_DBMODE_BLOB  &&
        db->DBType != UDM_DB_MYSQL &&
        db->DBType != UDM_DB_MSSQL &&
        db->DBType != UDM_DB_MIMER &&
        db->DBType != UDM_DB_ORACLE8 &&
        db->DBType != UDM_DB_DB2)
      return UDM_ERROR;
  }

  if(db->DBDriver==UDM_DB_IBASE || db->DBDriver==UDM_DB_SQLITE)
  {
    /*
      Ibase is a special case:
      It's database name consists of
      full path and file name        
    */
    db->DBName = (char*)UdmStrdup(UDM_NULL2EMPTY(addr.path));
  }
  else
  {
    /*
      ODBC Data Source Names may contain space and
      other tricky characters, let's unescape them.
    */
    size_t len= strlen(UDM_NULL2EMPTY(addr.path));
    char  *src= (char*)UdmMalloc(len+1);
    src[0]= '\0';
    sscanf(UDM_NULL2EMPTY(addr.path), "/%[^/]s", src);
    db->DBName= (char*)UdmMalloc(len+1);
    UdmUnescapeCGIQuery(db->DBName, src);
    UdmFree(src);
  }
ret:
  UdmURLFree(&addr);
  return rc;
}


__C_LINK int __UDMCALL UdmStatAction(UDM_AGENT *A, UDM_STATLIST *S)
{
  UDM_DB  *db;
  int  res=UDM_ERROR;
  size_t i, dbfrom = 0, dbto =  A->Conf->dbl.nitems;
  
  S->nstats = 0;
  S->Stat = NULL;

  for (i = dbfrom; i < dbto; i++)
  {
    db = &A->Conf->dbl.db[i];
#ifdef HAVE_SQL
    UDM_GETLOCK(A, UDM_LOCK_DB);
    res = UdmStatActionSQL(A, S, db);
    UDM_RELEASELOCK(A, UDM_LOCK_DB);
#endif
    if (res != UDM_OK) break;
  }
  if(res!=UDM_OK)
  {
    strcpy(A->Conf->errstr,db->errstr);
  }
  return res;
}

unsigned int UdmGetCategoryId(UDM_ENV *Conf, char *category)
{
  UDM_DB  *db;
  unsigned int rc = 0;
  size_t i, dbfrom = 0, dbto =  Conf->dbl.nitems;

  for (i = dbfrom; i < dbto; i++)
  {
    db = &Conf->dbl.db[i];
#ifdef HAVE_SQL
    rc = UdmGetCategoryIdSQL(Conf, category, db);
    if (rc != 0) return rc;
#endif
  }
  return rc;
}


int UdmTrack(UDM_AGENT * query, UDM_RESULT *Res)
{
  int rc = UDM_OK;
#ifdef HAVE_SQL
  size_t i, dbfrom = 0, dbto=  query->Conf->dbl.nitems; 
  
  for (i = dbfrom; i < dbto; i++)
  {
    const char *v;
    UDM_DB *db = &query->Conf->dbl.db[i];
    if((v= UdmVarListFindStr(&db->Vars,"trackquery",NULL)))
      rc = UdmTrackSQL(query, Res, db);
  }
#endif
  return rc;
}


UDM_RESULT * UdmCloneList(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc)
{
  size_t i, dbfrom = 0, dbto =  Indexer->Conf->dbl.nitems;
  UDM_DB    *db;
  UDM_RESULT  *Res;
  int    rc = UDM_OK;

  Res = UdmResultInit(NULL);
  
  for (i = dbfrom; i < dbto; i++)
  {
    db = &Indexer->Conf->dbl.db[i];
    switch(db->DBDriver)
    {
      case UDM_DB_SEARCHD:
        rc = UdmCloneListSearchd(Indexer, Doc, Res, db);
        break;
#ifdef HAVE_SQL
     default:
        rc = UdmCloneListSQL(Indexer, Doc, Res, db);
        break;
#endif
    }
    if (rc != UDM_OK) break;
  }
  if (Res->num_rows > 0) return Res;
  UdmResultFree(Res);
  return NULL;
}


int UdmCheckUrlid(UDM_AGENT *Agent, urlid_t id)
{
  size_t i, dbfrom = 0, dbto;
  UDM_DB    *db;
  int    rc = 0;

  UDM_GETLOCK(Agent, UDM_LOCK_CONF);
  dbto =  Agent->Conf->dbl.nitems;
  UDM_RELEASELOCK(Agent, UDM_LOCK_CONF);

  for (i = dbfrom; i < dbto; i++)
  {
    db = &Agent->Conf->dbl.db[i];
    UDM_GETLOCK(Agent, UDM_LOCK_DB); 
    switch(db->DBDriver)
    {
#ifdef HAVE_SQL
      default:
        rc = UdmCheckUrlidSQL(Agent, db, id);
        break;
#endif
    }
    UDM_RELEASELOCK(Agent, UDM_LOCK_DB);
    if (rc != 0) break;
  }
  return rc;
}


/********************************************************/


UDM_DBLIST * UdmDBListInit(UDM_DBLIST * List)
{
  bzero((void*)List, sizeof(*List));
  return(List);
}

size_t UdmDBListAdd(UDM_DBLIST *List, const char * addr, int mode)
{
  UDM_DB  *db;
  int res;
  db=List->db=(UDM_DB*)UdmRealloc(List->db,(List->nitems+1)*sizeof(UDM_DB));
  db+=List->nitems;
  UdmDBInit(db);
  res = UdmDBSetAddr(db, addr, mode);
  if (res == UDM_OK) List->nitems++;
  return res;
}

void UdmDBListFree(UDM_DBLIST *List)
{
  size_t  i;
  UDM_DB  *db=List->db;
  
  for(i = 0; i < List->nitems; i++)
  {
    UdmDBFree(&db[i]);
  }
  UDM_FREE(List->db);
  UdmDBListInit(List);
}

/********************* MultiCache stuff *************************/
UDM_MULTI_CACHE *UdmMultiCacheInit (UDM_MULTI_CACHE *cache)
{
  size_t i;

  if (! cache)
  {
    cache = UdmMalloc(sizeof(UDM_MULTI_CACHE));
    if (! cache) return(NULL);
    cache->free = 1;
  }
  else
  {
    cache->free = 0;
  }
  cache->nrecs = 0;

  for (i = 0; i <= MULTI_DICTS; i++)
  {
    cache->tables[i].nurls = 0;
    cache->tables[i].urls = NULL;
  }

  cache->nurls = 0;
  cache->urls = NULL;

  return(cache);
}

void UdmMultiCacheFree (UDM_MULTI_CACHE *cache)
{
  size_t w, s, u, t;

  if (! cache) return;
  for (t = 0; t <= MULTI_DICTS; t++)
  {
    UDM_MULTI_CACHE_TABLE *table = &cache->tables[t];
    for (u = 0; u < table->nurls; u++)
    {
      UDM_MULTI_CACHE_URL *url = &table->urls[u];
      for (s = 0; s < url->nsections; s++)
      {
        UDM_MULTI_CACHE_SECTION *section = &url->sections[s];
        for (w = 0; w < section->nwords; w++) {
          UDM_MULTI_CACHE_WORD *word = &section->words[w];
          UdmFree(word->word);
          UdmFree(word->intags);
        }
        UdmFree(section->words);
      }
      UdmFree(url->sections);
    }
    UdmFree(table->urls);
    cache->tables[t].nurls = 0;
    cache->tables[t].urls = NULL;
  }

  UdmFree(cache->urls);
  cache->nurls = 0;
  cache-> urls = NULL;
  cache->nrecs = 0;

  if (cache->free) UdmFree(cache);
}

int UdmWordCacheFlush (UDM_AGENT *Indexer)
{
#ifdef HAVE_SQL
  size_t i;

  for (i = 0; i < Indexer->Conf->dbl.nitems; i++)
  {
    int rc;
    UDM_DB *db = &Indexer->Conf->dbl.db[i];
    UDM_GETLOCK(Indexer, UDM_LOCK_DB);
    if (db->DBMode == UDM_DBMODE_MULTI)
    {
      if (UDM_OK != (rc= UdmWordCacheWrite(Indexer, db, 0)))
      {
        UdmLog(Indexer,UDM_LOG_ERROR,"%s",db->errstr);
        return rc;
      }
    }
    if (db->DBMode == UDM_DBMODE_FLY)
    {
      if (UDM_OK != (rc= UdmDBModeFlyWrite(Indexer, db, 0)))
      {
        UdmLog(Indexer,UDM_LOG_ERROR,"%s",db->errstr);
        return rc;
      }
    }
    UDM_RELEASELOCK(Indexer, UDM_LOCK_DB);
  }
#endif
  return UDM_OK;
}

int UdmMulti2Blob (UDM_AGENT *Indexer)
{
#ifdef HAVE_SQL
  size_t i;
  unsigned long ticks;

  UdmLog(Indexer,UDM_LOG_ERROR,"Converting to blob");
  ticks=UdmStartTimer();

  for (i = 0; i < Indexer->Conf->dbl.nitems; i++)
  {
    int rc;
    UDM_DB *db = &Indexer->Conf->dbl.db[i];
    UDM_GETLOCK(Indexer, UDM_LOCK_DB);
    if (db->DBMode == UDM_DBMODE_MULTI)
      rc= UdmMulti2BlobSQL(Indexer, db);
    else if (db->DBMode == UDM_DBMODE_SINGLE)
      rc= UdmSingle2BlobSQL(Indexer, db);
    else if (db->DBMode == UDM_DBMODE_FLY)
      rc= UdmDBModeFlyMerge(Indexer, db);
    UDM_RELEASELOCK(Indexer, UDM_LOCK_DB);
    if (rc != UDM_OK)
    {
      UdmLog(Indexer,UDM_LOG_ERROR,"%s",db->errstr); 
      return rc;
    }
  }

  ticks=UdmStartTimer()-ticks;
  UdmLog(Indexer,UDM_LOG_ERROR,"Converting to blob finished\t%.2f",(float)ticks/1000);
#endif
  return UDM_OK;
}

int UdmExport (UDM_AGENT *Indexer)
{
#ifdef HAVE_SQL
  size_t i;
  unsigned long ticks;

  UdmLog(Indexer,UDM_LOG_ERROR,"Starting export");
  ticks=UdmStartTimer();

  for (i = 0; i < Indexer->Conf->dbl.nitems; i++)
  {
    UDM_DB *db = &Indexer->Conf->dbl.db[i];
    UDM_GETLOCK(Indexer, UDM_LOCK_DB);
    UdmExportSQL(Indexer, db);
    UDM_RELEASELOCK(Indexer, UDM_LOCK_DB);
  }

  ticks=UdmStartTimer()-ticks;
  UdmLog(Indexer,UDM_LOG_ERROR,"Export finished\t%.2f",(float)ticks/1000);
#endif
  return(0);
}

static size_t UdmMultiCacheAddWord (UDM_MULTI_CACHE_WORD *cache, uint4 coord)
{
  uint4 *tmp;

  if (! cache) return(0);
  tmp = UdmRealloc(cache->intags, (cache->nintags + 1) * sizeof(uint4));
  if (! tmp) return(0);
  cache->intags = tmp;
  cache->intags[cache->nintags] = coord;
  cache->nintags++;
  return(1);
}

static size_t UdmMultiCacheAddSection (UDM_MULTI_CACHE_SECTION *cache, UDM_WORD *word)
{
  size_t i;
  uint4 coord = UDM_WRDPOS(word->coord) & 0xFFFF;

  if (! cache) return(0);

  for (i = 0; i < cache->nwords; i++)
  {
    if (! strcmp(cache->words[i].word, word->word))  break;
  }

  if (i == cache->nwords)
  {
    UDM_MULTI_CACHE_WORD *tmp;
    tmp = UdmRealloc(cache->words, (cache->nwords + 1) * sizeof(UDM_MULTI_CACHE_WORD));
    if (! tmp) return(0);
    cache->words = tmp;
    cache->words[cache->nwords].word = (char *)UdmStrdup(word->word);
    cache->words[cache->nwords].nintags = 0;
    cache->words[cache->nwords].intags = NULL;
    cache->nwords++;
  }

  return(UdmMultiCacheAddWord(&cache->words[i], coord));
}

static size_t UdmMultiCacheAddURL (UDM_MULTI_CACHE_URL *cache, UDM_WORD *word)
{
  size_t i;
  unsigned char secno = UDM_WRDSEC(word->coord);

  if (! cache) return(0);

  for (i = 0; i < cache->nsections; i++)
    if (cache->sections[i].secno == secno) break;

  if (i == cache->nsections)
  {
    UDM_MULTI_CACHE_SECTION *tmp;
    tmp = UdmRealloc(cache->sections, (cache->nsections + 1) * sizeof(UDM_MULTI_CACHE_SECTION));
    if (! tmp) return(0);
    cache->sections = tmp;
    cache->sections[cache->nsections].secno = secno;
    cache->sections[cache->nsections].nwords = 0;
    cache->sections[cache->nsections].words = NULL;
    cache->nsections++;
  }

  return(UdmMultiCacheAddSection(&cache->sections[i], word));
}

static size_t UdmMultiCacheAddTable (UDM_MULTI_CACHE_TABLE *cache, urlid_t url_id, unsigned char reindex, UDM_WORD *word)
{
  size_t i;

  if (! cache) return(0);
  for (i = 0; i < cache->nurls; i++)
  {
    if (cache->urls[i].url_id == url_id) break;
  }

  if (i == cache->nurls)
  {
    UDM_MULTI_CACHE_URL *tmp;
    tmp = UdmRealloc(cache->urls, (cache->nurls + 1) * sizeof(UDM_MULTI_CACHE_URL));
    if (! tmp) return(0);
    cache->urls = tmp;
    cache->urls[cache->nurls].url_id = url_id;
    cache->urls[cache->nurls].reindex = reindex;
    cache->urls[cache->nurls].nsections = 0;
    cache->urls[cache->nurls].sections = NULL;
    cache->nurls++;
  }

  return(UdmMultiCacheAddURL(&cache->urls[i], word));
}

size_t UdmMultiCacheAdd (UDM_MULTI_CACHE *cache, urlid_t url_id, unsigned char reindex, UDM_WORD *word)
{
  udmhash32_t table = UdmStrHash32(word->word) & MULTI_DICTS;
  size_t i;
  
  if (! cache) return(0);

  cache->nrecs++;

  if (reindex)
  {
    for (i = 0; i < cache->nurls; i++)
      if (cache->urls[i] == url_id) break;

    if (i == cache->nurls)
    {
      urlid_t *tmp;
      tmp = UdmRealloc(cache->urls, (cache->nurls + 1) * sizeof(urlid_t));
      if (! tmp) return(0);
      cache->urls = tmp;
      cache->urls[cache->nurls] = url_id;
      cache->nurls++;
    }
  }

  return(UdmMultiCacheAddTable(&cache->tables[table], url_id, reindex, word));
}

/******************** BlobCache stuff *******************/

UDM_BLOB_CACHE *UdmBlobCacheInit (UDM_BLOB_CACHE *cache)
{
  if (! cache)
  {
    cache = UdmMalloc(sizeof(UDM_BLOB_CACHE));
    if (! cache) return(NULL);
    cache->free = 1;
  }
  else
  {
    cache->free = 0;
  }
  cache->nwords = 0;
  cache->awords = 0;
  cache->words = NULL;

  return(cache);
}

void UdmBlobCacheFree (UDM_BLOB_CACHE *cache)
{
  size_t i;
  if (! cache) return;
  for (i= 0; i < cache->nwords; i++)
  {
    UDM_FREE(cache->words[i].word);
    UDM_FREE(cache->words[i].intags);
  }

  UdmFree(cache->words);
  cache->nwords = 0;
  cache->awords = 0;
  cache->words = NULL;

  if (cache->free) UdmFree(cache);
}

size_t UdmBlobCacheAdd (UDM_BLOB_CACHE *cache, urlid_t url_id,
                        unsigned char secno, const char *word,
                        size_t nintags, const char *intag, size_t intaglen)
{
  if (! cache)
  {
    fprintf(stderr, "Cache variable empty\n");
    return(0);
  }
  if (! url_id)
  {
    fprintf(stderr, "url_id variable empty\n");
    return(0);
  }
  if (! secno)
  {
    fprintf(stderr, "secno variable empty\n");
    return(0);
  }
  if (! word)
  {
    fprintf(stderr, "word variable empty\n");
    return(0);
  }
  if (! nintags)
  {
    fprintf(stderr, "nintags variable empty\n");
    return(0);
  }
  if (! intag)
  {
    fprintf(stderr, "intag variable empty\n");
    return(0);
  }

  if (cache->nwords == cache->awords)
  {
    UDM_BLOB_CACHE_WORD *tmp;
    tmp = UdmRealloc(cache->words, (cache->awords + 256) * sizeof(UDM_BLOB_CACHE_WORD));
    if (! tmp)
    {
      fprintf(stderr, "Realloc failed while adding word\n");
      return(0);
    }
    cache->words = tmp;
    cache->awords += 256;
  }
  cache->words[cache->nwords].secno = secno;
  cache->words[cache->nwords].word = UdmStrdup(word);
  cache->words[cache->nwords].url_id = url_id;
  cache->words[cache->nwords].nintags = nintags;
  cache->words[cache->nwords].intags = UdmMalloc(intaglen+1);
  memcpy(cache->words[cache->nwords].intags,intag,intaglen);
  cache->words[cache->nwords].intags[intaglen]= '\0';
  cache->words[cache->nwords].ntaglen = intaglen;

  cache->nwords++;

  return(1);
}

static int bccmpwrd (UDM_BLOB_CACHE_WORD *s1, UDM_BLOB_CACHE_WORD *s2)
{
  int _ = strcmp(s1->word, s2->word);
  if (! _) _ = s1->secno - s2->secno;
  if (! _) {
    if (s1->url_id > s2->url_id) _ = 1;
    else if (s1->url_id < s2->url_id) _ = -1;
    else _ = 0;
  }
  return(_);
}

void UdmBlobCacheSort (UDM_BLOB_CACHE *cache)
{
  qsort(cache->words, cache->nwords, sizeof(UDM_BLOB_CACHE_WORD), (qsort_cmp)bccmpwrd);
}

UDM_WORD_CACHE *UdmWordCacheInit (UDM_WORD_CACHE *cache)
{
  if (! cache)
  {
    cache = UdmMalloc(sizeof(UDM_WORD_CACHE));
    if (! cache) return(NULL);
    cache->free = 1;
  }
  else
  {
    cache->free = 0;
  }
  cache->nbytes = sizeof(UDM_WORD_CACHE);
  cache->nwords = 0;
  cache->awords = 0;
  cache->words = NULL;
  cache->nurls = 0;
  cache->aurls = 0;
  cache->urls = NULL;

  return(cache);
}

void UdmWordCacheFree (UDM_WORD_CACHE *cache)
{
  size_t i;

  if (! cache) return;
  for (i = 0; i < cache->nwords; i++) UDM_FREE(cache->words[i].word);
  UDM_FREE(cache->words);
  UDM_FREE(cache->urls);
  cache->nbytes = sizeof(UDM_WORD_CACHE);
  cache->nwords = 0;
  cache->awords = 0;
  cache->nurls = 0;
  cache->aurls = 0;

  if (cache->free) UdmFree(cache);
}

static int wccmpwrd (UDM_WORD_CACHE_WORD *s1, UDM_WORD_CACHE_WORD *s2)
{
  int _ = s1->seed - s2->seed;
  if (! _) _ = strcmp(s1->word, s2->word);
  if (! _) _ = s1->secno - s2->secno;
  if (! _) {
    if (s1->url_id > s2->url_id) _ = 1;
    else if (s1->url_id < s2->url_id) _ = -1;
    else _ = 0;
  }
  if (! _) _ = s1->coord - s2->coord;
  return(_);
}

void UdmWordCacheSort (UDM_WORD_CACHE *cache)
{
  UdmSort(cache->words, cache->nwords, sizeof(UDM_WORD_CACHE_WORD), (qsort_cmp)wccmpwrd);
}

int UdmWordCacheAdd (UDM_WORD_CACHE *cache, urlid_t url_id, const char *word, int intag)
{
  if (! word) return(UDM_OK);

  if (cache->nwords == cache->awords)
  {
    UDM_WORD_CACHE_WORD *tmp;
    tmp = UdmRealloc(cache->words, (cache->awords + 256) * sizeof(UDM_WORD_CACHE_WORD));
    if (! tmp) {
      fprintf(stderr, "Realloc failed while adding word\n");
      return(UDM_ERROR);
    }
    cache->words = tmp;
    cache->awords += 256;
    cache->nbytes += sizeof(UDM_WORD_CACHE_WORD) * 256;
  }

  cache->words[cache->nwords].word = UdmStrdup(word);
  if (! cache->words[cache->nwords].word) return(UDM_ERROR);
  cache->words[cache->nwords].url_id = url_id;
  cache->words[cache->nwords].secno = UDM_WRDSEC(intag) & 0xFF;
  cache->words[cache->nwords].coord = UDM_WRDPOS(intag) & 0xFFFF;
  cache->words[cache->nwords].seed = UdmStrHash32(word) & MULTI_DICTS;
  cache->nwords++;
  cache->nbytes += strlen(word) + 1;
  return(UDM_OK);
}

int UdmWordCacheAddURL (UDM_WORD_CACHE *cache, urlid_t url_id)
{
  if (cache->nurls == cache->aurls)
  {
    urlid_t *tmp;
    tmp = UdmRealloc(cache->urls, (cache->aurls + 256) * sizeof(urlid_t));
    if (!tmp)
    {
      fprintf(stderr, "Realloc failed while adding word\n");
      return(UDM_ERROR);
    }
    cache->urls = tmp;
    cache->aurls += 256;
    cache->nbytes += sizeof(urlid_t) * 256;
  }

  cache->urls[cache->nurls] = url_id;
  cache->nurls++;
  return(UDM_OK);
}
