/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#if (HAVE_PGSQL)
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "udm_common.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_utils.h"
#include "udm_vars.h"
#include "udm_sqldbms.h"

#include "udm_xmalloc.h"
#ifdef WIN32
#include <process.h>
#endif

static int UdmPgSQLInitDB(UDM_DB *db)
{
  char port[8];
  const char* DBUser= UdmVarListFindStr(&db->Vars, "DBUser", NULL);
  const char* DBPass= UdmVarListFindStr(&db->Vars, "DBPass", NULL);
  const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", NULL);
  const char* DBSock= UdmVarListFindStr(&db->Vars, "socket", NULL);
  int DBPort= UdmVarListFindInt(&db->Vars, "DBPort", 0);

  /*
    If no host given then check if an alternative
    Unix domain socket is specified.
  */
  DBHost= (DBHost && DBHost[0]) ? DBHost : DBSock;
  
  sprintf(port,"%d",DBPort);
  db->specific= (void*) PQsetdbLogin(DBHost, DBPort ? port : 0, 0, 0,
                                     db->DBName, DBUser, DBPass);
  if (PQstatus((PGconn*)db->specific) == CONNECTION_BAD)
  {
    db->errcode=1;
    sprintf(db->errstr, "%s", PQerrorMessage((PGconn*)db->specific));
    return UDM_ERROR;
  }
  db->connected=1;
  return UDM_OK;
}

static int UdmPgSQLQuery(UDM_DB *db, UDM_SQLRES *res, const char *q)
{
  size_t i;
  PGresult *PGres;
  PGconn *pgsql;
  
  db->errcode=0;
  if(!db->connected)
  {
    UdmPgSQLInitDB(db);
    if(db->errcode)
      return UDM_ERROR;
  }
  
  if (res)
  {
    bzero((void*) res, sizeof(UDM_SQLRES));
    res->db= db;
  }

  pgsql= (PGconn*) db->specific;
  
  if(!(PGres= PQexec(pgsql,q)))
  {
    sprintf(db->errstr, "%s", PQerrorMessage(pgsql));
    db->errcode=1;
    return UDM_ERROR;
  }
  if(PQresultStatus(PGres)==PGRES_COMMAND_OK)
  {
    /* Free non-SELECT query */
    PQclear(PGres);
    return UDM_OK;
  }
  
  if(PQresultStatus(PGres) != PGRES_TUPLES_OK)
  {
    PQclear(PGres);
    sprintf(db->errstr, "%s", PQerrorMessage(pgsql));
    if(strstr(db->errstr,"duplicate") ||
       strstr(db->errstr, "") ||
       strstr(db->errstr, "повторный") ||
       strcasestr(db->errstr,"Duplizierter"))
    {
      return UDM_OK;
    }
    else
    {
      db->errcode = 1;
      return UDM_ERROR;
    }
  }
  
  if (!res)
  {
    /* 
     Don't allow to call UdmPgSQLQuery
     returning data with NULL res pointer
    */
    sprintf(db->errstr, 
            "UdmPgSQLQuery executed with res=NULL returned result %d, %s",
             PQresultStatus(PGres),PQerrorMessage(pgsql));
    db->errcode=1;
    return UDM_ERROR;
  }
  
  res->pgsqlres= PGres;
  res->nCols=(size_t)PQnfields(res->pgsqlres);
  res->nRows=(size_t)PQntuples(res->pgsqlres);
  res->Fields=(UDM_SQLFIELD*)UdmMalloc(res->nCols*sizeof(UDM_SQLFIELD));
  for(i=0;i<res->nCols;i++)
    res->Fields[i].sqlname = (char*)UdmStrdup(PQfname(res->pgsqlres,(int)i));
  return UDM_OK;
}

static void UdmPgSQLClose(UDM_DB *db)
{
  PQfinish((PGconn*)db->specific);
  db->specific = NULL;
}


static int UdmPgSQLBegin(UDM_DB *db)
{
  return UdmPgSQLQuery(db,NULL,"BEGIN WORK");
}


static int UdmPgSQLCommit(UDM_DB *db)
{
  return UdmPgSQLQuery(db,NULL,"END WORK");
}

static
int UdmPgSQLFetchRow (UDM_DB *db, UDM_SQLRES *res, UDM_PSTR *buf)
{
  size_t j;

  if (res->curRow >= res->nRows)
    return UDM_ERROR;
  
  for (j = 0; j < res->nCols; j++)
  {
    buf[j].val= PQgetvalue(res->pgsqlres,(int)res->curRow,(int)(j));
    buf[j].len= 0;
  }
  res->curRow++;
  return(UDM_OK);
}


UDM_SQLDB_HANDLER udm_sqldb_pgsql_handler =
{
  NULL,
  UdmPgSQLQuery,
  UdmPgSQLClose,
  UdmPgSQLBegin,
  UdmPgSQLCommit,
  NULL,
  NULL,
  NULL,
  UdmPgSQLFetchRow,
  UdmSQLStoreResultSimple,
  UdmSQLFreeResultSimple,
  UdmPgSQLQuery
};

#endif
