#include <ctype.h>
#include <cassert>
#include <sstream>
#include <iomanip>


#include "Configuration.h"
#include "Tools.h"
#include "SDLTools.h"
#include "File.h"
#include "Tiles.h"


//----------------------------------------------------------------------------
TileException::TileException(Uint8 categoryId, Uint8 tileId)
{
    m_categoryId = categoryId;
    m_tileId = tileId;
}

//----------------------------------------------------------------------------
TileException::~TileException()
{
}

//----------------------------------------------------------------------------
std::string TileException::toString() const
{
    std::ostringstream s;
    s << "The tile '"
      << std::setfill('0') << std::setw(2) << std::hex << (int)m_tileId
      << "' doesn't exist in the category '"
      << std::setfill('0') << std::setw(2) << std::hex << (int)m_categoryId
      << "'" << std::ends;
    return s.str();
}



//----------------------------------------------------------------------------
Tiles *Tiles::sm_instance = NULL;


//----------------------------------------------------------------------------
Tiles::Tiles()
{
}

//----------------------------------------------------------------------------
Tiles::~Tiles()
{
    for (CategoryMap::iterator iter = m_map.begin();
         iter != m_map.end(); ++iter)
    {
        for (TileMap::iterator iter2 = iter->second.begin();
             iter2 != iter->second.end(); ++iter2)
        {
            SDL_CALLS::FreeSurface(iter2->second);
        }
    }

    m_map.clear();
}


//----------------------------------------------------------------------------
void Tiles::init()
{
    std::auto_ptr<Tiles> tiles(new Tiles());

    std::string name;
    name.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/tiles/")
        .append("categories.idx");

    File f(name.c_str(), "r");
    while (!f.isEOF())
    {
        char buffer[128] = "";
        f.readline(buffer, sizeof(buffer));

        if (isxdigit(buffer[0]) && isxdigit(buffer[1]))
        {
            unsigned int category = 0;
            if (sscanf(buffer, "%x", &category) != 1)  continue;

            tiles->initCategory(category, buffer+3);
        }
    }

    sm_instance = tiles.release();
}

//----------------------------------------------------------------------------
void Tiles::initCategory(Uint8 category, const char *dir)
{
    std::string name;
    name.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/tiles/").append(dir).append("/tiles.idx");

    File f(name.c_str(), "r");
    while (!f.isEOF())
    {
        char buffer[128] = "";
        f.readline(buffer, sizeof(buffer));

        if (isxdigit(buffer[0]) && isxdigit(buffer[1]))
        {
            unsigned int tile = 0;
            if (sscanf(buffer, "%x", &tile) != 1)  continue;

            std::string tileName;
            tileName.append(Configuration::getInstance()->getDataDir())
                .append("/gfx/tiles/").append(dir).append("/")
                .append(buffer+3);

            SDL_Surface *s = SDL_TOOLS::loadBMPWithColorKey(tileName.c_str());

            CTKey ctKey(category, tile);
            ABKey abKey(SDL_ALPHA_OPAQUE, 0x80);

            m_map[ctKey][abKey] = s;
        }
    }
}

//----------------------------------------------------------------------------
void Tiles::destroy()
{
    ZAP_POINTER(sm_instance);
}



//----------------------------------------------------------------------------
const SDL_Surface *Tiles::getTile(Uint8 category, Uint8 tile,
                                  Uint8 alpha, Uint8 brightness) const
{
    CTKey ctKey(category, tile);
    CategoryMap::iterator iter = m_map.find(ctKey);
    if (iter == m_map.end())
    {
        throw TileException(category, tile);
    }

    ABKey abKey(alpha, brightness);
    TileMap::const_iterator iter2 = iter->second.find(abKey);
    return iter2 != iter->second.end() ?
        iter2->second : createTile(alpha, brightness, iter);
}

//----------------------------------------------------------------------------
const SDL_Surface *Tiles::createTile(Uint8 alpha, Uint8 brightness,
                                     CategoryMap::iterator &iter) const
{
    ABKey abKey(alpha, brightness);
    ABKey refABKey(SDL_ALPHA_OPAQUE, 0x80);

    TileMap::const_iterator iter2 = iter->second.find(refABKey);
    if (abKey == refABKey || iter2 == iter->second.end())
    {
        throw TileException(iter->first.first, iter->first.second);
    }

    SDL_Surface *s = SDL_CALLS::CreateRGBSurface(
        SDL_SWSURFACE, iter2->second->w, iter2->second->h, 32);

    if (brightness <= 0x80)
    {
        // The brightness levels from normal to black are reached by setting
        // an apropriate alpha value and blitting to a black surface.

        SDL_CALLS::SetAlpha(iter2->second, SDL_SRCALPHA,
                            brightness == 0x80 ? 0xff : 2*brightness);
        SDL_CALLS::BlitSurface(iter2->second, NULL, s, NULL);
        SDL_CALLS::SetAlpha(iter2->second, SDL_SRCALPHA, SDL_ALPHA_OPAQUE);
    }
    else
    {
        // The brightness levels from normal to white are reached by setting
        // an apropriate alpha value and blitting to a white surface.

        SDL_Rect r = { 0, 0, s->w, s->h };
        SDL_CALLS::FillRect(s, &r, SDL_MapRGB(s->format, 0xff, 0xff, 0xff));

        SDL_CALLS::SetAlpha(iter2->second, SDL_SRCALPHA, 2*(0xff-brightness));
        SDL_CALLS::BlitSurface(iter2->second, NULL, s, NULL);
        SDL_CALLS::SetAlpha(iter2->second, SDL_SRCALPHA, SDL_ALPHA_OPAQUE);
    }

    SDL_CALLS::SetColorKey(
        s, SDL_SRCCOLORKEY, SDL_MapRGB(s->format, 0, 0, 0));
    SDL_CALLS::SetAlpha(s, SDL_SRCALPHA, alpha);

    iter->second[abKey] = s;

    return s;
}

