#ifndef TURRETS_H
#define TURRETS_H

#include <list>
#include <cassert>

#include "Tools.h"
#include "SinusTable.h"

#include "XMLVisitors.h"

#include "StaticDecorationBase.h"


//----------------------------------------------------------------------------
class TurretSurfaces : public SurfacesBase
{
    SURFACES_SINGLETON_OBJECT(TurretSurfaces);

  public:

    enum Turret
    {
        T_TUBE = 0,
        T_DOME = 1,
        T_BIGTUBE = 2,
        T_TOTAL_NUMBER = 3
    };

    inline const SDL_Surface *getSurface(
        OrientatingDecorationBase::Orientation o,
        Turret turret) const
    {
        assert(o >= 0 && o < 4);
        assert(turret >= 0);
        assert(turret < T_TOTAL_NUMBER);

        return m_turrets[o][turret];
    }

    Uint16 getTipOffset(Turret turret) const;

  private:

    /// All turret surfaces.
    SDL_Surface *m_turrets[4][T_TOTAL_NUMBER];
};


//----------------------------------------------------------------------------
class TurretBase;


//----------------------------------------------------------------------------
#undef DECLARE_UPDATE_STRATEGY
#define DECLARE_UPDATE_STRATEGY(c) \
    class c; \
    friend class c; \
    class c : public UpdateStrategy \
    { \
        STRATEGY_OBJECT(c); \
      public: \
        c() {} \
        ~c() {} \
        void update(TurretBase *t, TurretBarrel *b); \
    }


//----------------------------------------------------------------------------
class TurretBarrel
{
    //------------------------------------------------------------------------
    friend class TurretTest;

    //------------------------------------------------------------------------
    enum EnumUpdateStrategy
    {
        US_FIXED,
        US_RANDOM,
        US_SWEEP,
        US_SMART
    };

    //------------------------------------------------------------------------
    class UpdateStrategy
    {
      public:
        UpdateStrategy() {}
        virtual ~UpdateStrategy() {}

        static UpdateStrategy *get(EnumUpdateStrategy strategy);

        virtual void update(TurretBase *t, TurretBarrel *b) = 0;
    };

    DECLARE_UPDATE_STRATEGY(Fixed);
    DECLARE_UPDATE_STRATEGY(Random);
    DECLARE_UPDATE_STRATEGY(Sweep);
    DECLARE_UPDATE_STRATEGY(Smart);

  public:

    //------------------------------------------------------------------------
    class InitializationData
    {
      public:
        InitializationData(const XMLNode *barrelNode);
        ~InitializationData();

        EnumUpdateStrategy updateStrategy;

        int angle;
        int step;

        unsigned speed;
        unsigned delay;
    };

    ~TurretBarrel();

    inline int getAngle() const
    {
        return m_angle;
    }
    inline void setAngle(int angle)
    {
        m_angle = SinusTable::normalize(angle);
    }

    inline int getAngleStep() const
    {
        return m_angleStep;
    }

    inline unsigned getProjectileSpeed() const
    {
        return m_projectileSpeed;
    }

    void update(TurretBase *turret);

    inline void setStrategy(UpdateStrategy *strategy)
    {
        m_strategy = strategy;
    }

    static TurretBarrel *create(const XMLNode *barrelNode);

  protected:
    TurretBarrel(const InitializationData &init);

    int m_angle;
    int m_angleStep;

    unsigned m_projectileSpeed;

    unsigned m_frameDelay;
    unsigned m_frameCounter;

  private:
    UpdateStrategy *m_strategy;
};

typedef std::list<TurretBarrel*> TurretBarrelList;
typedef TurretBarrelList::iterator TurretBarrelListIter;
typedef TurretBarrelList::const_iterator TurretBarrelListCIter;



//----------------------------------------------------------------------------
class TurretBase : public StaticDecorationBase,
                   public OrientatingDecorationBase
{
    //------------------------------------------------------------------------
    friend class TurretTest;

    //------------------------------------------------------------------------
    class CopyTurretPropertiesConstVisitor : public XMLConstVisitor
    {
      public:
        //--------------------------------------------------------------------
        CopyTurretPropertiesConstVisitor(XMLNode &node) : m_node(node) {}
        ~CopyTurretPropertiesConstVisitor() {}

      private:
        //--------------------------------------------------------------------
        void do_visit(const XMLProperty *p);

        //--------------------------------------------------------------------
        XMLNode &m_node;
    };

    //------------------------------------------------------------------------
    class BarrelVisitor : public XMLConstVisitor
    {
      public:
        //--------------------------------------------------------------------
        BarrelVisitor(TurretBase *turret) : m_turret(turret) {}
        ~BarrelVisitor() { m_turret = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(const XMLNode *n);

        //--------------------------------------------------------------------
        TurretBase *m_turret;
    };

  public:
    //------------------------------------------------------------------------
    class InitializationData
        : public StaticDecorationBase::InitializationData,
          public OrientatingDecorationBase::InitializationData
    {
      public:
        InitializationData();
        InitializationData(const XMLNode *turretNode);
        ~InitializationData();

        static bool hasPresetTurretNode(const XMLNode *turretNode,
                                        XMLNode &processTurretNode);

        TurretSurfaces::Turret type;
        unsigned hitPoints;
    };


    //------------------------------------------------------------------------
    virtual ~TurretBase();

    //------------------------------------------------------------------------
    inline TurretSurfaces::Turret getType() const
    {
        return m_type;
    }

    /// Returns, whether the turret can fire to the given angle.
    bool isAngleInRange(int angle) const;
    virtual int getCenterAngle() const = 0;

    virtual OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const = 0;

    //------------------------------------------------------------------------
    /**
     * Called for a projectile collision to determine,
     * if the turret's hitpoints are reached.
     */
    inline bool decAndTestHitPoints()
    {
        return --m_hitPoints == 0;
    }

    //------------------------------------------------------------------------
    void update();

    //------------------------------------------------------------------------
    void updateSurface();

    //------------------------------------------------------------------------
    static TurretBase *create(const InitializationData &init);
    static TurretBase *create(const XMLNode *turretNode);

  protected:
    //------------------------------------------------------------------------
    TurretBase(const InitializationData &init);

    //------------------------------------------------------------------------
    Uint16 m_tipOffset;

  private:
    //------------------------------------------------------------------------
    DECLARE_OBJECT_VISITOR_API();

    //------------------------------------------------------------------------
    /// The type of turret, needed to access the ShipSurfaces factory.
    TurretSurfaces::Turret m_type;

    TurretBarrelList m_barrels;

    unsigned m_hitPoints;
};


//----------------------------------------------------------------------------
class TopTurret : public TurretBase
{
  public:
    TopTurret(const InitializationData &init);
    ~TopTurret();

    int getCenterAngle() const;
    OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const;
};

//----------------------------------------------------------------------------
class BottomTurret : public TurretBase
{
  public:
    BottomTurret(const InitializationData &init);
    ~BottomTurret();

    int getCenterAngle() const;
    OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const;
};

//----------------------------------------------------------------------------
class LeftTurret : public TurretBase
{
  public:
    LeftTurret(const InitializationData &init);
    ~LeftTurret();

    int getCenterAngle() const;
    OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const;
};

//----------------------------------------------------------------------------
class RightTurret : public TurretBase
{
  public:
    RightTurret(const InitializationData &init);
    ~RightTurret();

    int getCenterAngle() const;
    OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const;
};

#endif //TURRETS_H
