/*****************************************************************************/

/*
 *      main.c  --  Soundmodem main.
 *
 *      Copyright (C) 1999-2000
 *        Thomas Sailer (sailer@ife.ee.ethz.ch)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Please note that the GPL allows you to use the driver, NOT the radio.
 *  In order to use the radio, you need a license from the communications
 *  authority of your country.
 *
 */

/*****************************************************************************/

#define _GNU_SOURCE
#define _REENTRANT

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "soundio.h"
#include "simd.h"

#include "getopt.h"

#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <sys/uio.h>

#include <netinet/in.h>
#include <sys/socket.h>

/* libxml includes */
#include <tree.h>
#include <parser.h>

#ifdef HAVE_SCHED_H
#include <sched.h>
#endif

#include <sys/types.h>
#include <sys/mman.h>
#include <arpa/inet.h>

/* ---------------------------------------------------------------------- */

struct state state = {
	NULL, NULL, NULL, NULL
};

/* ---------------------------------------------------------------------- */

void audiowrite(struct modemchannel *chan, const int16_t *samples, unsigned int nr)
{
	struct audioio *audioio = chan->state->audio;

	if (!audioio->write)
		return;
	audioio->write(audioio, samples, nr);
}

void audioread(struct modemchannel *chan, int16_t *samples, unsigned int nr, u_int16_t tim)
{
	struct audioio *audioio = chan->state->audio;

	if (!audioio->read) {
		pthread_exit(NULL);
		return;
	}
	audioio->read(audioio, samples, nr, tim);
}

u_int16_t audiocurtime(struct modemchannel *chan)
{
	struct audioio *audioio = chan->state->audio;

	if (!audioio->curtime)
		return 0;
	return audioio->curtime(audioio);
}

/* ---------------------------------------------------------------------- */

static const struct modemparams stpparams[] = {
        { "udpport", "STP UDP Port", "UDP port number for Satellite Telemetry Protocol broadcasts; -1 disables STP",
          "-1", MODEMPAR_NUMERIC, { n: { -1, 65535, 100, 1000 } } },
        { "udpaddr", "STP UDP Address", "IP Address for UDP STP broadcasts", "127.0.0.1", MODEMPAR_STRING },
        { NULL }
};

static int stpsock = -1;

static int parseip(struct in_addr *ipaddr, const char *cp)
{
        unsigned int b1, b2, b3, b4;
        
        if (!cp || !*cp)
                return 0;
        if (sscanf(cp, "%u.%u.%u.%u", &b1, &b2, &b3, &b4) == 4 &&
            b1 < 256 && b2 < 256 && b3 < 256 && b4 < 256) {
                ipaddr->s_addr = htonl((b1 << 24) | (b2 << 16) | (b3 << 8) | b4);
                return 1;
        }
        ipaddr->s_addr = 0;
        logprintf(MLOG_ERROR, "stp: invalid IP address \"%s\"\n", cp);
        return 0;
}

static int stp_init(const char *params[])
{
	struct sockaddr_in sin;
        int port;
        
	if (stpsock != -1)
		close(stpsock);
        if (!params[0])
                return -1;
        port = strtol(params[0], NULL, 0);
        if (port < 0 || port > 65535)
                return -1;
	stpsock = socket(AF_INET, SOCK_DGRAM, 0);
	if (stpsock == -1)
		return -1;
	sin.sin_family = AF_INET;
        if (!parseip(&sin.sin_addr, params[1]))
                sin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	sin.sin_port = htons(port);
	if (connect(stpsock, &sin, sizeof(sin))) {
		close(stpsock);
		stpsock = -1;
		return -1;
	}
	return 0;
}

static int stp_send(const unsigned char *pkt)
{
	static unsigned char syncvec[4] = { 0x39, 0x15, 0xed, 0x30 };
	char hdr[256];
	int hdlen;
	char htime[128];
	struct iovec iov[3];
	struct tm *tm;
	time_t t;
	
	if (stpsock == -1)
		return -1;
	time(&t);
	tm = gmtime(&t);
	strftime(htime, sizeof(htime), "Date: %a, %d %b %Y %H:%M:%S UTC\r\n", tm);
	hdlen = snprintf(hdr, sizeof(hdr), 
			"Source: amsat.ao-40.ihu.standard\r\n"
			"%s"
			"Length: 4144\r\n\r\n", htime);
	if (hdlen >= sizeof(hdr) || hdlen <= 0)
		return -1;
	iov[0].iov_base = hdr;
	iov[0].iov_len = hdlen;
	iov[1].iov_base = &syncvec[0];
	iov[1].iov_len = 4;
	iov[2].iov_base = pkt;
	iov[2].iov_len = 514;
	if (stpsock == -1)
		return -1;
	if (writev(stpsock, iov, 3) == -1) {
		fprintf(stderr, "STP: write error\n");
		return -1;
	}
	return 0;
}

void p3dreceive(struct modemchannel *chan, const unsigned char *pkt, u_int16_t crc)
{
	unsigned int i;

        if (!crc)
                stp_send(pkt);
	printf("p3d: received new packet, CRC 0x%04x", crc & 0xffff);
	for (i = 0; i < 512+2; i++) {
		if (!(i & 15))
			printf("\n%04x:", i);
		printf(" %02x", pkt[i]);
	}
	for (i = 0; i < 512; i++) {
		if (!(i & 15))
			printf("\n%04x: ", i);
		if (pkt[i] < ' ' || pkt[i] >= 0x80)
			printf(".");
		else
			printf("%c", pkt[i]);
	}
	printf("\n\n");
}

void p3drxstate(struct modemchannel *chan, unsigned int synced, unsigned int carrierfreq)
{
	printf("p3d: modem state: %s  carrier frequency: %u\n",
               synced == 2 ? "RECEIVE (SYNCSCAN)" : synced ? "RECEIVE" : "SYNC HUNT", carrierfreq);
}

/* ---------------------------------------------------------------------- */

#define MAXPAR 16

static void getparam(xmlDocPtr doc, xmlNodePtr node, const struct modemparams *par, const char *parstr[MAXPAR])
{
	unsigned int i;

	memset(parstr, 0, sizeof(parstr));
	if (!par || !node)
		return;
	for (i = 0; i < MAXPAR && par->name; i++, par++)
		parstr[i] = xmlGetProp(node, par->name);
}

static void *demodthread(void *state)
{
	struct modemchannel *chan = state;
	chan->demod->demodulate(chan->demodstate);
	logprintf(MLOG_FATAL, "Receiver %s has returned\n", chan->demod->name);
	return NULL;
}

static void parsechannel(xmlDocPtr doc, xmlNodePtr node, struct state *state, unsigned int *samplerate)
{
	xmlNodePtr mod = NULL, demod = NULL;
	struct modulator *mc = NULL;
	struct demodulator *dc = NULL;
	struct modemchannel *chan;
	char *cp;
	const char *par[MAXPAR];
	unsigned int sr;

	for (; node; node = node->next) {
		if (!node->name)
			logprintf(MLOG_FATAL, "Node has no name\n");
		if (!strcmp(node->name, "mod")) {
			mod = node;
			continue;
		}
		if (!strcmp(node->name, "demod")) {
			demod = node;
			continue;
		}
		logprintf(MLOG_ERROR, "unknown node \"%s\"\n", node->name);
	}
	if (mod) {
		cp = xmlGetProp(mod, "mode");
		if (cp) {
			for (mc = modchain; mc && strcmp(mc->name, cp); mc = mc->next);
			if (!mc)
				logprintf(MLOG_ERROR, "Modulator \"%s\" unknown\n", cp);
		}
	}
	if (demod) {
		cp = xmlGetProp(demod, "mode");
		if (cp) {
			for (dc = demodchain; dc && strcmp(dc->name, cp); dc = dc->next);
			if (!dc)
				logprintf(MLOG_ERROR, "Demodulator \"%s\" unknown\n", cp);
		}
	}
	if (!mc && !dc)
		return;
	if (!(chan = malloc(sizeof(struct modemchannel))))
		logprintf(MLOG_FATAL, "out of memory\n");
        memset(chan, 0, sizeof(struct modemchannel));
	chan->next = state->channels;
	chan->state = state;
	chan->mod = mc;
	chan->demod = dc;
	chan->modstate = NULL;
	chan->demodstate = NULL;
	if (mc) {
		getparam(doc, mod, mc->params, par);
		sr = *samplerate;
		chan->modstate = mc->config(chan, &sr, par);
		if (sr > *samplerate)
			*samplerate = sr;
	}
	if (dc) {
		getparam(doc, demod, dc->params, par);
		sr = *samplerate;
		chan->demodstate = dc->config(chan, &sr, par);
		if (sr > *samplerate)
			*samplerate = sr;
	}
	state->channels = chan;
}

static int parsecfg(xmlDocPtr doc, xmlNodePtr node, struct state *state, unsigned int *schedrr)
{
	xmlNodePtr audio = NULL;
	xmlNodePtr ptt = NULL;
        xmlNodePtr stp = NULL;
	const char *par[MAXPAR];
	struct modemchannel *chan;
        pthread_attr_t rxattr;
        unsigned int samplerate = 5000, mode;

	for (; node; node = node->next) {
		if (!node->name)
			logprintf(MLOG_FATAL, "Node has no name\n");
		if (!strcmp(node->name, "audio")) {
			audio = node;
			continue;
		}
		if (!strcmp(node->name, "ptt")) {
			ptt = node;
			continue;
		}
		if (!strcmp(node->name, "stp")) {
			stp = node;
			continue;
		}
		if (!strcmp(node->name, "channel")) {
			if (node->childs)
				parsechannel(doc, node->childs, state, &samplerate);
			continue;
		}
		logprintf(MLOG_ERROR, "unknown node \"%s\"\n", node->name);
	}
	/* find audio mode */
	mode = 0;
	for (chan = state->channels; chan; chan = chan->next) {
		if (chan->demod && chan->demod->demodulate)
			mode |= IO_RDONLY;
		if (chan->mod && chan->mod->modulate)
			mode |= IO_WRONLY;
	}
	if (!state->channels || !mode) {
		logprintf(MLOG_ERROR, "no channels configured\n");
		return -1;
	}
	/* open PTT */
	getparam(doc, ptt, pttparams, par);
	if (pttinit(&state->ptt, par))
                logprintf(MLOG_ERROR, "cannot start PTT output\n");
        /* open STP */
        getparam(doc, ptt, stpparams, par);
        if (stp_init(par))
                logprintf(MLOG_INFO, "STP transmission disabled\n");
	/* open audio */
	getparam(doc, audio, ioparams_filein, par);
	if (par[0]) {
		if (!(state->audio = ioopen_filein(&samplerate, IO_RDONLY, par)))
			logprintf(MLOG_FATAL, "cannot start audio file input\n");
		if (schedrr)
			*schedrr = 0;
	} else {
		getparam(doc, audio, ioparams_sim, par);
		if (par[0]) {
			if (!(state->audio = ioopen_sim(&samplerate, IO_RDWR, par)))
				logprintf(MLOG_FATAL, "cannot start audio simulation\n");
			if (schedrr)
				*schedrr = 0;
		} else {
			getparam(doc, audio, ioparams_soundcard, par);
			if (!(state->audio = ioopen_soundcard(&samplerate, mode, par)))
				logprintf(MLOG_FATAL, "cannot start audio\n");
		}
	}
	for (chan = state->channels; chan; chan = chan->next) {
		if (chan->demod) {
			chan->demod->init(chan->demodstate, samplerate, &chan->rxbitrate);
                        if (pthread_attr_init(&rxattr))
				logerr(MLOG_FATAL, "pthread_attr_init");
#ifdef HAVE_SCHED_H
                        if (schedrr && *schedrr) {
                                struct sched_param schp;
                                memset(&schp, 0, sizeof(schp));
                                schp.sched_priority = sched_get_priority_min(SCHED_RR)+1;
                                if (pthread_attr_setschedpolicy(&rxattr, SCHED_RR))
                                        logerr(MLOG_ERROR, "pthread_attr_setschedpolicy");
                                if (pthread_attr_setschedparam(&rxattr, &schp))
                                        logerr(MLOG_ERROR, "pthread_attr_setschedparam");
                        }
#endif /* HAVE_SCHED_H */
			if (pthread_create(&chan->rxthread, &rxattr, demodthread, chan))
				logerr(MLOG_FATAL, "pthread_create");
                        pthread_attr_destroy(&rxattr);
		}
		if (chan->mod)
			chan->mod->init(chan->modstate, samplerate);
	}
	return 0;
}

/* ---------------------------------------------------------------------- */

static void parseopts(int argc, char *argv[])
{
	static const struct option long_options[] = {
		{ "config", 1, 0, 'c' },
		{ "syslog", 0, 0, 's' },
		{ "nosimd", 0, 0, 'S' },
		{ "daemonize", 0, 0, 'D' },
		{ 0, 0, 0, 0 }
	};
	char *configname = NULL, *cfgname, *filename = "/etc/ax25/soundmodem.conf";
	unsigned int verblevel = 2, tosyslog = 0, simd = 1, schedrr = 0, lockmem = 0, daemonize = 0;
        int c, err = 0;
	xmlDocPtr doc;
	xmlNodePtr node;
        int pfds[2];
	pid_t pid;
	unsigned char uch;	

        while ((c = getopt_long(argc, argv, "v:sSc:RMD", long_options, NULL)) != EOF) {
                switch (c) {
                case 'v':
                        verblevel = strtoul(optarg, NULL, 0);
                        break;

		case 's':
			tosyslog = 1;
			break;

		case 'S':
			simd = 0;
			break;

		case 'c':
			configname = optarg;
			break;

                case 'R':
                        schedrr = 1;
                        break;

                case 'M':
                        lockmem = 1;
                        break;
                        
		case 'D':
			daemonize = 1;
			break;

                default:
                        err++;
                        break;
                }
        }
	if (err) {
                fprintf(stderr, "usage: [-v <verblevel>] [-s] [-S] [-R] [-M] [-c <configname>] <configfile>\n");
                exit(1);
        }
	loginit(verblevel, tosyslog);
	if (daemonize) {
		if (pipe(pfds))
			logerr(MLOG_FATAL, "pipe");
		switch (pid = fork()) {
		case -1:
			logerr(MLOG_FATAL, "fork");

		case 0: /* child process */
			close(pfds[0]);
			setsid(); /* become a process group leader and drop controlling terminal */
			fclose(stdin); /* no more standard in */
			break;
			
		default: /* parent process */
			close(pfds[1]);
			err = read(pfds[0], &uch, sizeof(uch));
			if (err != sizeof(uch))
				logprintf(MLOG_FATAL, "SoundModem init failed\n");
			exit(0);
		}
        }
	initsimd(simd);
#if 0
	if (optind >= argc)
		logprintf(MLOG_FATAL, "no configuration file specified\n");
#endif
        if (optind < argc)
                filename = argv[optind];
	doc = xmlParseFile(filename);
	if (!doc || !doc->root || !doc->root->name)
		logprintf(MLOG_FATAL, "Error parsing config file \"%s\"\n", filename);
	if (strcmp(doc->root->name, "modem"))
		logprintf(MLOG_FATAL, "Config file does not contain modem data\n");
	for (node = doc->root->childs; node; node = node->next) {
		if (!node->name || strcmp(node->name, "configuration"))
			continue;
		if (!configname)
			break;
		cfgname = xmlGetProp(node, "name");
		if (cfgname && !strcmp(cfgname, configname))
			break;
	}
	if (!node)
		logprintf(MLOG_FATAL, "Configuartion not found\n");
	if (!node->childs)
		logprintf(MLOG_FATAL, "Configuration empty\n");
	err = parsecfg(doc, node->childs, &state, schedrr);
	xmlFreeDoc(doc);
	if (err)
		exit(1);
        /*
         * lock memory down
         */
        if (lockmem) {
                if (mlockall(MCL_CURRENT | MCL_FUTURE) == -1) 
                        logerr(MLOG_ERROR, "mlockall");
        }
#ifdef HAVE_SCHED_H
        if (schedrr) {
                struct sched_param schp;
                memset(&schp, 0, sizeof(schp));
                schp.sched_priority = sched_get_priority_min(SCHED_RR)+1;
                if (sched_setscheduler(0, SCHED_RR, &schp) != 0)
                        logerr(MLOG_ERROR, "sched_setscheduler");
        }
#endif /* HAVE_SCHED_H */
	if (daemonize) {
		uch = 0;
		if (write(pfds[1], &uch, sizeof(uch)) != sizeof(uch))
                        logerr(MLOG_ERROR, "write");
                close(pfds[1]);
	}
}

/* ---------------------------------------------------------------------- */

static unsigned int terminate = 0;

static RETSIGTYPE sigusr1()
{
	logprintf(MLOG_INFO, "SIGUSR1\n");
}

static RETSIGTYPE sigterm()
{
        terminate = 1;
}

void pkttransmitloop(struct state *state)
{
        signal(SIGUSR1, sigusr1);
        signal(SIGHUP, sigterm);
        while (!terminate) {
		usleep(250000);
	}
}

/* ---------------------------------------------------------------------- */

struct modulator *modchain = &p3dmodulator;
struct demodulator *demodchain = &p3ddemodulator;

int main(int argc, char *argv[])
{
        ioinit_filein();
	ioinit_soundcard();
	ioinit_sim();
       	parseopts(argc, argv);
	pkttransmitloop(&state);
	exit(0);
}
