/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_model.cpp 
 * @brief Implementation of the model that manages the visual representation in the generic canvas.
 * 
 */

#ifdef _WINDOWS
  #define  WIN32_LEAN_AND_MEAN
  #include <windows.h>
  #include <shlobj.h> // Font folder location
#else // !_WINDOWS
  #include <X11/Xlib.h>
  #include <GL/glx.h>
#endif // !_WINDOWS

#include <assert.h>
#include <GL/gl.h>

#include "myx_gc_font_manager.h"
#include "myx_gc_utilities.h"

//----------------------------------------------------------------------------------------------------------------------

// Converts the given string into a font weight value.
// Allowed values are: normal | bold | bolder | lighter | 100 | 200 | 300| 400 | 500 | 600 | 700 | 800 | 900 | inherit
int ConvertFontWeight(const string Weight)
{
  int Result = 0;
  if (Weight == "normal")
    Result = 400;
  else
    if (Weight == "bold")
      Result = 700;
    else
      if (Weight == "bolder")
        Result = 800;
      else
        if (Weight == "lighter")
          Result = 300;
        else
          if (Weight == "inherit")
            Result = 400;
          else
            Result = atoi(Weight.c_str());


  return Result;
}

//----------------- CFontManager ---------------------------------------------------------------------------------------

static private CFontManager* InternalManager;
static private LockCount = 0;

// Returns the current font manager (there is always only one).
CFontManager* FontManager(void)
{
  return InternalManager;
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * Increases the lock count of the font manager. If the manager does not yet exist it is created.
 *
 */
void LockFontManager(void)
{
  if (InternalManager == NULL)
    InternalManager = new CFontManager();
  LockCount++;
}

//----------------------------------------------------------------------------------------------------------------------

// Returns the current font manager (there is always only one).
void UnlockFontManager(void)
{
  if (LockCount > 0)
  {
    LockCount--;
    delete InternalManager;
    InternalManager = NULL;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Constructor of the class.
 */
CFontManager::CFontManager(void)
{
  // Fill our font file list with some predefined font files.
  FontFileEntry* Entry;

  // "Arial" normal, bold, italic, bold italic.
  Entry = new FontFileEntry;
  Entry->Entries[false][false] = "Arial.ttf"; Entry->Entries[false][true] = "Arialbd.ttf"; 
  Entry->Entries[true][false] = "Ariali.ttf";  Entry->Entries[true][true] = "Arialbi.ttf"; 
  Entry->UseCount = 5;
  FFiles["Arial"] = Entry;
  FFiles["Arial Normal"] = Entry;
  FFiles["Arial-Normal"] = Entry;
  FFiles["Arial Standard"] = Entry;
  FFiles["Arial-Standard"] = Entry;

  // Courier New
  Entry = new FontFileEntry;
  Entry->Entries[false][false] = "Cour.ttf"; Entry->Entries[false][true] = "Courbd.ttf"; 
  Entry->Entries[true][false] = "Couri.ttf";  Entry->Entries[true][true] = "Courbi.ttf"; 
  Entry->UseCount = 3;
  FFiles["Courier New"] = Entry;
  FFiles["Courier New Standard"] = Entry;
  FFiles["Courier"] = Entry;

  // Garamond
  Entry = new FontFileEntry;
  Entry->Entries[false][false] = "Gara.ttf"; Entry->Entries[false][true] = "Garabd.ttf"; 
  Entry->Entries[true][false] = "Garait.ttf";  Entry->Entries[true][true] = "Garabi.ttf";
  Entry->UseCount = 3;
  FFiles["Garamond"] = Entry;
  FFiles["Garamond Standard"] = Entry;
  FFiles["Garamond-Standard"] = Entry;

  // Palatino Linotype
  Entry = new FontFileEntry;
  Entry->Entries[false][false] = "Pala.ttf"; Entry->Entries[false][true] = "Palab.ttf"; 
  Entry->Entries[true][false] = "Palai.ttf";  Entry->Entries[true][true] = "Palabi.ttf";
  Entry->UseCount = 1;
  FFiles["Palation Linotype"] = Entry;

  // Tahoma
  Entry = new FontFileEntry;
  Entry->Entries[false][false] = "Tahoma.ttf"; Entry->Entries[false][true] = "Tahomabd.ttf"; 
  Entry->Entries[true][false] = "Tahomai.ttf";  Entry->Entries[true][true] = "Tahomabi.ttf";
  Entry->UseCount = 1;
  FFiles["Tahoma"] = Entry;

  // Tahoma-Bold, not an own font but Adobe Illustrator denotes it so.
  Entry = new FontFileEntry;
  Entry->Entries[false][false] = "Tahomabd.ttf"; Entry->Entries[false][true] = "Tahomabd.ttf"; 
  Entry->Entries[true][false] = "Tahomabdi.ttf";  Entry->Entries[true][true] = "Tahomabdi.ttf";
  Entry->UseCount = 1;
  FFiles["Tahoma-Bold"] = Entry;

  // Times New Roman
  Entry = new FontFileEntry;
  Entry->Entries[false][false] = "Times.ttf"; Entry->Entries[false][true] = "Timesbd.ttf"; 
  Entry->Entries[true][false] = "Timesi.ttf";  Entry->Entries[true][true] = "Timesbi.ttf";
  Entry->UseCount = 2;
  FFiles["Times New Roman"] = Entry;
  FFiles["Times"] = Entry;

  // Trebuchet MS
  Entry = new FontFileEntry;
  Entry->Entries[false][false] = "Trebuc.ttf"; Entry->Entries[false][true] = "Trebucbd.ttf"; 
  Entry->Entries[true][false] = "Trebucit.ttf";  Entry->Entries[true][true] = "Trebucbi.ttf";
  Entry->UseCount = 2;
  FFiles["Trebuchet MS"] = Entry;
  FFiles["Trebuchet"] = Entry;

  // Verdana
  Entry = new FontFileEntry;
  Entry->Entries[false][false] = "Verdana.ttf"; Entry->Entries[false][true] = "Verdanab.ttf"; 
  Entry->Entries[true][false] = "Verdanai.ttf";  Entry->Entries[true][true] = "Verdanaz.ttf";
  Entry->UseCount = 1;
  FFiles["Verdana"] = Entry;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Destructor of the class. Does some clean up.
 */
CFontManager::~CFontManager(void)
{
  Clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines platform dependantly the full path of a font file depending on some characteristics.
 *
 * @param Family The font family (e.g. Arial, Tahoma, Verdana).
 * @param Style The font style (normal, italic).
 * @param Weight The "boldness" of the font in the range of [100..900]. Currently values <= 400 are
 *               considered als normal font, everything else as bold.
 *
 * @return The full path and file name of a font that represents the given characteristics.
 *          A kind of intelligent replacement is done here, though. If there is no file with the given characteristics 
 *          (or cannot be found) then the one from the same family, but normal styles, is used instead. If there is no 
 *          entry for the family or even the standard style for a family cannot be found then Arial Standard is returned as 
 *          default. If this files is also not present on the system then the FTGL lib will throw an exception.
 * @note The returned file name is ANSI encoded as FTGL expects it so.
 */
string CFontManager::GetFontFile(string Family, string Style, int Weight)
{
  string Filename;
  FontFileEntry* Entry;

  // Adobe illustrator (and probably others too) put the family name in quotes.
  if (Family[0] == '"' || Family[0] == '\'')
    Family = Family.substr(1, Family.size() - 2);
  FontFiles::iterator Iterator = FFiles.find(Family);
  if (Iterator == FFiles.end())
  {
    // Family is unknown, use Arial as default.
    Entry = FFiles["Arial"];
  }
  else
  {
    Entry = Iterator->second;
  };

  bool Italic = (Style == "italic") || (Style == "oblique");
  bool Bold = Weight > 400;

  // Platform specific stuff to find the font folder of the system.
  #ifdef _WINDOWS
    LPITEMIDLIST pidl;
    wchar_t Path[MAX_PATH];
    if (SHGetSpecialFolderLocation(0, CSIDL_FONTS, &pidl) == NOERROR)
    {
      if (SHGetPathFromIDList(pidl, Path))
        Filename = Utf16ToANSI(Path);

      // We are required to free the returned pidl via the shell's IMalloc interface.
      LPMALLOC pMalloc;
      SHGetMalloc(&pMalloc);
      pMalloc->Free(pidl);
    };
  #else
    #error Write code to find the font folder.
  #endif // #fidef _WINDOWS

  string FontFile = Entry->Entries[Italic][Bold];
  string Testfile = Filename + '/' + FontFile;
  FILE* File = fopen(Testfile.c_str(), "rb");
  if (File != NULL)
  {
    fclose(File);
    Filename = Testfile;
  }
  else
  {
    // There is no file for the required style. Try the base entry.
    FontFile = Entry->Entries[false][false];
    Testfile = Filename + '/' + FontFile;
    File = fopen(Testfile.c_str(), "rb");
    if (File != NULL)
    {
      fclose(File);
      Filename = Testfile;
    }
    {
      // That's bad luck. No standard file either. Use the default font.
      Entry = FFiles["Arial"];
      Filename += '/' + Entry->Entries[false][false];
    };
  };

  return Filename;
}

//----------------------------------------------------------------------------------------------------------------------

void CFontManager::BoundingBox(const wstring& Output, string FontFamily, string FontStyle, int Weight, int FontSize, 
                               TBoundingBox* Box)
{
  FTFont* Font;
  string Key = CreateLookupKey(FontFamily, FontStyle, Weight);
  Fonts::iterator Iterator = FFonts.find(Key);

  if (Iterator == FFonts.end())
  {
    string FontFile = GetFontFile(FontFamily, FontStyle, Weight);
    Font = new FTGLPixmapFont(FontFile.c_str());
    Font->UseDisplayList(true);
    FFonts[Key] = Font;
  }
  else
    Font = Iterator->second;

  if (Font->FaceSize() != FontSize)
    Font->FaceSize(FontSize);

  Font->BBox(Output.c_str(), Box->UpperLeftX, Box->UpperLeftY, Box->UpperLeftZ, Box->LowerRightX, Box->LowerRightY, 
    Box->LowerRightZ);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Clears the font file list.
 */
void CFontManager::Clear(void)
{
  for (Fonts::iterator Iterator = FFonts.begin(); Iterator != FFonts.end(); Iterator++)
  {
    FTFont* Font = Iterator->second;
    delete Font;
  };
  FFonts.clear();

  for (FontFiles::iterator Iterator = FFiles.begin(); Iterator != FFiles.end(); Iterator++)
  {
    FontFileEntry* Entry = Iterator->second;
    if (--Entry->UseCount == 0)
      delete Entry;
  };
  FFiles.clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a string that can be used for lookup in the font list.
 * Either parameter can be NULL (or 0 for Weight) causing the manager to use the default values for each missing 
 * parameter. See header file for the list of default values.
 *
 * @param Family The font family (Arial, Courier etc.)
 * @param Style The font style.
 * @param Weight The boldness of the font (400 for normal).
 */
string CFontManager::CreateLookupKey(const string& Family, const string& Style, int Weight)
{
  char WeightString[30];

  // Construct a lookup string out of the font properties.
  string Key = Family + Style;
  #ifdef _WINDOWS
    // There is no null terminator written to the string so add one explicitely.
    _snprintf(WeightString, sizeof(WeightString), "%i\0", Weight);
  #else
    snprintf(WeightString, sizeof(WeightString), "%i\0", Weight);
  #endif // #ifdef _WINDOWS
  Key += WeightString;

  return Key;
}

//----------------------------------------------------------------------------------------------------------------------

void CFontManager::TextOut(const wstring& Output, string FontFamily, string FontStyle, int Weight, int FontSize)
{
  FTFont* Font;
  string Key = CreateLookupKey(FontFamily, FontStyle, Weight);
  Fonts::iterator Iterator = FFonts.find(Key);

  if (Iterator == FFonts.end())
  {
    string FontFile = GetFontFile(FontFamily, FontStyle, Weight);
    Font = new FTGLPixmapFont(FontFile.c_str());
    Font->UseDisplayList(true);
    
    FFonts[Key] = Font;
  }
  else
    Font = Iterator->second;

  if (Font->FaceSize() != FontSize)
    Font->FaceSize(FontSize);
  Font->Render(Output.c_str());
}

//----------------------------------------------------------------------------------------------------------------------

