/* nawm.c: startup and general-purpose routines */

/* Copyright (C) 1999 by the Massachusetts Institute of Technology.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in
 * advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 * M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <signal.h>
#include "nawm.h"
#include "lang.h"
#include "builtins.h"
#include "cache.h"

void usage(void);
int x_error(Display *, XErrorEvent *);
void turn_undead(void);
void nawm_set_window_manager(char *);

Display *dpy;
Window root;
int screen, screenwidth, screenheight, error_mode = SPEW_ON_ERROR;
int window_manager;
extern node *cmds;
extern int yydebug, cachedebug;
extern variable CURRENTWINDOW, ROOT, SCREENHEIGHT, SCREENWIDTH;
char **Argv;
long options;

int main(int argc, char **argv)
{
  extern char *optarg;
  char *home;
  int opt, readsomething = 0;
  struct sigaction sa;
  nawmval pw;

  if (!(dpy = XOpenDisplay(NULL)))
    {
      fprintf(stderr, "nawm: can't open display\n");
      exit(1);
    }

  Argv = argv;
  screen = DefaultScreen(dpy);
  assign_var(&ROOT, (nawmval)(root = RootWindow(dpy, screen)));
  assign_var(&SCREENWIDTH,
	     (nawmval)(screenwidth = DisplayWidth(dpy, screen)));
  assign_var(&SCREENHEIGHT,
	     (nawmval)(screenheight = DisplayHeight(dpy, screen)));
  XSetErrorHandler(x_error);

  sigemptyset(&sa.sa_mask);
  sa.sa_flags = 0;
  sa.sa_handler = turn_undead;
  sigaction(SIGCHLD, &sa, NULL);

  initgc();
  inittypes();
  initbindings();
  initcache();
  initbuiltins();
  initkeymap();
  initmodules();
  initlex();
  initrunscopes();

  /* Can't run pointerwindow_fun until after initcache() */
  pointerwindow_fun(0, NULL, &pw);
  assign_var(&CURRENTWINDOW, pw);

  if (getenv("WINDOW_MANAGER"))
    {
      char *wm = getenv("WINDOW_MANAGER");
      if (strchr(wm, '/'))
	wm = strrchr(wm, '/') + 1;
      
      nawm_set_window_manager(wm);
    }

  while ((opt = getopt(argc, argv, "d:e:f:i:o:w:")) != -1)
    {
      switch(opt)
	{
	case 'd':
	  if (strchr(optarg, 'c'))
	    {
	      sa.sa_handler = dumpcache;
	      sigaction(SIGINT, &sa, NULL);
	      cachedebug = 1;
	    }
	  if (strchr(optarg, 'y'))
	    yydebug = 1;
	  break;

	case 'e':
	  parse_string(optarg);
	  eval_cmds(cmds);
	  readsomething = 1;
	  break;

	case 'f':
	  parse_file(optarg);
	  readsomething = 1;
	  break;

	case 'i':
	  load_module(optarg);
	  break;

	case 'o':
	  do_option(optarg);
	  break;

	case 'w':
	  nawm_set_window_manager(optarg);
	  break;

	default:
	  usage();
	}
    }

  if (argc != optind)
    usage();

  if (!readsomething && (home = getenv("HOME")))
    {
      char file[MAXPATHLEN];
      sprintf(file, "%s/.nawmrc", home);
      parse_file(file);
    }

  run();
  XCloseDisplay(dpy);

  return 0;
}

void nawm_set_window_manager(char *wm)
{
  if (!strcmp(wm, "nawm"))
    window_manager = NAWM_WM_NAWM;
  else if (!strcmp(wm, "vtwm.gamma"))
    window_manager = NAWM_WM_VTWM_GAMMA;
  else
    window_manager = NAWM_WM_UNKNOWN;
}

void do_option(char *opt)
{
  if (!strcmp(opt, "nocapslock"))
    options |= NAWM_OPT_NOCAPSLOCK;
  else
    die("Unrecognized option \"%s\"", opt);
}

void usage(void)
{
  die("Usage: nawm [-i module] [-o option] [-w windowmanager]\n"
      "            [-f configfile] [-e commands] ...");
}

int x_error(Display *dpy, XErrorEvent *xee)
{
  if (xee->error_code == BadWindow)
    return 0;

  if (error_mode == BEEP_ON_ERROR)
    XBell(dpy, 100);
  else if(error_mode == SPEW_ON_ERROR)
    {
      char tmp[80];
      XGetErrorText(dpy, xee->error_code, tmp, 80);
      fprintf(stderr, "nawm X error: %s\n", tmp);
    }
  return 0;
}

void die(char *format, ...)
{
  va_list ap;

  va_start(ap, format);
  vfprintf(stderr, format, ap);
  va_end(ap);
  fprintf(stderr, "\n");
  exit(1);
}

void turn_undead(void)
{
  while (waitpid(-1, NULL, WNOHANG) > 0)
    ;
}
