/*
 *  Copyright (C) 2003  David A Knight
 *  
 *  based on ephy-shell.c,
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "neutrino-shell.h"
#include "neutrino-window.h"
#include "nomad-jukebox.h"

#include <libnjb.h>

#include <bonobo/bonobo-main.h>

#include <libgnome/gnome-macros.h>

#include <gtk/gtkwidget.h>
#include <gtk/gtkiconfactory.h>
#include <gtk/gtkstock.h>
     
GNOME_CLASS_BOILERPLATE( NeutrinoShell, neutrino_shell,
			GObject,
			G_TYPE_OBJECT )

struct NeutrinoShellPrivate
{
	GList *windows;
	GList *jukeboxes;

	njb_t njbs[ NJB_MAX_DEVICES ];
};

static void neutrino_shell_class_init( NeutrinoShellClass *klass );
static void neutrino_shell_instance_init( NeutrinoShell *shell );
static void neutrino_shell_finalize( GObject *obj );

static void neutrino_shell_add_jukebox( NeutrinoShell *shell,
					NomadJukebox *jukebox );
static void neutrino_shell_add_icons( NeutrinoShell *shell );
static gboolean neutrino_window_delete_cb( GtkWidget *widget,
					   GdkEvent *event,
					   NeutrinoShell *shell );
static void neutrino_window_destroy_cb( GObject *obj,
					NeutrinoShell *shell );

NeutrinoShell *neutrino_shell;

static void neutrino_shell_class_init( NeutrinoShellClass *klass )
{
	GObjectClass *obj_class;

	obj_class = G_OBJECT_CLASS( klass );

	obj_class->finalize = neutrino_shell_finalize;
}

static void neutrino_shell_instance_init( NeutrinoShell *shell )
{
	NeutrinoShell **ptr = &neutrino_shell;

	shell->priv = g_new0( NeutrinoShellPrivate, 1 );

	neutrino_shell = shell;
	g_object_add_weak_pointer( G_OBJECT( neutrino_shell ),
				  (gpointer *)ptr );
	
	neutrino_shell_add_icons( shell );	
	neutrino_shell_scan_jukeboxes( shell );
}

static void neutrino_shell_finalize( GObject *obj )
{
	NeutrinoShell *shell;
	NeutrinoShellPrivate *priv;
	GList *list;
	
	g_return_if_fail( obj != NULL );
	g_return_if_fail( NEUTRINO_IS_SHELL( obj ) );

	shell = NEUTRINO_SHELL( obj );

	g_return_if_fail( shell->priv != NULL );

	g_assert( neutrino_shell == NULL );

	priv = shell->priv;
	
	for( list = priv->jukeboxes; list; list = list->next ) {

		nomad_jukebox_release( NOMAD_JUKEBOX( list->data ) );
		
		g_object_unref( G_OBJECT( list->data ) );
	}

	
	GNOME_CALL_PARENT( G_OBJECT_CLASS, finalize, (obj) );
	
	g_free( priv );

	bonobo_main_quit();
}

static void neutrino_shell_add_jukebox( NeutrinoShell *shell,
					NomadJukebox *jukebox )
{
	NeutrinoShellPrivate *priv;

	g_return_if_fail( NEUTRINO_IS_SHELL( shell ) );
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );

	priv = shell->priv;

	priv->jukeboxes = g_list_prepend( priv->jukeboxes,
					  jukebox );
}

static void neutrino_shell_add_icons( NeutrinoShell *shell )
{
	static struct NeutrinoStockPixmap {
		guchar const *stock_id;
		guchar const *filename;
	} const icons[] = {
		{ "Neutrino_Play", "rhythmbox-play.png" },
		{ "Neutrino_Stop", "rhythmbox-stop.png" },
		{ "Neutrino_Next", "rhythmbox-next.png" },
		{ "Neutrino_Prev", "rhythmbox-previous.png" },
		{ "Neutrino_Shuffle", "rhythmbox-shuffle.png" },
		{ "Neutrino_Repeat", "rhythmbox-repeat.png" },
		{ "Neutrino_Library", "rhythmbox-library.png" },
		{ "Neutrino_Playlist", "rhythmbox-group.png" },
		{ "Neutrino_Local", "i-harddisk.png" },
		{ NULL, NULL }
	};
	static const GtkStockItem stockItems[] = {
		{ "Neutrino_Play", "Play", 0, 0, 0 },
		{ "Neutrino_Stop", "Stop", 0, 0, 0 },
		{ "Neutrino_Next", "Next", 0, 0, 0 },
		{ "Neutrino_Prev", "Previous", 0, 0, 0 },
		{ "Neutrino_Suffle", "Suffle", 0, 0, 0 },
		{ "Neutrino_Repeat", "Repeat", 0, 0, 0 },
		{ "Neutrino_Library", "Library", 0, 0, 0 },
		{ "Neutrino_Playlist", "Playlist", 0, 0, 0 },
		{ "Neutrino_Local", "Local", 0, 0, 0 }
	};

	GtkIconFactory *factory;
	gint i;

	factory = gtk_icon_factory_new();

	for( i = 0; icons[ i ].stock_id; ++ i ) {
		GtkIconSet *set;
		GtkIconSource *src;
		GdkPixbuf *pixbuf;
		gchar *filename;

		set = gtk_icon_set_new();
		src = gtk_icon_source_new();

		filename = g_strconcat( PIXMAP_PATH, G_DIR_SEPARATOR_S,
					icons[ i ].filename, NULL );
		pixbuf = gdk_pixbuf_new_from_file( filename, NULL );
		gtk_icon_source_set_pixbuf( src, pixbuf );

                gtk_icon_source_set_direction_wildcarded( src, TRUE );
                gtk_icon_source_set_state_wildcarded( src, TRUE );
                gtk_icon_source_set_size_wildcarded( src, TRUE );
                gtk_icon_source_set_size( src, GTK_ICON_SIZE_BUTTON );
                gtk_icon_set_add_source( set, src );

                gtk_icon_factory_add( factory, icons[ i ].stock_id, set );

                g_free( filename );

                g_object_unref( G_OBJECT( pixbuf ) );
                gtk_icon_set_unref( set );
                gtk_icon_source_free( src );
	}
        gtk_icon_factory_add_default( factory );

        gtk_stock_add( stockItems, G_N_ELEMENTS( stockItems ) );

        g_object_unref( G_OBJECT( factory ) );
}


NeutrinoShell *neutrino_shell_new( void )
{
	return NEUTRINO_SHELL( g_object_new( NEUTRINO_SHELL_TYPE, NULL ) );
}


gboolean neutrino_shell_scan_jukeboxes( NeutrinoShell *shell )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;
	guint number;
	guint boxes;
	GList *list;

	g_return_val_if_fail( NEUTRINO_IS_SHELL( shell ), FALSE );

	/* release all current jukeboxes, and destroy them,
	 * also requires we tell all windows they have no
	 * jukebox */
	for( list = neutrino_shell_get_window_list( shell );
			list; list = list->next ) {
		window = NEUTRINO_WINDOW( list->data );

		neutrino_window_set_jukebox( window, NULL );
	}
	number = neutrino_shell_get_jukeboxes( shell );
	while( number > 0 ) {
		jukebox = neutrino_shell_get_nth_jukebox( shell,
							number );

		nomad_jukebox_release( jukebox );

		g_object_unref( jukebox );
	}

	boxes = 0;
	NJB_Set_Unicode( NJB_UC_UTF8 );
	if( NJB_Discover( shell->priv->njbs, 0, &number ) == -1 ) {

		return FALSE;
	}

	/* aquire all jukeboxes connected to the system */
	for( boxes = number; number > 0; -- number ) {
		jukebox = nomad_jukebox_new( &shell->priv->njbs[ number - 1 ] );

		if( nomad_jukebox_aquire( jukebox ) ) {

			neutrino_shell_add_jukebox( shell, jukebox );
			
		} else {

			/* failed to aquire this jukebox */
			boxes --;			
		}
	}

	/* each open window now needs to be given a jukebox,
	 * we assign the first jukebox to all windows */
	jukebox = NULL;
	if( boxes > 0 ) {
		jukebox = neutrino_shell_get_nth_jukebox( shell, 0 );
	}
	for( list = neutrino_shell_get_window_list( shell );
			list; list = list->next ) {
		window = NEUTRINO_WINDOW( list->data );

		neutrino_window_scan_jukeboxes( window );
		neutrino_window_set_jukebox( window, jukebox );
	}

	/* now begin scanning the jukeboxes for tracks / playlists */
	for( number = 0; number < boxes; ++ number ) {
		jukebox = neutrino_shell_get_nth_jukebox( shell,
							number );

		nomad_jukebox_getusage( jukebox );
		nomad_jukebox_build_tracklist( jukebox );
		nomad_jukebox_build_playlist( jukebox );
	}

	return ( boxes != 0 );
}


guint neutrino_shell_get_jukeboxes( NeutrinoShell *shell )
{
	g_return_val_if_fail( NEUTRINO_IS_SHELL( shell ),
				0 );

	return g_list_length( shell->priv->jukeboxes );
}

NomadJukebox *neutrino_shell_get_nth_jukebox( NeutrinoShell *shell,
						 guint index )
{
	NomadJukebox *box;
	
	g_return_val_if_fail( NEUTRINO_IS_SHELL( shell ), NULL );
	g_return_val_if_fail( index < neutrino_shell_get_jukeboxes( shell ), NULL );
	
	
	box = NOMAD_JUKEBOX( g_list_nth( shell->priv->jukeboxes,
					index )->data );
	
	return box;
}

NomadJukebox *neutrino_shell_get_jukebox_by_id( NeutrinoShell *shell,
						const gchar *id )
{
	NeutrinoShellPrivate *priv;
	NomadJukebox *ret;
	NomadJukebox *box;
	GList *list;
	const gchar *jid;

	g_return_val_if_fail( NEUTRINO_IS_SHELL( shell ), NULL );
	g_return_val_if_fail( id != NULL, NULL );

	priv = shell->priv;
	ret = NULL;
	for( list = priv->jukeboxes; list; list = list->next ) {

		box = NOMAD_JUKEBOX( list->data );

		jid = nomad_jukebox_get_idstring( box );

		if( ! strcmp( jid, id ) ) {
			ret = box;
			break;
		}
	}
	
	return ret;
}

GList *neutrino_shell_get_window_list( NeutrinoShell *shell )
{
	g_return_val_if_fail( NEUTRINO_IS_SHELL( shell ), NULL );

	return shell->priv->windows;
}

void neutrino_shell_close_all_windows( NeutrinoShell *shell )
{
	NeutrinoShellPrivate *priv;
	g_return_if_fail( NEUTRINO_IS_SHELL( shell ) );

	priv = shell->priv;
	while( priv->windows && priv->windows->data ) {
		neutrino_window_close( NEUTRINO_WINDOW( priv->windows->data ) );
	}
}
	
NeutrinoWindow *neutrino_shell_create_window( NeutrinoShell *shell )
{
	NeutrinoWindow *window;
	
	g_return_val_if_fail( NEUTRINO_IS_SHELL( shell ), NULL );

	g_object_ref( shell );

	window = NEUTRINO_WINDOW( g_object_new( NEUTRINO_TYPE_WINDOW,
						"shell", shell,
						NULL ) );
	
	g_signal_connect( G_OBJECT( window ), "delete_event",
			  G_CALLBACK( neutrino_window_delete_cb ),
			  shell );
	g_signal_connect( G_OBJECT( window ), "destroy",
			  G_CALLBACK( neutrino_window_destroy_cb ),
			  shell );

	shell->priv->windows = g_list_prepend( shell->priv->windows,
						window );

	return window;
}

static gboolean neutrino_window_delete_cb( GtkWidget *widget,
					   GdkEvent *event,
					   NeutrinoShell *shell )
{
	NeutrinoWindow *window;

	if( NEUTRINO_IS_WINDOW( widget ) ) {
	
		window = NEUTRINO_WINDOW( widget );

		gdk_threads_leave();
		neutrino_window_close( window );
		gdk_threads_enter();
	}

	return TRUE;
}

static void neutrino_window_destroy_cb( GObject *obj,
					NeutrinoShell *shell )
{
	NeutrinoShellPrivate *priv;

	priv = shell->priv;

	priv->windows = g_list_remove( priv->windows, obj );
	
	g_object_unref( shell );
}
