/*  Neutrino:  neutrino-window-menus.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <gconf/gconf-client.h>

#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-about.h>
#include <libgnomeui/gnome-stock-icons.h>

#include "neutrino-shell.h"
#include "neutrino-window.h"
#include "neutrino-window-private.h"

static void add_widget( EggMenuMerge *merge, GtkWidget *widget,
			NeutrinoWindow *window )
{
	if( GTK_IS_MENU_SHELL( widget ) ) {
		gtk_box_pack_start( GTK_BOX( window->content_hbox ),
				    widget, FALSE, TRUE, 0 );
		gtk_box_reorder_child( GTK_BOX( window->content_hbox ),
					widget, 0 );
	} else if( EGG_IS_TOOLBAR( widget ) ) {
		guint toolbars;

		toolbars = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( merge ), "toolbars" ) );
		++ toolbars;

		egg_toolbar_set_show_arrow( EGG_TOOLBAR( widget ),
						TRUE );
		egg_toolbar_set_tooltips( EGG_TOOLBAR( widget ),
						TRUE );
		gtk_box_pack_start( GTK_BOX( window->content_hbox ),
				widget, FALSE, TRUE, 0 );
		gtk_box_reorder_child( GTK_BOX( window->content_hbox ),
					widget,
					toolbars );
		g_object_set_data( G_OBJECT( merge ), "toolbars",
				GUINT_TO_POINTER( toolbars ) );
	}
}

static void
file_menu_close_window_callback( EggAction *action,
				 gpointer user_data )
{
	neutrino_window_close( NEUTRINO_WINDOW( user_data ) );
}

static void
file_menu_exit_callback( EggAction *action, gpointer user_data )
{
	NeutrinoShell *app;
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );
	app = window->shell;

	gdk_threads_leave();
	neutrino_shell_close_all_windows( app );
	gdk_threads_enter();
}


static void
edit_menu_eax_callback( EggAction *action, gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_edit_eax( window );
}

static void
edit_menu_preferences_callback( EggAction *action, gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_edit_preferences( window );
}

static void
edit_menu_selectall_callback( EggAction *action, gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_select_all( window );
}

static void
view_menu_sidebar_callback( EggAction *action, gpointer user_data )
{	
	GConfClient *client;
	gboolean state;

	state = egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) );

	client = gconf_client_get_default();

	gconf_client_set_bool( client, "/apps/neutrino/ui/show_sidebar",
			       state, NULL );
	g_object_unref( client );
}

static void
view_menu_browser_callback( EggAction *action, gpointer user_data )
{	
	GConfClient *client;
	gboolean state;

	state = egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) );

	client = gconf_client_get_default();

	gconf_client_set_bool( client, "/apps/neutrino/ui/show_browser",
			       state, NULL );
	g_object_unref( client );
}

static void
view_menu_playback_callback( EggAction *action, gpointer user_data )
{	
	GConfClient *client;
	gboolean state;

	state = egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) );

	client = gconf_client_get_default();

	gconf_client_set_bool( client, "/apps/neutrino/ui/show_playback",
			       state, NULL );
	g_object_unref( client );
}

static void
view_menu_toolbar_callback( EggAction *action, gpointer user_data )
{	
	GConfClient *client;
	gboolean state;

	state = egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) );

	client = gconf_client_get_default();

	gconf_client_set_bool( client, "/apps/neutrino/ui/show_toolbar",
			       state, NULL );
	g_object_unref( client );
}

static void
help_menu_about_neutrino_callback( EggAction *action,
				   gpointer user_data )
{
	static GtkWidget *about = NULL;

	const gchar *COPYRIGHT = "(C) 2002, 2003 David A Knight, parts (C) 2002 Linus Walleij";
	const gchar *AUTHORS[] = {
		"David A Knight <david@ritter.demon.co.uk>",
		"Linus Walleij <triad@df.lth.se> - GNOMAD author",
		NULL
	};
	const gchar *DESCRIPTION = "\
Neutrino is a GNOME shell for managing your Creative Nomad/D.A.P Jukebox\n The jukebox code was derived from that of GNOMAD, a GTK+ shell with similar goals.\n\nNeutrino is licensed under the GNU General Public License version 2 or above";
        GdkPixbuf *logo = NULL;
	const gchar *DOCUMENTERS[] = {
		NULL
	};
	const gchar *TRANSLATERS = _( "translater credits" );

	if( ! about ) {
		logo = NULL;
		about = gnome_about_new( PACKAGE, VERSION,
					 COPYRIGHT,
					 DESCRIPTION,
					 AUTHORS,
				 	DOCUMENTERS,
				 	TRANSLATERS,
				 	logo );
        	g_signal_connect_swapped( G_OBJECT( about ), "delete_event",
					  G_CALLBACK( g_nullify_pointer ),
					  &about );
        	g_signal_connect_swapped( G_OBJECT( about ), "destroy",
                	                  G_CALLBACK( g_nullify_pointer ),
					  &about );
        }

	gtk_window_set_transient_for( GTK_WINDOW( about ),
				      GTK_WINDOW( user_data ) );
	gtk_widget_show( about );
	gdk_window_raise( GDK_WINDOW( about->window ) );
}

static void commands_upload_callback( EggAction *action,
				    gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_upload( window );
}

static void commands_download_callback( EggAction *action,
					gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_download( window );
}

static void commands_cancel_callback( EggAction *action,
				      gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_cancel( window );
}


static void commands_delete_callback( EggAction *action,
				      gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_delete( window );
}

static void commands_scan_callback( EggAction *action,
				    gpointer user_data )
{
	NeutrinoWindow *window;
	NeutrinoShell *shell;

	window = NEUTRINO_WINDOW( user_data );
	shell = window->shell;

	gdk_threads_leave();
	neutrino_shell_scan_jukeboxes( shell );
	gdk_threads_enter();
}

static void commands_edit_meta_callback( EggAction *action,
					 gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_edit_meta( window );
}

static void commands_add_to_playlist_callback( EggAction *action,
					       gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_add_to_playlist( window );
}

static void commands_new_playlist_callback( EggAction *action,
					    gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_new_playlist( window );
}

static void commands_delete_playlist_callback( EggAction *action,
					       gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_delete_playlist( window );
}

static void commands_rename_playlist_callback( EggAction *action,
					       gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_rename_playlist( window );
}

static void commands_jukebox_info_callback( EggAction *action,
					    gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_jukebox_info( window );
}

static void commands_playback_callback( EggAction *action,
					gpointer user_data )
{
	neutrino_window_playback( NEUTRINO_WINDOW( user_data ) );
}

static void commands_previous_callback( EggAction *action,
					gpointer user_data )
{
	neutrino_window_play_prev( NEUTRINO_WINDOW( user_data ) );
}

static void commands_stop_callback( EggAction *action,
				    gpointer user_data )
{
	neutrino_window_play_close( NEUTRINO_WINDOW( user_data ) );
}

static void commands_next_callback( EggAction *action,
				    gpointer user_data )
{
	neutrino_window_play_next( NEUTRINO_WINDOW( user_data ) );
}

static void commands_shuffle_callback( EggAction *action,
				       gpointer user_data )
{
	gboolean state;

	state = egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) );
	g_object_set_data( G_OBJECT( user_data ), "shuffle",
			GINT_TO_POINTER( state ) );
}

static void commands_repeat_callback( EggAction *action,
				       gpointer user_data )
{
	gboolean state;

	state = egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) );
	g_object_set_data( G_OBJECT( user_data ), "repeat",
			GINT_TO_POINTER( state ) );

}

static void commands_queue_callback( EggAction *action,
				     gpointer user_data )
{
	neutrino_window_queue( NEUTRINO_WINDOW( user_data ) );
}


static EggActionGroupEntry neutrino_window_action_entries[] = {

  {"File", N_("_File"), NULL, NULL, NULL, NULL, NULL},
  {"Edit", N_("_Edit"), NULL, NULL, NULL, NULL, NULL},
  {"View", N_("_View"), NULL, NULL, NULL, NULL, NULL},
  {"Music", N_("_Music"), NULL, NULL, NULL, NULL, NULL},
  {"Playlists", N_("_Playlists"), NULL, NULL, NULL, NULL, NULL},
  {"Help", N_("_Help"), NULL, NULL, NULL, NULL, NULL},
  {"Customize Interface", N_("_Customize Interface" ), NULL, NULL, NULL, NULL, NULL },
  {"tracks", N_("tracks"), NULL, NULL, NULL, NULL, NULL},


  {"Upload", N_("Transfer _To"),
   GTK_STOCK_GO_FORWARD, NULL,
   N_("Transfer file(s) to jukebox"), 
   G_CALLBACK (commands_upload_callback), NULL},

  {"Download", N_("Transfer _From"),
   GTK_STOCK_GO_BACK, NULL,
   N_("Transfer file(s) from jukebox"), 
   G_CALLBACK (commands_download_callback), NULL},

  {"Cancel", N_("_Cancel Transfer"),
   GTK_STOCK_STOP, NULL,
   N_("Cancel transfer of files"), 
   G_CALLBACK (commands_cancel_callback), NULL},

  {"Quit", N_( "_Quit" ),
   GTK_STOCK_QUIT, "<control>q",
   N_( "Quit Neutrino" ),
   G_CALLBACK (file_menu_exit_callback), NULL},

  {"Preferences", N_( "Preferences" ),
   GTK_STOCK_PREFERENCES, NULL,
   N_( "Edit Preferences" ),
   G_CALLBACK (edit_menu_preferences_callback), NULL},

  {"Delete", N_("_Delete"),
   GTK_STOCK_DELETE, NULL,
   N_("Delete track or playlist from jukebox"), 
   G_CALLBACK (commands_delete_callback), NULL},

  {"Scan", N_("Scan for Jukeboxes"),
   GTK_STOCK_REFRESH, NULL,
   N_("(Re)Scan for attached Jukeboxes"), 
   G_CALLBACK (commands_scan_callback), NULL},

  {"EditMeta", N_("_Edit Track Properties"),
   GTK_STOCK_PROPERTIES, NULL,
   N_
   ("Edit the information for the track(s), e.g. artist, title, album etc."),
   G_CALLBACK (commands_edit_meta_callback), NULL},

  {"NewPlaylist", N_("_New Playlist..."),
   GTK_STOCK_NEW, NULL,
   N_("Create a new playlist"), 
   G_CALLBACK (commands_new_playlist_callback), NULL},

  {"AddToPlaylist", N_("_Add to Playlist"),
   GTK_STOCK_ADD, NULL,
   N_("Add the selected tracks / albums to a playlist"), 
   G_CALLBACK (commands_add_to_playlist_callback),
   NULL},

  {"DeletePlaylist", N_("_Delete Playlist"),
   GTK_STOCK_DELETE, NULL,
   N_("Delete the selected playlist"), 
   G_CALLBACK (commands_delete_playlist_callback), NULL},

  {"RenamePlaylist", N_("_Rename Playlist..."),
   NULL, NULL,
   N_("Rename the selected playlist"), 
   G_CALLBACK (commands_rename_playlist_callback), NULL},

  {"JukeboxInfo", N_("Jukebox Info"),
   GTK_STOCK_DIALOG_INFO, NULL,
   N_("Show Jukebox Information"), 
   G_CALLBACK (commands_jukebox_info_callback), NULL},

  {"EAX", N_("E_AX"),
   GTK_STOCK_PROPERTIES, NULL,
   N_("Edit EAX Settings"), 
   G_CALLBACK (edit_menu_eax_callback), NULL},

  {"playback", N_("Play"),
   "Neutrino_Play", NULL,
   N_("Playback selected tracks / playlists"), 
   G_CALLBACK (commands_playback_callback), NULL},

  {"stop", N_("Stop"),
   "Neutrino_Stop", NULL,
   N_("Stop Playback"), 
   G_CALLBACK (commands_stop_callback), NULL},

  {"prev", N_("Previous"),
   "Neutrino_Prev", NULL,
   N_("Play previous track"), 
   G_CALLBACK (commands_previous_callback), NULL},

  {"next", N_("Next"),
   "Neutrino_Next", NULL,
   N_("Play next track"),
   G_CALLBACK (commands_next_callback), NULL},

  {"Shuffle", N_("Shuffle"),
   "Neutrino_Shuffle", NULL,
   N_("Shuffle Mode"), 
   G_CALLBACK (commands_shuffle_callback), NULL, TOGGLE_ACTION},

  {"Repeat", N_("Repeat"),
   "Neutrino_Repeat", NULL,
   N_("Repeat in non Shuffle Mode"), 
   G_CALLBACK (commands_repeat_callback), NULL, TOGGLE_ACTION},

  {"ViewSidebar", N_("View Sidebar"),
   NULL, NULL,
   N_("Show/Hide the Sidebar"), 
   G_CALLBACK (view_menu_sidebar_callback), NULL, TOGGLE_ACTION},

  {"ViewBrowser", N_("View Browser"),
   NULL, NULL,
   N_("Show/Hide the Browser"), 
   G_CALLBACK (view_menu_browser_callback), NULL, TOGGLE_ACTION},

  {"ViewPlayback", N_("View Playback"),
   NULL, NULL,
   N_("Show/Hide the Playback display"), 
   G_CALLBACK (view_menu_playback_callback), NULL, TOGGLE_ACTION},

  {"ViewToolbar", N_("View Toolbar"),
   NULL, NULL,
   N_("Show/Hide the Toolbar"), 
   G_CALLBACK (view_menu_toolbar_callback), NULL, TOGGLE_ACTION},

  {"Queue", N_("Queue"),
   NULL, NULL,
   N_("Queue track(s) for playback"), 
   G_CALLBACK (commands_queue_callback), NULL},

  {"Selectall", N_("Select All"),
   NULL, "<control>A",
   N_("Select All tracks in the track list"), 
   G_CALLBACK (edit_menu_selectall_callback), NULL},

  {"About", N_("About Neutrino"),
  GNOME_STOCK_ABOUT, NULL,
  N_( "Display credits for the creators of Neutrino" ),
  G_CALLBACK (help_menu_about_neutrino_callback), NULL }

};

static guint
neutrino_window_action_n_entries = G_N_ELEMENTS( neutrino_window_action_entries );

void
neutrino_window_initialise_menus( NeutrinoWindow *window )
{
	gint i;
	GError *error;

	window->details->group = egg_action_group_new( "NeutrinoActions" );
	/* set user data field in action table */
	for( i = 0; i < neutrino_window_action_n_entries; ++ i ) {
		neutrino_window_action_entries[ i ].user_data = window;
	}

	egg_action_group_add_actions( window->details->group,
				neutrino_window_action_entries,
				neutrino_window_action_n_entries );

	window->details->merge = egg_menu_merge_new();
	egg_menu_merge_insert_action_group( window->details->merge,
					    window->details->group,
					    0 );
	g_signal_connect( window->details->merge,
			  "add_widget",
			  G_CALLBACK( add_widget ),
			  window );
	error = NULL;
	egg_menu_merge_add_ui_from_file( window->details->merge,
					 UIDATADIR"/neutrino-ui.xml",
					 &error );
	gtk_window_add_accel_group( GTK_WINDOW( window ),
				window->details->merge->accel_group );

	if( error ) {
		g_print( "ERROR: %s\n", error->message );
	}
}


