/*
 * audio-win32.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/audio/audio-win32.cc,v 1.16 2002/02/03 03:10:46 lim Exp $";

#include <assert.h>
#include <windows.h>
#include <winsock.h>
#include <mmsystem.h>
#include <mmreg.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "audio.h"
#include "ss.h"
#include "tclcl.h"

#define BLKS_PER_READ 4
#define READ_AHEAD 8
#define BLKS_PER_WRITE 4
#define WRITE_AHEAD 20

class audMux {
    public:
    u_int id_;
	MIXERCONTROLDETAILS select_[8];
	MIXERCONTROLDETAILS vol_[8];
	u_char mcnt_;
	u_char vcnt_;
	char vmap_[8];
	u_char isOut_;
};

class Win32Audio : public Audio {
    public:
	Win32Audio();
	~Win32Audio();
	void Write(u_char *);
	int FrameReady();
	u_char* Read();
	void SetRGain(int);
	void SetPGain(int);
	void OutputPort(int);
	void InputPort(int);
	void Obtain();
	void Release();
	void RMute();
	void RUnmute();
	int HalfDuplex() const;

    protected:
	int OpenOut();
	void CloseOut();
	int outErr(int) const;
	int OpenIn();
	void CloseIn();
	int inErr(int) const;
	int getVolCtrl(MIXERLINE& ml, audMux&mux);
	void addPorts(int lineVolIdx, MIXERLINE& ml, MIXERCONTROL& mc,
		      audMux& mux);
	void setupMux(audMux& mux, DWORD ctype);
	void getMixerCtrls(MIXERLINE&, audMux&);

	HWAVEOUT out_;
	HWAVEIN  in_;

	u_char* rbuf_;
	u_char* rbufStart_;
	u_char* rbufEnd_;
	u_char* zbuf_;
	u_int lastmean_;
	u_int ibindx_;
	u_char* ibufStart_;
	u_char* ibufEnd_;

	u_char* obuf_;
	u_char* obufStart_;
	u_char* obufEnd_;

	u_short iblen_;
	u_short oblen_;

	WAVEHDR iwhdr_[READ_AHEAD];
	WAVEHDR owhdr_[BLKS_PER_WRITE * WRITE_AHEAD];
	const WAVEFORMATEX* iformat_;
	const WAVEFORMATEX* oformat_;

	audMux omux_;
	audMux imux_;

	int portChangePending_;
};

static class Win32AudioClass : public TclClass {
    public:
	Win32AudioClass() : TclClass("Audio/Win32") {}
	TclObject* create(int, const char*const*) {
                return new Win32Audio();
		return (0);
	}
} win32_audio_class;

extern "C" const u_char lintomulawX[];
extern "C" const short mulawtolin[];

static const WAVEFORMATEX lin16fmt = {
	WAVE_FORMAT_PCM,
	1,
	8000,
	2 * 8000,
	2,
	16,
	0
};
static const WAVEFORMATEX lin8fmt = {
	WAVE_FORMAT_PCM,
	1,
	8000,
	8000,
	1,
	8,
	0
};


Win32Audio::Win32Audio() :
	out_(0),
	in_(0),
	lastmean_(0)
{
	zbuf_ = new u_char[blksize_];
	memset(zbuf_, ULAW_ZERO, blksize_);

	u_int len = blksize_ * BLKS_PER_READ;
	rbufStart_ = new u_char[len];
	rbufEnd_ = rbufStart_ + len;
	rbuf_ = rbufEnd_;

	/*
	 * figure out what input format is available.
	 */
	int sts = waveInOpen(0, WAVE_MAPPER, &lin16fmt, 0, 0,
			     WAVE_FORMAT_QUERY);
	if (sts == WAVERR_BADFORMAT) {
		/* can't do 16 bit audio, try 8 bit */
		sts = waveInOpen(0, WAVE_MAPPER, &lin8fmt, 0, 0,
				 WAVE_FORMAT_QUERY);
		if (sts) {
			fprintf(stderr,
    "vat: soundcard supports neither 16 nor 8 bit 8KHz PCM audio input (%d)\n",
				sts);
			abort();
		}
		iformat_ = &lin8fmt;
		iblen_ = len;
		len *= READ_AHEAD;
		ibufStart_ = new u_char[len];
		ibufEnd_ = ibufStart_ + len;
	} else {
		iformat_ = &lin16fmt;
		iblen_ = len * sizeof(short);
		len *= READ_AHEAD;
		ibufStart_ = (u_char*)new short[len];
		ibufEnd_ = ibufStart_ + len * sizeof(short);
	}

	/*
	 * figure out what output format is available.
	 */
	sts = waveOutOpen(0, WAVE_MAPPER, &lin16fmt, 0, 0,
			  WAVE_FORMAT_QUERY);
	if (sts == WAVERR_BADFORMAT) {
		/* can't do 16 bit audio, try 8 bit */
		sts = waveOutOpen(0, WAVE_MAPPER, &lin8fmt, 0, 0,
				  WAVE_FORMAT_QUERY);
		if (sts) {
			fprintf(stderr,
    "vat: soundcard supports neither 16 nor 8 bit 8KHz PCM audio output (%d)\n",
				sts);
			abort();
		}
		oformat_ = &lin8fmt;
		len = blksize_ * BLKS_PER_WRITE;
		oblen_ = len;
		len *= WRITE_AHEAD;
		obufStart_ = new u_char[len];
		obufEnd_ = obufStart_ + len;
	} else {
		oformat_ = &lin16fmt;
		len = blksize_ * BLKS_PER_WRITE;
		oblen_ = len * sizeof(short);
		len *= WRITE_AHEAD;
		obufStart_ = (u_char*)new short[len];
		obufEnd_ = obufStart_ + len * sizeof(short);
	}

    memset(&imux_, 0, sizeof(imux_));
    imux_.id_ = 0;
    memset(&omux_, 0, sizeof(omux_));
    omux_.id_ = 0;
    omux_.isOut_ = 1;

	if (mixerGetNumDevs()) {
		/* set up the mixer controls for input & out select & gain */
        HMIXER mixerHdl;
        HWAVEIN waveInHdl;
        if (!waveInOpen(&waveInHdl, WAVE_MAPPER, iformat_, 0, 0, 0)) {
            if (!mixerOpen(&mixerHdl, (UINT) waveInHdl, 0, 0, MIXER_OBJECTF_HWAVEIN)) {
                mixerGetID((HMIXEROBJ) mixerHdl, &imux_.id_, MIXER_OBJECTF_HMIXER);
                mixerClose(mixerHdl);
            }
            waveInClose(waveInHdl);
        }
		setupMux(imux_, MIXERLINE_COMPONENTTYPE_DST_WAVEIN);

        HWAVEOUT waveOutHdl;
        if (!waveOutOpen(&waveOutHdl, WAVE_MAPPER, oformat_, 0, 0, 0)) {
            if (!mixerOpen(&mixerHdl, (UINT) waveOutHdl, 0, 0, MIXER_OBJECTF_HWAVEOUT)) {
                mixerGetID((HMIXEROBJ) mixerHdl, &omux_.id_, MIXER_OBJECTF_HMIXER);
                mixerClose(mixerHdl);
            }
            waveOutClose(waveOutHdl);
        }
		setupMux(omux_, MIXERLINE_COMPONENTTYPE_DST_SPEAKERS);
	} else {
        /* Put in default names if there is no mixer. */
        output_names_ = new char[8];
        strcpy(output_names_, "speaker");
        input_names_ = new char[5];
        strcpy(input_names_, "mike");
	}

	/* FIXME: should enable full-duplex for soundcards that are capable of
	 * doing that */
	duplex_ = 0;
}

Win32Audio::~Win32Audio()
{
	CloseIn();
	CloseOut();

	delete zbuf_;
	delete rbufStart_;
	delete ibufStart_;
	delete obufStart_;
	delete[] input_names_;
	input_names_ = 0;
	delete[] output_names_;
	output_names_ = 0;
}

int Win32Audio::HalfDuplex() const
{
	/*FIXME*/
	return 1;
}

int Win32Audio::inErr(int error) const
{
	if (error) {
		char errorText[MAXERRORLENGTH];
		waveInGetErrorText(error, errorText, sizeof(errorText));
		fprintf(stderr, "vat - input error: %s\n", errorText);
	}
	return (error);
}

int Win32Audio::outErr(int error) const
{
	if (error) {
		char errorText[MAXERRORLENGTH];
		waveOutGetErrorText(error, errorText, sizeof(errorText));
		fprintf(stderr, "vat - output error: %s\n", errorText);
	}
	return (error);
}

int Win32Audio::OpenOut()
{
	int error = 0;

	if (out_ == 0) {
		error = waveOutOpen(&out_, WAVE_MAPPER, oformat_,
				    NULL, NULL, CALLBACK_NULL);

		/*
		 * Maybe we failed because someone is playing sound already.
		 * Shut any sound off then try once more before giving up.
		 */
		if (error) {
			sndPlaySound(NULL, 0);
			if (outErr(waveOutOpen(&out_, WAVE_MAPPER, oformat_,
					       NULL, NULL, CALLBACK_NULL))) {
				return 1;
			}
		}
		/* restore the gain to what the user set on this vat window */
		SetPGain(pgain_);

		if (portChangePending_) {
			OutputPort(oport_);
			portChangePending_ = 0;
		}

		/* (re-)initialize the output buffer descriptors */
		memset(owhdr_, 0, sizeof(owhdr_));
		u_char* bp = obufStart_;
		obuf_ = bp;
		u_int len = oblen_;
		int i;
		for (i = 0; i < WRITE_AHEAD; ++i) {
			/*
			 * we mark the last hdr of the group of hdrs
			 * associated with this write block as 'writeable'
			 * (by setting its DONE bit) but make the address
			 * in the hdr indicate the start of the group of
			 * blocks.
			 */
			WAVEHDR* whp = &owhdr_[(i + 1) * BLKS_PER_WRITE - 1];
			whp->dwFlags = 0;
			whp->dwBufferLength = oblen_;
			whp->lpData = (char*)bp;
			outErr(waveOutPrepareHeader(out_, whp, sizeof(*whp)));
			whp->dwFlags |= WHDR_DONE;
			bp += len;
		}
		/*
		 * do initial write to generate a backlog to avoid
		 * dropouts due to scheduling delays.
		 */
		for (i = BLKS_PER_WRITE; --i >= 0; )
			Write(zbuf_);
	}
	return error;
}

int Win32Audio::OpenIn()
{
	if (in_ == 0) {
		if (inErr(waveInOpen(&in_, WAVE_MAPPER, iformat_,
				     NULL, NULL, CALLBACK_NULL)))
			return (1);

		/* restore the gain to what the user set on this vat window */
		SetRGain(rgain_);
		if (portChangePending_) {
			InputPort(iport_);
			portChangePending_;
		}
		/* (re-)initialize the input buffer descriptors */
		memset(iwhdr_, 0, sizeof(iwhdr_));
		ibindx_ = 0;
		rbuf_ = rbufEnd_;
		u_char* bp = ibufStart_;
		u_int len = iblen_;
		memset(bp, 0, len * READ_AHEAD);
		for (int i = 0; i < READ_AHEAD; ++i) {
			WAVEHDR* whp = &iwhdr_[i];
			whp->dwFlags = 0;
			whp->dwBufferLength = len;
			whp->lpData = (char*)bp;
			bp += len;
			waveInPrepareHeader(in_, whp, sizeof(*whp));
			if (inErr(waveInAddBuffer(in_, whp, sizeof(*whp)))) {
				CloseIn();
				return (1);
			}
		}
		waveInStart(in_);
	}
	return 0;
}

void Win32Audio::CloseOut()
{
	if (out_) {
		waveOutReset(out_);
		for (int i = 1; i < WRITE_AHEAD + 1; ++i) {
			WAVEHDR* whp = &owhdr_[i * BLKS_PER_WRITE - 1];
			if (whp->dwFlags & WHDR_PREPARED)
				waveOutUnprepareHeader(out_, whp, sizeof(*whp));
		}
		waveOutClose(out_);
		out_ = 0;
	}
}

void Win32Audio::CloseIn()
{
	if (in_) {
		waveInStop(in_);
		waveInReset(in_);
		for (int i = 0; i < READ_AHEAD; ++i) {
			WAVEHDR* whp = &iwhdr_[i];
			if (whp->dwFlags & WHDR_PREPARED)
				waveInUnprepareHeader(in_, whp, sizeof(*whp));
		}
		waveInClose(in_);
		in_ = 0;
	}
}

void Win32Audio::Obtain()
{
	int error;

	if (haveaudio())
		abort();

	if (rmute_ & 1) {
		error = OpenOut();
	} else {
		error = OpenIn();
	}
	/* FIXME: hack to make HaveAudio method work */
	if (error)
		fd_ = -1;
	else {
		fd_ = 0;
		SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_HIGHEST);
	}
	notify();
}

void Win32Audio::Release()
{
	CloseOut();
	CloseIn();
	SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_NORMAL);
	Audio::Release();
}

void Win32Audio::Write(u_char *cp)
{
	if (out_) {
		/*
		 * copy the new data into our circular output buffer,
		 * converting from ulaw to linear as we go.
		 */
		WAVEHDR* whp;
		if (oformat_ == &lin8fmt) {
			u_char* sp = obuf_;
			u_int bindx = (sp - obufStart_) / blksize_;
			whp = &owhdr_[bindx];

            /* Check if device is done with buffer before copying into it. */
            u_int waveHdrIndex = bindx;
            while (!owhdr_[waveHdrIndex].lpData) waveHdrIndex++;
            if (!(owhdr_[waveHdrIndex].dwFlags & WHDR_DONE)) return;

			u_char* ep = sp + blksize_;
			const short* u2l = mulawtolin;
			while (sp < ep)
				*sp++ = (u2l[*cp++] >> 8) ^ 0x80;
			obuf_ = (ep >= obufEnd_)? obufStart_ : ep;
		} else {
			short* sp = (short*)obuf_;
			u_int bindx = (sp - (short*)obufStart_) / blksize_;
			whp = &owhdr_[bindx];

            /* Check if device is done with buffer before copying into it. */
            u_int waveHdrIndex = bindx;
            while (!owhdr_[waveHdrIndex].lpData) waveHdrIndex++;
            if (!(owhdr_[waveHdrIndex].dwFlags & WHDR_DONE)) return;

			short* ep = sp + blksize_;
			const short* u2l = mulawtolin;
			while (sp < ep)
				*sp++ = u2l[*cp++];
			obuf_ = ((u_char*)ep >= obufEnd_)? obufStart_ : (u_char*)ep;
		}

		/*
		 * if the buffer descriptor associated with this block
		 * is marked ready, ship it.
		 */
        if (whp->lpData) {
			whp->dwFlags &=~ WHDR_DONE;
			outErr(waveOutWrite(out_, whp, sizeof(*whp)));
		}
	}
}

int Win32Audio::FrameReady()
{
	if (in_ && rbuf_ >= rbufEnd_) {
		/* mulaw conversion buffer is empty - see if a read finished */
		u_int i = ibindx_;
		WAVEHDR* whp = iwhdr_ + i;
		if ((whp->dwFlags & WHDR_DONE) == 0)
			return (0);

		/* read finished - move input to ulaw buffer */
		rbuf_ = rbufStart_;
		ibindx_ = (i + 1) % READ_AHEAD;

		const u_char* l2u = lintomulawX;
		u_int* ip = (u_int*)rbuf_;
		int smean = lastmean_;
		if (iformat_ == &lin8fmt) {
			u_char* sp = (u_char*)whp->lpData;
			u_char* ep = sp + whp->dwBytesRecorded;
			for ( ; sp < ep; sp += 4) {
				register int mean, dif;
				register u_int res;
				register int s0 = (int(sp[0]) - 0x80) << 8;
				register int s1 = (int(sp[1]) - 0x80) << 8;
				register int s2 = (int(sp[2]) - 0x80) << 8;
				register int s3 = (int(sp[3]) - 0x80) << 8;

				mean = smean >> 13;
				dif = s0 - mean;
				smean += dif;
				res = l2u[dif & 0x1ffff] << 0;

				mean = smean >> 13;
				dif = s1 - mean;
				smean += dif;
				res |= l2u[dif & 0x1ffff] << 8;

				mean = smean >> 13;
				dif = s2 - mean;
				smean += dif;
				res |= l2u[dif & 0x1ffff] << 16;

				mean = smean >> 13;
				dif = s3 - mean;
				smean += dif;
				res |= l2u[dif & 0x1ffff] << 24;

				*ip++ = res;
			}
		} else {
			short* sp = (short*)whp->lpData;
			short* ep = (short*)((char*)sp + whp->dwBytesRecorded);
			for ( ; sp < ep; sp += 4) {
				register int mean, dif;
				register u_int res;
				register int s0 = sp[0];
				register int s1 = sp[1];
				register int s2 = sp[2];
				register int s3 = sp[3];

				mean = smean >> 13;
				dif = s0 - mean;
				smean += dif;
				res = l2u[dif & 0x1ffff] << 0;

				mean = smean >> 13;
				dif = s1 - mean;
				smean += dif;
				res |= l2u[dif & 0x1ffff] << 8;

				mean = smean >> 13;
				dif = s2 - mean;
				smean += dif;
				res |= l2u[dif & 0x1ffff] << 16;

				mean = smean >> 13;
				dif = s3 - mean;
				smean += dif;
				res |= l2u[dif & 0x1ffff] << 24;

				*ip++ = res;
			}
		}
		lastmean_ = smean;
		whp->dwFlags &=~ WHDR_DONE;
		inErr(waveInAddBuffer(in_, whp, sizeof(*whp)));
	}
	return (1);
}

u_char* Win32Audio::Read()
{
	u_char* cp;

	if (in_) {
		cp = rbuf_;
		rbuf_ = cp + blksize_;
	} else
		cp = zbuf_;
	return (cp);
}

void Win32Audio::SetRGain(int level)
{
	rgain_ = level;
	if (in_) {
		if (level > 255)
			level = 255;
		level <<= 8;
		MIXERCONTROLDETAILS& mcd = imux_.vol_[imux_.vmap_[iport_]];
		for (u_int i = 0; i < mcd.cChannels; ++i)
		   ((MIXERCONTROLDETAILS_UNSIGNED*)mcd.paDetails + i)->dwValue =
			level;
		MMRESULT result =
			mixerSetControlDetails((HMIXEROBJ) imux_.id_, &mcd,
					       MIXER_SETCONTROLDETAILSF_VALUE);
		if (result != MMSYSERR_NOERROR) {
			fprintf(stderr, "cannot set rgain (err=%d)\n", result);
		}
	}
}

void Win32Audio::SetPGain(int level)
{
	pgain_ = level;
	/* FIXME: shouldn't we set output volume found in mux_.vol_[oport_] as well? */
	if (out_) {
		if (level > 255)
			level = 255;
		level <<= 8;
		DWORD vol = level | (level << 16);
		outErr(waveOutSetVolume(out_, vol));
	}
}

void Win32Audio::OutputPort(int p)
{
	assert( (omux_.mcnt_==0 || p < omux_.mcnt_) || "selecting invalid oport");
	if (!out_ && oport_ != p) {
		portChangePending_ = 1;
	}
	oport_ = p;
	if (out_ && p < omux_.mcnt_) {
		MMRESULT status=
			mixerSetControlDetails((HMIXEROBJ) omux_.id_, &omux_.select_[p],
					MIXER_SETCONTROLDETAILSF_VALUE);
		if (status != MMSYSERR_NOERROR)
			fprintf(stderr, "output port select failed (err=%d)\n",
			status);
	}
}

void Win32Audio::InputPort(int p)
{
	assert( (imux_.mcnt_==0 || p <= imux_.mcnt_) || "selecting invalid iport");
	if (!in_ && iport_ != p) {
		portChangePending_ = 1;
	}
	iport_ = p;
	if (in_ && p < imux_.mcnt_) {
		MMRESULT status =
			mixerSetControlDetails((HMIXEROBJ) imux_.id_, &imux_.select_[p],
					MIXER_SETCONTROLDETAILSF_VALUE);
		if (status != MMSYSERR_NOERROR)
			fprintf(stderr, "input port select failed (err=%d)\n",
			status);
	}
}

void Win32Audio::RMute()
{
	CloseIn();
	if (OpenOut() == 0)
		rmute_ |= 1;
}

void Win32Audio::RUnmute()
{
	CloseOut();
	if (OpenIn() == 0)
		rmute_ &=~ 1;
}

void Win32Audio::addPorts(int lineVolIdx, MIXERLINE& ml, MIXERCONTROL& mc,
			audMux& mux)
{
	MIXERCONTROLDETAILS mcd;

	mcd.cbStruct = sizeof(mcd);
	mcd.dwControlID = mc.dwControlID;
	mcd.cChannels = 1;
	mcd.cMultipleItems = mc.cMultipleItems;

	for (u_int i = 0; i < mc.cMultipleItems; ++i) {
		int p = mux.mcnt_++;
		if (mux.mcnt_ >= 8) {
			fprintf(stderr, "oops, too many ports, "
				"ignoring the rest\n");
			break;
		}
		u_int n = mcd.cMultipleItems * mcd.cChannels;
		MIXERCONTROLDETAILS_BOOLEAN* mcdb =
			new MIXERCONTROLDETAILS_BOOLEAN[n];
		/* mute the rest */
		memset(mcdb, 0, n * sizeof(*mcdb));
		/* set this line to 1 when selected */
		for (u_int j = 0; j < mcd.cChannels; ++j)
			mcdb[j*mc.cMultipleItems+i].fValue = 1;
		mcd.cbDetails = sizeof(mcdb[0]);
		mcd.paDetails = mcdb;
		mux.select_[p] = mcd;
		/* if we don't have any volume control from the line yet,
		 * try to find one from the source lines */
		/* FIXME: there don't seem to be a way to
		 * associate the mux controls with the lines,
		 * so we will assume they are in the same order */
		int volIdx = lineVolIdx;
		if (volIdx == -1 && i < ml.cConnections) {
			MIXERLINE src;
			memset(&src, 0, sizeof(src));
			src.cbStruct = sizeof(src);
			src.dwSource = i;
			src.dwDestination = ml.dwDestination;
			MMRESULT status =
				mixerGetLineInfo((HMIXEROBJ) mux.id_, &src,
						 MIXER_GETLINEINFOF_SOURCE);
			if (status == MMSYSERR_NOERROR)
				volIdx = getVolCtrl(src, mux);
			else
				fprintf(stderr, "cannot get vol controls for srcf line %d, err=%d\n", i, status);
		}
		mux.vmap_[p]=volIdx;
	}
}

/* gets the volume control for a line */
int Win32Audio::getVolCtrl(MIXERLINE& ml, audMux& mux)
{
	MIXERLINECONTROLS mlc;
	MIXERCONTROL mc;

	memset(&mlc, 0, sizeof(mlc));
	memset(&mc, 0, sizeof(mc));
	mlc.cbStruct = sizeof(mlc);
	mlc.cbmxctrl = sizeof(mc);
	mlc.pamxctrl = &mc;
	mlc.dwLineID = ml.dwLineID;
	mlc.cControls = ml.cControls;

	/* get the volume control */
	mlc.dwControlType = MIXERCONTROL_CONTROLTYPE_VOLUME;
	MMRESULT status;
	int volIdx = -1;
	status = mixerGetLineControls((HMIXEROBJ) mux.id_, &mlc,
				      MIXER_GETLINECONTROLSF_ONEBYTYPE);
	if (MMSYSERR_NOERROR != status) {
#if 0
		fprintf(stderr, "cannot get volume control for line, err=%d\n",
			status);
#endif
	} else {
		MIXERCONTROLDETAILS mcd;
		mcd.cbStruct = sizeof(mcd);
		mcd.dwControlID = mc.dwControlID;
		mcd.cChannels = ml.cChannels;
		mcd.cMultipleItems = mc.cMultipleItems;
		volIdx = mux.vcnt_++;
		if (volIdx >= 8) {
			fprintf(stderr, "too many volume controls,"
				"ignoring the rest\n");
			/* return the first one */
			return -1;
		}
		MIXERCONTROLDETAILS_UNSIGNED* mcdu =
			new MIXERCONTROLDETAILS_UNSIGNED[mcd.cChannels];
		memset(mcdu, 0, mcd.cChannels * sizeof(*mcdu));
		mux.vol_[volIdx] = mcd;
		mux.vol_[volIdx].cbDetails = sizeof(*mcdu);
		mux.vol_[volIdx].paDetails = mcdu;
	}
	return volIdx;
}

void Win32Audio::getMixerCtrls(MIXERLINE& ml, audMux& mux)
{
	MIXERLINECONTROLS mlc;
	memset(&mlc, 0, sizeof(mlc));
	mlc.cbStruct = sizeof(mlc);
	mlc.dwLineID = ml.dwLineID;
	mlc.cControls = ml.cControls;

	if (!ml.cControls) /* just in case */
		return;

	int lineVolIdx = getVolCtrl(ml, mux);

	/* find all ports */
	mlc.dwControlType = 0;
	MIXERCONTROL *mcs = new MIXERCONTROL[ml.cControls];
	memset(mcs, 0, sizeof(mcs));
	mlc.pamxctrl = &mcs[0];
	mlc.cbmxctrl = sizeof(mcs[0]);
	MMRESULT status = mixerGetLineControls((HMIXEROBJ) mux.id_, &mlc,
					       MIXER_GETLINECONTROLSF_ALL);
	if (MMSYSERR_NOERROR != status)
		fprintf(stderr, "cannot get controls for line %d, err=%d\n",
			status);

	for (u_int i = 0; i < mlc.cControls; ++i) {
		switch (mcs[i].dwControlType) {
		case MIXERCONTROL_CONTROLTYPE_MUX:
		case MIXERCONTROL_CONTROLTYPE_MIXER:
		case MIXERCONTROL_CONTROLTYPE_SINGLESELECT:
		case MIXERCONTROL_CONTROLTYPE_MULTIPLESELECT:
			addPorts(lineVolIdx, ml, mcs[i], mux);
			break;
		}
	}
	free(mcs);
}

void Win32Audio::setupMux(audMux& mux, DWORD ctype)
{
	MIXERLINE l;
	memset(&l, 0, sizeof(l));
	l.cbStruct = sizeof(l);
	l.dwComponentType = ctype;
	int s = mixerGetLineInfo((HMIXEROBJ) mux.id_, &l, MIXER_GETLINEINFOF_COMPONENTTYPE);
	if (s == MMSYSERR_NOERROR)
		getMixerCtrls(l, mux);
	else
		fprintf(stderr, "mixerGetLineInfo failed: %d\n", s);

	/* initilialize the list of names for the ports */
	int i, len=0;

	char** arNames = (mux.mcnt_) ? new char*[mux.mcnt_] : 0;
	for (i=0; i < mux.mcnt_; i++) {
		/* too bad we cannot get names of the individual items using
		 * the mcdt, so we get them multiple times even though some of
		 * the calls are repeated, no big deal */
		MIXERCONTROLDETAILS mcd;
		mcd.cbStruct = sizeof(mcd);
		mcd.dwControlID = mux.select_[i].dwControlID;
		mcd.cChannels = mux.select_[i].cChannels;
		mcd.cMultipleItems = mux.select_[i].cMultipleItems;
		MIXERCONTROLDETAILS_LISTTEXT* mcdt =
			new MIXERCONTROLDETAILS_LISTTEXT[mcd.cChannels*mcd.cMultipleItems];
		mcd.cbDetails = sizeof(mcdt[0]);
		mcd.paDetails = mcdt;
		if (mixerGetControlDetails((HMIXEROBJ) mux.id_, &mcd,
					   MIXER_GETCONTROLDETAILSF_LISTTEXT))
			fprintf(stderr, "cannot get list text in audio\n");
		/* if multiple channels, use the first channel's name */
		arNames[i] = strdup(mcdt[i].szName);
		len += strlen(arNames[i]) + 3;
		delete[] mcdt;
	}

	char* szNameList = 0;
	for (i=0; i < mux.mcnt_; i++) {
		if (!szNameList) {
			szNameList = new char[len];
			strcpy(szNameList, "{");
			strcat(szNameList, arNames[i]);
			strcat(szNameList, "}");
		} else {
			strcat(szNameList, " {");
			strcat(szNameList, arNames[i]);
			strcat(szNameList, "}");
		}
		free(arNames[i]);
	}
	assert( (szNameList==0 || len >= (int)strlen(szNameList)) &&
	       "incorrect algorithm for len");
	if (arNames)
		delete[] arNames;

	if (mux.mcnt_ == 0) {
		/* put in some defaults */
		if (mux.isOut_) {
			output_names_ = new char[8];
			strcpy(output_names_, "speaker");
		} else {
			input_names_ = new char[5];
			strcpy(input_names_, "mike");
		}
		mux.vmap_[0] = 0;
	} else {
		if (mux.isOut_) output_names_ = szNameList;
		else input_names_ = szNameList;
	}
}
