/*
 * mb-canv.h --
 *
 *      MediaBoard Canvas
 *      This is now a wrapper around tkCanvas, ultimately, we want to implement
 *      our own canvas.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef MBCANV_H
#define MBCANV_H

#include "mb/mb.h"
#include "mb/mb-items.h"
#include "mb/mb-obj.h"
#include "mb/error.h"
#include "misc/transform.h"

// forward declarations
class Page;

//
// This is the type for canvas items
//
typedef n_long CanvItemId;

//
// how much space in bytes we need to reserve for a canvas itemid
//
#define MAX_CANVID_CHAR MAX_N_LONG_CHAR

//
// Since we are using Tcl, save them as strings this is faster
//     properties are saved as is from the canvas at first
//     on the first call the GetProperties, we change it to the
//     -varname1 value1 -varname2 -value2 form
//
//     points are stored in double pairs since we need to convert them
//     between the normalized form and the canvas coordinates
//
class MBCanvItem {
public:
	MBCanvItem(const char* szType, Point* arCoords, int count);
	virtual ~MBCanvItem();

	/* accessors */
	char * getType()   { return szType_; }
	Point *getCoords() { return arCoords_; }
	int getCount()     { return count_; }

	void setProp(const char* szProp) {
		// we should propably keep the reference instead, but this is
		// cleaner since we want to keep the tcl specfic things in one
		// place.
		::AllocNCopy(&szProp_, szProp);
	}
	char * getProp() { return szProp_; }

	// raises afId to just before b4Id
	void raiseAfter(CanvItemId b4Id, CanvItemId afId);

	// converts to a page item
	virtual PageItem* toPageItem();

private:
	Point *arCoords_;
	int count_;
	char *szType_;
	char *szProp_;
};

struct MBCanvItemInfo {
	MBBaseRcvr* pRcvr_;
	long cmdId;
};

/*
 *  This is currently just an encapsulation of the tk canvas class
 * Ultimately, we might write a real canvas class
 */
class MBBaseCanvas : public TclObject {
public:
	MBBaseCanvas();
	virtual ~MBBaseCanvas();

	// return a record of the item
	virtual MBCanvItem* getItem(const CanvItemId canvId)=0;

	// delete an item
	virtual Bool deleteItem(const CanvItemId canvId, MBCmd *pCmd)=0;

	// re-create an item from a record of the item
	virtual CanvItemId createItem(Page* pPage, MBCmd* pCmd,
				      MBCanvItem* pItem)=0;

	// create an item from raw data
	virtual CanvItemId createItem(Page* pPage, MBCmd* pCmd,
				      PageItem* pPgItem)=0;

	virtual Bool moveItem(CanvItemId targetId,  MBCmd* pCmd, Coord dx,
			      Coord dy)=0;

	// convert result to a numerical value
	CanvItemId getResultAsLong() {
		char *pEnd=NULL;
		char *szCanvId=Tcl::instance().result();
		unsigned long newCanvId = strtoul(szCanvId, &pEnd, 10);
		assert(newCanvId!=0);
		if (pEnd==szCanvId)
			return 0; // invalid result item
		return (CanvItemId) newCanvId;
	}

	virtual void raiseAfter(CanvItemId b4Id, CanvItemId afId)=0;
	virtual void raise(CanvItemId cid)=0;

	virtual char *getText(CanvItemId cid, int& len)=0;
	virtual void setText(CanvItemId cid, MBCmd* pCmd,const char* szText)=0;

	virtual CanvItemId createImage(Page* pPage,  MBCmd* pCmd,
				       ImageItem* pItem,
				       const char* szFileName)=0;

	virtual int command(int argc, const char*const* argv) {
		return TclObject::command(argc, argv);
	}

	/* returns 1 if new hash entry is created */
	int assocItem(CanvItemId cid, MBCmd* pCmd);

	/* returns FALSE if cannot find entry */
	int forgetItem(CanvItemId cid);

	/* return NULL if not found */
	MBCmd* canvId2cmd(CanvItemId cid);
	ulong canvId2itemId(CanvItemId cid);

private:
	Tcl_HashTable htCmds_; /* keyed by canvItemIds */
};



/*
 *  This is currently just an encapsulation of the tk canvas class
 * Ultimately, we might write a real canvas class
 */
class MBCanvas : public MBBaseCanvas {
public:
	MBCanvas();
	virtual ~MBCanvas();
	void setPathName(const char *szPathName) {
		pathObj_ = Tcl_NewStringObj(CONST_CAST(char*)(szPathName), -1);
		Tcl_IncrRefCount(pathObj_);
	}

	// return a record of the item
	virtual MBCanvItem* getItem(const CanvItemId canvId);

	// delete an item
	virtual Bool deleteItem(const CanvItemId canvId, MBCmd *pCmd);

	// re-create an item from a record of the item
	virtual CanvItemId createItem(Page* pPage, MBCmd* pCmd,
				      MBCanvItem* pItem);

	// create an item from raw data
	virtual CanvItemId createItem(Page* pPage, MBCmd* pCmd,
				      PageItem* pPgItem);

	// change co-ord of an item
	// REVIEW: should be able to get rid of this function, since the
	//         coord cmd is never sent...
	Bool itemCoord(CanvItemId targetId, MBCmd* pCmd, Coord x1, Coord y1,
		       Coord x2, Coord y2);

	virtual Bool moveItem(CanvItemId targetId,  MBCmd* pCmd, Coord dx,
			      Coord dy);

	// returns the next nearest item within dist of (x,y)
	CanvItemId nextNearest(Coord x, Coord y, Coord dist, CanvItemId cid);

	// returns whether the item overlaps ( (x1,y1) (x2,y2) )
	Bool overlap(Coord x1, Coord y1, Coord x2, Coord y2, CanvItemId cid);

	virtual void raiseAfter(CanvItemId b4Id, CanvItemId afId);
	virtual void raise(CanvItemId cid);

	virtual char *getText(CanvItemId cid, int& len);
	virtual void setText(CanvItemId cid, MBCmd* pCmd, const char* szText);

	virtual CanvItemId createImage(Page* pPage,  MBCmd* pCmd,
				       ImageItem* pItem,
				       const char* szFileName);

	int visibleSize();
	int visibleSize(int& width, int& height, int& inset);

	int resize(int width, int height);
	int setsize(int width, int height);
	int rescale(double scale, int sizeChanged);
	int fit(double width, double height);
	int getScale();

	virtual int command(int argc, const char*const* argv);

	int redirectCommand(int argc, const char*const* argv);
	int resetOrigin();

	/* stucture for font table entries, remembered each time
	 * scaleFont is called, so that we can scale them up to the
	 * right size */
	struct FontStruct {
		FontStruct(char* szNamedFont, int size) : size_(size) {
			::AllocNCopy(&szNamedFont_, szNamedFont);
		}
		~FontStruct() { delete[] szNamedFont_; }
		int size_;
		char* szNamedFont_;
	};
	char* scaleFont(Tk_Font tkfont);
	int rescaleAllFonts();

	int showOwner(CanvItemId cid);

private:
	const char *pathName() const  {
		if (!pathObj_) return NULL;
		return Tcl_GetStringFromObj(pathObj_, NULL);
	}
	Tcl_Obj* pathObj() const { return pathObj_; }
	Tcl_Obj* pathObj_;
	Transform matrix_; /* maps normalized coordinates to canvas
			    * coordinates */
	double xOrigin_, yOrigin_; /*canvas coordinate of top left corner*/
	/* width and height in normalized coordinates that fit into
	 * the current window */
	double norm_width_, norm_height_;
	int fontIdx_;
	Tcl_HashTable htFonts_;
};


#endif /* #ifdef MBCANV_H */
