/*
 * mb-cmdexec.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-cmdexec.cc,v 1.27 2002/02/03 03:16:30 lim Exp $
 */

#include "mb/mb-cmd.h"
#include "mb/mb-page.h"
#include "mb/error.h"

//
//  MBCmd::execute --
//     execute a command on "pPage"
//     "oldtime" is the current time the page is on
//     "newtime" is the new time we want the command to display itself
//     on
//
//     Normally the function optimizes by doing nothing if it is not
//     transiting to a different state, but it can be forced to apply
//     the command when "force" is set to 1
//
//     hint: to execute a command for the first time,
//           set oldtime to cMBTimeNegInf
//
int MBCmd::execute(Page* pPage, const MBTime& oldtime,
		   const MBTime& newtime)
{
	if (!pPage) {
		assert(pPage && "page is null, should not happen");
		return MB_EXE_OK;
	}
	if (!activeAt(oldtime)) {
		if (activeAt(newtime))
		    return apply(pPage, newtime);
	} else if (!activeAt(newtime)) {
		return reverse(pPage, newtime);
	}
	return MB_EXE_OK; /* no changes */
}

int MBCmd::removeAssocItem(Page* pPage, int reportErr)
{
	CanvItemId cid = pPage->itemId2canvId(getSeqno());
	if (!cid) {
		if (reportErr)
			return errorDup(pPage);
		else
			return MB_EXE_OK; /* item already removed */
	}
	MBBaseCanvas* pCanv= pPage->getCanvas();
	if (!pCanv) return MB_EXE_ERROR;
	pCanv->deleteItem(cid, NULL);
	pPage->removeCanvItem(getSeqno());
	return MB_EXE_OK;
}

int MBCmd::errorDup(Page* pPage)
{
	char* szPgId = PgId2Str(pPage->getId());
	perr(("reapplying cmd: (%s, %ld) (%s)",
	      szPgId, getSeqno(), achCmdNames[getType()] ));
	delete[] szPgId;
	return MB_EXE_ERROR;
}

int MBCmdCreate::apply(Page* pPage, const MBTime& /* t */)
{
	if (0 == pPage->itemId2canvId(getSeqno())) {
		CanvItemId canvId =
			pPage->getCanvas()->createItem(pPage, this, pItem_);
		if (canvId!=0) {          // create successful
			// get new sequence number
			if (MBCmd::Instantiate(pPage)==MB_EXE_OK) {
				pPage->addCanvItem(this, getSeqno(), canvId);
				return MB_EXE_OK;
			}
		}
		// error occurred if reached here
		return MB_EXE_ERROR;
	} else {
		return errorDup(pPage);
	}
}

int
MBCmdMLine::apply(Page* pPage, const MBTime& /* t */)
{
	if (pPage->itemId2canvId(getSeqno())) {
		return errorDup(pPage);
	}
	MBBaseCanvas *pCanv = pPage->getCanvas();
	CanvItemId canvId;

	// loop through all the items, store the coords
	// don't want to delete the items on the 1st pass since error
	// could occur
	if (snEnd_ < snStart_) {
		SignalError(("Error: snEnd(=%d) < snStart(=%d)!",
			     snEnd_, snStart_));
		return MB_EXE_ERROR;
	}
	Point *pPoints= new Point[snEnd_ - snStart_ + 2];
	ulong i, index=0;
	MBCmdCreate* pCmd = NULL;
	const PageItem *pPgItem = NULL;
	// the first points of all line segments (except the first) are
	// redundant, add up the line segments by adding all 2nd & 1st
	// pts of 1st seg
#ifdef MB_DEBUG
	// check that the line segments are contigous
	Point lastPt;
#endif

	for (i=snStart_; i<=snEnd_; i++) {
		pCmd=(MBCmdCreate*)pPage->getCmd(i);
		if (!pCmd) break;
		if (pCmd->getType()!=CCreateItem) {
			disperr("invalid cmd type: %d", pCmd->getType());
			break;
		}
		pPgItem = pCmd->getItem();
		if (pPgItem->getType()!=PgItemLine) {
			disperr("invalid page type: %d",
				   pPgItem->getType());
			break;
		}
		if (pPgItem->getNumPoints()!=2) {
			SignalError(("invalid number of points!"));
			goto err;
		}
		if (i==snStart_) {
			pPoints[index++]=pPgItem->getPointByIdx(0);
		}
		// use default copy constructors
		pPoints[index++]=pPgItem->getPointByIdx(1);
#ifdef MB_DEBUG
		if (i!=snStart_) {
			assert(lastPt==pPgItem->getPointByIdx(0)
			       && "non-cont multi-line");
		}
		lastPt = pPgItem->getPointByIdx(1);
#endif
	} // for (i=snStart_; ...
	assert(index == snEnd_ - snStart_ + 2);
	assert(pPage->getCmd(snStart_) && pPage->getCmd(snStart_)->getItem() &&
	       "no pItem_ for first line seg!");
	pPage->getCmd(snStart_)->getItem()->getCopy(&pItem_);
	// note that pItem_ will delete pPoints in destr
	pItem_->setPoints(pPoints,index);
	// everything is okay, loop through to delete all items
	for (i=snStart_; i<=snEnd_; i++) {
		canvId=pPage->itemId2canvId(i);
		if (canvId) {
			if (!pCanv->deleteItem(canvId, NULL)) {
				SignalError(("Impossible: item found before,"
					     " now not there?"));
				goto err;
			}
			// successfully deleted, remove the mapping
			pPage->removeCanvItem(i);
		} else {
			disperr("cannot find canvid");
		}
	}

	canvId = pCanv->createItem(pPage, this, pItem_);
	if (!canvId) {
		assert(FALSE && "canvId null after createitem!");
		goto err;          // create unsuccessful
	}
	// get new sequence number
	if (MBCmd::Instantiate(pPage) == MB_EXE_OK)
		pPage->addCanvItem(this, getSeqno(), canvId);

	return MB_EXE_OK;

 err:
	delete[] pPoints;
	return MB_EXE_ERROR;
}

int
MBCmdMLine::reverse(Page* pPage, const MBTime& t)
{
	CanvItemId cid = pPage->itemId2canvId(getSeqno());
	if (0 == cid) {
		return errorDup(pPage);
	}
	// recreate back all the required original line segments items
	if (snEnd_ < snStart_) {
		SignalError(("Error: snEnd(=%d) < snStart(=%d)!",
			     snEnd_, snStart_));
		return MB_EXE_ERROR;
	}
	MBCmdCreate* pCmd;
	for (ulong i=snStart_; i<=snEnd_; i++) {
		pCmd = (MBCmdCreate*)pPage->getCmd(i);
		if (!pCmd) break;
		if (pCmd->getType()!=CCreateItem) break;
		if (pCmd->activeAt(t)) {
			pCmd->apply(pPage, t);
		}
	}
	MBBaseCanvas* pCanv = pPage->getCanvas();
	pCanv->deleteItem(cid, NULL);
	pPage->removeCanvItem(getSeqno());
	return MB_EXE_OK;
}

//
// Assemble the image file if neccessary then create the image
//
int
MBCmdImage::apply(Page* pPage, const MBTime& /* t */)
{
	CanvItemId canvId = pPage->itemId2canvId(getSeqno());
	if (0 != canvId) {
		return errorDup(pPage);
	}

	if (snEnd_ < snStart_) {
		SignalError(("Error: snEnd(=%d) < snStart(=%d)!",
			     snEnd_, snStart_));
		return MB_EXE_ERROR;
	}
	// assemble the file from fragments if neccessary
	MBFrag** ppFrags = NULL;
	const char* szFN;
	if (!szFileName_) {
		int nFrag = snEnd_ - snStart_ + 1;
		ppFrags = new MBFrag*[nFrag];
		for (ulong index=snStart_; index <=snEnd_; index++) {
			MBCmdFrag* pCmdFrag=(MBCmdFrag*)pPage->getCmd(index);
			if (pCmdFrag->getType()!=CFrag) {
				disperr("Invalid fragments received"
					" for image!");
				goto err;
			}
			ppFrags[index-snStart_]=pCmdFrag;
		}
		szFN = MBFile::Assemble(nFrag, ppFrags);
		if (!szFN) SignalError(("Could not assemble fragments"));
		::AllocNCopy(&szFileName_, szFN);
#ifdef WIN32	//FIXME: is this neccessary anymore?
		for (int i=0; szFileName_[i]!='\0'; i++)
			if (szFileName_[i]=='\\')
				szFileName_[i]='/';
#endif
	}
	assert(pItem_);
	canvId = pPage->getCanvas()->
		createImage(pPage, this, (ImageItem*)pItem_, szFileName_);
	if (MB_EXE_OK != MBCmd::Instantiate(pPage)) goto err;
	pPage->addCanvItem(this, getSeqno(), canvId);

	delete[] ppFrags;
	return MB_EXE_OK;

 err:
	delete[] ppFrags;
	return MB_EXE_ERROR;
}

//
// Assemble the postscript file if neccessary then create the image
//
int
MBCmdPS::apply(Page* pPage, const MBTime& /* t */)
{
	CanvItemId canvId = pPage->itemId2canvId(getSeqno());
	if (0 != canvId) {
		return errorDup(pPage);
	}

	if (snEnd_ < snStart_) {
		SignalError(("Error: snEnd(=%d) < snStart(=%d)!",
			     snEnd_, snStart_));
		return MB_EXE_ERROR;
	}
	// assemble the file from fragments if neccessary
	MBFrag** ppFrags = NULL;
	const char* szFN=((PSItem*)pItem_)->getFileName();
	if (!szFN) {
		// REVIEW: put the assemble to file code in MBCmdGroup?
		int nFrag = snEnd_ - snStart_ + 1;
		ppFrags = new MBFrag*[nFrag];
		for (ulong index=snStart_; index <=snEnd_; index++) {
			MBCmdFrag* pCmdFrag=(MBCmdFrag*)pPage->getCmd(index);
			if (pCmdFrag->getType()!=CFrag) {
				disperr("Invalid fragments received "
					"for postscript!");
				goto err;
			}
			ppFrags[index-snStart_]=pCmdFrag;
		}
		szFN = MBFile::Assemble(nFrag, ppFrags);
		if (!szFN) SignalError(("Could not assemble fragments"));
		((PSItem*)pItem_)->setFileName(szFN);
	}
	assert(pItem_);
	canvId = pPage->getCanvas()->createItem(pPage, this, pItem_);
	if (MB_EXE_OK != MBCmd::Instantiate(pPage)) goto err;
	pPage->addCanvItem(this, getSeqno(), canvId);

	delete[] ppFrags;
	return MB_EXE_OK;

 err:
	delete[] ppFrags;
	return MB_EXE_ERROR;
}

//
// Creates the actual text from the characters,
// note that before this command is called
// all commands from snStart_ to snEnd_ should have arrived.
//
int MBCmdText::apply(Page* pPage, const MBTime& /* targetTime */)
{
	CanvItemId canvId = pPage->itemId2canvId(getSeqno());
	if (0 != canvId) {
		return errorDup(pPage);
	}

	// The current state of text might be in consistent
	// Loop thru, execute them in sequence to ensure we get the correct
	// state
	MBCmd* pCmd=pPage->getCmd(snStart_);
	if (!pCmd) return MB_EXE_ERROR;
	if (pCmd->getType()!=CCreateItem) {
		disperr("Invalid packet: Invalid grouping of text!");
		return MB_EXE_ERROR;
	}
	const PageItem *pItem=pCmd->getItem();
	assert(pItem && "found initial text command with no pItem_!");
	pItem->getCopy(&pItem_);
	assert(pItem_ && "getCopy unsuccessful!");

	canvId = pPage->itemId2canvId(snStart_);
	if (!canvId) {
		// the grouping will change the item to a new item
		// there should not be any command that changes the
		// first item
		disperr("Error: first text item deleted: should not occur");
		return MB_EXE_ERROR;
	}
	if (snStart_ > snEnd_) {
		disperr("Invalid packet: starting seq# "
			"larger than ending seq#");
		return MB_EXE_ERROR;
	}
	char *szText=new char[snEnd_ - snStart_ + 1];
	*szText=cchNull;
	u_short index=0;
	MBCmdChar *pCmdChar=NULL;
	// get the coord from the first cmd
	for (ulong sn=snStart_+1; sn<=snEnd_; sn++) {
		pCmdChar = (MBCmdChar*) pPage->getCmd(sn);
		if (pCmdChar->getType()!=CChar) {
			disperr("Invalid packet: text group "
				"contains of non-char items!");
			/* try next char */
			break;
		}
		char chr=pCmdChar->getChar();
		if (chr==cchDel) {
			if (index>0) index -= 1;
		} else {
			if (index!=pCmdChar->getIndex()) {
				disperr("Inconsistent data: index is "
					"incorrect");
				/* don't want to overwrite data */
				if (index >= snEnd_ - snStart_ + 1)
					break;
			}
			szText[index++]=chr;
		}
	}
	szText[index]=cchNull;            // null terminate the string
	assert(index < snEnd_ - snStart_ + 1 &&
	       "memory could be overwritten");

	pPage->getCanvas()->setText(canvId, this, szText);
	assert(pItem_->getType()==PgItemText);

	// deletion will be done by TextItem destr
	((TextItem*)pItem_)->setText(szText);

	if (MB_EXE_OK != MBCmd::Instantiate(pPage))
		goto error;

	// change the mapping to map new item to old item
	pPage->removeCanvItem(snStart_);
	pPage->addCanvItem(this, getSeqno(), canvId);
	return MB_EXE_OK;

 error:
	delete[] szText;
	return MB_EXE_ERROR;
}

/*virtual*/
int MBCmdText::reverse(Page* pPage, const MBTime& /* t */)
{
	CanvItemId canvId = pPage->itemId2canvId(getSeqno());
	if (0 == canvId) {
		return errorDup(pPage);
	}
	/* associate the item with the starting text command rather
	 * than this command */
	pPage->removeCanvItem(getSeqno());
	MBCmd* pCmd = pPage->getCmd(snStart_);
	if (!pCmd) {
		assert(0 && "cmdtext::reverse called when incomplete?");
		return MB_EXE_ERROR;
	}
	pPage->addCanvItem(pCmd, snStart_, canvId);
	return MB_EXE_OK;
}

int MBCmdGroup::apply(Page* /* pPage */, const MBTime& /* targetTime */)
{
	assert(FALSE && "only children of cmdgroup should be executed");
	return MB_EXE_ERROR;
}

int MBCmdGroup::reverse(Page* /* pPage */, const MBTime& /* targetTime */)
{
	assert(FALSE && "only children of cmdgroup should be reversed");
	return MB_EXE_ERROR;
}

//
// Just update the character, will worry about consistency when the ending
// group functions arrive.
//

/* virtual */
int
MBCmdChar::apply(Page* pPage, const MBTime& /* targetTime */)
{
	// FIXME: right now we cannot check if the command has been
	// applied twice in a row!
	CanvItemId cid = pPage->itemId2canvId(targetItem_);
	if (!cid) {
		disperr("Inserting char into non-existent text item");
		return MB_EXE_ERROR;
	}

	MBBaseCanvas* pCanv=pPage->getCanvas();
	int len = index_ + 2;
	char *szText=pCanv->getText(cid, len);
	// take care of delete, right now just do this:
	// if index is in range, delete it, else ignore
	// (should defer, but...)
	char* szTmp = szText;
	if (char_ == cchDel) {
		if ((index_ <= len) && (index_ > 0)) {
			szTmp[index_-1] = cchNull;
			szText=Concat(szTmp, szTmp+index_);
		}
	} else {
		// expand with spaces if needed, either way set the character
		int slen = strlen(szTmp);
		if (index_ >= slen) {
			assert(len >= index_ + 2 &&
			       "getText should guarantee this");
			char *pChr = szText + slen;
			for (; pChr < szText + index_; pChr++) {
				*pChr = cchSpc;
			}
			*(pChr+1) = cchNull;
		}
		szText[index_]=char_;
	}
	/* don't assoc item with this command yet, this will be done
	 * at the grouptext command */
	pCanv->setText(cid, 0, szText);
	if (szTmp != szText) {
		delete[] szText;
	}
	delete[] szTmp;
	return MBCmd::Instantiate(pPage);
}

int
MBCmdChar::reverse(Page* pPage, const MBTime& /* t */)
{
	/* PRECONDITION: all commands that came before this command
	 * MUST be reversed! */

	CanvItemId cid = pPage->itemId2canvId(targetItem_);
	if (!cid) {
		disperr("MBCmdChar::reverse cannot find target item!");
		return MB_EXE_ERROR;
	}
	MBBaseCanvas* pCanv = pPage->getCanvas();
	if (!pCanv) return MB_EXE_ERROR;
	int len = index_ + 2;
	char *szText=pCanv->getText(cid, len);

	if (char_ == cchDel) {
		// we are reversing so we check to make sure we are at
		// the state where the command is...
		if ((long)strlen(szText) != index_ - 1) {
			perr(("mbcmdchar::reverse: del char with wrong len"));
			perr(("    duplicated execution?"));
			return errorDup(pPage);
		}

		if ((index_ >= len) || (index_ < 1)) {
			/* FIXME: ignore? */
			return MB_EXE_OK;
		}
		MBCmd* pCmd;
		MBCmdChar* pCmdChar=NULL;
		ulong i;
		for (i=getSeqno(); i >= targetItem_; i--) {
			pCmd=pPage->getCmd(i);
			if (!pCmd || pCmd->getType()!=CChar) {
				disperr("Invalid packet: text group "
					"contains of non-char items!");
				/* try next char */
				break;
			}
			pCmdChar = DYN_CAST(MBCmdChar*)(pCmd);
			if (pCmdChar->getIndex() == (u_int)index_ - 1) {
				break;
			}
		}
		if (i< targetItem_) {
			disperr("error: reverse cannot find the character!");
			return MB_EXE_OK; /* ignore error for now */
		}
		assert((pCmdChar && ((u_int)index_ - 1 == pCmdChar->getIndex()) )
		       || !"check above loop");
		szText[index_-1] = pCmdChar->getChar();
		szText[index_] = cchNull;
	} else {
		/* non-space character, just delete the character */
		if (strlen(szText) <= index_) {
			perr(("mbcmdchar::reverse: len of string too small"));
			perr(("    possible duplicate reverse?"));
			return errorDup(pPage);
		}

		/* use a tmp copy for the rest of the string since
		 * strcpy must have unspecified behavior when the
		 * parameters are from a same string */
		char *szTmp;
		::AllocNCopy(&szTmp, szText+index_+1);
		strcpy(szText+index_, szTmp);
		delete[] szTmp;
	}
	pCanv->setText(cid, 0, szText);
	delete[] szText;
	return MB_EXE_OK;
}


int MBCmdMove::executeInteractive(Page* pPage)
{
	assert(pPage && "page is null, should not happen");
	CanvItemId canvId=pPage->itemId2canvId(targetId_);
	if ( 0 != canvId ) {        // found it
		if (pPage->getCanvas()->moveItem(canvId, NULL, dx_, dy_)) {
				pPage->RaiseToFront(targetId_);
				return MB_EXE_OK;
		}
	}
	perror(("Item not found, cannot move"));
	return MB_EXE_ERROR;
}

int MBCmdMove::apply(Page* pPage, const MBTime& /* targetTime */)
{
	if (pPage->itemId2canvId(getSeqno()))
		return errorDup(pPage);
	CanvItemId canvId=pPage->itemId2canvId(targetId_);
	if ( 0 != canvId ) {        // found it
		if (pPage->getCanvas()->moveItem(canvId, this, dx_, dy_)) {
			pPage->removeCanvItem(targetId_);
			if (MB_EXE_OK == MBCmd::Instantiate(pPage)) {
				pPage->addCanvItem(this, getSeqno(),canvId);
				MTrace(trcMB|trcExcessive,
				       ("move %lu to %lu",
					targetId_, getSeqno()));
				return MB_EXE_OK;
			}
		}
	} else {
		perr(("Item not found, cannot move (duplicate apply calls?)"));
		return errorDup(pPage);
	}
	return MB_EXE_OK;
}

int MBCmdMove::reverse(Page* pPage, const MBTime& /* targetTime */)
{
	CanvItemId canvId=pPage->itemId2canvId(targetId_);
	if ( 0 == canvId ) {        // target item not there yet
		canvId = pPage->itemId2canvId(getSeqno());
		if (canvId==0) {
			disperr("cannot find own item in reverse move");
			return MB_EXE_ERROR;
		}
		MBCmd* pTgtCmd = pPage->getCmd(targetId_);
		if (pPage->getCanvas()->moveItem(canvId, pTgtCmd, -dx_, -dy_)) {
			pPage->removeCanvItem(getSeqno());
			pPage->addCanvItem(this, targetId_, canvId);
			MTrace(trcMB|trcVerbose,
			       ("rev move %lu to %lu", getSeqno(), targetId_));
			return MB_EXE_OK;
		}
	} else {
		perr(("target item found, dup reverse calls?)"));
		return errorDup(pPage);
	}
	return MB_EXE_OK;
}

int MBCmdDup::apply(Page* pPage, const MBTime& /* targetTime */)
{
	CanvItemId cid = pPage->itemId2canvId(getSeqno());
	if (0 != cid) {
		return errorDup(pPage);
	}

	if (pItem_ == NULL) {
		// item might have changed since
		// re-construct the item by:
		// - thread through the commands until we find one with a copy
		//   of the item
		// - execute the thread of commands to recover the item
		MBCmd* pCmd = pPage->getCmd(targetId_);
		MBStack<MBCmd*> *pStack = new MBStack<MBCmd*>(100);
		while (!pCmd->getItem()) {
			// for now we should only have moves in-between
			if (pCmd->getType()!=CMove) {
				assert(pCmd->getType()==CMove);
				return MB_EXE_ERROR;
			}
			pStack->Push(pCmd);
			pCmd=pPage->getCmd(((MBCmdMove*)pCmd)->getTargetId());
		}
		PageItem* pItem=NULL;
		pCmd->getItem()->getCopy(&pItem);
		assert(pItem);
		while (pStack->Pop(pCmd)) {
			((MBCmdMove*)pCmd)->Apply(pItem);
		}
		delete pStack;
		pItem_ = pItem;
	}
	CanvItemId newCanvItem=
		pPage->getCanvas()->createItem(pPage, this, pItem_);

	if (MB_EXE_OK!=MBCmd::Instantiate(pPage))
		return MB_EXE_ERROR;
	pPage->addCanvItem(this, getSeqno(), newCanvItem);
	return MB_EXE_OK;
}

int MBCmdDup::reverse(Page* pPage, const MBTime& /* targetTime */)
{
	return removeAssocItem(pPage);
}

/* virtual */
int MBCmdDel::apply(Page* pPage, const MBTime& /* targetTime */)
{
	MBBaseCanvas *pCanv = pPage->getCanvas();
	// if we are deleting an existing item, delete it
	if (!itemId_) {
		SignalError(("Error deleting null item"));
		return MB_EXE_ERROR;
	}
	CanvItemId canvId=pPage->itemId2canvId(itemId_);
	if (canvId==0) {
		return errorDup(pPage);
	}
	/* FIXME: make canvas return PageItem right away and throw away
	 * the canvItem code */
	MBCanvItem *pCanvItem = pCanv->getItem(canvId);
	if (!pCanvItem) return MB_EXE_ERROR;
	pPageItem_ = pCanvItem->toPageItem();

	if (!pCanv->deleteItem(canvId, this)) {
		return MB_EXE_ERROR;
	}
	pPage->removeCanvItem(itemId_);

	// success
	return ( MBCmd::Instantiate(pPage) );
}

/* virtual */
int MBCmdDel::reverse(Page* pPage, const MBTime& /* targetTime */)
{
	//  - save a copy of the pageitem, in case we need to recreate it
	//    when there is a subsequent copy of the item.
	MBCmd* pCmd = pPage->getCmd(itemId_);
	assert(pCmd || !"CmdDel::reverse called when incomplete!");
	CanvItemId canvId =
		pPage->getCanvas()->createItem(pPage, pCmd, pPageItem_);
	if (canvId==0) {
		return MB_EXE_OK;
	}
	pPage->addCanvItem(pCmd, itemId_, canvId);

	assert(pPageItem_);
	return MB_EXE_OK;
}

/* virtual */
int
MBCmdPgName::apply(Page* pPage, const MBTime& /* targetTime */)
{
	if (!pPage) {
		SignalError(("page is null, should not happen"));
		return MB_EXE_ERROR;
	}
	pPage->ChangeName(szPgName_);
	return (MBCmd::Instantiate(pPage));
}

int
MBCmdPgName::reverse(Page* /* pPage */, const MBTime& /* targetTime */)
{
	// does not do any thing now, it effect it should notify the
	// ui of the hiding of a page.
	return MB_EXE_OK;
}

#if 0
//
// MBCmdCoord member functions
//
///////////

//
// Returns 0 if it is completed, or the first item that we depend on
//
/* virtual */
n_long MBCmdCoord::Incomplete(Page* pPage) const
{
	MBCmd *pCmd=NULL;
	// only depends on the targetItem_
	if (!(pCmd=pPage->getCmd(targetId_)))
		return targetId_;
	return pCmd->Incomplete(pPage);
}

/* virtual */
Bool MBCmdCoord::execute(Page* pPage)
{
	assert(pPage && "page is null, should not happen");
	n_long canvId=pPage->itemId2canvId(targetId_);
	if ( 0 != canvId ) {        // found it
		if (pPage->getCanvas()->ItemCoord(canvId, x1_, y1_, x2_, y2_))
			return ( MBCmd::Instantiate(pPage) );
	}
	return MB_EXE_ERROR;
}

/* virtual */
Bool MBCmdCoord::executeInteractive(Page* pPage)
{
	assert(pPage && "page is null, should not happen");
	n_long canvId=pPage->itemId2canvId(targetId_);
	if ( 0 != canvId ) {        // found it
		// don't instantiate since it's interactive
		if (pPage->getCanvas()->ItemCoord(canvId, x1_, y1_, x2_, y2_))
			return MB_EXE_OK;
	}
	return MB_EXE_ERROR;
}

// packetization
Byte* MBCmdCoord::Extract(Byte* pb) {
	Pkt_CmdCoord *pPkt = (Pkt_CmdCoord*) pb;
	targetId_ = net2host(pPkt->targetId);
	x1_ = net2host(pPkt->x1);
	y1_ = net2host(pPkt->y1);
	x2_ = net2host(pPkt->x2);
	y2_ = net2host(pPkt->y2);
	return (Byte*)NULL;                // return value is not needed
}

Byte* MBCmdCoord::Packetize(Byte* /*pb*/)
{
	assert(FALSE && "obsolete function!");
	return NULL;
	Byte* pbCurr = MBCmd::Packetize(pb);
	Pkt_CmdCoord *pPkt = (Pkt_CmdCoord*) pbCurr;
	pPkt->targetId = host2net(targetId_);
	pPkt->x1 = host2net(x1_);
	pPkt->y1 = host2net(y1_);
	pPkt->x2 = host2net(x2_);
	pPkt->y2 = host2net(y2_);
	return (Byte*)(pPkt+1);
}

#endif // 0

