/*
 * session-rtcp.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <math.h>
#include <errno.h>
#include <string.h>
#ifdef WIN32
extern "C" int getpid();
#endif
#include "source.h"
#include "tclcl.h"
#include "ntp-time.h"
#include "session-rtcp.h"

RTCP_Session::RTCP_Session() {
  sm_ = 0;

  for(int i=0; i < NLAYER; i++) {
    ch_[i].manager(this);
  }

  pktbuf_ = new u_char[2 * RTP_MTU];
};

RTCP_Session::~RTCP_Session()
{
	delete pktbuf_;
}

int
RTCP_Session::command(int argc, const char*const* argv)
{
  Tcl& tcl = Tcl::instance();
  char* cp = tcl.buffer();
  if (argc == 2) {
    if (strcmp(argv[1], "sm") == 0) {
      if (sm_ != 0) {
	strcpy(cp, sm_->name());
      } else {
	strcpy(cp, "");
      }
      tcl.result(cp);
      return (TCL_OK);
    }
    if (strcmp(argv[1], "active") == 0) {
      sm_->sortactive(cp);
      tcl.result(cp);
      return (TCL_OK);
    }
  } else if (argc == 3) {
    if (strcmp(argv[1], "sm") == 0) {
      sm_ = (SourceManager*)TclObject::lookup(argv[2]);
      return (TCL_OK);
    }
    if (strcmp(argv[1], "data-net") == 0) {
      return (TCL_OK);
    }
    if (strcmp(argv[1], "ctrl-net") == 0) {
      ch_[0].net((Network*)TclObject::lookup(argv[2]));
      return (TCL_OK);
    }
  } else if (argc == 4) {
    if (strcmp(argv[1], "data-net") == 0) {
      return (TCL_OK);
    }
    if (strcmp(argv[1], "ctrl-net") == 0) {
      u_int layer = atoi(argv[3]);
      if (layer >= NLAYER)
	abort();
      if (*argv[2] == 0) {
	ch_[layer].net(0);
	return (TCL_OK);
      }
      Network* net = (Network*)TclObject::lookup(argv[2]);
      if (net == 0) {
	tcl.resultf("no network %s", argv[2]);
	return (TCL_ERROR);
      }
      if (net->rchannel() < 0) {
	tcl.resultf("network not open");
	return (TCL_ERROR);
      }
      ch_[layer].net(net);
      return (TCL_OK);
    }
  }
  return TclObject::command(argc, argv);
}

void
RTCP_Session::recv(DataHandler * /* dh */)
{
  return;
}

void
RTCP_Session::recv(CtrlHandler *ch)
{
  u_int32_t src;
  int port;
  int cc = ch->recv(pktbuf_, 2 * RTP_MTU, src, port);
  if (cc <= 0)
    return;

  rtcphdr* rh = (rtcphdr*)pktbuf_;
  if (cc < int(sizeof(*rh))) {
    // Runt packet
    return;
  }
  /*
   * try to filter out junk: first thing in packet must be
   * sr, rr or bye & version number must be correct.
   */
  switch(ntohs(rh->rh_flags) & 0xc0ff) {
  case RTP_VERSION << 14 | RTCP_PT_SR:
  case RTP_VERSION << 14 | RTCP_PT_RR:
  case RTP_VERSION << 14 | RTCP_PT_BYE:
    break;
  default:
    /*
     * FIXME should further categorize this error -- it is
     * likely that people mis-implement applications that
     * don't put something other than SR,RR,BYE first.
     */
    // Bad version
    return;
  }
  /*
   * at this point we think the packet's valid.  Update our average
   * size estimator.  Also, there's valid ssrc so charge errors to it
   */
  ch->sample_size(cc);
  u_int32_t addr = src;

  /*
   * First record in compound packet must be the ssrc of the
   * sender of the packet.  Pull it out here so we can use
   * it in the sdes parsing, since the sdes record doesn't
   * contain the ssrc of the sender (in the case of mixers).
   */
  u_int32_t ssrc = rh->rh_ssrc;
  Source* ps = sm_->lookup(this, ssrc, ssrc, addr);
  if (ps == 0)
    return;

  int layer = ch - ch_;
  /*
   * Outer loop parses multiple RTCP records of a "compound packet".
   * There is no framing between records.  Boundaries are implicit
   * and the overall length comes from UDP.
   */
  u_char* epack = (u_char*)rh + cc;
  while ((u_char*)rh < epack) {
    u_int len = (ntohs(rh->rh_len) << 2) + 4;
    u_char* ep = (u_char*)rh + len;
    if (ep > epack) {
      ps->badsesslen(1);
      return;
    }
    u_int flags = ntohs(rh->rh_flags);
    if (flags >> 14 != RTP_VERSION) {
      ps->badsessver(1);
      return;
    }
    switch (flags & 0xff) {

    case RTCP_PT_SR:
      parse_sr(rh, flags, ep, ps, addr, layer);
      break;

    case RTCP_PT_RR:
      parse_rr(rh, flags, ep, ps, addr, layer);
      break;

    case RTCP_PT_SDES:
      parse_sdes(rh, flags, ep, ps, addr, ssrc, layer);
      break;

    case RTCP_PT_BYE:
      parse_bye(rh, flags, ep, ps);
      break;

    case RTCP_PT_APP:
      parse_app(rh, flags, ep, ps);
      break;

    default:
      ps->badsessopt(1);
      break;
    }
    rh = (rtcphdr*)ep;
  }
  sm_->CheckActiveSources(5000);

}

void RTCP_Session::parse_app(rtcphdr*, int, u_char*, Source*)
{
}

void RTCP_Session::parse_rr_records(u_int32_t, rtcp_rr*, int,
				      const u_char*, u_int32_t)
{
}

void RTCP_Session::parse_sr(rtcphdr* rh, int flags, u_char*ep,
			    Source* ps, u_int32_t addr, int layer)
{
	rtcp_sr* sr = (rtcp_sr*)(rh + 1);
	Source* s;
	u_int32_t ssrc = rh->rh_ssrc;
	if (ps->srcid() != ssrc)
		s = sm_->lookup(this, ssrc, ssrc, addr);
	else
		s = ps;

	Source::Layer& sl = s->layer(layer);
	sl.lts_ctrl(unixtime());
	sl.sts_ctrl(ntohl(sr->sr_ntp.upper) << 16 |
		    ntohl(sr->sr_ntp.lower) >> 16);

	int cnt = flags >> 8 & 0x1f;

	Tcl& tcl = Tcl::instance();
	tcl.evalf("%s notify_observers sr_received %s", sm_->name(),
		  s->name());

	parse_rr_records(ssrc, (rtcp_rr*)(sr + 1), cnt, ep, addr);
}

void RTCP_Session::parse_rr(rtcphdr* rh, int flags, u_char* ep,
			      Source* ps, u_int32_t addr, int layer)
{
	Source* s;
	u_int32_t ssrc = rh->rh_ssrc;
	if (ps->srcid() != ssrc)
		s = sm_->lookup(this, ssrc, ssrc, addr);
	else
		s = ps;

	s->layer(layer).lts_ctrl(unixtime());
	int cnt = flags >> 8 & 0x1f;
	parse_rr_records(ssrc, (rtcp_rr*)(rh + 1), cnt, ep, addr);
}

int RTCP_Session::sdesbody(u_int32_t* p, u_char* ep, Source* ps,
			     u_int32_t addr, u_int32_t ssrc, int layer)
{
	Source* s;
	u_int32_t srcid = *p;
	if (ps->srcid() != srcid)
		s = sm_->lookup(this, srcid, ssrc, addr);
	else
		s = ps;
	if (s == 0)
		return (0);
	/*
	 * Note ctrl packet since we will never see any direct ctrl packets
	 * from a source through a mixer (and we don't want the source to
	 * time out).
	 */
	s->layer(layer).lts_ctrl(unixtime());

	u_char* cp = (u_char*)(p + 1);
	while (cp < ep) {
		char buf[256];

		u_int type = cp[0];
		if (type == 0) {
			/* end of chunk */
			return (((cp - (u_char*)p) >> 2) + 1);
		}
		u_int len = cp[1];
		u_char* eopt = cp + len + 2;
		if (eopt > ep)
			return (0);

		if (type >= RTCP_SDES_MIN && type <= RTCP_SDES_MAX) {
			memcpy(buf, (char*)&cp[2], len);
			buf[len] = 0;
			s->sdes(type, buf);
		} else
			/*FIXME*/;

		cp = eopt;
	}
	return (0);
}

void RTCP_Session::parse_sdes(rtcphdr* rh, int flags, u_char* ep, Source* ps,
				u_int32_t addr, u_int32_t ssrc, int layer)
{
	int cnt = flags >> 8 & 0x1f;
	u_int32_t* p = (u_int32_t*)&rh->rh_ssrc;
	while (--cnt >= 0 && (u_char*)p < ep) {
		int n = sdesbody(p, ep, ps, addr, ssrc, layer);
		if (n == 0)
			break;
		p += n;
	}
	if (cnt >= 0)
		ps->badsdes(1);
}

void RTCP_Session::parse_bye(rtcphdr* rh, int flags, u_char* ep, Source* ps)
{
	int cnt = flags >> 8 & 0x1f;
	u_int32_t* p = (u_int32_t*)&rh->rh_ssrc;

	while (--cnt >= 0) {
		if (p >= (u_int32_t*)ep) {
			ps->badbye(1);
			return;
		}
		Source* s;
		if (ps->srcid() != rh->rh_ssrc)
			s = sm_->consult(*p);
		else
			s = ps;
		if (s != 0)
			s->lts_done(unixtime());
		++p;
	}
}



void
RTCP_Session::announce(CtrlHandler * /* ch */)
{
  return;
}

static class RTCPSessionClass : public TclClass {
public:
  RTCPSessionClass() : TclClass("Session/RTCP") {}
  TclObject *create(int, const char*const*) {
    return (new RTCP_Session);
  }
} rtcp_session_class_;

