# room-manager.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1996-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/room-manager/room-manager.tcl,v 1.10 2002/02/03 04:22:27 lim Exp $


import mashutils Trace HashTable RendezvousManager Servent

#Trace on ; Trace add LocalHostManager

#
# The LocalHostManager
# - defines new virtual scopes (ie, for use as
#   and argmument to a -rendez option somewhere)
# - accepts querys against the cached list of messages sent on scopes it
#   is listening on
# - advertises the scope (if it chooses to define one) and itself
# - advertises that it can allocate certain services that
#   are machine-specific and thus not for use with a HostManager.
#   These are defined in a "srv" file, which defaults to
#   ~/.mash/localsrvs/FQDN_OF_LOCALMACHINE .  (should be /etc/lhm-srvs).
#   Override with the -srvfile arg.
# - accepts requests to allocate these certain "services"
#<br>
# It advertises the new scopes (`scope: ...') and
# itself (`rv-cache: ...') in the top-level virtual scope
#<br>
# The hard-coded rendez spec is the default *global*
# rendezvous spec, ie, the toplevel "virtual scope".  The default
# can be overridden to allow a multi-level hierarchy; i.e., the
# -rendez option address is considered the "top-level" rendezvous
# channel, and this agent listens/advertises only on it and in "lower"
# channels advertised on it.
#
#
Class LocalHostManager -superclass Observer -configuration {
    rendez 224.2.127.253/1202
}

#
#
LocalHostManager public init {} {
    $self next
    $self read_srv_file
    $self init_ctrl
    $self init_network
    $self init_srvs
    $self announce_proxysrvs
}

#
LocalHostManager public destroy {} {
    $self instvar rv_
    $rv_ detach_observer $self
    $self next
}


# reads file that lists which servents can be allocated by
# this LocalHostManager.  File formatted as lines, each consisting of: <br>
#
# {advert for srv} {http://... srv script URL} {machine} {args}
#
LocalHostManager private read_srv_file {} {
    Trc $class "--> ${class}::$proc"
    $self instvar srvs_

    set srvs_ ""
    set f [$self get_option srvfile]
    if {$f != ""} {
	if [file readable $f] {
	    puts "reading in srvs from `$f'"
	    set fd [open $f r]
	    set filedata [read $fd]
	    set linedata [split $filedata "\n"]
	    foreach i $linedata {
		set j [string trim $i]
		if {$j == ""} {continue}
		if {[string index $j 0] == "#"} {continue}
		lappend srvs_ [new Servent $j]
	    }
	    close $fd
	} else {
	    puts "Srv file `$f' not found/readable..."
	}
    } else {
	puts "No srv file specified."
    }
}


# reads file that lists proxy announcements and start announcing them
#
LocalHostManager private announce_proxysrvs {} {
    Trc $class "--> ${class}::$proc"
    $self instvar rv_

    set proxy_srvs_ ""
    set f [$self get_option proxysrvfile]
    if {$f != ""} {
	if [file readable $f] {
	    puts "reading in proxy srvs from `$f'"
	    set fd [open $f r]
	    set filedata [read $fd]
	    set linedata [split $filedata "\n"]
	    foreach i $linedata {
		set j [string trim $i]
		if {$j == ""} {continue}
		if {[string index $j 0] == "#"} {continue}
		Trc $class "proxy announcing `$j'"

		if {[$self myvscopes] != ""} {
		    foreach vs [$self myvscopes] {
			$rv_ start $vs $j
		    }
		} else {
		    $rv_ start [$rv_ get_local_rv] $j
		}

	    }
	    close $fd
	} else {
	    puts "ProxySrv file `$f' not found/readable..."
	}
    } else {
	puts "No proxysrv file specified."
    }
}


#
LocalHostManager private init_ctrl {} {
    Trc $class "--> ${class}::$proc"
    $self instvar unicastmgr_ myport_
    set myport_ [$self get_option ctrlport]
    if {$myport_ == ""} {
	set myport_ [$self alloc_port]
    }
    set unicastmgr_ [new LocalHostManagerCtrl $myport_ $self]
    puts "LocalHostManagerCtrl listening on [localaddr]/$myport_"
}


# announce the new "virtual scopes" defined by this LocalHostManager
# (if there are any), and advertise that it caches and accepts
# queries against msgs send on these scopes.
# <br>
# The vscopes option is assumed to be a list of the form
# <br> `scopeName scopeSpec'.
LocalHostManager private init_network {} {
    Trc $class "--> ${class}::$proc"
    $self instvar rv_

    set r [$self get_option rendez]
    set rv_ [new RendezvousManager $r]  ;# (r != "") because of default
    $rv_ attach_observer $self

    set vsopt [$self get_option vscope]
    set vsl ""
    foreach v $vsopt {eval lappend vsl $v}
    foreach {vsn vss} $vsl {
	puts "vscope = `$vsn' on `$vss'"
	$rv_ add_spec $vss
	$self announce_vs $vsn $vss
	$self announce_cache $vss
    }
}


# announce `rv-cache:' msgs on and for spec `s'
LocalHostManager private announce_cache {s} {
    Trc $class "--> ${class}::$proc"
    $self instvar rv_ myport_
    $rv_ start $s "rv-cache: rspec=$s ctrladdr=[localaddr]/$myport_"
}

# announce virtual scope with name `name' and spec `spec'
# on the toplevel rv channel and in the new vs
LocalHostManager private announce_vs {name spec} {
    Trc $class "--> ${class}::$proc"
    $self instvar rv_

    set r [$self get_option rendez]
    puts "announcing scope msgs on `$r' and `$spec'"
    $rv_ start $r "scope: name=$name spec=$spec"
    $rv_ start $spec "scope: name=$name spec=$spec"
}

# init and assert `can-allocate:' msgs
#
LocalHostManager private init_srvs {} {
    Trc $class "--> ${class}::$proc"
    $self instvar srvs_ rv_ url2file_ srv_ads_
    set url2file_ [new HashTable]

    if {$srvs_==""} {return}

    foreach s $srvs_ {
	append msg "can-allocate: "
	append msg [$s rv_msg]
	append msg " uniqid=[$s uniqid]\n"
    }

    if {[$self myvscopes] == ""} {
	$rv_ start [$rv_ get_local_rv] $msg
	set srv_ads_([$rv_ get_local_rv]) $msg
    } else {
	foreach vs [$self myvscopes] {
	    $rv_ start $vs $msg
	}
    }
}

#
LocalHostManager public myvscopes {} {
    Trc $class "--> ${class}::$proc"
    set reply ""
    set vsl [$self get_option vscope]
    foreach s $vsl {
	lappend reply [lindex $s 1]
    }
    return $reply
}

#
LocalHostManager instproc alloc_port {  } {
    # Random number U[8192, 16384]
    set r01 [expr [random]/double(0x7fffffff)]
    set r02 [expr round(8392 + $r01 * 8192)]
    if {$r02 % 2 == 1} {incr r02 -1}
    return $r02
}

#
LocalHostManager instproc alloc_mcast_addr {  } {
    set lo1 round([expr [random]/double(0x7fffffff) * 256])
    set lo2 round([expr [random]/double(0x7fffffff) * 256])
    return "224.3.[expr round($lo1)].[expr round($lo2)]"
}


# recv a scope msg -- see if we need to update which channel
# our announcements are sent on.  If we are not "asserting"
# virtual scope(s), keep announcements on the "local" vscope
#
LocalHostManager public rendez_recv_scope {rspec addr port data size} {
    Trc $class "--> ${class}::$proc"
    if {[$self myvscopes] == ""} {
	$self instvar rv_ srv_ads_
	set l [$rv_ get_local_rv]
	Trc $class "local_rv = $l"

	if ![info exists srv_ads_($l)] {
	    foreach i [array names srv_ads_] {
		set msg $srv_ads_($i)
		$rv_ stop $i $msg
		unset srv_ads_($i)
	    }
	    set srv_ads_($l) $msg
	    $rv_ start $l $msg
	}
    }
}


#----------------------------------------
# query resolution
#----------------------------------------



# called when msg of type "query:" is received. If it was received
# on a rv channel, the rspec arg is set to its spec, otherwise it
# was received via unicast and rspec is set to "-".
#<br>
# The msg payload is assumed to be of the form:
#<br>
#   "query: ?vs=(scopename|scopespec)? ?rport=num? query=queryString"
#<br>
# Replies are sent either to the issuing rv channel or via unicast
# as appropriate.  [FIXME- this currently ignores scopename/scopespec
#                   until RendezvousManager is fixed to handle it.]
#
LocalHostManager public rendez_recv_query {rvmsg} {
    Trc $class "--> ${class}::$proc"
    $self instvar rv_

    set queryrspec [$rvmsg get_field vs]
    set querystring [$rvmsg get_field query]
    set reply "query-reply: `"
    append reply [$rv_ query $querystring]
    append reply "' for [list $querystring]"
    if {$queryrspec != ""} {
	append reply " on $queryrspec"
    }

    Trc $class "sending `$reply'"
    if {[$rvmsg rspec] == "-"} {
	# unicast the response
	set rport [$rvmsg get_field rport]
	if {$rport==""} {
	    set rport [$rvmsg sender_port]
	}
	set u [new UDPChannel [$rvmsg sender_addr]/$rport]
	$u send $reply
	delete $u
    } else {
	# multicast the response
	# FIXME don't want to keep periodically transmitting query replies
	#     too long...
	$rv_ start [$rvmsg rspec] $reply
    }
}



#----------------------------------------
# exec local processes
#----------------------------------------



# Accept allocation requests for services we have advertised via
# "can-allocate"
# <p>
# Check if the allocation msg refers to a service in the srv file,
# and check that said srv hasn't been heard from ever/in a while.
# If both are true, exec the srv.
# <p>
# Note that `-rendez FIXME -uniqid YYY' is added to execargs.
#
LocalHostManager public rendez_recv_allocate {rvmsg} {
    Trc $class "--> ${class}::$proc [$rvmsg data]"
    $self instvar rv_ url2file_

    set req_srv [$self get_srvref $rvmsg]
    if {$req_srv == ""}  {
	Trc $class "allocation request:\n `$data'\n for non-localhost"
	return
    }

    if ![$self srv_running $req_srv] {
	# if it was exec'd too recently, wait a bit to retry
	if {[$req_srv pid] > 0} {
	    Trc $class "checking if exec'd < 10s ago"
	    Trc $class "=> [expr [clock seconds]-[$req_srv exectime]]s. ago"
	    if {[expr [clock seconds]-[$req_srv exectime]] < 10} {
		return
	    }
	}

	# get script
	set url [$req_srv url]
	set localfile [$req_srv scriptfile]
	if {$localfile == ""} {
	    set localfile [$url2file_ exists $url]
	    if {$localfile == -1} {
		set localfile [$self generate_filename]
		$url2file_ create $url $localfile
	    }
	    $req_srv set_scriptfile $localfile
	}
	if ![file exists $localfile] {
	    if {[$self copyURLtoFile $url $localfile] == 0} {return}
	    #file attributes $localfile -permissions 00755
	}

	# exec
	set filename [$req_srv scriptfile]
        set tmppath [$self get_option tmppath]
	#set outputf ">& $tmppath/[file tail $filename]-[$req_srv uniqid].out"
	set outputf ""

        set path [$self get_option execpath]
	# FIXME
	set rspec [$rvmsg rspec]
	if {$rspec == "-"} {set rspec [lindex [$self myvscopes] 0]}
	set execargs "-rendez $rspec -uniqid [$req_srv uniqid]"
	set execargs "$execargs [$req_srv execargs]"
        set execstr "$path/[$req_srv execcmd] $filename $execargs $outputf"

        puts " -- exec'ing new servent"
	puts " `$execstr'"
        if { [catch "eval exec [list $execstr] &" pid] != 0 } {
	    global errorCode
	    puts "Error in exec of '$execstr': $errorCode"
	    $req_srv set_pid -1
	    return -1
        } else {
	    Trc $class "exec successful, pid = $pid"
        }
	$req_srv set_pid $pid
	$req_srv set_exectime [clock seconds]
        return 0
    }
    return 0
}

# returns a ref to the Servent object indicated in
# `msg', or {} if it is either invalid (improperly formatted)
# does not refer to a Servent we can allocate
#
LocalHostManager private get_srvref {msg} {
    Trc $class "--> ${class}::$proc"
    Trc detail-$class "`[$msg get_msg]'"
    $self instvar srvs_

    set uid [$msg get_field uniqid]
    foreach s $srvs_ {
	#Trc $class "uid=$uid ?==? srvuid=[$s uniqid]"
	if {$uid == [$s uniqid]} {
	    return $s
	}
    }
    return ""
}

# see if servent `srv' is running by checking if it was exec'd
# and if it has been heard from less then `timeout' secs ago.
#
LocalHostManager private srv_running {srv {timeoutsecs 20}} {
    Trc $class "--> ${class}::$proc $srv"
    $self instvar rv_

    set pid [$srv pid]
    Trc $class "srv pid=$pid"
    if {$pid >= 0} {
	# exec'd it before -- still alive?
	set qry "will-provide: & uniqid=[$srv uniqid]"
	Trc $class "srv query is `$qry'"
	set q [$rv_ query $qry]
	if {$q == ""} {
	    # its gone!
	    Trc $class "srv gone: no announcement(s) found"
	    return 0
	} else {
	    Trc $class "matching msg= $q"
	    # its here, but check against (possibily tighter) `timeoutsecs'
	    set q [$rv_ query_metadata $qry]
	    Trc $class "msg metadata= [$q get_metadata]"
	    set t [$q get_meta_field time]
	    set currTime [clock seconds]
	    if {[expr $currTime - $t] > $timeoutsecs} {
		Trc $class "srv not heard from in >$timeoutsecs secs"
		return 0
	    }
	    Trc $class "srv here recently"
	}
	return 1
    } else {
	if {$pid == -1} {puts "Warning: attempting to redo a failed exec."}
    }
    return 0
}


# FIXME - needs to check that the URL is valid
#
#
LocalHostManager private copyURLtoFile {url file {chunk 4096} } {
    Trc $class "--> ${class}::$proc $url $file"

    set urlhost [lindex [split $url /] 2]
    set urlhost [lindex [split $urlhost :] 0]
    if {[gethostbyname $urlhost] == ""} {
	puts "copyURLtoFile: Bad hostname in URL: $urlhost"
	return 0
    }

    puts "Retreiving URL $url ..."
    set out [open $file w]
    set token [::http::geturl $url -channel $out -blocksize $chunk]
    close $out
    set retVal [::http::code $token]
    if {[lindex $retVal 1] != 200} {
	puts "Error: HTTP request failed: $retVal"
	file delete $file
	return 0
    }
    ::http::reset $token
    return $token
}

#
LocalHostManager private generate_filename {} {
    $self instvar uniq_idx_
    if ![info exists uniq_idx_] {set uniq_idx_ 0}
    set r "/var/tmp/servent-${uniq_idx_}.mash"
    while [file exists $r] {
	incr uniq_idx_
	set r "/var/tmp/servent-${uniq_idx_}.mash"
    }
    return $r
}



# Recv a "map-WPI-WRI" msg, which indicates the client wants us
# to generate a shepard from the WRI that maps to the WPI.
#
LocalHostManager public rendez_recv_map-WPI-WRI {rvmsg} {
    Trc $class "--> ${class}::$proc [$rvmsg get_msg]"

    puts "request for shepard generation (WPI to WRI mapping)"
}


#-----------------------------------------------------------------


import UDPServer mashutils

# accept unicast queries and control messages
#
Class LocalHostManagerCtrl -superclass UDPServer

#
LocalHostManagerCtrl public init {spec parent} {
    Trc $class "--> ${class}::$proc"
    $self next $spec
    $self instvar parent_
    set parent_ $parent
}

# unicast forwarder to the LocalHostManager, setting rspec to "-"
#
LocalHostManagerCtrl public recv {addr port data size} {
    Trc $class "--> ${class}::$proc"
    $self instvar parent_
    set newmsg [new RVMsg $data "-" $addr/$port]
    set t [$newmsg get_type]
    set theMethod rendez_recv_$t
    if [$parent_ has_method $theMethod] {
	$parent_ $theMethod $newmsg
    }
}

