# nam-network.tcl --
#
#       NamNetwork: this is a "graph" of edges and nodes, and their
#           characteristics
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#   @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/atobj/nam-network.tcl,v 1.6 2002/02/03 04:25:26 lim Exp $


####################################################################
# NamNetwork --
#
# This is a "graph" that knows about edges and nodes, and their
# characteristics. Note that it is different from the edges and nodes in
# the animation itself.
#
#---

Class NamNetwork


#NamNetwork set pi_ 3.14159265358979323846
NamNetwork set pi_ 3.14159

NamNetwork instproc init {} {
        # REVIEW: decide how to handle initialization
        nam_config $self
}

# for backward compatibility
NamNetwork instproc node {id shape} {
        $self new_node $id $shape
}

NamNetwork instproc new_node {id shape}  {
        $self instvar nodes_ nodeIds_
        set nodes_(shape,$id) $shape
        set nodes_(size,$id) 0
        lappend nodeIds_ $id
}

NamNetwork instproc new_link {srcId destId bw delay angle} {
        $self instvar edges_ nodes_

        set pi [NamNetwork set pi_]
        set edges_($srcId,$destId) [list $bw $delay [expr $angle*$pi]]

        # DbgOut edge($srcId,$destId) "is" $edges_($srcId,$destId)
        # DbgOut id says $edges_($srcId,$destId) is \
                        # [[$edges_($srcId,$destId) src] get_id] \
                        # [[$edges_($srcId,$destId) dest] get_id]
}

NamNetwork instproc nodes {} {
        return [$self set nodeIds_]
}

# the format returned is "n0,n1 n1,n2 ...." (note the comma in between)
NamNetwork instproc edges {} {
        return [$self array names edges_]
}

# pkt_size is in byes, bw in bits per sec, hence the << 3
NamNetwork instproc xmitTime {src dest pkt_size} {
        set bw [$self edgeBW $src $dest]
        return [expr (($pkt_size<<3)/$bw)]
}

NamNetwork instproc hasEdge {src dest} {
        $self instvar edges_
        return [info exists edges_($src,$dest)]
}

NamNetwork instproc edgeBW {src dest} {
        $self instvar edges_
        if [info exists edges_($src,$dest)] {
                return [lindex $edges_($src,$dest) 0]
        } else {
                return ""
        }
}

NamNetwork instproc edgeDelay {src dest} {
        $self instvar edges_
        if [info exists edges_($src,$dest)] {
                return [lindex $edges_($src,$dest) 1]
        } else {
                return ""
        }
}

NamNetwork instproc edgeAngle {src dest} {
        $self instvar edges_
        if [info exists edges_($src,$dest)] {
                return [lindex $edges_($src,$dest) 2]
        } else {
                return ""
        }
}

NamNetwork instproc setEdgePos {src dest x0 y0 x1 y1} {
        $self set edge_pos_($src,$dest) [list $x0 $y0 $x1 $y1]
}

NamNetwork instproc edgePos {src dest} {
        return [$self set edge_pos_($src,$dest)]
}


NamNetwork instproc setPktHt {src dest pktHt} {
        $self set edges_pktHt($src,$dest) $pktHt
}

NamNetwork instproc pktHt {src dest} {
        return [$self set edges_pktHt($src,$dest)]
}

NamNetwork instproc adjNodes {nId} {
        $self instvar edges_
        set neighbors ""
        foreach pair [array names edges_ "$nId,*"] {
                lappend neighbors [lindex [split $pair ","] 1]
        }
        return $neighbors
}

NamNetwork instproc queue {args} {
#        puts stderr "TODO: implement queue!"
}

NamNetwork instproc setNodePos {node pos} {
        $self set nodes_(pos,$node) $pos
}

NamNetwork instproc nodePos {node} {
        return [$self set nodes_(pos,$node)]
}

NamNetwork instproc nodePlaced {node} {
        $self instvar nodes_
        return [info exists nodes_(pos,$node)]
}

NamNetwork instproc setNodeSize {nId size} {
        # DbgOut node $nId size is $size
        $self set nodes_(size,$nId) $size
}

NamNetwork instproc nodeSize {nId} {
        $self set nodes_(size,$nId)
}

NamNetwork instproc layout {} {
        $self instvar nodes_ edges_ nodeIds_ bbox_

        $self scale_estimate
        $self setNodePos [lindex $nodeIds_ 0] {0.0 0.0}
        while 1 {
                set did_something 0
                foreach nId $nodeIds_ {
                        set did_something [expr $did_something | \
                                        [$self traverse $nId]]
                }
                if {$did_something == 0} break
        }

        foreach nId $nodeIds_ {
                foreach neighbor [$self adjNodes $nId] {
                        $self placeEdge $nId $neighbor
                }
        }

}

NamNetwork instproc placeEdge {src dest} {
        $self instvar edge_pos_
        if {![info exist edge_pos_($src,$dest)]} {
                set angle [$self edgeAngle $src $dest]
                set s [expr sin($angle)]
                if {[expr abs($angle - 3.14159) < 0.00001]} {
			set c -1.0
		} elseif {[expr abs($angle) < 0.00001]} {
                        set c 1.0
                } else {
			set c [expr cos($angle)]
		}
		set nsin [ expr 0.75 * $s ]
                set ncos [ expr 0.75 * $c ]
                set srcPos [$self nodePos $src]
                set dstPos [$self nodePos $dest]
                set srcSize [$self nodeSize $src]
                set destSize [$self nodeSize $dest]
                set x0 [expr [lindex $srcPos 0] + $srcSize * $ncos]
                set y0 [expr [lindex $srcPos 1] + $srcSize * $nsin]
                set x1 [expr [lindex $dstPos 0] - $destSize * $ncos]
                set y1 [expr [lindex $dstPos 1] - $destSize * $nsin]
                # place the queues here later
                $self setEdgePos $src $dest $x0 $y0 $x1 $y1
        }
}

#
# move {x y} using a vector in polar co-ordinates
#     {$angle, displacement ($disp)}
#
NamNetwork instproc move { pos angle disp } {
        set x [lindex $pos 0]
        set y [lindex $pos 1]
        return [list [expr $x + ($disp * cos($angle))] \
                        [expr $y + ($disp * sin($angle))] ]
}

#
# Traverse node n's neighbors and place them based on the
# delay of their links to n.  The two branches of the if..else
# are to handle unidirectional links -- we place ourselves if
# we haven't been placed & our downstream neighbor has.
#
NamNetwork instproc traverse {node} {
        $self instvar edges_

        set did_something 0
        foreach n [$self adjNodes $node] {
                lappend edges [list $node $n]
        }

        set nodes $node
        #    DbgOut "edges:$edges"
        while {1} {
                if {[llength $edges]==0} {
                        break
                }
                set edge [removeFirst edges]
                # DbgOut traversing edge: ($edge = [$edge src]  \
                                # [$edge dest] angle: \
                                # [expr [$edge get_angle]/3.1415926]
                set node [lindex $edge 0]
                set neighbor [lindex $edge 1]

                # DbgOut "neighbor: [$neighbor get_id]"

                if {[$self nodePlaced $node] && [$self nodePlaced $neighbor]} {
                        continue
                }

                # one of the nodes is not placed
                # 0.75 is to allow space for the nodes themselves
                set d [expr [$self edgeDelay $node $neighbor] + \
                                (0.75 * ([$self nodeSize $neighbor] + \
                                [$self nodeSize $node]))]
                if [$self nodePlaced $neighbor] {
                        set place [$self move \
                                        [$self nodePos $neighbor] \
                                        [$self edgeAngle $node $neighbor] \
                                        [expr -1 * $d]]
                        $self setNodePos $node $place
                        set did_something 1
                } elseif [$self nodePlaced $node] {
                        set place [$self move \
                                        [$self nodePos $node] \
                                        [$self edgeAngle $node $neighbor] $d]
                        $self setNodePos $neighbor $place

                        # Note: we doing breadth first search here instead of
                        #       depth first as in the original NAM
                        foreach nn [$self adjNodes $neighbor] {
                                lappend edges [list $neighbor $nn]
                        }
                        # REVIEW: compute nymax_ and nymin_ (see orig NAM) ??
                } else {
                        error "check the algorithm ($node,$neighbor)"
                }
        }
        return $did_something
}

#
# Compute reasonable defaults for missing node or edge sizes
# based on the maximum link delay.
#
NamNetwork instproc scale_estimate {} {
        $self instvar nodes_ edges_ nodeIds_
        # Determine the maximum link delay
        set max 0.0
        foreach nodeId $nodeIds_ {
                foreach n [$self adjNodes $nodeId] {
                        if {[$self edgeDelay $nodeId $n] > $max} {
                                set max [$self edgeDelay $nodeId $n]
                        }
                }
        }
        # DbgVar max

        # Check for missing node or edge sizes. If any are found,
        # compute a reasonable default based on the maximum edge
        # dimension.

        foreach node $nodeIds_ {
                if {[$self nodeSize $node] <= 0} {
                        # DbgOut Setting node $node to size [expr $max * 0.1]
                        $self setNodeSize $node [expr $max * 0.1]
                }
                foreach n [$self adjNodes $node] {
                        $self setPktHt $node $n [expr $max * 0.03]
                }
        }
}

NamNetwork instproc color {index color} {
        $self instvar colors_
        set colors_($index) $color
}

NamNetwork instproc get_color {colorindex} {
        $self instvar colors_
        if [info exists colors_($colorindex)] {
                return $colors_($colorindex)
        } else {
                return black
        }
}



#--------------------------------------------------------------------------
# Helper functions for nam, used to be backward compatible with
# the standalone NAM
#

#
# helper functions
#
proc nam_angle { v } {
        switch $v {
                up-right -
                right-up	{ return 0.25 }
                up		{ return 0.5 }
                up-left -
                left-up		{ return 0.75 }
                left		{ return 1. }
                left-down -
                down-left	{ return 1.25 }
                down		{ return 1.5 }
                down-right -
                right-down	{ return 1.75 }
                default		{ return 0.0 }
        }
}

if 0 {
        NamNetwork set scales_ {m u k M}
        NamNetwork set uscale_(m) 1e-3
        NamNetwork set uscale_(u) 1e-6
        NamNetwork set uscale_(k) 1e3
        NamNetwork set uscale_(M) 1e6
}

NamNetwork array set scales_ {
        m 1e-3
        u 1e-6
        k 1e3
        M 1e6
}

proc time2real {v} {
        foreach u [NamNetwork array names scales_] {
                set k [string first $u $v]
                if { $k >= 0 } {
                        set scale [NamNetwork set scales_($u)]
                        break
                }
        }
        if { $k > 0 } {
                set v [string range $v 0 [expr $k - 1]]
                set v [expr $scale * $v]
        }
        return $v
}

#FIXME
proc bw2real {v} {
        return [time2real $v]
}

proc mklink { net n0 n1 bandwidth delay angle } {
        global delay01
        set th [nam_angle $angle]
        set result [$net new_link $n0 $n1 \
                        [bw2real $bandwidth]  [time2real $delay] $th]
        $net new_link $n1 $n0 \
                        [bw2real $bandwidth] [time2real $delay] [expr $th + 1]
        if { $n0 == 0 && $n1 == 1 } {
                set delay01 $result
        }
}
