/*
 * announce-listen.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/srmv2/net/announce-listen.cc,v 1.4 2002/02/03 03:03:19 lim Exp $";
#endif

#include <sys/types.h>
#include "tclcl.h"
#include "net.h"
#include "sap.h"

/*
 * <otcl> Class AnnounceListenManager
 * 
 * This module implements a basic manager for <i>announce-listen</i> style
 * string-based protocols.  The object supplies hooks for announcing and 
 * receiving messages based on a timer that can be set in the derived classes
 * (see the Timer class page).
 * 
 * The AnnounceListenManager class is an abstract class and developers should 
 * derive their classes from the base class or the 
 * AnnounceListenManager/SAP class for SAP encapsulation.  When a message
 * is received, the OTcl method  <tt>recv_announcement</tt> is called 
 * with three asrguments: the originating address, data, and message length.
 * For the "announce" side of the protocol, the <tt>build_announcement</tt>
 * OTcl method will be called every time that the associated timer times out.
 */
class AnnounceListenManager : public SessionHandler, public TclObject {
public:
	AnnounceListenManager(int mtu);
	~AnnounceListenManager();
	virtual void recv(DataHandler* dh);
	virtual int command(int argc, const char*const* argv);
protected:
	DataHandler rdh_;
	DataHandler sdh_;
	u_char* pktbuf_;
	int mtu_;
	virtual int announce(const char* s);

	virtual void recv(CtrlHandler*) {}
	virtual void announce(CtrlHandler*) {}
};

/*
 * <otcl> Class AnnounceListenManager/SAP -superclass AnnounceListenManager
 * 
 * An AnnounceListenManager that enapsulates each message with a SAP header 
 * and decapsulates each received messages from its SAP header before
 * passing it on to <tt>recv_announcement</tt> (see AnnounceListenManager
 * manual page).
 */
class SAPAnnounceListenManager : public AnnounceListenManager {
public:
	SAPAnnounceListenManager(int);
	virtual void recv(DataHandler* dh);
protected:
	virtual int announce(const char* s);
};

static class AnnounceListenManagerClass : public TclClass {
public:
	AnnounceListenManagerClass() : TclClass("AnnounceListenManager") {}
	TclObject* create(int, const char*const* argv) {
		return (new AnnounceListenManager(atoi(argv[4])));
	}
} al_class;

static class SAPAnnounceListenManagerClass : public TclClass {
public:
	SAPAnnounceListenManagerClass() : 
		TclClass("AnnounceListenManager/SAP") {}
	TclObject* create(int, const char*const* argv) {
		return (new SAPAnnounceListenManager(atoi(argv[4])));
	}
} al_sap;

AnnounceListenManager::AnnounceListenManager(int mtu)
	:mtu_(mtu)
{
	pktbuf_ = new u_char[2 * mtu];
	rdh_.manager(this);
	sdh_.manager(this);
}

AnnounceListenManager::~AnnounceListenManager()
{
	delete[] pktbuf_;
}

void AnnounceListenManager::recv(DataHandler* dh)
{
	u_int32_t addr;
	int cc = dh->recv(pktbuf_, 2 * mtu_, addr);
	if (cc < 0) {
		perror("hm: recv");
		return;
	}
	if (cc == 0)
		return;

	Tcl& tcl = Tcl::instance();
	u_char* s = pktbuf_;
	s[cc] = '\0';

	//FIXME tcl.evalf() uses a fixed buffer that may be too small.
	//    this doesn't really solve the problem.
	char buf[4*1024];
	sprintf(buf, "%s recv_announcement %s {%s} %d",
		name(), intoa(addr), s, cc);
	tcl.eval(buf);
}

int AnnounceListenManager::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 3) {
		/*
		 * <otcl> AnnounceListenManager public announce { msg }
		 * Announce the message <i>msg</i>.
		 */
		if (strcmp(argv[1], "announce") == 0) {
			tcl.resultf("%d", announce(argv[2]));
			return (TCL_OK);
		}
		/*
		 * <otcl> AnnounceListenManager private network { net }
		 * Install <i>net</i> as the network for the object.
		 */
		 if (strcmp(argv[1], "network") == 0) {
			Network* net = (Network*)TclObject::lookup(argv[2]);
			sdh_.net(net);
			rdh_.net(net);
			return (TCL_OK);
		}
		/*
		 * <otcl> AnnounceListenManager private send-network { net }
		 * Install <i>net</i> as the send network for the object.
		 */
		if (strcmp(argv[1], "send-network") == 0) {
			Network* net = (Network*)TclObject::lookup(argv[2]);
			sdh_.net(net);
			return (TCL_OK);
		}
		/*
		 * <otcl> AnnounceListenManager private recv-network { net }
		 * Install <i>net</i> as the receive network for the object.
		 */	
		if (strcmp(argv[1], "recv-network") == 0) {
			Network* net = (Network*)TclObject::lookup(argv[2]);
			rdh_.net(net);
			return (TCL_OK);
		}
	}
		
	return (TclObject::command(argc, argv));
}

int AnnounceListenManager::announce(const char* s)
{
	int len = strlen(s);
	sdh_.send((u_char*)s, len);
	return (len);
}

SAPAnnounceListenManager::SAPAnnounceListenManager(int mtu)
	: AnnounceListenManager(mtu)
{}

int SAPAnnounceListenManager::announce(const char* s)
{
	sap_hdr* h = (sap_hdr*)pktbuf_;
	/*  v = 1, e = 0, c = 0 for now */
	h->flags = 0x20;
	h->authlen = 0;
	h->msgid = 0;
	h->src = LookupLocalAddr();

	strcpy((char *)(h + 1), s);
	
#ifdef notdef
printf("ALM: send {%s}\n", s);
#endif
	int len = sizeof(sap_hdr) + strlen(s);
	sdh_.send(pktbuf_, len);
	return (len);
}

void SAPAnnounceListenManager::recv(DataHandler* dh)
{
	u_int32_t addr;
	int cc = dh->recv(pktbuf_, 2 * mtu_, addr);
	if (cc < 0) {
		perror("hm: recv");
		return;
	}
	if (cc == 0)
		return;

	Tcl& tcl = Tcl::instance();
	sap_hdr* h = (sap_hdr*)pktbuf_;
	char* s = (char*)(h + 1);

	cc -= sizeof(sap_hdr);
	s[cc] = '\0';

	//FIXME tcl.evalf() uses a fixed buffer that may be too small.
	//    this doesn't really solve the problem.
	char buf[4*1024];
	sprintf(buf, "%s recv_announcement %s {%s} %d",
		name(), intoa(addr), s, cc);
	tcl.eval(buf);
}
