# OpenVerse URL Display Module
# 
# Contains functions which are related to displaying
# and clicking on URLS offered by people.
#
# Module Name		- URL Module
# Current Maintainter 	- Cruise <cruise@openverse.org>
# Sourced By		- InitMain.tcl
#
# Copyright (C) 1999 David Gale <cruise@openverse.org>
# For more information visit http://OpenVerse.org/
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
# USA.

#NOTE
# Now using curley braces around functiosn for greatest speeds


# URL's use icons to identify them. We pre-create the photos of them at
# startup.

image create photo OpenVerse_Image_URL_UNKNOWN -file "$MV(icondir)/unknown.gif"
image create photo OpenVerse_Image_URL_DCCFILE -file "$MV(icondir)/dccfile.gif"
image create photo OpenVerse_Image_URL_OPENVERSE -file "$MV(icondir)/mv2.gif"
image create photo OpenVerse_Image_URL_WEB -file "$MV(icondir)/wwwurl.gif"
image create photo OpenVerse_Image_URL_EMAIL -file "$MV(icondir)/emailurl.gif"
image create photo OpenVerse_Image_URL_FTP -file "$MV(icondir)/ftpurl.gif"

proc ProcURL {who what} {
	global MV

	# Get a random screen position (allowing 50sq around it.)
	# This gives us a grid 16x12.

	set foundspot 0
	set tries 0
	while {$foundspot == 0 && $tries < 164} {
		set x [expr int(rand() * 15)]
		set y [expr int(rand() * 11)]
		if !$MV(URLS.$x.$y) {
			set MV(URLS.$x.$y) 1
			set foundspot 1
		}
		update idletasks
		incr tries
	}
	DebugIt "(URL) Using $x $y" other

	set display 0
	set type [URLType $what]
	#
	# Process Plugins
	#
	foreach plugin $MV(plugin.traps.ProcURL) {
		if ![$MV(plugin.traps.ProcURL.$plugin) $who $what $type] {return}
	}
	switch -exact -- $type {
		"FILE" {
			set display 1
			set img "OpenVerse_Image_URL_DCCFILE"
		}
		"OpenVerse" {
			set display 1
			set img "OpenVerse_Image_URL_OPENVERSE"
		}
		"WEB" {
			set display 1
			set img "OpenVerse_Image_URL_WEB"
		}
		"EMAIL" {
			set display 1
			set img "OpenVerse_Image_URL_EMAIL"
		}
		"FTP" {
			set display 1
			set img "OpenVerse_Image_URL_FTP"
		}
		default {
			set display 1
			set img "OpenVerse_Image_URL_UNKNOWN"
		}
	}

	if $display {
		set MV(urlicon.$x.$y) [.top.c create image [lindex $MV(URLS.x) $x] [lindex $MV(URLS.y) $y] -image $img]
		if $MV(bell) {bell}
		InsertIntoChat 1 "\0030,4***\0030,12 URL \0030,4***\00399,99 [Trns from] $who $what"
		after [expr $MV(urltimeout) * 1000] "DeleteUrl $x $y"
		.top.c bind $MV(urlicon.$x.$y) <Button> "UrlClick $x $y"
		set MV(url.$x.$y.from) $who
		set MV(url.$x.$y.type) $type
		set MV(url.$x.$y.what) $what
		update idletasks
		set dir 0
		set yy 0
		set xx 0
		for {set c 0} {$c < 50} {incr c} {
			switch $dir {
				"0" {
					set xx 15
					set yy 0
					incr dir
				}
				"1" {
					set xx 0
					set yy -15
					incr dir
				}
				"2" {
					set xx 30
					set yy 0
					incr dir
				}
				"3" {
					set xx 0
					set yy 30
					incr dir
				}
				"4" {
					set xx -30
					set yy 0
					incr dir
				}
				"5" {
					set xx 0
					set yy -30
					set dir 2
				}

			}
			.top.c move $MV(urlicon.$x.$y) $xx $yy
			update idletasks
		}
	}
}


#
# Sends a url
#
proc SendURL {who what} {
	global MV

	if {$who == "" || $what == ""} {
		ProcChat $MV(nick) [Trns url_help] 0 1 $MV(colors.system.baloon)
		bell
		return
	}

	if {$who == "*"} {
		SendToServer "URL * $what"
	} else {
		if {[lsearch -exact $MV(people) $who] != -1} {
			SendToServer "URL $who $what"
			ProcURL $MV(nick) "$what"
		}
	}
}


proc UrlClick {x y} {
	global MV

	switch -exact -- $MV(url.$x.$y.type) {
		"FILE" {
			set Type [Trns file_offer]
			set cmd "URL_DCCGet"
		}
		"EMAIL" {
			set Type [Trns email_address]
			set cmd "URL_OpenWeb"
		}
		"FTP" {
			set Type [Trns ftp_address]
			set cmd "URL_OpenWeb"
		}
		"WEB" {
			set Type [Trns www_address]
			set cmd "URL_OpenWeb"
		}
		"OpenVerse" {
			set Type [Trns ov_server_link]
			set cmd "URL_OpenVerse"
		}
		default {
			set Type [Trns unknown_url]
			set cmd "URL_KillOpts"
		}
	}

	set where n
	set dir left
	if {$x < 7 && $y < 5} {set where nw;set dir left}
	if {$x < 7 && $y > 5} {set where sw;set dir left}
	if {$x < 7 && $y == 5} {set where w;set dir left}
	if {$x > 7 && $y == 5} {set where e;set dir right}
	if {$x > 7 && $y < 5} {set where ne;set dir right}
	if {$x > 7 && $y > 5} {set where se;set dir right}
	if {$x == 7 && $y > 5} {set where s;set dir left}
	if {$x == 7 && $y < 5} {set where n;set dir left}
	if {$x == 7 && $y == 5} {set where center;set dir left}

	if {[winfo exists .top.c.dlopts]} {
		destroy .top.c.dlopts
		.top.c delete $MV(urlopt.$MV(urlopt.selected.x).$MV(urlopt.selected.y))
		catch {unset MV(urlopt.$MV(urlopt.selected.x).$MV(urlopt.selected.y))}
		catch {unset MV(urlopt.selected.x)}
		catch {unset MV(urlopt.selected.y)}
	}
	frame .top.c.dlopts -relief raised -borderwidth 2 \
		-bg $MV(colors.url.frames.bg)

	set MV(urlopt.$x.$y) [.top.c create window [lindex $MV(URLS.x) $x] [lindex $MV(URLS.y) $y] -anchor $where -window .top.c.dlopts]
	set MV(urlopt.selected.x) $x
	set MV(urlopt.selected.y) $y
	frame .top.c.dlopts.info1 -relief sunken -borderwidth 2 \
		-bg $MV(colors.url.frames.bg)
	frame .top.c.dlopts.info2 -relief sunken -borderwidth 2 \
		-bg $MV(colors.url.frames.bg)
	frame .top.c.dlopts.info3 -relief sunken -borderwidth 2 \
		-bg $MV(colors.url.frames.bg)

	label .top.c.dlopts.info1.txt -relief raised -borderwidth 2 \
		-text [Trns from] -width 15 \
		-fg $MV(colors.url.labels.fg) \
		-bg $MV(colors.url.labels.bg)
	label .top.c.dlopts.info2.txt -relief raised -borderwidth 2 \
		-text "URL" -width 15 \
		-fg $MV(colors.url.labels.fg) \
		-bg $MV(colors.url.labels.bg)
	label .top.c.dlopts.info3.txt -relief raised -borderwidth 2 \
		-text [Trns type] -width 15 \
		-fg $MV(colors.url.labels.fg) \
		-bg $MV(colors.url.labels.bg)
	label .top.c.dlopts.info1.info -relief sunken -borderwidth 2 \
		-text "$MV(url.$x.$y.from)" \
		-fg $MV(colors.url.entries.fg) \
		-bg $MV(colors.url.entries.bg)
	label .top.c.dlopts.info2.info -relief sunken -borderwidth 2 \
		-text "$MV(url.$x.$y.what)" \
		-fg $MV(colors.url.entries.fg) \
		-bg $MV(colors.url.entries.bg)
	label .top.c.dlopts.info3.info -relief sunken -borderwidth 2 \
		-text "$Type" \
		-fg $MV(colors.url.entries.fg) \
		-bg $MV(colors.url.entries.bg)

	button .top.c.dlopts.get -text [Trns open_save] \
		-command "$cmd $x $y" \
		-fg $MV(colors.url.buttons.fg) \
		-bg $MV(colors.url.buttons.bg) \
		-activeforeground $MV(colors.url.buttons.afg) \
		-activebackground $MV(colors.url.buttons.abg)
	button .top.c.dlopts.ign -text [Trns ignore] \
		-command "URL_KillOpts $x $y" \
		-fg $MV(colors.url.buttons.fg) \
		-bg $MV(colors.url.buttons.bg) \
		-activeforeground $MV(colors.url.buttons.afg) \
		-activebackground $MV(colors.url.buttons.abg)
	button .top.c.dlopts.ref -text [Trns refuse] \
		-command "URL_KillOpts $x $y;DeleteUrl $x $y" \
		-fg $MV(colors.url.buttons.fg) \
		-bg $MV(colors.url.buttons.bg) \
		-activeforeground $MV(colors.url.buttons.afg) \
		-activebackground $MV(colors.url.buttons.abg)

	pack .top.c.dlopts.info1 -fill both -expand y
	pack .top.c.dlopts.info2 -fill both -expand y
	pack .top.c.dlopts.info3 -fill both -expand y
	pack .top.c.dlopts.info1.txt -side left -fill y
	pack .top.c.dlopts.info2.txt -side left -fill y
	pack .top.c.dlopts.info3.txt -side left -fill y
	pack .top.c.dlopts.info1.info -side left -expand y -fill both
	pack .top.c.dlopts.info2.info -side left -expand y -fill both
	pack .top.c.dlopts.info3.info -side left -expand y -fill both

	pack .top.c.dlopts.get .top.c.dlopts.ign .top.c.dlopts.ref -side $dir

}

proc DeleteUrl {x y} {
	global MV

	catch {.top.c delete $MV(urlicon.$x.$y)}
	set MV(URLS.$x.$y) 0
	#
	# Clean up the memory from this item.
	#
	catch {unset MV(url.$x.$y.what)}
	catch {unset MV(url.$x.$y.from)}
	catch {unset MV(url.$x.$y.type)}
	catch {unset MV(urlicon.$x.$y)}
}

proc URLType {url} {
	global MV

	set parms [split [string trim $url] ":"]
	switch -exact -- [string tolower [lindex $parms 0]] {
		"file" {set type "FILE"}
		"http" {set type "WEB"}
		"https" {set type "WEB"}
		"ftp" {set type "FTP"}
		"mailto" {set type "EMAIL"}
		"openverse" {set type "OpenVerse"}
		default {set type "UNKNOWN"}
	}
	return $type
}

proc URL_DCCGet {x y} {
	global MV

	destroy .top.c.dlopts
	.top.c delete $MV(urlopt.$x.$y)
	catch {unset MV(urlopt.$x.$y)}
	catch {unset MV(urlopt.selected.x)}
	catch {unset MV(urlopt.selected.y)}
	set parms [split $MV(url.$x.$y.what) "/"]
	set hp [split [lindex $parms 2] ":"]
	set fs [split [lindex $parms 3] ":"]
	DCCGet [lindex $hp 0] [lindex $hp 1] [lindex $fs 0] [lindex $fs 1] DOWNLOAD $MV(url.$x.$y.from)
	DeleteUrl $x $y
}

# URL_OpenVerse
#
# Usage: URL_OpenVerse x y
#
# x and y are the id's for this url on the screen. this is normally called
# internally when a url is clicked on. If you call this without a url on
# the screen, it will generate an error cause it's going to delete any
# urls on the screen first. It will validate urls somewhat, being sure
# that it has a valid number as a port and that the hostname contains at
# least 3 bytes including a period. VALID == a.c
#
proc URL_OpenVerse {x y} {
	global MV

	destroy .top.c.dlopts
	.top.c delete $MV(urlopt.$x.$y)
	catch {unset MV(urlopt.$x.$y)}
	catch {unset MV(urlopt.selected.x)}
	catch {unset MV(urlopt.selected.y)}
	set parms [split $MV(url.$x.$y.what) "/"]
	set hp [split [lindex $parms 2] ":"]
	DeleteUrl $x $y
	set host [lindex $hp 0]
	set port [lindex $hp 1]
	if {$port == "" || [TestPosNum $y] || [string first "." $host] == -1 || [string length $host] < 3} {
		ThrowError [Trns ov_url_invalid]
		unset host port hp parms
		return
	}
	Disconnect
	set MV(roomhost) $host
	set MV(roomport) $port
	Connect
	unset host port hp parms
}

proc URL_KillOpts {x y} {
	global MV
	destroy .top.c.dlopts
	.top.c delete $MV(urlopt.$x.$y)
	#
	# Clean up memory used by this item.
	#
	catch {unset MV(urlopt.$x.$y)}
	catch {unset MV(urlopt.selected.x)}
	catch {unset MV(urlopt.selected.y)}
}

proc URL_OpenWeb {x y} {
	global MV

	if ![file exists "$MV(homedir)/tmp"] {file mkdir "$MV(homedir)/tmp"}
	set outfile [open "$MV(homedir)/tmp/WEB.htm" "w"]
	puts $outfile "<html><title>[Trns please_wait_loading_url]</title><head>"
	puts $outfile "<meta HTTP-EQUIV=\"Refresh\" CONTENT=\"0; URL=$MV(url.$x.$y.what)\""
	puts $outfile "</head><body BGCOLOR=\"#FFFFFF\" TEXT=\"#000000\" LINK=\"#0000FF\" VLINK=\"#0000FF\" ALINK=\"#FF0000\">"
	puts $outfile "<CENTER><H1><a href=\"http://www.openverse.com/\">WWW.OPENVERSE.COM</a></H1></CENTER>"
	puts $outfile "<P>"
	puts $outfile "<TABLE WIDTH=\"100%\" BORDER=\"0\" CELLPADDING=\"3\" CELLSPACING=\"3\">"
	puts $outfile "<TR>"
	puts $outfile "<TD BGCOLOR=\"#6699CC\"><FONT COLOR=\"white\" FACE=\"Arial,Helvetica,Geneva,Swiss,SunSans-Regular\"><B>"
	puts $outfile "[Trns please_wait_loading_url]</B></FONT></TD></TR>"
	puts $outfile "<TR><TD ALIGN=\"top\"><FONT FACE=\"Arial,Helvetica,Geneva,Swiss,SunSans-Regular\">"
	puts $outfile "[Trns if_failure_msg]<P>"
	puts $outfile "<a href=\"$MV(url.$x.$y.what)\">$MV(url.$x.$y.what)</a>"
	puts $outfile "</TR></TD></TABLE></body></html>"
	close $outfile
	set fullpath "[file nativename $MV(homedir)/tmp/WEB.htm]"
	catch {exec "$MV(browser_cmd)" $fullpath &}
	#
	# Negative numbers are for built in URLs and they do not
	# need to be erased from the screen (they are forced)
	#
	if {$x >= 0} {
		URL_KillOpts $x $y
		DeleteUrl $x $y
	}
}
