#ifndef FLOW_H
#define FLOW_H

/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdint.h>
#include <sys/types.h>
#include <sys/time.h>
#include <event.h>

#include "list.h"

#define FLOW_FIN	0x01
#define FLOW_ACK	0x02
#define FLOW_RST	0x04
#define FLOW_FRAG	0x08

struct flow
{
   time_t timestamp;			/* when flow was created			*/
   uint64_t octets,packets;		/* bytes, packets transferred		*/
   uint8_t proto;				/* protocol							*/
   uint8_t tos;					/* type of service					*/
   uint16_t id;					/* identification number			*/
   uint32_t saddr,daddr;		/* source, destination address		*/
   uint16_t sport,dport;		/* source, destination port			*/
   uint8_t flags;				/* some [tcp] flags					*/
   double bitrate;				/* average bitrate since creation	*/
   struct event active;
   struct event inactive;
   struct list_head time;
   struct list_head size;
};

/*
 * Allocate memory for storing flows. The number of stored flows
 * will be limited to the limit specified here. Returns 0 if
 * successful, -1 if we ran out of memory.
 */
extern int flow_open (size_t entries,time_t active,time_t inactive);

/*
 * Free resources allocated for flow repository.
 */
extern void flow_close (void);

/*
 * Insert a flow. If the flow is part of an existing
 * flow, it will be merged, else a new flow will be created.
 *
 * If there are no space left to store this flow, the
 * oldest flow will automatically be removed to make
 * space for this one.
 */
extern void flow_insert (const struct flow *flow);

/*
 * Remove a flow. The specified flow should be the
 * result of an earlier call to flow_find().
 */
extern void flow_remove (struct flow *flow);

/*
 * Find a flow.
 */
extern struct flow *flow_find (const struct flow *flow);

/*
 * Sort the flows by size (descending order) and call
 * the specified function for each flow.
 */
extern void flow_process (void (*process) (const struct flow *));

#endif	/* #ifndef FLOW_H */
