/*****
*
* Copyright (C) 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

/*
 * this code is inspired from the NIDSfindshellcode program by
 * Fermin J. Serna <fjserna@ngsec.com>.
 *
 * Here is what this rew-write provide you compared to the original code:
 *
 * - serialized all architecture test within the same buffer walking loop.
 *   was 3 * O(n * m) and is now 1 * O(n * m).
 *
 * - use an hash table for pointing to opcode we are now O(n lg m).
 *
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <inttypes.h>


#include "detect.h"
#include "shellcode.h"

/*
 * port-list stuff.
 */
#include "plugin-protocol.h"



#define HASH_SIZE 256


#define get_tbl_size(tbl) \
        sizeof(tbl) / sizeof(struct junks)


struct arch_tbl {
        int nop_max;
        int nop_counter;
        int continue_checking;
        int dlen;
        const unsigned char *ptr;
        idmef_string_t desc;
};


struct hash_item {
        struct junks *opcode;
        struct hash_item *next;
};



static int is_enabled = 0;
static plugin_detect_t plugin;
static int nops_raise_error = 60;
static subscribtion_t subscribtion[] = {
        { p_data, NULL },
        { p_end, NULL  }
};


static port_list_t *port_list = NULL;
struct hash_item *hppa_hash[HASH_SIZE];
struct hash_item *intel_hash[HASH_SIZE];
struct hash_item *sparc_hash[HASH_SIZE];



static int hash_entry(const unsigned char *code) 
{
        return *code % HASH_SIZE;
}




static int add_entry(struct hash_item **tbl, struct junks *junk) 
{
        int key;
        struct hash_item *item;

        item = malloc(sizeof(*item));
        if ( ! item ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        item->opcode = junk;
        key = hash_entry(junk->code);
        
        item->next = tbl[key];
        tbl[key] = item;

        return 0;
}




static void raise_alert(packet_container_t *packet, struct arch_tbl *arch)
{
        static nids_alert_t alert;
        static int initialized = 0;
        static idmef_impact_t impact;

        if ( ! initialized ) {
                initialized = 1;
                nids_alert_init(&alert);
                
                alert.impact = &impact;
                impact.type = other;
                impact.severity = impact_high;
                impact.description.string = NULL;
        }
        
        alert.classification.name.len = arch->desc.len;
        alert.classification.name.string = arch->desc.string;

        nids_alert((plugin_generic_t *)&plugin, packet, &alert,
                   "%d NOPs in packet payload", arch->nop_counter);
}



static void check_tbl(packet_container_t *packet, const unsigned char *end,
                      struct arch_tbl *arch, struct hash_item **hash_tbl) 
{
        struct junks *opcode;
        struct hash_item *item;
        int len, found = 0, key, ret;
        
        if ( arch->continue_checking < 0 || arch->ptr >= end )
                return;
        
        key = hash_entry(arch->ptr);
        
        for ( item = hash_tbl[key]; item != NULL; item = item->next ) {
                     
                opcode = item->opcode;
                len = (opcode->dyn == 0) ? opcode->len : opcode->dynint;

                if ( ! opcode->noppad || arch->dlen < len )
                        continue;
                
                ret = memcmp(arch->ptr, opcode->code, len);
                if ( ret == 0 ) {
                        found = 1;
                        arch->nop_counter++;
                        arch->ptr += opcode->len - 1;
                        arch->dlen -= opcode->len - 1;
                        break;
                }
        }

        /*
         * we want contiguous chain of NOP. reset counter.
         */
        if ( ! found )
                arch->nop_counter = 0;
        
        if ( arch->nop_counter == arch->nop_max ) {
                arch->continue_checking = -1;
                raise_alert(packet, arch);
        }
}




static void detect_shellcode(packet_container_t *packet, const char *buffer, size_t len) 
{
        const unsigned char *end = buffer + len;
        struct arch_tbl intel_arch, hppa_arch, sparc_arch;
        
        idmef_string_set_constant(&hppa_arch.desc, "HPPA shellcode found");
        idmef_string_set_constant(&intel_arch.desc, "IA32 shellcode found");
        idmef_string_set_constant(&sparc_arch.desc, "SPARC shellcode found");

        intel_arch.nop_max = nops_raise_error;
        hppa_arch.nop_max = sparc_arch.nop_max = nops_raise_error / 4;

        intel_arch.ptr = hppa_arch.ptr = sparc_arch.ptr = buffer;        
        intel_arch.nop_counter = hppa_arch.nop_counter = sparc_arch.nop_counter = 0;
        intel_arch.continue_checking = hppa_arch.continue_checking = sparc_arch.continue_checking = 0;

        intel_arch.dlen = sparc_arch.dlen = hppa_arch.dlen = len;
        
        while ( len-- ) {

                check_tbl(packet, end, &hppa_arch, hppa_hash);
                check_tbl(packet, end, &intel_arch, intel_hash);
                check_tbl(packet, end, &sparc_arch, sparc_hash);
                
                hppa_arch.ptr++; hppa_arch.dlen--;
                intel_arch.ptr++; intel_arch.dlen--;
                sparc_arch.ptr++; sparc_arch.dlen--;
        }
}



static void shellcode_run(packet_container_t *pc) 
{
        int depth;
        packet_t *packet;
        
        packet = &pc->packet[pc->application_layer_depth];

        if ( packet->len < nops_raise_error )
                return;

        depth = pc->transport_layer_depth;
        
        if ( port_list && depth != -1 ) {
                int ret = -1;

                if ( pc->packet[depth].proto == p_tcp )
                        ret = extract_uint16(&pc->packet[depth].p.tcp->th_dport);

                else if ( pc->packet[depth].proto == p_udp )
                        ret = extract_uint16(&pc->packet[depth].p.udp_hdr->uh_dport);

                if ( ret != -1 ) {
                        
                        ret = protocol_plugin_is_port_ok(port_list, ret);
                        if ( ret < 0 ) 
                                return;
                }
        }

        detect_shellcode(pc, packet->p.data, packet->len);
}




static int set_shellcode_state(prelude_option_t *opt, const char *optarg) 
{
        int ret;
        
        if ( is_enabled == 1 ) {
                ret = plugin_unsubscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;        
                is_enabled = 0;
        }

        else {
                ret = plugin_subscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;
                is_enabled = 1;
        }
        
        return prelude_option_success;
}



static int get_shellcode_state(char *buf, size_t size) 
{
        snprintf(buf, size, "%s", (is_enabled == 1) ? "enabled" : "disabled");
        return prelude_option_success;
}



static int set_port_list(prelude_option_t *opt, const char *arg) 
{
        int ret;
        
        port_list = protocol_plugin_port_list_new();
        if ( ! port_list )
                return prelude_option_error;

        ret = protocol_plugin_add_string_port_to_list(port_list, arg);
        if ( ret < 0 )
                return prelude_option_error;

        return prelude_option_success;
}



static int set_nop_number(prelude_option_t *opt, const char *arg)
{
        nops_raise_error = atoi(arg);
        return prelude_option_success;
}



static int get_nop_number(char *buf, size_t size) 
{
        snprintf(buf, size, "%d", nops_raise_error);
        return prelude_option_success;
}



static int setup_hash_tbl(struct hash_item **hashtbl, struct junks *junk, size_t tblsize) 
{
        int i, ret;
        
        for ( i = 0; i < tblsize; i++ ) {
                ret = add_entry(hashtbl, &junk[i]);
                if ( ret < 0 )
                        return -1;
        }

        return 0;
}





plugin_generic_t *plugin_init(int argc, char **argv)
{
        prelude_option_t *opt;

        /*
         * setup architecture hash table.
         */
        memset(hppa_hash, 0, sizeof(hppa_hash));
        setup_hash_tbl(hppa_hash, hppa_njunk, get_tbl_size(hppa_njunk));

        memset(sparc_hash, 0, sizeof(sparc_hash));
        setup_hash_tbl(sparc_hash, sparc_njunk, get_tbl_size(sparc_njunk));

        memset(intel_hash, 0, sizeof(intel_hash));
        setup_hash_tbl(intel_hash, intel_njunk, get_tbl_size(intel_njunk));
        
        opt = prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 0, "shellcode",
                                 "Set Shellcode plugin option", no_argument,
                                 set_shellcode_state, get_shellcode_state);

        prelude_option_add(opt, CLI_HOOK|CFG_HOOK, 'p', "port-list",
                           "Analyze data on specified port only", required_argument,
                           set_port_list, NULL);
        
        prelude_option_add(opt, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'n', "nops-raise-error",
                           "Set maximum number of contiguous NOP before raising an alert",
                           required_argument, set_nop_number, get_nop_number);
        
        plugin_set_name(&plugin, "Shellcode");
        plugin_set_author(&plugin, "Yoann Vandoorselaere");
        plugin_set_contact(&plugin, "yoann@prelude-ids.org");
        plugin_set_desc(&plugin, "The shellcode plugin will warn you in case NOP number in a packet "
                        "reach a certain threshold.");
        plugin_set_running_func(&plugin, shellcode_run);
        plugin_set_subscribtion(&plugin, subscribtion);
        
        return (plugin_generic_t *) &plugin;
}













