#ifndef PACKET_H
#define PACKET_H

/*****
*
* Copyright (C) 1998 - 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

/* 
 * NOTE: On some systems (namely FreeBSD) including this file leads to 
 * including <sys/queue.h>. If you include also <libprelude/list.h>
 * in your code that will cause a conflict of LIST_HEAD macro
 * from <libprelude/list.h> and <sys/queue.h>. Therefore you should include
 * this file ALWAYS BEFORE <libprelude/list.h> so the macro defined in
 * <libprelude/list.h> takes precedence. Also see comment in 
 * <libprelude/list.h>
 */

#include <pcap.h>
#include <sys/time.h>
#include "nethdr.h"


typedef enum {
        p_null,
	p_raw,
        p_ether,
        p_ip,
        p_ipopts,
        p_ipencap,
        p_ipicmp,
        p_arp,
        p_rarp,
        p_udp,
        p_tcp,
        p_tcpopts,
        p_icmp,
        p_igmp,
        p_data,
        p_all,
        p_external,
        p_end
} proto_enum_t;



/*
 * Max number of packet that can be kept in memory by Prelude.
 */
#define MAX_PKTINUSE 1000 


/*
 *
 */
#define MAX_PKTDEPTH 20


/*
 * Packet stuff
 */


union proto_u {
        uint32_t *null_hdr;
        etherhdr_t *ether_hdr;
        iphdr_t *ip;
        tcphdr_t *tcp;
        udphdr_t *udp_hdr;
        icmphdr_t *icmp_hdr;
        igmphdr_t *igmp_hdr;
        etherarphdr_t *arp_hdr;
        unsigned char *data, *opts;
};



typedef struct {
        uint16_t len;
        uint8_t proto;
        union proto_u p;
} packet_t;



typedef struct {
        struct pcap_pkthdr *pcap_hdr;
        
	/*
	 * Current depth of the packet (used in order to index the headers array).
	 */
        int8_t depth;
        
        /*
	 * Used by the stream reassembly module and the signature engine
	 * to directly address the packet array without having to search through it.
	 */
        int8_t network_layer_depth;     /* Ip */
        int8_t transport_layer_depth;   /* Tcp, udp */
        int8_t application_layer_depth; /* application data */
        
	/*
	 * How many place are currently referencing this packet.
	 */
        int refcount;
        
	/*
	 * Pointer on the raw data packet buffer.
	 */
	unsigned char *captured_data;

	/*
	 * If data allocation was needed in order to store this packet after the capture,
	 * (for exemple for IP defragmentation) it is referenced here.
	 */
        unsigned char *allocated_data;

        /*
         * this is where an eventual TCP allocate buffer is reference (TCP stream
         * reassembly use this).
         */
        unsigned char *tcp_allocated_data;
        
	/*
	 * Protocol plugin related data, not part of the headers array, 
	 * because only the plugin know about theses data (private, but still packet related).
	 */
	int8_t protocol_plugin_id;
	void *protocol_plugin_data;
        
	/*
	 * Array containing the headers.
	 */ 
	packet_t packet[MAX_PKTDEPTH];

        /*
         *
         */
        uint32_t paws_tsval;
        uint32_t paws_tsecr;

        uint8_t tcp_sack;
        
} packet_container_t;


#define packet_2_container(packet) packet - offsetof(packet_container_t, packet)


#endif





