/*****
*
* Copyright (C) 2001, 2002, 2003 Jeremie Brebec <flagg@ifrance.com>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Jeremie Brebec <flagg@ifrance.com>
*
*****/


#include <stdlib.h>
#include <stdio.h>

#include "rules.h"

#include <libprelude/prelude-log.h>
#include <libprelude/list.h>

#include "pconfig.h"
#include "rules-type.h"
#include "rules-operations.h"
#include "rules-default.h"



/* "global variable" */
static rule_t *_rule_added;
static run_f_t _run_added;
static rules_node_t *_used_root;




/*
 * make a new node
 */
static rules_node_t *make_new_node(var_t *var, rules_node_t *child, rules_node_t *brother) 
{
	rules_node_t *node;

	node = malloc(sizeof(rules_node_t));
	if (! node) {
                log(LOG_ERR, "memory exhausted.\n");
		return NULL;
	}

        node->ref = 0;
        node->run = NULL;
	node->match_packet = NULL;
        
        node->child = child;
	if ( child )
                child->ref++;

	node->brother = brother;
	if ( brother )
                brother->ref++;
        
	if (var) {
		generic_type_t *type;

                copy_var(&node->var, var);
                
		type = signature_engine_get_type_by_id(node->var.id);
		if ( type )
			node->match_packet = type->match_packet;
	} else {
                node->var.id = ID_LEAF_NODE;
		node->var.set = NULL;
	}
        
	return node;
}




static rules_node_t *r_rule_to_node(test_t *test)
{
	rules_node_t *node;
        
	if (!test) {

		node = make_new_node(NULL, NULL, NULL);

		/* Ignore Rule ? */
		if (!_run_added)
			node->var.id = ID_IGNORE_NODE;
		else {

			l_run_t *run;

                        run = malloc(sizeof(l_run_t));
			if ( ! run ) {
                                log(LOG_ERR, "memory exhausted.\n");
				return NULL;
			}
			
			run->run = _run_added;
			run->data = _rule_added->data;
			run->leaf_match = _rule_added->leaf_match;
			run->next = NULL;
                        
			node->run = run;
		}
	} else {
		if (test->inversed)
			node = make_new_node(&test->var, NULL, r_rule_to_node(test->next));
		else
			node = make_new_node(&test->var, r_rule_to_node(test->next), NULL);
	}

	return node;
		
}




static rules_node_t *rule_to_node(rule_t *rule) 
{
	return r_rule_to_node(rule->test);
}




#if 0
static int are_nodes_equal(rules_node_t *node1, rules_node_t *node2) 
{
	l_run_t *run1, *run2;

	if ( node1 == node2 )
		return 0;

	if ( ! node1 || ! node2 )
		return -1;

	if ( node1->var.id != node2->var.id )
		return -1;

	if ( node1->var.id >= 0 ) {
		generic_type_t *type;

                type = signature_engine_get_type_by_id(node1->var.id);
                if ( ! type->equal(node1->var.set, node2->var.set) )
			return -1;
	}

	run1 = node1->run;
	run2 = node2->run;

	while ( run1 == run2 && run1 ) {
		run1 = run1->next;
		run2 = run2->next;
	}

	if ( run1 || run2 )
		return -1;

	return are_nodes_equal(node1->child, node2->child) &&
                are_nodes_equal(node1->brother, node2->brother);

}



static rules_node_t *search_same_node(rules_node_t *ref_node, rules_node_t *tree)
{
	generic_type_t *type, *ref_type;
	
	if ( ! tree || ref_node == tree || ref_node->var.id < 0 || tree->var.id < 0 )
		return NULL;

	ref_type = signature_engine_get_type_by_id(ref_node->var.id);
	type = signature_engine_get_type_by_id(tree->var.id);

	if ( ref_type == type ) {

		int m_ref_tree, m_tree_ref;

		if ( are_nodes_equal(ref_node, tree) ) 
			return tree;
		
		m_ref_tree = type->match(ref_node, tree);
		m_tree_ref = type->match(tree, ref_node);
		
		if ( m_ref_tree && m_tree_ref ) {
			rules_node_t *r;
                        
			if ( (r = search_same_node(ref_node, tree->child)) )
				return r;
                        
			return search_same_node(ref_node, tree->brother);
		}
		
		if ( m_ref_tree )
			return NULL;
		
		if ( m_tree_ref )
			return search_same_node(ref_node, tree->child);
		
		return search_same_node(ref_node, tree->brother);
	}
		
	if (type->priority < ref_type->priority) {
		rules_node_t *r;
		
		if ((r = search_same_node(ref_node, tree->child)) )
			return r;
                
		return search_same_node(ref_node, tree->brother);
	}

	return NULL;
}
#endif


/*******************************************************
 * Add a rule to a tree
 *******************************************************/


static rules_node_t *_r_insert_node( rules_node_t *current, 
				     rules_node_t *test, 
				     rules_node_t *new_child_node, int shared );

static rules_node_t *process_child(rules_node_t *current, rules_node_t *test, 
                                   rules_node_t *new_child_node, int shared)
{
	rules_node_t *tmp;

        tmp = _r_insert_node(current->child, test, new_child_node, shared);

	if ( tmp != current->child ) {

		if ( shared )
			current = copy_rules_node(current);

		if ( tmp )
                        tmp->ref++;

                delete_rules_node(current->child);
		current->child = tmp;

                /*
                 * if (shared && (tmp = search_same_node( current, _used_root->child )))
                 *      return tmp;
                 */
	}
	return current;
}




static rules_node_t *process_brother(rules_node_t *current, rules_node_t *test, 
                                     rules_node_t *new_child_node, int shared)
{
	rules_node_t *tmp;

        tmp = _r_insert_node(current->brother, test, new_child_node, shared);

	if ( tmp != current->brother ) {

		if ( shared )
			current = copy_rules_node(current);

		if ( tmp )
                        tmp->ref++;
                
		delete_rules_node(current->brother);
		current->brother = tmp;

                /*
                 * if ( shared && (tmp = search_same_node( tmp, _used_root->child )))
                 *      return tmp;
                 */
	}
	return current;
}



static rules_node_t *ri_next_node(rules_node_t *current, rules_node_t *test, int shared)
{
	if ( test->brother )
		return process_brother(current, test->brother, NULL, shared);
	else
		return process_child(current, test->child, NULL, shared);
}



static rules_node_t *ri_broadcast_test(rules_node_t *current, rules_node_t *test, int shared)
{
	current = process_child(current, test, NULL, shared);
	current = process_brother(current, test, NULL, shared);
	
	return current;
}



/*
 * Add a run function to a node
 */
static rules_node_t *ri_add_run_fun(rules_node_t *current, int shared) 
{
	l_run_t *run;

	for( run = current->run; run!= NULL; run = run->next ) 
		if ( run->run ==_run_added &&
                     run->data == _rule_added->data &&
                     run->leaf_match == _rule_added->leaf_match )
			return current;

	if ( shared )
		current = copy_rules_node(current);

	/* add the run function */
	run = malloc(sizeof(l_run_t));
	if ( ! run ) {
		log(LOG_ERR, "memory exhausted.\n");
		return current;
	}

	run->run = _run_added;
	run->data = _rule_added->data;
	run->leaf_match = _rule_added->leaf_match;
		
	run->next = current->run;
	current->run = run;

	return current;

}



static rules_node_t *ri_add_new_node_to_leaf(rules_node_t *current, rules_node_t *test, 
                                             rules_node_t *new_child_node, int shared)
{

	generic_type_t *type;

	if ( shared )
		current = copy_rules_node(current);
	
	type = signature_engine_get_type_by_id(test->var.id);
	if ( ! type ) 
                return NULL;
	
	/*
         * copy test, this delete the leaf status
         */
	copy_var(&current->var, &test->var);
	current->child = new_child_node;
        
	if ( new_child_node )
                new_child_node->ref++;
	
	/*
         * add the match_packet function
         */
	current->match_packet = type->match_packet;
	
	/*
         * next test
         */
	return ri_next_node(current, test, shared);
}



/*
 * Make current node an ignore node
 */
static rules_node_t *ri_make_ignore_node(rules_node_t *current, int shared)
{
	generic_type_t *type;

	if ( shared )
		current = copy_rules_node(current);

	type = signature_engine_get_type_by_id(current->var.id);
	if ( type )
		type->delete(current->var.set);

	current->var.id = ID_IGNORE_NODE;
	current->run = NULL;

	delete_rules_node(current->child);
	delete_rules_node(current->brother);
	
	current->child = NULL;
	current->brother = NULL;
	
	return current;
}




/*
 * Insert a new node 'test' before current
 */
static rules_node_t *ri_insert_new_node(rules_node_t *current, rules_node_t *test, 
                                        rules_node_t *new_child_node, int shared)
{
        rules_node_t *n, **last_child = &new_child_node;


	/*
         * copy current to the end of the new_child_list.
         */
	if ( new_child_node )
		current->ref++;

	while ( *last_child )
		last_child = &(*last_child)->brother;

        *last_child = current;

	n = make_new_node(&test->var, new_child_node, current);
        
	return ri_next_node(n, test, shared);
} 




/*
 * insert 'test' as a sub-test of 'current'
 */
static rules_node_t *ri_insert_sub_test(rules_node_t *current, rules_node_t *test, int shared) 
{
	current = process_child(current, test, NULL, shared);
		
	/*
         * if this test is a 'not' test, we should handle brother too.
         */
	if ( test->brother )
		current = process_brother(current, test, NULL, shared);

	return current;
}




/*
 * make current a sub-test of 'test'
 */
static rules_node_t *ri_insert_test(rules_node_t *current, rules_node_t *test, 
                                    rules_node_t *new_child_node, int shared)
{
	rules_node_t *next_node;

        next_node = current->brother;

	/*
         * link current to the child list
         */
	if ( shared )
		current = copy_rules_node(current);

	if ( new_child_node )
                new_child_node->ref++;
        
	if ( current->brother )
                current->brother->ref--;
        
	current->brother = new_child_node;

	/*
         * process next brother
         */
	return _r_insert_node(next_node, test, current, shared);
}



/*
 * merge 'current' and 'test' node
 */
static rules_node_t *ri_merge_test(rules_node_t *current, rules_node_t *test, 
                                   rules_node_t *new_child_node, int shared)
{
        var_t inter;
        rules_node_t *n;
        void *test_rest_set;
	void *current_rest_set;
	rules_node_t *alt_test;
	generic_type_t *type;

        type = signature_engine_get_type_by_id(current->var.id);
	if ( ! type )
		return NULL;

	type->split(test->var.set, current->var.set,
                    &inter.set, &test_rest_set, &current_rest_set);

	/*
         * break the current test
         */
	if ( shared )
		current = copy_rules_node(current);

	type->delete(current->var.set);
	current->var.set = current_rest_set;
		
	/* 
	 * break the rule test 
	 * we can't modify 'test' because of recursivity
	 * so create an another test list
	 */
	alt_test = copy_rules_node(test);
	alt_test->var.set = test_rest_set;
	
	/*
         * make a new node for the intersection test
         */
	inter.id = test->var.id;
	n = make_new_node(&inter, current->child, current->brother );
	type->delete(inter.set);

	/*
         * link the intersection to the current node
         */
	n->ref++;
	if ( current->brother )
                current->brother->ref--;
	current->brother = n;
	
	/*
         * if test is a 'not'-test, fusion with current, else with the intersection
         */
	if ( test->brother )
		current = process_child(current, test->brother, NULL, shared);
	else
		n = process_child(n, test->child, NULL, shared);
	
	/*
         * process next brother
         */
	n = process_brother(n, alt_test, new_child_node, shared);
		
	/*
         * delete alt_test
         */
	delete_rules_node(alt_test);
	
	return current;
}
	



/*
 *
 */
static rules_node_t *_r_insert_node(rules_node_t *current, rules_node_t *test, 
                                    rules_node_t *new_child_node, int shared) 
{
	generic_type_t *current_type, *test_type;
	int match_current_test, match_test_current;

	/* 
	 * current node
	 * if NULL, create a leaf node
	 */
	if ( ! current ) {
		if ( new_child_node ) {
			shared = 0;
			current = make_new_node(NULL, NULL, NULL);
		} else
			return test;
	} else
		shared |= current->ref > 1;

	/*
	 * Ignore Node -> stop here
	 */
	if ( current->var.id == ID_IGNORE_NODE ) 
		return current;

	/*
	 * if test is an ignore node, delete current child and brother
	 */
	if ( test->var.id == ID_IGNORE_NODE )
		return ri_make_ignore_node(current, shared);

	/*
	 * if test is NULL, we should stop here
	 */
	if ( test->var.id == ID_LEAF_NODE )
		return ri_add_run_fun(current, shared);

	/*
	 * if current is a leaf,
         * we are at the end of an exploration we should add a new node.
	 */
	if ( current->var.id == ID_LEAF_NODE )
		return ri_add_new_node_to_leaf(current, test, new_child_node, shared);

        
        test_type = signature_engine_get_type_by_id(test->var.id);
	if ( ! test_type )
		return current;
        
        current_type = signature_engine_get_type_by_id(current->var.id);
        if ( ! current_type )
		return NULL;

	/*
	 * priority of (current_type) > priority of (test_type)
	 * 2 cases: we have reached an "else" condition while traversing the brother list
	 * or we have reached a test less important that the one we want to check 
	 *
	 * we should add the new node here
	 */
	if ( current_type->priority > test_type->priority )
		return ri_insert_new_node(current, test, new_child_node, shared);


	/*
	 * priority of (current_type) < priority of (rule_type)
	 * so we don't have find in our rule list a test for this priority
	 * the rule match all this priority
	 */
	if ( current_type->priority < test_type->priority )
		return ri_broadcast_test(current, test, shared);


	/*
	 * here, priority of (current_type) = priority of (rule_type)
	 * we should handle collision
	 */

	/*
	 * case 1, current = rule, so the test already exist
	 * nothing to do
	 */
	if ( current_type->equal(current->var.set, test->var.set) )
		return ri_next_node(current, test, shared);

	match_current_test = current_type->match(current->var.set, test->var.set);
	match_test_current = current_type->match(test->var.set, current->var.set);

	/*
	 * case 2, current match test, and test match current
	 * there is no order relation between this type
	 * we can't decide where to go
	 * -> add to child and brother, and then check for integrity
	 */
	if ( match_current_test && match_test_current )
		return ri_broadcast_test(current, test, shared);

	/*
	 * case 3, the rule test is included in the current test
	 * so we should add a sub node
	 */
	if ( match_current_test )
		return ri_insert_sub_test(current, test, shared);

	/*
	 * case 4, the current test is included in the rule test,
	 * so we should make the current a sub test of our new rule
	 * and process to next brother
	 */
	if ( match_test_current )
		return ri_insert_test(current, test, new_child_node, shared);

	/*
	 * case 5, intersection of the rule test and the current test is not empty
	 * so we should fusion the intersection and break the rule test
	 */
	if ( current_type->intersection(test->var.set, current->var.set) )
		return ri_merge_test(current, test, new_child_node, shared);

	/*
	 * else continue with the brother
	 * if test is a 'not'-test, fusion with current child
	 */
	if ( test->brother )
		current = process_child(current, test->brother, new_child_node, shared);

	current = process_brother(current, test, new_child_node, shared);

	return current;
}




/*
 *
 */
static int add_rule(rules_node_t *root, rule_t *rule, run_f_t run) 
{
	rules_node_t *rule_tmp;

	/*
         * Check for real root
         */
	if ( ! root || root->var.id != ID_ROOT_NODE ) {
                log(LOG_ERR, "rule isn't root\n");
		return -1;
        }

	if ( check_rule_integrity(rule) < 0 ) {
                log(LOG_ERR, "integrity check failed.\n");
		return -1;
        }

	/*
         * set global variable
         */        
        _rule_added = rule;
	_run_added = run;
	_used_root = root;

	rule_tmp = rule_to_node(rule);
	rule_tmp->ref++;

	/* begin the recursion */
	process_child(root, rule_tmp, NULL, 0);

	delete_rules_node(rule_tmp);

	return 0;
}




/*
 * Add a rules to the tree.
 */
int signature_engine_add_rules(rules_node_t *root, rules_t *rules, run_f_t run) 
{        
	if ( add_rule(root, rules->rule, run) < 0 )
		return -1;

        if ( rules->next )
                return signature_engine_add_rules(root, rules->next, run);
        
        return 0;
}



/*
 * Process a packet on a tree
 */
#ifdef DEBUG
static void debug_print_node(rules_node_t *node, int matched, int depth) 
{
        int i;
        generic_type_t *type;
        
        type = signature_engine_get_type_by_id(node->var.id);
        if ( type ) {
                for (i = 0; i < depth; i++ )
                        fprintf(stderr, "\t");
                
                fprintf(stderr, "[%s id=%d] ", matched == 0 ? "matched" : "not matched", node->var.id);
                type->print(node->var.set);
        }
}
#endif


/* 
 * Report only one matching rule
 * (the most specific rule)
 */

#define MAX_RULES_CALLED 1000
extern Pconfig_t config;


void signature_engine_process_packet(rules_node_t *root, packet_container_t *packet) 
{
        int depth = 0;
        rules_node_t *node;
        int rules_nb = -1, ignore_nb = -1, i, ret;
        static l_run_t *runtbl[MAX_RULES_CALLED];
	static l_run_t *ignoretbl[MAX_RULES_CALLED];

        signature_engine_start_new_match();
        
	if ( ! root || root->var.id != ID_ROOT_NODE )
		return;

	node = root->child;
	
	while ( node ) {
		l_run_t *run;
                
		/* call each run function after leaf_match testing */
		for ( run = node->run; run != NULL; run = run->next ) {
                        /*
                         * possible "ignore"
                         */
                        if ( ! run->run ) {
				if ( ignore_nb < MAX_RULES_CALLED - 1 )
                                        ignore_nb++;
                                ignoretbl[ignore_nb] = run;

                        } else {
				if ( rules_nb < MAX_RULES_CALLED - 1 )
                                        rules_nb++;
				runtbl[rules_nb] = run;
			}
		}

		/* Ignore node stop the processing */
		if ( node->var.id == ID_IGNORE_NODE ) 
			goto out;

		/*
		 * Check test if node is valid ie: not a leaf or an ignore_node
		 */
                if ( node->var.id < 0 )
                        break;

                ret = node->match_packet(packet, node->var.set);
#ifdef DEBUG
                debug_print_node(node, ret, depth);
#endif
                if ( ret < 0 )
                        node = node->brother;
                else {
                        depth++;
                        node = node->child;	       
                }
        }
        

	/* Check Ignore */
	for ( i = ignore_nb; i >= 0; i-- ) {
		l_leaf_match_t *leaf_match;
                
		for ( leaf_match = ignoretbl[i]->leaf_match;
                      leaf_match != NULL; leaf_match = leaf_match->next ) {
                    
                        ret = leaf_match->leaf_match(packet, leaf_match->var.set);
                        ret = (ret == 0) ? 1 : 0;
                        ret = ret ^ leaf_match->inversed;
                        
                        if ( ! ret )
                                break;
		}

		if (! leaf_match )
			goto out; /* Ignore */
	}

	/* Check Run */
	for ( i = rules_nb; i >= 0; i-- ) {
		l_leaf_match_t *leaf_match;
                
		for ( leaf_match = runtbl[i]->leaf_match;
                      leaf_match != NULL; leaf_match = leaf_match->next ){

                        ret = leaf_match->leaf_match(packet, leaf_match->var.set);
                        ret = (ret == 0) ? 1 : 0;
                        ret = ret ^ leaf_match->inversed;
                        
                        if ( ! ret )
                                break;
		}

		if ( ! leaf_match ) {
			runtbl[i]->run(packet, runtbl[i]->data) ; 
                        if ( ! config.report_all ) 
                                goto out;
		}
	}

  out:
#ifdef DEBUG
        fprintf(stderr, "\n\n**** Packet processed ****\n\n");
#endif

        return;
}
