/* -*- Mode: c; c-basic-offset: 2 -*-
 *
 * sparql_parser.y - Rasqal SPARQL parser over tokens from sparql_lexer.l
 *
 * $Id: sparql_parser.y,v 1.76 2005/03/21 14:41:56 cmdjb Exp $
 *
 * Copyright (C) 2004-2005, David Beckett http://purl.org/net/dajobe/
 * Institute for Learning and Research Technology http://www.ilrt.bristol.ac.uk/
 * University of Bristol, UK http://www.bristol.ac.uk/
 * 
 * This package is Free Software and part of Redland http://librdf.org/
 * 
 * It is licensed under the following three licenses as alternatives:
 *   1. GNU Lesser General Public License (LGPL) V2.1 or any newer version
 *   2. GNU General Public License (GPL) V2 or any newer version
 *   3. Apache License, V2.0 or any newer version
 * 
 * You may not use this file except in compliance with at least one of
 * the above three licenses.
 * 
 * See LICENSE.html or LICENSE.txt at the top of this package for the
 * complete terms and further detail along with the license texts for
 * the licenses in COPYING.LIB, COPYING and LICENSE-2.0.txt respectively.
 *
 * References:
 *   SPARQL Query Language for RDF
 *   http://www.w3.org/TR/rdf-sparql-query/
 *
 * Editor's draft of above http://www.w3.org/2001/sw/DataAccess/rq23/
 *
 */

%{
#ifdef HAVE_CONFIG_H
#include <rasqal_config.h>
#endif

#ifdef WIN32
#include <win32_rasqal_config.h>
#endif

#include <stdio.h>
#include <stdarg.h>

#include <rasqal.h>
#include <rasqal_internal.h>

#include <sparql_parser.h>

#define YY_DECL int sparql_lexer_lex (YYSTYPE *sparql_parser_lval, yyscan_t yyscanner)
#define YY_NO_UNISTD_H 1
#include <sparql_lexer.h>

#include <sparql_common.h>


/* Make verbose error messages for syntax errors */
/*
#ifdef RASQAL_DEBUG
#define YYERROR_VERBOSE 1
#endif
*/
#define YYERROR_VERBOSE 1

/* Slow down the grammar operation and watch it work */
#if RASQAL_DEBUG > 2
#define YYDEBUG 1
#endif

/* the lexer does not seem to track this */
#undef RASQAL_SPARQL_USE_ERROR_COLUMNS

/* Missing sparql_lexer.c/h prototypes */
int sparql_lexer_get_column(yyscan_t yyscanner);
/* Not used here */
/* void sparql_lexer_set_column(int  column_no , yyscan_t yyscanner);*/


/* What the lexer wants */
extern int sparql_lexer_lex (YYSTYPE *sparql_parser_lval, yyscan_t scanner);
#define YYLEX_PARAM ((rasqal_sparql_query_engine*)(((rasqal_query*)rq)->context))->scanner

/* Pure parser argument (a void*) */
#define YYPARSE_PARAM rq

/* Make the yyerror below use the rdf_parser */
#undef yyerror
#define yyerror(message) sparql_query_error((rasqal_query*)rq, message)

/* Make lex/yacc interface as small as possible */
#undef yylex
#define yylex sparql_lexer_lex


static int sparql_parse(rasqal_query* rq, const unsigned char *string);
static void sparql_query_error(rasqal_query* rq, const char *message);
static void sparql_query_error_full(rasqal_query *rq, const char *message, ...);
%}


/* directives */


%pure-parser


/* Interface between lexer and parser */
%union {
  raptor_sequence *seq;
  rasqal_variable *variable;
  rasqal_literal *literal;
  rasqal_triple *triple;
  rasqal_expression *expr;
  rasqal_graph_pattern *graph_pattern;
  double floating;
  raptor_uri *uri;
  unsigned char *name;
}


/*
 * shift/reduce conflicts
 */
%expect 2

/* word symbols */
%token SELECT SOURCE FROM WHERE AND
%token OPTIONAL PREFIX DESCRIBE CONSTRUCT ASK DISTINCT LIMIT UNION
%token BASE LOAD BOUND STR LANG DATATYPE ISURI ISBLANK ISLITERAL
%token GRAPH WITH

/* expression delimitors */

%token ',' '(' ')' '[' ']' '{' '}'
%token '?' '$'

/* function call indicator */
%token '&'

/* SC booleans */
%left SC_OR SC_AND

/* string operations */
%left STR_EQ STR_NE STR_MATCH STR_NMATCH

/* operations */
%left EQ NEQ LT GT LE GE

/* arithmetic operations */
%left '+' '-' '*' '/' '%'

/* unary operations */
%left '~' '!'

/* literals */
%token <literal> FLOATING_POINT_LITERAL
%token <literal> STRING_LITERAL PATTERN_LITERAL INTEGER_LITERAL
%token <literal> BOOLEAN_LITERAL
%token <literal> NULL_LITERAL
%token <uri> URI_LITERAL URI_LITERAL_BRACE
%token <name> QNAME_LITERAL BLANK_LITERAL QNAME_LITERAL_BRACE

%token <name> IDENTIFIER


%type <seq> SelectClause ConstructClause DescribeClause
%type <seq> PrefixDeclOpt GraphClauseOpt WhereClauseOpt
%type <seq> VarList VarOrURIList ArgList URIList
%type <seq> ConstructPattern TriplePatternList

%type <seq> GraphPattern GraphOrPattern GraphAndPattern 

%type <graph_pattern> PatternElement PatternElementConstraint
%type <graph_pattern> NamedGraphPattern OptionalGraphPattern

%type <expr> Expression ConditionalAndExpression ValueLogical
%type <expr> EqualityExpression RelationalExpression AdditiveExpression
%type <expr> MultiplicativeExpression UnaryExpression
%type <expr> PrimaryExpression BuiltinExpression

%type <literal> Literal URI BNode
%type <literal> VarOrLiteral VarOrURI
%type <literal> VarOrLiteralOrBNode VarOrURIOrBNode
%type <literal> URIBrace

%type <variable> Var

%type <triple> TriplePattern TripleTemplate


%%

/* Below here, grammar terms are numbered from
 * http://www.w3.org/TR/2005/WD-rdf-sparql-query-20050217/
 * except where noted
 */

/* SPARQL Grammar: [1] Query */
Query : BaseDeclOpt PrefixDeclOpt ReportFormat LoadClauseOpt GraphClauseOpt
        NamedGraphClauseOpt WhereClauseOpt LimitClauseOpt
{
}
;

/* NEW Grammar Term pulled out of [1] Query */
ReportFormat : SelectClause
{
  ((rasqal_query*)rq)->selects=$1;
}
|  ConstructClause
{
  ((rasqal_query*)rq)->constructs=$1;
}
|  DescribeClause
{
  ((rasqal_query*)rq)->select_is_describe=1;
  ((rasqal_query*)rq)->describes=$1;
}
| AskClause
{
  ((rasqal_query*)rq)->ask=1;
}
;

/* SPARQL Grammar: [2] Prolog - merged into Query */


/* SPARQL Grammar: [3] SelectClause */
SelectClause : SELECT DISTINCT VarList
{
  $$=$3;
  ((rasqal_query*)rq)->distinct=1;
}
| SELECT DISTINCT '*'
{
  $$=NULL;
  ((rasqal_query*)rq)->select_all=1;
  ((rasqal_query*)rq)->distinct=1;
}
| SELECT VarList
{
  $$=$2;
}
| SELECT '*'
{
  $$=NULL;
  ((rasqal_query*)rq)->select_all=1;
}
;


/* SPARQL Grammar: [4] DescribeClause */
DescribeClause : DESCRIBE VarOrURIList
{
  $$=$2;
}
| DESCRIBE '*'
{
  $$=NULL;
}
;

/* SPARQL Grammar: [5] rq23 ConstructClause */
ConstructClause : CONSTRUCT ConstructPattern
{
  $$=$2;
}
| CONSTRUCT '*'
{
  $$=NULL;
  ((rasqal_query*)rq)->construct_all=1;
}
;


/* SPARQL Grammar: [6] AskClause */
AskClause : ASK 
{
}


/* SPARQL Grammar: [7] LoadClause - renamed for clarity */
LoadClauseOpt : LOAD URIList
{
  if($2) {
    int i;
    
    for(i=0; i < raptor_sequence_size($2); i++) {
      raptor_uri* uri=(raptor_uri*)raptor_sequence_get_at($2, i);
      rasqal_query_add_data_graph((rasqal_query*)rq, uri, uri, RASQAL_DATA_GRAPH_NAMED);
    }
    
    raptor_free_sequence($2);
  }
}
| /* empty */
{
}
;


/* SPARQL Grammar: [7] GraphClause - renamed for clarity */
GraphClauseOpt : WITH URIList
{
  if($2) {
    int i;
    
    for(i=0; i < raptor_sequence_size($2); i++) {
      raptor_uri* uri=(raptor_uri*)raptor_sequence_get_at($2, i);
      rasqal_query_add_data_graph((rasqal_query*)rq, uri, NULL, RASQAL_DATA_GRAPH_BACKGROUND);
    }
    
    raptor_free_sequence($2);
  }
}
| /* empty */
{
}
;  


/* SPARQL Grammar: [8] NamedGraphClause - renamed for clarity */
NamedGraphClauseOpt : FROM URIList
{
  if($2) {
    int i;
    
    for(i=0; i < raptor_sequence_size($2); i++) {
      raptor_uri* uri=(raptor_uri*)raptor_sequence_get_at($2, i);
      rasqal_query_add_data_graph((rasqal_query*)rq, uri, uri, RASQAL_DATA_GRAPH_NAMED);
    }
    raptor_free_sequence($2);
  }
}
| /* empty */
{
}
;


/* SPARQL Grammar: [9] SourceSelector - junk */

/* SPARQL Grammar: [10] WhereClause - remained for clarity */
WhereClauseOpt :  WHERE GraphPattern
{
  if($2) {
    rasqal_graph_pattern* gp=((rasqal_query*)rq)->query_graph_pattern;
    if(gp->graph_patterns) {
      raptor_sequence_join(gp->graph_patterns, $2);
      raptor_free_sequence($2);
    } else
      gp->graph_patterns=$2;
  }
}
| /* empty */
{
}
;


/* SPARQL Grammar: [11] LimitClause - remained for clarity */
LimitClauseOpt :  LIMIT INTEGER_LITERAL
{
  if($2 != NULL)
    ((rasqal_query*)rq)->limit=$2->value.integer;
}
| /* empty */
{
}
;

/* SPARQL Grammar: [12] GraphPattern */
GraphPattern : GraphOrPattern
{
  $$=$1;
}
;


/* SPARQL Grammar: [13] PatternGroup - merged into GraphPattern */

/* SPARQL Grammar: [14] GraphOrPattern */
GraphOrPattern : GraphAndPattern
{
  $$=$1;
}
| GraphAndPattern UNION GraphAndPattern 
{
  /* FIXME - union graph pattern type */
  sparql_syntax_warning(((rasqal_query*)rq), "SPARQL UNION ignored");
  $$=$1;
}
;


/* SPARQL Grammar: [15] GraphAndPattern */
GraphAndPattern : GraphAndPattern PatternElementConstraint
{
#if RASQAL_DEBUG > 1  
  printf("GraphAndPattern 1\n  graphpattern=");
  raptor_sequence_print($1, stdout);
  printf(", patternelement=");
  if($2)
    rasqal_graph_pattern_print($2, stdout);
  else
    fputs("NULL", stdout);
  fputs("\n\n", stdout);
#endif

  $$=$1;
  if($2)
    raptor_sequence_push($$, $2);
}
| PatternElementConstraint
{
#if RASQAL_DEBUG > 1  
  printf("GraphAndPattern 2\n  patternelement=");
  if($1)
    rasqal_graph_pattern_print($1, stdout);
  else
    fputs("NULL", stdout);
  fputs("\n\n", stdout);
#endif

  $$=raptor_new_sequence((raptor_sequence_free_handler*)rasqal_free_graph_pattern, (raptor_sequence_print_handler*)rasqal_graph_pattern_print);
  if($1)
    raptor_sequence_push($$, $1);
}
;


/* SPARQL Grammar: [16] PatternElement */
PatternElement : TriplePatternList
{
#if RASQAL_DEBUG > 1  
  printf("PatternElement 1\n  triplepatternlist=");
  raptor_sequence_print($1, stdout);
  fputs("\n\n", stdout);
#endif

  if($1) {
    raptor_sequence *s=((rasqal_query*)rq)->triples;
    int offset=raptor_sequence_size(s);
    int triple_pattern_size=raptor_sequence_size($1);
    
    raptor_sequence_join(s, $1);
    raptor_free_sequence($1);

    $$=rasqal_new_graph_pattern_from_triples((rasqal_query*)rq, s, offset, offset+triple_pattern_size-1, 0);
  } else
    $$=NULL;
}
| '{' GraphPattern '}' /*  GroupGraphPattern inlined */
{
#if RASQAL_DEBUG > 1  
  printf("PatternElement 2\n  graphpattern=");
  raptor_sequence_print($2, stdout);
  fputs("\n\n", stdout);
#endif
  
  $$=rasqal_new_graph_pattern_from_sequence((rasqal_query*)rq, $2, 0);
}
| NamedGraphPattern
{
  $$=$1;
}
| OptionalGraphPattern
{
  $$=$1;
}
;


/* NEW Grammar Term */
/* Ensures a constraint gets added to it's adjacent graph pattern */
PatternElementConstraint : PatternElement
{
  $$=$1;
}
| PatternElement AND Expression
{
  $$=$1;
  rasqal_graph_pattern_add_constraint($$, $3);
}
;


/* NEW Grammar Term */
TriplePatternList : TriplePatternList TriplePattern
{
  $$=$1;
  raptor_sequence_push($$, $2);
}
| TriplePattern
{
  $$=raptor_new_sequence((raptor_sequence_free_handler*)rasqal_free_triple, (raptor_sequence_print_handler*)rasqal_triple_print);
  raptor_sequence_push($$, $1);
}
;


/* SPARQL Grammar: [17] PatternElementAsGroup - junk */

/* SPARQL Grammar: [18] GroupGraphPattern - inlined into PatternElement */

/* SPARQL Grammar: [19] NamedGraphPattern */
NamedGraphPattern: GRAPH '*' PatternElementConstraint
{
#if RASQAL_DEBUG > 1  
  printf("NamedGraphPattern 1\n  patternelement=");
  rasqal_graph_pattern_print($3, stdout);
  fputs("\n\n", stdout);
#endif

  /* FIXME */
  sparql_syntax_warning(((rasqal_query*)rq), "SPARQL GRAPH * ignored");

  $$=$3;
}
| GRAPH VarOrURI PatternElementConstraint
{
#if RASQAL_DEBUG > 1  
  printf("NamedGraphPattern 2\n  varoruri=");
  rasqal_literal_print($2, stdout);
  printf(", patternelement=");
  rasqal_graph_pattern_print($3, stdout);
  fputs("\n\n", stdout);
#endif

  rasqal_graph_pattern_set_origin($3, $2);

  rasqal_free_literal($2);
  $$=$3;
}
|  SOURCE '*' PatternElementConstraint
{
#if RASQAL_DEBUG > 1  
  printf("NamedGraphPattern 3\n  patternelement=");
  rasqal_graph_pattern_print($3, stdout);
  fputs("\n\n", stdout);
#endif

  /* FIXME */
  sparql_syntax_warning(((rasqal_query*)rq), "SPARQL SOURCE * ignored");

  $$=$3;
}
| SOURCE VarOrURI PatternElementConstraint
{
#if RASQAL_DEBUG > 1  
  printf("NamedGraphPattern 4\n  varoruri=");
  rasqal_literal_print($2, stdout);
  printf(", patternelement=");
  rasqal_graph_pattern_print($3, stdout);
  fputs("\n\n", stdout);
#endif

  /* FIXME - deprecated */
  sparql_syntax_warning(((rasqal_query*)rq), "Use GRAPH instead of SOURCE in SPARQL (2005-02-17 WD)");

  rasqal_graph_pattern_set_origin($3, $2);

  rasqal_free_literal($2);
  $$=$3;
}
;


/* SPARQL Grammar: [20] OptionalGraphPattern */
OptionalGraphPattern: OPTIONAL PatternElementConstraint
{
  int i;
  raptor_sequence *s=$2->graph_patterns;

#if RASQAL_DEBUG > 1  
  printf("PatternElementForms 4\n  graphpattern=");
  rasqal_graph_pattern_print($2, stdout);
  fputs("\n\n", stdout);
#endif

  if(s) {
    /* Flag all the triples in GraphPattern1 as optional */
    for(i=0; i < raptor_sequence_size(s); i++) {
      rasqal_triple *t=(rasqal_triple*)raptor_sequence_get_at(s, i);
      t->flags |= RASQAL_TRIPLE_FLAGS_OPTIONAL;
    }
  }
  $$=$2;
  $$->flags |= RASQAL_PATTERN_FLAGS_OPTIONAL;
}
| '[' GraphPattern ']' /* from OptionalGraphPattern */
{
  int i;

#if RASQAL_DEBUG > 1  
  printf("PatternElementForms 5\n  graphpattern=");
  if($2)
    raptor_sequence_print($2, stdout);
  else
    fputs("NULL", stdout);
  fputs("\n\n", stdout);
#endif

  /* FIXME - deprecated */
  sparql_syntax_warning(((rasqal_query*)rq), "Use OPTIONAL {} instead of [] for optional triples in SPARQL (2005-02-17 WD)");

  if($2) {
    /* Make all graph patterns in GraphPattern, optional */
    for(i=0; i < raptor_sequence_size($2); i++) {
      rasqal_graph_pattern *gp=(rasqal_graph_pattern*)raptor_sequence_get_at($2, i);
      gp->flags |= RASQAL_PATTERN_FLAGS_OPTIONAL;
    }
    $$=rasqal_new_graph_pattern_from_sequence((rasqal_query*)rq, $2, RASQAL_PATTERN_FLAGS_OPTIONAL);
  } else
    $$=NULL;
}
;


/* SPARQL Grammar: [21] ConstraintPattern - inlined into PatternElement */

/* SPARQL Grammar: [22] TriplePattern */
TriplePattern : '(' VarOrURI VarOrURI VarOrLiteral ')'
{
  $$=rasqal_new_triple($2, $3, $4);
}
;


/* SPARQL Grammar: [23] ConstructPattern */
ConstructPattern : ConstructPattern TripleTemplate
{
  $$=$1;
  raptor_sequence_push($$, $2);
}
| TripleTemplate
{
  $$=raptor_new_sequence((raptor_sequence_free_handler*)rasqal_free_triple, (raptor_sequence_print_handler*)rasqal_triple_print);
  raptor_sequence_push($$, $1);
}
;


/* SPARQL Grammar: [24] TripleTemplate */
TripleTemplate : '(' VarOrURIOrBNode VarOrURI VarOrLiteralOrBNode ')'
{
  $$=rasqal_new_triple($2, $3, $4);
}
;


/* NEW Grammar Term */
VarOrURIList : VarOrURIList Var
{
  $$=$1;
  raptor_sequence_push($$, rasqal_new_variable_literal($2));
}
| VarOrURIList ',' Var
{
  $$=$1;
  raptor_sequence_push($$, rasqal_new_variable_literal($3));
}
| VarOrURIList URI
{
  $$=$1;
  raptor_sequence_push($$, $2);
}
| VarOrURIList ',' URI
{
  $$=$1;
  raptor_sequence_push($$, $3);
}
| Var 
{
  /* rasqal_variable* are freed from the raptor_query field variables */
  $$=raptor_new_sequence((raptor_sequence_free_handler*)rasqal_free_literal, (raptor_sequence_print_handler*)rasqal_literal_print);
  raptor_sequence_push($$, rasqal_new_variable_literal($1));
}
| URI
{
  $$=raptor_new_sequence((raptor_sequence_free_handler*)rasqal_free_literal, (raptor_sequence_print_handler*)rasqal_literal_print);
  raptor_sequence_push($$, $1);
}
;

/* NEW Grammar Term */
VarList : VarList Var
{
  $$=$1;
  raptor_sequence_push($$, $2);
}
| VarList ',' Var
{
  $$=$1;
  raptor_sequence_push($$, $3);
}
| Var 
{
  /* The variables are freed from the raptor_query field variables */
  $$=raptor_new_sequence(NULL, (raptor_sequence_print_handler*)rasqal_variable_print);
  raptor_sequence_push($$, $1);
}
;

/* NEW Grammar Term */
URIList : URIList URI
{
  raptor_uri* uri=rasqal_literal_as_uri($2);
  $$=$1;
  if(uri)
    raptor_sequence_push($$, raptor_uri_copy(uri));
  rasqal_free_literal($2);
}
| URIList ',' URI
{
  raptor_uri* uri=rasqal_literal_as_uri($3);
  $$=$1;
  if(uri)
    raptor_sequence_push($$, raptor_uri_copy(uri));
  rasqal_free_literal($3);
}
| /* empty */
{
  $$=raptor_new_sequence((raptor_sequence_free_handler*)raptor_free_uri, (raptor_sequence_print_handler*)raptor_sequence_print_uri);
}
;


/* SPARQL Grammar: [26] VarOrURI */
VarOrURI : Var
{
  $$=rasqal_new_variable_literal($1);
}
| URI
{
  $$=$1;
}
;

/* SPARQL Grammar: [27] VarOrLiteral */
VarOrLiteral : Var
{
  $$=rasqal_new_variable_literal($1);
}
| Literal
{
  $$=$1;
}
;

/* SPARQL Grammar: [28] VarAsNnode - junk */

/* SPARQL Grammar: [29] VarAsExpr - junk */


/* NEW Grammar Term */
VarOrURIOrBNode : VarOrURI
{
  $$=$1;
}
| BNode
{
  $$=$1;
}
;

/* NEW Grammar Term */
VarOrLiteralOrBNode : VarOrLiteral
{
  $$=$1;
}
| BNode
{
  $$=$1;
}
;


/* SPARQL Grammar: [30] BaseDecl */
BaseDeclOpt : BASE URI_LITERAL
{
  if(((rasqal_query*)rq)->base_uri)
    raptor_free_uri(((rasqal_query*)rq)->base_uri);
  ((rasqal_query*)rq)->base_uri=$2;
}
| /* empty */
{
  /* nothing to do */
}
;


/* SPARQL Grammar: [31] PrefixDecl */
PrefixDeclOpt : PrefixDeclOpt PREFIX IDENTIFIER URI_LITERAL
{
  rasqal_prefix *p=rasqal_new_prefix($3, $4);
  $$=((rasqal_query*)rq)->prefixes;
  raptor_sequence_push($$, p);
  rasqal_engine_declare_prefix(((rasqal_query*)rq), p);
}
| /* empty */
{
  /* nothing to do, rq->prefixes already initialised */
}
;


/* SPARQL Grammar: [32] Expression */
Expression : ConditionalAndExpression SC_OR Expression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_OR, $1, $3);
}
| ConditionalAndExpression
{
  $$=$1;
}
;

/* SPARQL Grammar: [33] ConditionalOrExpression - merged into Expression */

/* SPARQL Grammar: [34] ConditionalXorExpression - merged into ConditionalOrExpression */


/* SPARQL Grammar: [35] ConditionalAndExpression */
ConditionalAndExpression: ValueLogical SC_AND ConditionalAndExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_AND, $1, $3);
;
}
| ValueLogical
{
  $$=$1;
}
;

/* SPARQL Grammar: [36] ValueLogical */
ValueLogical : EqualityExpression STR_EQ EqualityExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_STR_EQ, $1, $3);
}
| EqualityExpression STR_NE EqualityExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_STR_NEQ, $1, $3);
}
| EqualityExpression STR_MATCH PATTERN_LITERAL
{
  $$=rasqal_new_string_op_expression(RASQAL_EXPR_STR_MATCH, $1, $3);
}
| EqualityExpression STR_NMATCH PATTERN_LITERAL
{
  $$=rasqal_new_string_op_expression(RASQAL_EXPR_STR_NMATCH, $1, $3);
}
| EqualityExpression
{
  $$=$1;
}
;

/* SPARQL Grammar: [37] StringEqualityExpression - merged into ValueLogical */

/* SPARQL Grammar: [38] NumericalLogical - merged into ValueLogical */

/* SPARQL Grammar: [39] EqualityExpression */
EqualityExpression : RelationalExpression EQ RelationalExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_EQ, $1, $3);
}
| RelationalExpression NEQ RelationalExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_NEQ, $1, $3);
}
| RelationalExpression
{
  $$=$1;
}
;

/* SPARQL Grammar: [40] RelationalExpression */
RelationalExpression : AdditiveExpression LT AdditiveExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_LT, $1, $3);
}
| AdditiveExpression GT AdditiveExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_GT, $1, $3);
}
| AdditiveExpression LE AdditiveExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_LE, $1, $3);
}
| AdditiveExpression GE AdditiveExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_GE, $1, $3);
}
| AdditiveExpression
{
  $$=$1;
}
;

/* SPARQL Grammar: [41] NumericExpression - merged into RelationalExpression */

/* SPARQL Grammar: [42] AdditiveExpression */
AdditiveExpression : MultiplicativeExpression '+' AdditiveExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_PLUS, $1, $3);
}
| MultiplicativeExpression '-' AdditiveExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_MINUS, $1, $3);
}
| MultiplicativeExpression
{
  $$=$1;
}
;

/* SPARQL Grammar: [43] MultiplicativeExpression */
MultiplicativeExpression : UnaryExpression '*' MultiplicativeExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_STAR, $1, $3);
}
| UnaryExpression '/' MultiplicativeExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_SLASH, $1, $3);
}
| UnaryExpression '%' MultiplicativeExpression
{
  $$=rasqal_new_2op_expression(RASQAL_EXPR_REM, $1, $3);
}
| UnaryExpression
{
  $$=$1;
}
;

/* SPARQL Grammar: [44] UnaryExpression */
UnaryExpression :  '-' BuiltinExpression
{
  $$=rasqal_new_1op_expression(RASQAL_EXPR_UMINUS, $2);
}
|'~' BuiltinExpression
{
  $$=rasqal_new_1op_expression(RASQAL_EXPR_BANG, $2);
}
|'!' BuiltinExpression
{
  $$=rasqal_new_1op_expression(RASQAL_EXPR_BANG, $2);
}
| '+' BuiltinExpression
{
  $$=$2;
}
| BuiltinExpression
{
  $$=$1;
}
;

/* SPARQL Grammar: [45] BuiltInExpression */
BuiltinExpression : BOUND '(' Var ')'
{
  rasqal_literal *l=rasqal_new_variable_literal($3);
  rasqal_expression *e=rasqal_new_literal_expression(l);
  $$=rasqal_new_1op_expression(RASQAL_EXPR_BOUND, e);
}
| STR '(' VarOrLiteral ')'
{
  rasqal_expression *e=rasqal_new_literal_expression($3);
  $$=rasqal_new_1op_expression(RASQAL_EXPR_STR, e);
}
| LANG '(' VarOrLiteral ')'
{
  rasqal_expression *e=rasqal_new_literal_expression($3);
  $$=rasqal_new_1op_expression(RASQAL_EXPR_LANG, e);
}
| DATATYPE '(' VarOrLiteral ')'
{
  rasqal_expression *e=rasqal_new_literal_expression($3);
  $$=rasqal_new_1op_expression(RASQAL_EXPR_DATATYPE, e);
}
| ISURI '(' VarOrLiteral ')'
{
  rasqal_expression *e=rasqal_new_literal_expression($3);
  $$=rasqal_new_1op_expression(RASQAL_EXPR_ISURI, e);
}
| ISBLANK '(' VarOrLiteral ')'
{
  rasqal_expression *e=rasqal_new_literal_expression($3);
  $$=rasqal_new_1op_expression(RASQAL_EXPR_ISBLANK, e);
}
| ISLITERAL '(' VarOrLiteral ')'
{
  rasqal_expression *e=rasqal_new_literal_expression($3);
  $$=rasqal_new_1op_expression(RASQAL_EXPR_ISLITERAL, e);
}
| URIBrace Expression ')' 
{
  raptor_uri* uri=rasqal_literal_as_uri($1);
  $$=rasqal_new_cast_expression(uri, $2);
}
| PrimaryExpression
{
  $$=$1;
}
;


/* SPARQL Grammar: [46] PrimaryExpression */
PrimaryExpression: Var
{
  rasqal_literal *l=rasqal_new_variable_literal($1);
  $$=rasqal_new_literal_expression(l);
}
| Literal
{
  $$=rasqal_new_literal_expression($1);
}
| '&' QNAME_LITERAL '(' ArgList ')'
{
  /* FIXME - do something with the function name, args */
  raptor_free_sequence($4);

  $$=NULL;
}
| '(' Expression ')'
{
  $$=$2;
}
;

/* SPARQL Grammar: [47] FunctionCall - merged into PrimaryExpression */

/* SPARQL Grammar: [48] ArgList */
ArgList : ArgList VarOrLiteral
{
  $$=$1;
  raptor_sequence_push($$, $2);
}
| /* empty */
{
  $$=raptor_new_sequence((raptor_sequence_free_handler*)rasqal_free_literal, (raptor_sequence_print_handler*)rasqal_literal_print);
}
;


/* SPARQL Grammar: [49] VarOrLiteralAsExpr - junk */

/* SPARQL Grammar: [50] Literal */
Literal : URI
{
  $$=$1;
}
| INTEGER_LITERAL
{
  $$=$1;
}
| FLOATING_POINT_LITERAL
{
  $$=$1;
}
| STRING_LITERAL
{
  $$=$1;
}
| BOOLEAN_LITERAL
{
  $$=$1;
}
| NULL_LITERAL
{
  $$=$1;
}
;

/* SPARQL Grammar: [51] NumericLiteral - merged into Literal */

/* SPARQL Grammar: [52] TextLiteral - merged into Literal */

/* SPARQL Grammar: [53] PatternLiteral - made into terminal PATTERN_LITERAL */

/* SPARQL Grammar: [54] URI */
URI : URI_LITERAL
{
  $$=rasqal_new_uri_literal($1);
}
| QNAME_LITERAL
{
  $$=rasqal_new_simple_literal(RASQAL_LITERAL_QNAME, $1);
  if(rasqal_literal_expand_qname((rasqal_query*)rq, $$)) {
    sparql_query_error_full((rasqal_query*)rq,
                            "QName %s cannot be expanded", $1);
    rasqal_free_literal($$);
    $$=NULL;
  }
}

/* NEW Grammar Term made from SPARQL Grammer: [54] URI + '(' expanded */
URIBrace : URI_LITERAL_BRACE
{
  $$=rasqal_new_uri_literal($1);
}
| QNAME_LITERAL_BRACE
{
  $$=rasqal_new_simple_literal(RASQAL_LITERAL_QNAME, $1);
  if(rasqal_literal_expand_qname((rasqal_query*)rq, $$)) {
    sparql_query_error_full((rasqal_query*)rq,
                            "QName %s cannot be expanded", $1);
    rasqal_free_literal($$);
    $$=NULL;
  }
}

/* SPARQL Grammar: [55] QName - made into terminal QNAME_LITERAL */

/* SPARQL Grammar: [56] BNode */
BNode : BLANK_LITERAL
{
  $$=rasqal_new_simple_literal(RASQAL_LITERAL_BLANK, $1);
}
;

/* SPARQL Grammar: [57] QuotedURI - made into terminal URI_LITERAL */

/* SPARQL Grammar: [58] Integer - made into terminal INTEGER_LITERAL */

/* SPARQL Grammar: [59] FloatingPoint - made into terminal FLOATING_POINT_LITERAL */


/* SPARQL Grammar: [60] onwards are all lexer items with similar
 * names or are inlined except for:
 */

/* SPARQL Grammer: [65] <VAR> */
Var : '?' IDENTIFIER
{
  $$=rasqal_new_variable((rasqal_query*)rq, $2, NULL);
}
| '$' IDENTIFIER
{
  $$=rasqal_new_variable((rasqal_query*)rq, $2, NULL);
}
;




%%


/* Support functions */


/* This is declared in sparql_lexer.h but never used, so we always get
 * a warning unless this dummy code is here.  Used once below in an error case.
 */
static int yy_init_globals (yyscan_t yyscanner ) { return 0; };



/**
 * rasqal_sparql_query_engine_init - Initialise the SPARQL query engine
 *
 * Return value: non 0 on failure
 **/
static int
rasqal_sparql_query_engine_init(rasqal_query* rdf_query, const char *name) {
  /* rasqal_sparql_query_engine* sparql=(rasqal_sparql_query_engine*)rdf_query->context; */

  /* Initialise rdf, rdfs, owl and xsd prefixes and namespaces */
  raptor_namespaces_start_namespace_full(rdf_query->namespaces, 
                                         (const unsigned char*)"rdf",
                                         (const unsigned char*)RAPTOR_RDF_MS_URI,0);
  raptor_namespaces_start_namespace_full(rdf_query->namespaces, 
                                         (const unsigned char*)"rdfs", 
                                         (const unsigned char*)RAPTOR_RDF_SCHEMA_URI,0);
  raptor_namespaces_start_namespace_full(rdf_query->namespaces,
                                         (const unsigned char*)"xsd",
                                         (const unsigned char*)RAPTOR_XMLSCHEMA_DATATYPES_URI, 0);
  raptor_namespaces_start_namespace_full(rdf_query->namespaces,
                                         (const unsigned char*)"owl",
                                         (const unsigned char*)RAPTOR_OWL_URI, 0);

  return 0;
}


/**
 * rasqal_sparql_query_engine_terminate - Free the SPARQL query engine
 *
 * Return value: non 0 on failure
 **/
static void
rasqal_sparql_query_engine_terminate(rasqal_query* rdf_query) {
  rasqal_sparql_query_engine* sparql=(rasqal_sparql_query_engine*)rdf_query->context;

  if(sparql->scanner_set) {
    sparql_lexer_lex_destroy(sparql->scanner);
    sparql->scanner_set=0;
  }

}


static int
rasqal_sparql_query_engine_prepare(rasqal_query* rdf_query) {
  /* rasqal_sparql_query_engine* sparql=(rasqal_sparql_query_engine*)rdf_query->context; */
  int rc;
  
  if(!rdf_query->query_string)
    return 1;
  
  rc=sparql_parse(rdf_query, rdf_query->query_string);
  if(rc)
    return rc;

  /* FIXME - should check remaining query parts  */
  if(rasqal_engine_sequence_has_qname(rdf_query->triples) ||
     rasqal_engine_sequence_has_qname(rdf_query->constructs) ||
     rasqal_engine_query_constraints_has_qname(rdf_query)) {
    sparql_query_error(rdf_query, "SPARQL query has unexpanded QNames");
    return 1;
  }

  return rasqal_engine_prepare(rdf_query);
}


static int
rasqal_sparql_query_engine_execute(rasqal_query* rdf_query,
                                   rasqal_query_results *results) 
{
  /* rasqal_sparql_query_engine* sparql=(rasqal_sparql_query_engine*)rdf_query->context; */
  
  /* nothing needed here */
  return 0;
}


static int
sparql_parse(rasqal_query* rq, const unsigned char *string) {
  rasqal_sparql_query_engine* rqe=(rasqal_sparql_query_engine*)rq->context;
  raptor_locator *locator=&rq->locator;
  char *buf=NULL;
  size_t len;
  void *buffer;

  if(!string || !*string)
    return yy_init_globals(NULL); /* 0 but a way to use yy_init_globals */

  locator->line=1;
  locator->column= -1; /* No column info */
  locator->byte= -1; /* No bytes info */

#if RASQAL_DEBUG > 2
  sparql_parser_debug=1;
#endif

  rqe->lineno=1;

  sparql_lexer_lex_init(&rqe->scanner);
  rqe->scanner_set=1;

  sparql_lexer_set_extra(((rasqal_query*)rq), rqe->scanner);

  /* This
   *   buffer= sparql_lexer__scan_string((const char*)string, rqe->scanner);
   * is replaced by the code below.  
   * 
   * The extra space appended to the buffer is the least-pain
   * workaround to the lexer crashing by reading EOF twice in
   * sparql_copy_regex_token; at least as far as I can diagnose.  The
   * fix here costs little to add as the above function does
   * something very similar to this anyway.
   */
  len= strlen((const char*)string);
  buf= (char *)RASQAL_MALLOC(cstring, len+3);
  strncpy(buf, (const char*)string, len);
  buf[len]= ' ';
  buf[len+1]= buf[len+2]='\0'; /* YY_END_OF_BUFFER_CHAR; */
  buffer= sparql_lexer__scan_buffer(buf, len+3, rqe->scanner);

  sparql_parser_parse(rq);

  if(buf)
    RASQAL_FREE(cstring, buf);

  sparql_lexer_lex_destroy(rqe->scanner);
  rqe->scanner_set=0;

  /* Parsing failed */
  if(rq->failed)
    return 1;
  
  return 0;
}


static void
sparql_query_error(rasqal_query *rq, const char *msg) {
  rasqal_sparql_query_engine* rqe=(rasqal_sparql_query_engine*)rq->context;

  rq->locator.line=rqe->lineno;
#ifdef RASQAL_SPARQL_USE_ERROR_COLUMNS
  /*  rq->locator.column=sparql_lexer_get_column(yyscanner);*/
#endif

  rasqal_query_error(rq, "%s", msg);
}


static void
sparql_query_error_full(rasqal_query *rq, const char *message, ...) {
  va_list arguments;
  rasqal_sparql_query_engine* rqe=(rasqal_sparql_query_engine*)rq->context;

  rq->locator.line=rqe->lineno;
#ifdef RASQAL_SPARQL_USE_ERROR_COLUMNS
  /*  rq->locator.column=sparql_lexer_get_column(yyscanner);*/
#endif

  va_start(arguments, message);

  rasqal_query_error_varargs(rq, message, arguments);

  va_end(arguments);
}


int
sparql_syntax_error(rasqal_query *rq, const char *message, ...)
{
  rasqal_sparql_query_engine *rqe=(rasqal_sparql_query_engine*)rq->context;
  va_list arguments;

  rq->locator.line=rqe->lineno;
#ifdef RASQAL_SPARQL_USE_ERROR_COLUMNS
  /*  rp->locator.column=sparql_lexer_get_column(yyscanner);*/
#endif

  va_start(arguments, message);
  rasqal_query_error_varargs(rq, message, arguments);
  va_end(arguments);

   return (0);
}


int
sparql_syntax_warning(rasqal_query *rq, const char *message, ...)
{
  rasqal_sparql_query_engine *rqe=(rasqal_sparql_query_engine*)rq->context;
  va_list arguments;

  rq->locator.line=rqe->lineno;
#ifdef RASQAL_SPARQL_USE_ERROR_COLUMNS
  /*  rq->locator.column=sparql_lexer_get_column(yyscanner);*/
#endif

  va_start(arguments, message);
  rasqal_query_warning_varargs(rq, message, arguments);
  va_end(arguments);

   return (0);
}


static void
rasqal_sparql_query_engine_register_factory(rasqal_query_engine_factory *factory)
{
  factory->context_length = sizeof(rasqal_sparql_query_engine);

  factory->init      = rasqal_sparql_query_engine_init;
  factory->terminate = rasqal_sparql_query_engine_terminate;
  factory->prepare   = rasqal_sparql_query_engine_prepare;
  factory->execute   = rasqal_sparql_query_engine_execute;
}


void
rasqal_init_query_engine_sparql(void) {
  rasqal_query_engine_register_factory("sparql", 
                                       "SPARQL W3C DAWG RDF Query Language",
                                       NULL,
                                       (const unsigned char*)"http://www.w3.org/TR/rdf-sparql-query/",
                                       &rasqal_sparql_query_engine_register_factory);
}



#ifdef STANDALONE
#include <stdio.h>
#include <locale.h>

#define SPARQL_FILE_BUF_SIZE 2048

int
main(int argc, char *argv[]) 
{
  const char *program=rasqal_basename(argv[0]);
  char query_string[SPARQL_FILE_BUF_SIZE];
  rasqal_query *query;
  FILE *fh;
  int rc;
  char *filename=NULL;
  raptor_uri* base_uri=NULL;
  unsigned char *uri_string;

#if RASQAL_DEBUG > 2
  sparql_parser_debug=1;
#endif

  if(argc > 1) {
    filename=argv[1];
    fh = fopen(argv[1], "r");
    if(!fh) {
      fprintf(stderr, "%s: Cannot open file %s - %s\n", program, filename,
              strerror(errno));
      exit(1);
    }
  } else {
    filename="<stdin>";
    fh = stdin;
  }

  memset(query_string, 0, SPARQL_FILE_BUF_SIZE);
  fread(query_string, SPARQL_FILE_BUF_SIZE, 1, fh);
  
  if(argc>1)
    fclose(fh);

  rasqal_init();

  query=rasqal_new_query("sparql", NULL);

  uri_string=raptor_uri_filename_to_uri_string(filename);
  base_uri=raptor_new_uri(uri_string);

  rc=rasqal_query_prepare(query, (const unsigned char*)query_string, base_uri);

  rasqal_query_print(query, stdout);

  rasqal_free_query(query);

  raptor_free_uri(base_uri);

  raptor_free_memory(uri_string);

  rasqal_finish();

  return rc;
}
#endif
