/*  readline.c: interacting with the GNU readline library */

/*  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License , or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  hlub@knoware.nl
*/


#include "rlwrap.h"

/* global vars */
int remember_for_completion =  FALSE; /* whether we should put al words from in/output on the list */ 
struct rl_state  saved_rl_state ={"","",0,0}; /* saved state of readline */
static char return_key;               /* Key pressed to enter line */



char previous_line[BUFFSIZE]="";      /* previous input line */

/* forward declarations */
void line_handler(char * );
void my_add_history(char *);
int my_accept_line(int,int);


void init_readline(char *prompt) {
  rl_add_defun ("rlwrap_accept_line", my_accept_line, -1);
  rl_add_defun ("insert_close", rl_insert_close, -1);
   
  rl_bind_key('\n', my_accept_line); 
  rl_bind_key('\r', my_accept_line);

  rl_bind_key(')', rl_insert_close);
  rl_bind_key('}', rl_insert_close);
  rl_bind_key(']', rl_insert_close);

  rl_initialize(); /* This has to happen *before* we set rl_redisplay_function, otherwise
		      readline will not bother to call tgetent(), will be agnostic about terminal
		      capabilities and hence not be able to honour e.g. a set horizontal-scroll-mode off
		      in .inputrc */
  using_history();
  rl_redisplay_function = my_redisplay;
  rl_already_prompted = 1;  /* we'll always write our own prompt */
  rl_completion_entry_function = (rl_compentry_func_t  *) & my_completion_function;
  rl_catch_signals = FALSE;
  rl_catch_sigwinch = FALSE;
}


void save_rl_state (struct rl_state *state) {
  
  mystrlcpy (state ->text, rl_line_buffer, sizeof(state -> text));  /* save text*/
  mystrlcpy(state -> prompt, rl_prompt, sizeof(state -> prompt));
  state -> point = rl_point;                           /* and point    */
  rl_line_buffer[0] = '\0';
  if (state->  already_saved) 
    return;
  state -> already_saved = 1;
  rl_delete_text(0, rl_end);                           /* clear line  (after prompt) */
  rl_point = 0;
  my_redisplay();                                      /* and redisplay */
  rl_callback_handler_remove ();                       /* restore old term settings */
  rl_deprep_terminal();
 
}

 
void restore_rl_state (struct rl_state *state) {
  char *newprompt;
  newprompt = state -> prompt; 

  DPRINTF1(DEBUG_READLINE, "restore prompt: '%s'", newprompt);
  rl_expand_prompt (newprompt); /* We have to call this because rl_already_prompted is set */
  
  mirror_slaves_echo_mode(); /* don't show passwords etc */	    
  rl_callback_handler_install (newprompt , &line_handler);
  rl_insert_text(state -> text);
  rl_point = state -> point;
  state -> already_saved = 0;
  cr();
  rl_forced_update_display ();
  rl_prep_terminal(1);
}

void line_handler(char * line) {  
  if (line == NULL) {                               /* EOF, forward it  */
    DPRINTF1(DEBUG_READLINE,"EOF detected, writing character %d", term_eof);
    write(master_pty_fd, &term_eof, 1); 
  } else { 
    if (*line && strlen(line) > 1 && redisplay)  {  /* don't save empty lines,
						       single keystrokes or passwords*/
         my_add_history (line);
    }  
    write(master_pty_fd, line, strlen(line));
    DPRINTF2(DEBUG_READLINE, "writing %d bytes %s", strlen(line), mangle_string_for_debug_log(line,40)); 
    write(master_pty_fd, (return_key ? &return_key : "\n"), 1);
    free(line);                                      /* we're done with it  */
    
    return_key = 0;
    within_line_edit = FALSE;
    
    rl_callback_handler_remove ();
    set_echo(FALSE);
    saved_rl_state.text[0]= '\0';
    saved_rl_state.prompt[0] = '\0';
    saved_rl_state.point = 0;
    saved_rl_state.already_saved = TRUE;
    
  }
}

/* this function (drop-in replacement for readline's own accept-line())
   will be bound to RETURN key: */
int my_accept_line(int count, int key) {
  rl_point = 0;                             /* leave cursor on predictable place */
  my_redisplay();
  rl_done = 1;
  return_key = (char) key;  
  return 0;
}


/* Homegrown redisplay function - erases current line and prints the
   new one.  Used for passwords (where we want to show **** instead of
   user input) and whenever HOMEGROWN_REDISPLAY is defined (for
   systems where rl_redisplay() misbehaves, like sometimes on
   Solaris). Otherwise we use the much faster and smoother
   rl_redisplay() This function cannot display multiple lines: it will
   only scroll horizontally (even if horizontal-scroll-mode is off in .inputrc)
*/



static void my_homegrown_redisplay(int hide_passwords) {
  static int line_start = 0; /* at which position of prompt_plus_line does the printed line start? */
  static int line_extends_right = 0;
  static int line_extends_left = 0;
  static char *previous_line = NULL;
  static int previous_curpos = 0;
  
  int width = window_size.ws_col;
  int skip = max(1,min(width/5, 10)); /* jumpscroll this many positions when cursor reaches edge of terminal */
  int promptlen = strlen(rl_prompt);
  char *prompt_plus_line = add2strings(rl_prompt,rl_line_buffer);
  char *new_line;
  int total_length = strlen (prompt_plus_line);
  int curpos =   promptlen + rl_point;
  int i, printed_length, new_curpos, keep_old_line;
  
  if (hide_passwords)
    for (i = promptlen; i <  total_length; i++)
      prompt_plus_line[i] = '*';  /* make user input unreadable  */
  
 
  if (rl_point == 0)  /* re)set  at program start and after accept_line (where rl_point is zeroed) */ 
    line_start = 0;
  if (curpos  - line_start > width - line_extends_right)      	/* cursor falls off right edge ?   */ 
    line_start = (curpos  - width + line_extends_right) + skip; /* jumpscroll left                 */

  if (curpos < line_start + line_extends_left)                 	/* cursor falls off left edge ?    */ 
    if (curpos == total_length)              			/* .. but still at end of line?    */
      line_start = max(0, total_length - width); 		/* .. try to display entire line   */
    else                                                        /* in not at end of line ..        */
      line_start = curpos -  line_extends_left - skip;		/* ... jumpscroll right ..         */
  if (line_start < 0)
    line_start = 0;                                             /* ... but not past start of line! */
  
  printed_length =  min(width, total_length - line_start); 	/* never print more than width     */
  
  
  
  /* some invariants :     0 <= line_start <= curpos <= line_start + printed_length <= total_length */
  /* these are interesting:   ^                                                      ^              */
                     
  assert(0 <= line_start);
  assert(line_start <= curpos);
  assert(curpos <= line_start + printed_length);               /* yes <=: cursor may be past eol   */
  assert(line_start + printed_length <= total_length);
  
 
  new_line = prompt_plus_line + line_start;
  new_line[printed_length] = '\0';
  new_curpos = curpos - line_start;
  
  /* indicate whether line extends past right or left edge  (i.e. whether the "interesting
     inequalities marked ^ above are really unequal) */
  
  line_extends_left = (line_start > 0 ? 1 : 0);
  line_extends_right = (total_length - line_start > width ? 1 : 0);
  if (line_extends_left)                      
    new_line[0] = '<';
  if (line_extends_right)               
     new_line[width - 1] = '>'; 

 
  
  keep_old_line = FALSE;
  if (term_cursor_hpos) {
    if (previous_line && strcmp(new_line, previous_line) == 0) {
      keep_old_line = TRUE;
    } else {
      if(previous_line)
	free(previous_line);
      previous_line = mysavestring(new_line);
    }
  }
    

  if (!keep_old_line) { 
    clear_line();
    cr();
    write(STDOUT_FILENO, new_line, printed_length);
  } 
  free(prompt_plus_line);
  assert (term_cursor_hpos || !keep_old_line); /* if we cannot position cursor, we must have reprinted ... */
  
  if (term_cursor_hpos) 
    cursor_hpos(new_curpos);
  else                                         /* ... so we know we're 1 past last position on line */
    backspace(printed_length - new_curpos); 
}
  


void my_redisplay() {
#ifndef HOMEGROWN_REDISPLAY
  if (redisplay) 
    rl_redisplay();
  else
#endif	
    my_homegrown_redisplay(!redisplay);
}


void my_add_history (char * line) { /* add line to history list, avoiding duplicates */
  if (strncmp (line, previous_line, sizeof(previous_line))) {
    mystrlcpy (previous_line, line, sizeof(previous_line));
    add_history (line);               
  }
}
  















