#include "module_tree.h"

#include <algorithm>
using namespace std;

#include "module.h"
#include "base_node.h"
#include "module_registry.h"
#include "root-portal.h"
#include "utils.h"

ModuleTree* ModuleTree::c_instance = 0;

ModuleTree::~ModuleTree()
{
        delete_recursive(Path());
}

void ModuleTree::delete_recursive(const Path& path)
{
        try {
                int num_children = child_count(path);
                Path cpath = path;
                cpath.add_last(0);
                for (int i = 0; i < num_children; i++) {
                        delete_recursive(cpath);
                        issue_command(path, "delete_child", "0");
                }
        } catch (ModuleError ex) {
                cerr << "Module exception: " << ex.description << endl;
        }
}

void ModuleTree::init()
{
        base_module.postStartup();
}

void ModuleTree::shutdown()
{
        base_module.preShutdown();
}

void ModuleTree::service()
{
        base_module.service();
}

string ModuleTree::get_type(const Path& target_module) throw (ModuleError)
{
        return issue_command(target_module, "get_type", "");
}

DataSet ModuleTree::get_value(const Path& target_module, const string& varname) throw (ModuleError)
{
        string result = issue_command(target_module, "get_value", varname);
        return DataSet(result);
}

void ModuleTree::set_value(const Path& target_module, const string& varname,
                           const DataSet& value) throw (ModuleError)
{
        issue_command(target_module, "set_value", varname + "=" + value.asString());
}

void ModuleTree::set_value(const Path& target_module, const string& varname,
                           const string& value) throw (ModuleError)
{
        issue_command(target_module, "set_value", varname + "=" + value);
}

void ModuleTree::create_child(const Path& target_module, const string& child_type) throw (ModuleError)
{
        issue_command(target_module, "create_child", child_type);
}

void ModuleTree::delete_child(const Path& target_module, int child_num) throw (ModuleError)
{
        issue_command(target_module, "delete_child", int_to_string(child_num));
}

int ModuleTree::child_count(const Path& target_module) throw (ModuleError)
{
        string result = issue_command(target_module, "child_count", "");
        assert(result != "");
        return atoi(result.c_str());
}

const DataSetMap *ModuleTree::module_info(const Path& target_module) throw (ModuleError)
{
        string mtype = get_type(target_module);
        if (mtype == "")
                return 0;
        return &(ModuleRegistry::instance()->module_info(mtype).module_info());
}

void ModuleTree::post_startup(const Path& target_module) throw (ModuleError)
{
        issue_command(target_module, "post_startup", "");
}

void ModuleTree::post_startup_recursive(const Path& path)
{
        recursive_command(path, "post_startup", "");
}

void ModuleTree::pre_shutdown(const Path& target_module) throw (ModuleError)
{
        issue_command(target_module, "pre_shutdown", "");
}

void ModuleTree::pre_shutdown_recursive(const Path& path)
{
        recursive_command(path, "pre_shutdown", "");
}

void ModuleTree::overlay_defaults(const Path& target_module) throw (ModuleError)
{
        issue_command(target_module, "overlay_defaults", "");
}

string ModuleTree::issue_command(const Path& target_module, const string& command,
                                 const string& parameter) throw (ModuleError)
{
//          rpdbgmsg("issue_command(" << target_module.to_string() << "," << command << "," << parameter << ")");
        if (output_command_file.is_open())
                output_command_file << target_module.to_string() << command
                                    << "/" << parameter << endl;
        string result = base_module.issue_command(target_module, command, parameter);
//          rpdbgmsg("result = " << result);
        if (!strncmp(result.c_str(), "!!ERROR!!", 9))
                throw (ModuleError(string(result, 9)));
        return result;
}

void ModuleTree::recursive_command(const Path& path, const string& command,
                                   const string& parameter)
{
        try {
                int num_children = ModuleTree::instance()->child_count(path);
                for (int i = 0; i < num_children; i++) {
                        Path cpath = path;
                        cpath.add_last(i);
                        recursive_command(cpath, command, parameter);
                }
                issue_command(path, command, parameter);
        } catch (ModuleError ex) {
                cerr << "Module exception: " << ex.description << endl;
        }
        
}

/* command has the following format:
 * "/childnum/childnum/.../childnum/command name/parameters"
 * get_value : varname
 * set_value : varname=value
 * get_type
 * create_child : type
 * delete_child : child_number
 * child_count
 * post_startup
 * overlay_defaults
 * pre_shutdown
 */
string ModuleTree::issue_string_command(const string& cmdstring) throw (ModuleError)
{
        string command;
        Path path;
        string parameter;
        string tmp;
        for (int i = 0; i < int(cmdstring.length()); i++) {
                if (cmdstring[i] == '/') {
                        if (tmp.length() > 0) {
                                if (command.length() > 0) {
                                        if (parameter.length() > 0)
                                                parameter += '/';
                                        parameter += tmp;
                                } else {
                                        if (isdigit(tmp[0])) {
                                                path.add_last(atoi(tmp.c_str()));
                                        } else {
                                                if (command.length() > 0)
                                                        command += '/';
                                                command += tmp;
                                        }
                                }
                                tmp = "";
                        }
                } else {
                        tmp += cmdstring[i];
                }
        }
        if (parameter.length() > 0)
                parameter += '/';
        parameter += tmp;
        return issue_command(path, command, parameter);
}

void ModuleTree::return_message(const DataSet& msg, const Path& path)
{
        for (int i = 0; i < int(retmsg_listeners.size()); i++) {
                retmsg_listeners[i]->return_message(msg, path);
        }
}

void ModuleTree::register_listener(ModuleTreeListener *l)
{
        retmsg_listeners.push_back(l);
}

void ModuleTree::unregister_listener(ModuleTreeListener *l)
{
        retmsg_listeners.erase(find(retmsg_listeners.begin(),
                                    retmsg_listeners.end(),
                                    l));
}

