// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __framebase_h__
#define __framebase_h__

#include "widget.h"

#include "contourscale.h"
#include "marker.h"
#include "hist.h"
#include "list.h"
#include "coord.h"
#include "grid.h"
#include "frscale.h"

#define SCALESIZE 16384
#define HISTEQUSIZE 16384
#define MULTWCS 27

#define DEFAULTLOW 1
#define DEFAULTHIGH 100

extern int DebugAST;
extern int DebugMosaic;
extern int DebugPerf;
extern int DebugWCS;
extern int DebugBin;
extern int DebugGZ;
extern int DebugRGB;

class FitsImage;
class Contour;
class FVContour;
class Grid;
class Attribute;
class Marker;
class BaseBox;
class BaseCircle;
class BaseEllipse;
class BaseMarker;
class Circle;
class Annulus;
class Panda;
class Ellipse;
class EllipseAnnulus;
class Box;
class BoxAnnulus;
class Polygon;
class Point;
class Line;
class Text;
class Ruler;
class Compass;
class Projection;
class CirclePoint;
class BoxPoint;
class DiamondPoint;
class CrossPoint;
class ExPoint;
class ArrowPoint;
class BoxCirclePoint;

// Frame

class FrameBase : public Widget {
  friend class FitsImage;
  friend class Contour;
  friend class FVContour;
  friend class Grid;
  friend class Attribute;
  friend class Marker;
  friend class BaseBox;
  friend class BaseCircle;
  friend class BaseEllipse;
  friend class BaseMarker;
  friend class Circle;
  friend class Annulus;
  friend class Panda;
  friend class Ellipse;
  friend class EllipseAnnulus;
  friend class Box;
  friend class BoxAnnulus;
  friend class Polygon;
  friend class Point;
  friend class Line;
  friend class Text;
  friend class Ruler;
  friend class Compass;
  friend class Projection;
  friend class CirclePoint;
  friend class BoxPoint;
  friend class DiamondPoint;
  friend class CrossPoint;
  friend class ExPoint;
  friend class ArrowPoint;
  friend class BoxCirclePoint;

public:
  enum Precision {DEFAULT, FIXED, SCIENTIFIC, INTEGER};
  enum MemType {ALLOC, ALLOCGZ, CHANNEL, MMAP, SMMAP, MMAPINCR, 
		SHARE, SSHARE, SOCKET, SOCKETGZ, VAR, HIST};
  enum MarkerFormat {DS9, CIAO, SAOTNG, SAOIMAGE, PROS, RAWXY};
  enum FileNameType {ROOTBASE, FULLBASE, ROOT, FULL};
  enum PSColorSpace {BW, GRAY, RGB, CMYK};
  enum ShmType {SHMID,KEY};
  enum ColormapMode {CENTER, CLICK};
  enum UndoMarkerType {NONE, MOVE, EDIT, DELETE};
  enum DisplayMode {SINGLE, MOSAIC};
  enum LoadMethod {LOADALL, INCR};
  enum MosaicType {NOMOSAIC, IRAF, WCSMOSAIC, WFPC2};
  enum CompressType {NOCOMPRESS, GZ};

protected:
  enum UpdateType {MATRIX, BASE, PIXMAP, NOUPDATE};

  FitsImage** keyFits;       // primary fits channel
  FitsImage** channelFits;   // current fits channel
  FitsImage* currentFits;    // current fits image
  int* currentCount;         // current count of fits images
  FrScale* currentScale;     // current scale parameters
  DisplayMode* currentMode;  // display mode
  MosaicType* currentMosaic; // mosaic mode
  CoordSystem* currentMosaicSystem; // mosaic wcs system

  Vector* currentWCScdelt;   // needed for wcs mosaics
  double* currentWCSrot;
  Orientation* currentWCSorientation;
  BBox* currentWCSbb;        // WCS bbox in IMAGE coords
  Matrix* currentWCSmatrix;

  Matrix dataToImage;
  Matrix imageToData;

  int slice;                 // current slice
  int mosaicFast;            // flag for fast mosaic render

  XImage* baseXImage;        // ximage of image
  Pixmap basePixmap;         // pixmap of image
  UpdateType needsUpdate;    // updateFlags

  Pixmap panPM;              // tmp pixmap for pan motion
  Vector panCursor;          // tmp cursor for interactive panning
  GC panGCXOR;               // GC for interactive panning
  Pixmap pannerPixmap;       // pixmap for panner
  XImage* pannerXImage;      // ximage for panner
  int pannerWidth;           // panner width
  int pannerHeight;          // panner height
  char pannerName[32];       // panner widget name
  CoordSystem pannerSystem;  // panner wcs coord system
  SkyFrame pannerSky;        // panner wcs sky frame
  int usePanner;             // panner status

  Pixmap magnifierPixmap;    // pixmap for magnifier
  XImage* magnifierXImage;   // ximage for magnifier
  int magnifierWidth;        // magnifier width
  int magnifierHeight;       // magnifier height
  double magnifierZoom;      // magnifier zoom factor
  Vector magnifierCursor;     // we need to save the last cursor used
  char magnifierName[32];    // magnifer widget name
  int useMagnifier;          // magnifer status
  int useMagnifierGraphics;  // magnifer marker status
  int useMagnifierCursor;    // magnifier cursor status
  GC magnifierGC;            // GC for magnifier
				  
  Vector cursor;             // current cursor position in USER coords
  double zoom;               // current image zoom value
  Orientation orientation;   // current image orientation
  Matrix orientationMatrix;  // current image orientation matrix

  double rotation;           // current image rotation angle
  double rotateRotation;     // tmp cursor for interactive rotation
  XImage* rotateSrcXM;       // rotate src ximage
  XImage* rotateDestXM;      // rotate dest ximage
  Pixmap rotatePM;           // rotate pixmap
  GC rotateGCXOR;            // GC for interactive rotation

  int preservePan;           // flag to indicate preserve between loads

  int wcsAlignment;          // flag to indicate wcs alignment
  CoordSystem wcsSystem;     // wcs system
  SkyFrame wcsSky;           // wcs sky frame
  double wcsRotation;        // wcs image rotation angle
  Orientation wcsOrientation; // wcs image orientation
  Matrix wcsOrientationMatrix; // wcs image orientation matrix

  Contour** currentcontour;  // main contour
  List<Contour> auxcontours; // aux contours
  GC contourGC;

  Grid* grid;
  GC gridGC;

  int cmapArea;              // current colormap update size
  ColormapMode cmapMode;     // current colormap update mode
  Vector cmapOffset;         // current colormap update offset

  Vector crosshair;          // crosshair location in REF coords
  int useCrosshair;          // flag to use crosshair
  GC crosshairGC;            // crosshair gc
  GC crosshairGCXOR;         // crosshair gcxor

  GC highliteGC;             // gc for frame highlite
  int useHighlite;           // flag to draw highlite box

  List<Marker> fgMarkers;    // foreground marker stack
  List<Marker> undofgMarkers;// remember marker to undo
  List<Marker> pastefgMarkers;// cut/copy/paste
  List<Marker> bgMarkers;    // background marker stack
  List<Marker> undobgMarkers;// remember marker to undo
  List<Marker> pastebgMarkers;// cut/copy/paste
  List<Marker>* markers;     // pointer to current marker stack
  List<Marker>* undoMarkers; // pointer to current marker stack
  List<Marker>* pasteMarkers;// pointer to current marker stack
  UndoMarkerType undoMarkerType; // type
  UpdateType markerUpdate;   // update level for marker

  int* showMarkers;          // display markers
  int showfgMarkers;         // display markers
  int showbgMarkers;         // display markers
  int preserveMarkers;       // flag to indicate preserve between loads
  
  GC markerGC;               // marker gc
  GC selectGCXOR;            // select gc
  XRectangle rectWidget[1];  // clip rect
  XRectangle rectWindow[1];  // clip rect

  Vector markerBegin;        // marker move begin in image coords
  Marker* editMarker;        // pointer to marker currently being edited
  Marker* rotateMarker;      // pointer to marker currently being rotated
  Vector regionBegin;        // select region begin in canvas coords
  Vector regionEnd;          // select region end in canvas coords
  CoordSystem markerDefaultSystem; // default marker sys (RAWXY)
  SkyFrame markerDefaultSky; // default marker sky (RAWXY)

  int psInterpolate;         // postscript level 2 interpolation flag
  int psResolution;          // postscript resolution
  int psLevel;               // postscript level
  PSColorSpace psColorSpace; // postscript color space

  Vector iisLastCursor;      // iis cursor state info
  Vector iisSaveCursor;
  Window iisSaveRoot;
  int iisInteractive;
				  
  FitsHist::Function binFunction_;
  Vector binFactor_;
  int binBufferSize_;
  int binDepth_;
  int binSmooth_;
  FitsHist::Smooth binSmoothFunc_;
  int binSmoothRadius_;

  char* bgColorName;
  XColor* bgColor;

  char* nanColorName;
  XColor* nanColor;

  Matrix canvasToRef;
  Matrix canvasToUser;
  Matrix canvasToWidget;
  Matrix canvasToWindow;
  Matrix refToCanvas;
  Matrix refToPanner;
  Matrix refToUser;
  Matrix refToWidget;
  Matrix refToWindow;
  Matrix pannerToRef;
  Matrix pannerToUser;
  Matrix pannerToWidget;
  Matrix userToCanvas;
  Matrix userToRef;
  Matrix userToMagnifier;
  Matrix userToMagnifierCursor;
  Matrix userToPanner;
  Matrix userToWidget;
  Matrix widgetToCanvas;
  Matrix widgetToPanner;
  Matrix widgetToRef;
  Matrix widgetToUser;
  Matrix widgetToWindow;
  Matrix windowToCanvas;
  Matrix windowToRef;
  Matrix windowToWidget;
  Matrix ximageToMagnifier;

protected:
  Vector imageSize() {return imageBBox().size();}
  Vector imageCenter() {return imageBBox().center();}

  // Virtual routines with base component
  // framebase.C

  virtual void reset();
  virtual void unloadFits();
  virtual void updateMatrices();

  // Virtual routines

  virtual void setCurrentFits(FitsImage*) =0;
  virtual void setBinCursor(const Vector&) =0;
  virtual void updateBin(const Matrix&) =0;
  virtual BBox imageBBox() =0;
  virtual int doRender() =0;
  virtual void unloadAllFits() =0;
  virtual void updateMagnifierMatrices(Matrix&) =0;

  virtual void updateColorScale() =0;
  virtual int validColorScale() =0;

  virtual int isIIS() =0;

  virtual void psLevel1(PSColorSpace, int, int, float) =0;
  virtual void psLevel2(PSColorSpace, int, int, float) =0;

  virtual void buildXImage(XImage*, int, int, int, int,
			   double* (FitsImage::*)()) =0;

  virtual void rotateMotion() =0;
  virtual void align() =0;

  // framebase.C
  void resetClip(int);
  void resetValues();

  void update(UpdateType);         // schedule update
  void update(UpdateType,BBox);    // schedule update
  void updateNow(UpdateType);      // schedule update
  void updateNow(UpdateType,BBox); // schedule update
  void updateClip(int);
  void updateWCSbb();

  double calcZoom(int,int,Vector);  // calculate the zoom factor
  void calcHistEqu();
  void setColorScale(FrScale::ColorScaleType);
  ContourScale* contourScale(int, float, float, FrScale::ColorScaleType);

  void centerImage();
  void setScanModeIncr(LoadMethod);

  // frcommandbase.C
  void printDouble(double, Precision);
  void printInteger(int);
  void printVector(const Vector&, Precision);
  void printCoordSystem(CoordSystem);

  int hasWCS(CoordSystem);
  int hasWCSEqu(CoordSystem); 
  int hasWCSLinear(CoordSystem);
  int hasContour() {return (currentcontour && *currentcontour);}

  virtual void getInfoSingle(const Vector&, InternalSystem,
			     Matrix& (FitsImage::*)(),Matrix& (FitsImage::*)(),
			     SkyFrame, SkyFormat, char*);
  virtual void getInfoMosaic(const Vector&, InternalSystem,
			     Matrix& (FitsImage::*)(), 
			     Matrix& (FitsImage::*)(),
			     SkyFrame, SkyFormat, char*);

  // frload.C
  void load(MemType, const char*, FitsImage*);
  void loadDataCube(MemType, const char*, FitsImage*);
  void loadMosaicImage(MemType, const char*, FitsImage*);
  void loadMosaicImageNext(MemType, const char*, FitsImage*);
  void loadMosaicWFPC2(MemType, const char*, FitsImage*);
  void loadMosaic(FitsImage*);
  void loadFinish();
  FitsHead* parseWCS(istream&);

  // frsave.C
  void saveFitsResample(OutFitsStream&);
  void saveFitsResampleSingle(OutFitsStream&);
  void saveFitsResampleMosaic(OutFitsStream&);
  void saveFitsResampleKeyword(OutFitsStream&, FitsHead&);

  // print.C
  void psLevel1Head(PSColorSpace, int, int);
  Matrix psMatrix(float, int, int);
#if __GNUC__ >= 3
  void psFix(ostringstream& ostr);
#endif
  void psFix(char*,int);

  void updateMarkerCoords(const Matrix&); // update marker coords
  void updateBinFileNames();

  // frmarker.C
  void updateCBMarkers();           // execute any update callbacks

private:
  Vector getClip(FrScale::ClipMode, float);

  void bltHist(char*, char*);

  void invalidPixmap();      // update all pixmaps
  int updatePixmap(const BBox&); // renders image/graphics into pixmap
  void updateBase(const BBox&);    // generate base pixmap
  void updatePM(const BBox&);      // update part of pixmap
  void updateGCs();          // update gcs
  void ximageToPixmap(Pixmap, XImage*, int, int, int, int, 
		      double* (FitsImage::*)());

  void updatePanner();       // render panner
  double calcPannerZoom();  // calculate the zoom factor for the panner
  void pannerPush();         // send panner and update message

  void updateMagnifier(); // render magnifier
  void updateMagnifier(const Vector&); // render magnifier
  void updateMagnifierMatrix();
  void updateMagnifierZoom(const Vector&); // render magnifier
  void renderMagnifierCursor(const Vector&); // render magnifier cursor

  void getInfo(const Vector&, InternalSystem,
	       Matrix& (FitsImage::*)(), Matrix& (FitsImage::*)(),
	       SkyFrame, SkyFormat, char*);

  FitsImage* isInFits(const Vector&, Matrix& (FitsImage::*)(), Vector*);

  void printFromRef(FitsImage*, const Vector&, CoordSystem, SkyFrame,
		    SkyFormat, Precision);

  void doubleToTclArray(double, Precision, const char*, const char*, 
			const char*);
  void coordToTclArray(FitsImage*, const Vector&, InternalSystem, CoordSystem, 
		       Precision, const char*, const char*);

  char* varcat(char* buf, char* var, char r);

  int hasLTMV();
  int hasATMV();
  int hasDTMV();

  void printSkyFrame(SkyFrame);
  void printSkyFormat(SkyFormat);

  void markerPrintCoord(const Vector&, InternalSystem);
  void markerPrintCoord(const Vector&, const Vector&,
			CoordSystem, SkyFrame, SkyFormat);

  void markerPrintDouble(double, InternalSystem);
  void markerPrintDouble(const Vector&, double, CoordSystem, SkyFormat);
  void markerPrintDouble(const Vector&, const Vector&, const Vector&, 
			 CoordSystem, SkyFormat);

  void markerPrintVector(const Vector&,InternalSystem);
  void markerPrintVector(const Vector&,const Vector&,CoordSystem,SkyFormat);

  void psMarkers(int);              // generate ps for all markers
  void renderfgMarkers(const BBox&);  // render the markers into pixmap
  void renderbgMarkers(const BBox&);  // render the markers into pixmap
  void renderMagnifierMarkers(const Matrix&, const BBox& bb);
  void unselectMarkers();
  void updateMarkers();             // update marker bbox
  void markerUndo(Marker*, UndoMarkerType);

  void markerListHeader(ostream&, CoordSystem, SkyFrame, SkyFormat, char, int);
  void markerListCiaoHeader(ostream&, CoordSystem, SkyFrame, SkyFormat, 
			    char, int);
  void markerListSAOtngHeader(ostream&,CoordSystem, SkyFrame, SkyFormat,
			      char, int);

  void createMarker(Marker*);       // process new marker
  void parseMarker(MarkerFormat,istream&);

  void renderHighLite();     // render a hightlite bbox

  void renderCrosshair();    // render the crosshair
  void psCrosshair(PSColorSpace);
  void drawCrosshair();
  void eraseCrosshair();

  void renderContours(Pixmap, const Matrix&, const BBox&);
  void psContours(PSColorSpace);

  void bltProjection(char*, char*, char*, char*, 
		     CoordSystem, SkyFrame,
		     const Vector&, const Vector&, double, int);
  void bltProjSingle(double*, double*, double*, double*, int,
		     CoordSystem, SkyFrame,
		     const Vector&, const Vector&, double, int);
  void bltProjMosaic(double*, double*, double*, double*, int,
		     CoordSystem, SkyFrame,
		     const Vector&, const Vector&, double, int);
  void bltCut(char*, char*, Orientation, const Vector&);
  void bltCutSingle(double*, double*, int, Vector&, Orientation);
  void bltCutMosaic(double*, double*, int, Vector&, Orientation);

  int postscriptProc(int);   // generate postscript
  Vector psOrigin();

  double getZoom() {return zoom;}

  double getMagnifierZoom() {return zoom*magnifierZoom;}
  double getRotation() {return rotation+wcsRotation;}

  double getWCSRotation(CoordSystem, SkyFrame);
  Orientation getWCSOrientation(CoordSystem, SkyFrame);

  Matrix& getCanvasToWidget() {return canvasToWidget;}
  Matrix& getRefToCanvas() {return refToCanvas;}
  Matrix& getRefToWidget() {return refToWidget;}
  Matrix& getRefToWindow() {return refToWindow;}
  Matrix& getWidgetToCanvas() {return widgetToCanvas;}

  FitsImage* findFits(const Vector&);
  int findFits(FitsImage*);

  Vector mapToRef(const Vector&, InternalSystem);
  Vector mapFromRef(const Vector&, InternalSystem);
  double mapLenFromRef(double, InternalSystem);
  Vector mapLenFromRef(const Vector&, InternalSystem);
  double mapAngleFromRef(double, CoordSystem);
  double mapAngleFromReff(double, CoordSystem);

public:
  FrameBase(Tcl_Interp*, Tk_Canvas, Tk_Item*);
  virtual ~FrameBase();

#if __GNUC__ >= 3
  int parse(istringstream&);
#else
  int parse(istrstream&);
#endif

  FitsImage* findFits(int);
  double mapAngleToRef(double, CoordSystem);
  Matrix& getCanvasToRef() {return canvasToRef;}
  SkyFrame getMarkerDefaultSky() {return markerDefaultSky;}
  CoordSystem getMarkerDefaultSystem() {return markerDefaultSystem;}
  int hasFitsMosaic() {return *currentMode == MOSAIC ? 1 : 0;}
  virtual int hasFitsCube() =0;

  // Misc Functions

  virtual void bgColorCmd(const char*) =0;
  void clearCmd();
  void colorScaleCmd(FrScale::ColorScaleType);
  void convertPathCmd(const char*);
  void getBitpixCmd();
  void getBgColorCmd();
  void getHeightCmd();
  void getInfoCmd(char*);
  void getInfoCmd(const Vector&, InternalSystem, SkyFrame, SkyFormat, 
		  char*);
  void getNANColorCmd();
  virtual void getTypeCmd() =0;
  void getValueCmd(const Vector&, InternalSystem);
  void getWidthCmd();
  void highliteCmd(int);
  void mosaicFastCmd(int);
  virtual void nanColorCmd(const char*);
  void warpCmd(const Vector&);
  void warpToCmd(const Vector&);

  // Clip Commands

  void clipMinMaxModeCmd(FrScale::MinMaxMode);
  void clipMinMaxParamCmd(int);
  void clipModeCmd(float);
  void clipModeCmd(FrScale::ClipMode);
  void clipPreserveCmd(int);
  void clipScopeCmd(FrScale::ClipScope);
  void clipUserCmd(double, double);
  void clipUserLowCmd(double);
  void clipUserHighCmd(double);
  void clipZScaleParamCmd(float, int, int);
  void getClipCmd();
  void getClipCmd(float);
  void getClipCmd(FrScale::ClipMode);
  void getClipMinMaxModeCmd();
  void getClipMinMaxSampleCmd();
  void getClipModeCmd();
  void getClipPreserveCmd();
  void getClipScopeCmd();
  void getClipUserCmd();
  void getClipZScaleParamCmd();
  void getHistogramCmd(char*, char*);
  void getMinMaxCmd();
  void hasDATAMINCmd();
  void hasIRAFMINCmd();

  // Colormap Commands

  void colormapAreaCmd(int s, ColormapMode m) {cmapArea = s; cmapMode = m;}
  virtual void colormapBeginCmd() =0;
  virtual void colormapBeginCmd(const Vector&) =0;
  virtual void getColormapCmd() =0;
  void getColorScaleCmd();

  // CrossHair Commands

  void crosshairCmd(int);
  void crosshairCmd(const Vector&, InternalSystem);
  void crosshairCmd(const Vector&, CoordSystem, SkyFrame);
  void crosshairBeginCmd(const Vector&, InternalSystem);
  void crosshairBeginCmd(const Vector&, CoordSystem);
  void crosshairMotionCmd(const Vector&, InternalSystem);
  void crosshairMotionCmd(const Vector&, CoordSystem);
  void crosshairWarpCmd(const Vector&);
  void getCrosshairCmd(InternalSystem);
  void getCrosshairCmd(CoordSystem, SkyFrame, SkyFormat, Precision);
  void getCrosshairStatusCmd();

  void getHorzCutCmd(char*, char*, const Vector&, InternalSystem);
  void getVertCutCmd(char*, char*, const Vector&, InternalSystem);

  // DATASEC Commands

  void DATASECCmd(int);
  void getDATASECCmd();
  void hasDATASECCmd();

  // Export Commands

  void exportJPEGCmd(const char*, float) {}

  // Fits Commands

  void getFitsCountCmd() {printInteger(*currentCount);}
  void getFitsDepthCmd();
  void getFitsHeaderCmd(int);
  void getFitsHeaderKeywordCmd(int,const char*);
  void getFitsFileNameCmd(FileNameType);
  void getFitsFileNameCmd(int,FileNameType);
  void getFitsFileNameCmd(const Vector&, InternalSystem, FileNameType);
  void getFitsObjectNameCmd(int);
  void getFitsMasterCmd();
  virtual void getFitsSizeCmd() =0;
  virtual void getFitsSizeCmd(CoordSystem, SkyFormat) =0;
  void getFitsSliceCmd();

  void hasFitsCmd();
  void hasFitsCubeCmd();
  void hasFitsExtCmd(int);
  void hasFitsImageCmd();
  void hasFitsMosaicCmd();
  void hasFitsTableCmd();

  void loadFitsAllocCmd(const char*);
  void loadFitsAllocGZCmd(const char*);
  void loadFitsChannelCmd(const char*, const char*);
  void loadFitsMMapCmd(const char* fn, LoadMethod);
  void loadFitsSMMapCmd(const char* hdr, const char* fn, LoadMethod);
  void loadFitsMMapIncrCmd(const char* fn, LoadMethod);
  void loadFitsShareCmd(ShmType, int, const char*, LoadMethod);
  void loadFitsSShareCmd(ShmType, int, int, const char*, LoadMethod);
  void loadFitsSocketCmd(int, const char* fn);
  void loadFitsSocketGZCmd(int, const char* fn);
  void loadFitsVarCmd(const char*, const char*, LoadMethod);

  void loadArrAllocCmd(const char*);
  void loadArrAllocGZCmd(const char*);
  void loadArrChannelCmd(const char*, const char*);
  void loadArrMMapCmd(const char* fn);
  void loadArrMMapIncrCmd(const char* fn);
  void loadArrShareCmd(ShmType, int, const char*);
  void loadArrSocketCmd(int, const char* fn);
  void loadArrSocketGZCmd(int, const char* fn);
  void loadArrVarCmd(const char*, const char*);

  void loadDataCubeAllocCmd(const char*);
  void loadDataCubeAllocGZCmd(const char*);
  void loadDataCubeChannelCmd(const char*, const char*);
  void loadDataCubeMMapCmd(const char* fn, LoadMethod);
  void loadDataCubeMMapIncrCmd(const char* fn, LoadMethod);
  void loadDataCubeShareCmd(ShmType, int, const char*, LoadMethod);
  void loadDataCubeSocketCmd(int, const char* fn);
  void loadDataCubeSocketGZCmd(int, const char* fn);
  void loadDataCubeVarCmd(const char*, const char*, LoadMethod);

  void loadMosaicImageIRAFAllocCmd(const char*);
  void loadMosaicImageIRAFAllocGZCmd(const char*);
  void loadMosaicImageIRAFChannelCmd(const char*, const char*);
  void loadMosaicImageIRAFMMapCmd(const char*, LoadMethod);
  void loadMosaicImageIRAFMMapIncrCmd(const char*, LoadMethod);
  void loadMosaicImageIRAFShareCmd(ShmType, int, const char*, LoadMethod);
  void loadMosaicImageIRAFSocketCmd(int, const char*);
  void loadMosaicImageIRAFSocketGZCmd(int, const char*);
  void loadMosaicImageIRAFVarCmd(const char*,const char*, LoadMethod);

  void loadMosaicIRAFAllocCmd(const char*);
  void loadMosaicIRAFAllocGZCmd(const char*);
  void loadMosaicIRAFChannelCmd(const char*, const char*);
  void loadMosaicIRAFMMapCmd(const char*, LoadMethod);
  void loadMosaicIRAFSMMapCmd(const char*, const char*, LoadMethod);
  void loadMosaicIRAFMMapIncrCmd(const char*, LoadMethod);
  void loadMosaicIRAFShareCmd(ShmType, int, const char*, LoadMethod);
  void loadMosaicIRAFSShareCmd(ShmType, int, int, const char*, LoadMethod);
  void loadMosaicIRAFSocketCmd(int, const char*);
  void loadMosaicIRAFSocketGZCmd(int, const char*);
  void loadMosaicIRAFVarCmd(const char*, const char*, LoadMethod);

  void loadMosaicImageWCSAllocCmd(CoordSystem, const char*);
  void loadMosaicImageWCSAllocGZCmd(CoordSystem, const char*);
  void loadMosaicImageWCSChannelCmd(CoordSystem, const char*, const char*);
  void loadMosaicImageWCSMMapCmd(CoordSystem, const char*, LoadMethod);
  void loadMosaicImageWCSMMapIncrCmd(CoordSystem, const char*, LoadMethod);
  void loadMosaicImageWCSShareCmd(CoordSystem, ShmType, int, const char*, 
				  LoadMethod);
  void loadMosaicImageWCSSocketCmd(CoordSystem, int, const char*);
  void loadMosaicImageWCSSocketGZCmd(CoordSystem, int, const char*);
  void loadMosaicImageWCSVarCmd(CoordSystem, const char*,const char*, 
				LoadMethod);

  void loadMosaicImageNextWCSAllocCmd(CoordSystem, const char*);
  void loadMosaicImageNextWCSAllocGZCmd(CoordSystem, const char*);
  void loadMosaicImageNextWCSChannelCmd(CoordSystem, const char*, const char*);
  void loadMosaicImageNextWCSMMapCmd(CoordSystem, const char*, LoadMethod);
  void loadMosaicImageNextWCSMMapIncrCmd(CoordSystem, const char*, LoadMethod);
  void loadMosaicImageNextWCSShareCmd(CoordSystem, ShmType, int, const char*, 
				      LoadMethod);
  void loadMosaicImageNextWCSSocketCmd(CoordSystem, int, const char*);
  void loadMosaicImageNextWCSSocketGZCmd(CoordSystem, int, const char*);
  void loadMosaicImageNextWCSVarCmd(CoordSystem, const char*,const char*, 
				    LoadMethod);

  void loadMosaicWCSAllocCmd(CoordSystem, const char*);
  void loadMosaicWCSAllocGZCmd(CoordSystem, const char*);
  void loadMosaicWCSChannelCmd(CoordSystem, const char*, const char*);
  void loadMosaicWCSMMapCmd(CoordSystem, const char*, LoadMethod);
  void loadMosaicWCSSMMapCmd(CoordSystem, const char*, const char*,LoadMethod);
  void loadMosaicWCSMMapIncrCmd(CoordSystem, const char*, LoadMethod);
  void loadMosaicWCSShareCmd(CoordSystem, ShmType, int, 
			     const char*, LoadMethod);
  void loadMosaicWCSSShareCmd(CoordSystem, ShmType, int, int, 
			     const char*, LoadMethod);
  void loadMosaicWCSSocketCmd(CoordSystem, int, const char*);
  void loadMosaicWCSSocketGZCmd(CoordSystem, int, const char*);
  void loadMosaicWCSVarCmd(CoordSystem, const char*, const char*, LoadMethod);

  void loadMosaicImageWFPC2AllocCmd(const char*);
  void loadMosaicImageWFPC2AllocGZCmd(const char*);
  void loadMosaicImageWFPC2ChannelCmd(const char*, const char*);
  void loadMosaicImageWFPC2MMapCmd(const char* fn, LoadMethod);
  void loadMosaicImageWFPC2MMapIncrCmd(const char* fn, LoadMethod);
  void loadMosaicImageWFPC2ShareCmd(ShmType, int, const char*,LoadMethod);
  void loadMosaicImageWFPC2SocketCmd(int, const char* fn);
  void loadMosaicImageWFPC2SocketGZCmd(int, const char* fn);
  void loadMosaicImageWFPC2VarCmd(const char*, const char*, LoadMethod);

  virtual void loadRGBCubeAllocCmd(const char*) =0;
  virtual void loadRGBCubeAllocGZCmd(const char*) =0;
  virtual void loadRGBCubeChannelCmd(const char*, const char*) =0;
  virtual void loadRGBCubeMMapCmd(const char*, LoadMethod) =0;
  virtual void loadRGBCubeSMMapCmd(const char*, const char*, LoadMethod) =0;
  virtual void loadRGBCubeMMapIncrCmd(const char*, LoadMethod) =0;
  virtual void loadRGBCubeShareCmd(ShmType, int, const char*, LoadMethod) =0;
  virtual void loadRGBCubeSShareCmd(ShmType,int,int,const char*,LoadMethod) =0;
  virtual void loadRGBCubeSocketCmd(int, const char*) =0;
  virtual void loadRGBCubeSocketGZCmd(int, const char*) =0;
  virtual void loadRGBCubeVarCmd(const char*, const char*, LoadMethod) =0;

  virtual void loadRGBCubeSlaveCmd(const char*, FitsFile*) =0;

  virtual void loadRGBImageAllocCmd(const char*) =0;
  virtual void loadRGBImageAllocGZCmd(const char*) =0;
  virtual void loadRGBImageChannelCmd(const char*, const char*) =0;
  virtual void loadRGBImageMMapCmd(const char*, LoadMethod) =0;
  virtual void loadRGBImageMMapIncrCmd(const char*, LoadMethod) =0;
  virtual void loadRGBImageShareCmd(ShmType, int, const char*, LoadMethod) =0;
  virtual void loadRGBImageSocketCmd(int, const char*) =0;
  virtual void loadRGBImageSocketGZCmd(int, const char*) =0;
  virtual void loadRGBImageVarCmd(const char*, const char*, LoadMethod) =0;

  virtual void loadArrRGBCubeAllocCmd(const char*) =0;
  virtual void loadArrRGBCubeAllocGZCmd(const char*) =0;
  virtual void loadArrRGBCubeChannelCmd(const char*, const char*) =0;
  virtual void loadArrRGBCubeMMapCmd(const char*) =0;
  virtual void loadArrRGBCubeMMapIncrCmd(const char*) =0;
  virtual void loadArrRGBCubeShareCmd(ShmType, int, const char*) =0;
  virtual void loadArrRGBCubeSocketCmd(int, const char*) =0;
  virtual void loadArrRGBCubeSocketGZCmd(int, const char*) =0;
  virtual void loadArrRGBCubeVarCmd(const char*, const char*) =0;

  void loadIncrDataCmd(int, int, int, int, int);
  void loadIncrMinMaxCmd(int, int, int, int, int);
  void loadIncrEndCmd();

  void loadFitsSlaveCmd(const char*, FitsFile*);

  void saveFitsImageFileCmd(const char*, int);
  void saveFitsImageChannelCmd(const char*, int);
  void saveFitsImageSocketCmd(int, int);

  void saveFitsTableFileCmd(const char*, int);
  void saveFitsTableChannelCmd(const char*, int);
  void saveFitsTableSocketCmd(int, int);

  void saveArrayFileCmd(const char*);
  void saveArrayChannelCmd(const char*);
  void saveArraySocketCmd(int);

  void saveFitsResampleFileCmd(const char*, int);
  void saveFitsResampleChannelCmd(const char*, int);
  void saveFitsResampleSocketCmd(int, int);

  virtual void sliceCmd(int) =0;

  void updateFitsCmd(int);
  void updateFitsCmd(int, BBox, int);
  void unloadFitsCmd();

  // Fitsy Commands

  void fitsyHasExtCmd(const char*);

  // Coordinate Commands

  void getCoordCmd(const Vector&, InternalSystem, CoordSystem, 
		   SkyFrame, SkyFormat);

  void getCursorCmd(InternalSystem);
  void getCursorCmd(CoordSystem, SkyFrame, SkyFormat, Precision);

  void hasSystemCmd(CoordSystem);

  void hasAmplifierCmd();
  void hasDetectorCmd();
  void hasPhysicalCmd();

  void hasWCSCmd(CoordSystem);
  void hasWCSEquCmd(CoordSystem);
  void hasWCSLinearCmd(CoordSystem);
  void getWCSNameCmd(CoordSystem);

  // IIS Commands

  virtual void iisCmd(int, int) =0;
  virtual void iisEraseCmd() =0;
  virtual void iisGetCmd(char*, int, int, int, int) =0;
  virtual void iisSetCmd(const char*, int, int, int, int) =0;
  virtual void iisWCSCmd(const Matrix&, const Vector&, int) =0;

  void iisMessageCmd(const char*);
  void iisUpdateCmd() {updateNow(MATRIX);}
  void iisSetCursorCmd(const Vector&, InternalSystem);
  void iisSetCursorCmd(const Vector&, CoordSystem);
  void iisGetCursorCmd();
  void iisCursorModeCmd(int);
  void hasIISCmd();

  void iisGetFileNameCmd();
  void iisGetFileNameCmd(int);
  void iisGetFileNameCmd(const Vector&);
  void iisSetFileNameCmd(const char*);
  void iisSetFileNameCmd(const char*,int);

  // Marker Commands

  void createCircleCmd(const Vector&, double,
		       const char*, int, const char*, 
		       const char*, unsigned short, const char*,
		       const List<Tag>&);
  void createAnnulusCmd(const Vector&, double, double, int,
			const char*, int, const char*, 
			const char*, unsigned short, const char*,
			const List<Tag>&);
  void createAnnulusCmd(const Vector&, double*, int,
			const char*, int, const char*, 
			const char*, unsigned short, const char*,
			const List<Tag>&);

  void createEllipseCmd(const Vector&, const Vector&, 
			double, const char*, int, const char*, 
			const char*, unsigned short, const char*,
			const List<Tag>&);
  void createEllipseAnnulusCmd(const Vector&, const Vector&, 
			       const Vector&, int, double,
			       const char*, int, const char*, 
			       const char*, unsigned short, const char*,
			       const List<Tag>&);
  void createEllipseAnnulusCmd(const Vector&, double, int, Vector*,
			       const char*, int,  const char*, 
			       const char*, unsigned short, const char*,
			       const List<Tag>&);

  void createBoxCmd(const Vector&, const Vector&, 
		    double, const char*, int, const char*, 
		    const char*, unsigned short, const char*,
		    const List<Tag>&);
  void createBoxAnnulusCmd(const Vector&, const Vector&, 
			   const Vector&, int, double,
			   const char*, int, const char*, 
			   const char*, unsigned short, const char*,
			   const List<Tag>&);
  void createBoxAnnulusCmd(const Vector&, double, int, Vector*,
			   const char*, int, const char*, 
			   const char*, unsigned short, const char*,
			   const List<Tag>&);

  void createPandaCmd(const Vector&, double, double, int, double, double, int,
		      const char*, int, const char*, 
		      const char*, unsigned short, const char*,
		      const List<Tag>&);
  void createPandaCmd(const Vector&, double*, int, double*, int,
		      const char*, int, const char*, 
		      const char*, unsigned short, const char*,
		      const List<Tag>&);

  void createPolygonCmd(const Vector&, const Vector&,
			const char*, int, const char*, 
			const char*, unsigned short, const char*,
			const List<Tag>&);
  void createPolygonCmd(const List<Vertex>&,
			const char*, int, const char*, 
			const char*, unsigned short, const char*,
			const List<Tag>&);
  void createContourPolygonCmd(const char*, int, const char*, 
			       const char*, unsigned short, const char*,
			       const List<Tag>&);
  void createLineCmd(const Vector&, const Vector&, int, int,
		     const char*, int, const char*,
		     const char*, unsigned short, const char*,
		     const List<Tag>&);
  void createTextCmd(const Vector&, double,
		     const char*, int, const char*, 
		     const char*, unsigned short, const char*,
		     const List<Tag>&);
  void createRulerCmd(const Vector&, const Vector&, 
		      CoordSystem, SkyFrame, CoordSystem, SkyFormat,
		      const char*, int, const char*,
		      const char*, unsigned short, const char*,
		      const List<Tag>&);
  void createCompassCmd(const Vector&, double, const char*, const char*,
			int, int,
			CoordSystem, SkyFrame,
			const char*, int, const char*,
			const char*, unsigned short, const char*,
			const List<Tag>&);
  void createProjectionCmd(const Vector&, const Vector&,
			   double, const char*, const char*,
			   const char*, int, const char*,
			   const char*, unsigned short, const char*,
			   const List<Tag>&);
  void createCirclePointCmd(const Vector&,
			    const char*, int, const char*,
			    const char*, unsigned short, const char*,
			    const List<Tag>&);
  void createBoxPointCmd(const Vector&,
			 const char*, int, const char*,
			 const char*, unsigned short, const char*,
			 const List<Tag>&);
  void createDiamondPointCmd(const Vector&,
			     const char*, int, const char*, 
			     const char*, unsigned short, const char*,
			     const List<Tag>&);
  void createCrossPointCmd(const Vector&, 
			   const char*, int, const char*,
			   const char*, unsigned short, const char*,
			   const List<Tag>&);
  void createExPointCmd(const Vector&,
			const char*, int, const char*,
			const char*, unsigned short, const char*,
			const List<Tag>&);
  void createArrowPointCmd(const Vector&, 
			   const char*, int, const char*,
			   const char*, unsigned short, const char*,
			   const List<Tag>&);
  void createBoxCirclePointCmd(const Vector&,
			       const char*, int, const char*,
			       const char*, unsigned short, const char*,
			       const List<Tag>&);

  void getMarkerAngleCmd(int, CoordSystem);
  void getMarkerAnnulusRadiusCmd(int, CoordSystem, SkyFormat);
  void getMarkerBoxAnnulusSizeCmd(int, CoordSystem, SkyFormat);
  void getMarkerBoxSizeCmd(int, InternalSystem);
  void getMarkerBoxSizeCmd(int, CoordSystem, SkyFormat);
  void getMarkerCenterCmd(int, InternalSystem);
  void getMarkerCenterCmd(int, CoordSystem, SkyFrame, SkyFormat);
  void getMarkerCircleRadiusCmd(int, InternalSystem);
  void getMarkerCircleRadiusCmd(int, CoordSystem, SkyFormat);
  void getMarkerColorCmd();
  void getMarkerColorCmd(const char*);
  void getMarkerColorCmd(int);
  void getMarkerCompassArrowCmd(int);
  void getMarkerCompassLabelCmd(int);
  void getMarkerCompassRadiusCmd(int, InternalSystem);
  void getMarkerCompassRadiusCmd(int, CoordSystem, SkyFormat);
  void getMarkerEllipseRadiusCmd(int, CoordSystem, SkyFormat);
  void getMarkerEllipseAnnulusRadiusCmd(int, CoordSystem, SkyFormat);
  void getMarkerFontCmd();
  void getMarkerFontCmd(const char*);
  void getMarkerFontCmd(int);
  void getMarkerHandleCmd(const Vector&);
  void getMarkerIdCmd(const Vector&);
  void getMarkerIdAllCmd();
  void getMarkerLineCmd(int, CoordSystem, SkyFrame, SkyFormat);
  void getMarkerLineArrowCmd(int);
  void getMarkerLineDistanceCmd(int, CoordSystem, SkyFormat);
  void getMarkerLineWidthCmd();
  void getMarkerLineWidthCmd(int);
  void getMarkerMapLenFromRefCmd(int, double, CoordSystem, SkyFormat);
  void getMarkerPandaAnglesCmd(int, CoordSystem);
  void getMarkerPandaRadiusCmd(int, CoordSystem, SkyFormat);
  void getMarkerPolygonSegmentCmd(const Vector&);
  void getMarkerPreserveCmd();
  void getMarkerProjectionCmd(int, char*, char*, char*, char*);
  void getMarkerProjectionPointsCmd(int, CoordSystem, SkyFrame, SkyFormat);
  void getMarkerProjectionDistanceCmd(int, CoordSystem, SkyFormat);
  void getMarkerProjectionMethodCmd(int);
  void getMarkerProjectionWidthCmd(int, CoordSystem, SkyFormat);
  void getMarkerProjectionSystemCmd(int);
  void getMarkerProjectionSkyFrameCmd(int);
  void getMarkerPropertyCmd(unsigned short);
  void getMarkerPropertyCmd(const char*, unsigned short);
  void getMarkerPropertyCmd(int, unsigned short);
  void getMarkerRulerDistanceCmd(int, CoordSystem, SkyFormat);
  void getMarkerRulerPointCmd(int, CoordSystem, SkyFrame, SkyFormat);
  void getMarkerSelectedCmd();
  void getMarkerSelectedCmd(const Vector&);
  void getMarkerSelectedNumberCmd();
  void getMarkerShowCmd();
  void getMarkerTagCmd(const char*);
  void getMarkerTagCmd(int);
  void getMarkerTagCmd(int,int);
  void getMarkerTagDefaultNameCmd();
  void getMarkerTagNumberCmd(const char*);
  void getMarkerTagsCmd();
  void getMarkerTextCmd(int);
  void getMarkerTypeCmd(int);

  void hasMarkerSelectedCmd();
  void hasMarkerPasteCmd();
  void hasMarkerUndoCmd();

  void markerfg() {
    markers = &fgMarkers;
    undoMarkers = &undofgMarkers;
    pasteMarkers = &pastefgMarkers;
    showMarkers = &showfgMarkers;
    markerUpdate = PIXMAP;
  }
  void markerbg() {
    markers = &bgMarkers;
    undoMarkers = &undobgMarkers;
    pasteMarkers = &pastebgMarkers;
    showMarkers = &showbgMarkers;
    markerUpdate = BASE;
  }

  void markerAngleCmd(int, double);
  void markerAngleCmd(int, double, CoordSystem);
  void markerAnnulusCreateRadiusCmd(int, const Vector&);
  void markerAnnulusDeleteRadiusCmd(int, int);
  void markerAnnulusRadiusCmd(int, double, double, int, CoordSystem,SkyFormat);
  void markerAnnulusRadiusCmd(int, const char*, CoordSystem, SkyFormat);
  void markerBackCmd();
  void markerBackCmd(const char*);
  void markerBackCmd(int);
  void markerBoxAnnulusSizeCmd(int, const Vector&, const Vector&, int, 
			       CoordSystem, SkyFormat);
  void markerBoxAnnulusSizeCmd(int, const char*, CoordSystem, SkyFormat);
  void markerBoxAnnulusCreateSizeCmd(int, const Vector&);
  void markerBoxAnnulusDeleteSizeCmd(int, int);
  void markerBoxSizeCmd(int, const Vector&, CoordSystem, SkyFormat);
  void markerCallBackCmd(int, Marker::Callback, const char*, const char*);
  void markerCircleRadiusCmd(int, double, CoordSystem, SkyFormat);
  void markerColorCmd(const char*);
  void markerColorCmd(const char*, const char*);
  void markerColorCmd(int, const char*);
  void markerCompassArrowCmd(int, int, int);
  void markerCompassLabelCmd(int, const char*, const char*);
  void markerCompassRadiusCmd(int, double, CoordSystem, SkyFormat);
  void markerCompassSystemCmd(int, CoordSystem, SkyFrame);
  void markerCommandCmd(MarkerFormat, const char*);
  void markerCommandVarCmd(MarkerFormat, const char*);
  void markerDeleteCmd();
  void markerDeleteCmd(const char*);
  void markerDeleteCmd(int);
  void markerDeleteAllCmd();
  void markerDeleteLastCmd();
  void markerDeleteCallBackCmd(int, Marker::Callback, const char*);
  void markerDeleteTagCmd(int);
  void markerDeleteTagCmd(int, const char*);
  void markerDeleteTagCmd(int, int);
  void markerEditBeginCmd(int, int);
  void markerEditBeginCmd(const Vector&, int);
  void markerEditMotionCmd(const Vector&, int h);
  void markerEditEndCmd();
  void markerEllipseRadiusCmd(int, const Vector&, CoordSystem, SkyFormat);
  void markerEllipseAnnulusRadiusCmd(int, const Vector&, const Vector&, int, 
				     CoordSystem, SkyFormat);
  void markerEllipseAnnulusRadiusCmd(int, const char*, 
				     CoordSystem, SkyFormat);
  void markerEllipseAnnulusCreateRadiusCmd(int, const Vector&);
  void markerEllipseAnnulusDeleteRadiusCmd(int, int);
  void markerFontCmd(const char*);
  void markerFontCmd(const char*, const char*);
  void markerFontCmd(int, const char*);
  void markerFrontCmd();
  void markerFrontCmd(const char*);
  void markerFrontCmd(int);
  void markerLineCmd(int, const Vector&, const Vector&, CoordSystem, SkyFrame);
  void markerLineArrowCmd(int, int, int);
  void markerLineWidthCmd(int);
  void markerLineWidthCmd(int, int);
  void markerListCmd(MarkerFormat, CoordSystem, SkyFrame, SkyFormat, 
		     int, int, int,
		     unsigned short, unsigned short, List<Tag>&);
  void markerLoadCmd(MarkerFormat,const char*);
  void markerLoadCmd(MarkerFormat,const char*,CoordSystem,SkyFrame);
  void markerLoadCmd(MarkerFormat,int);
  void markerLoadCmd(MarkerFormat,int,CoordSystem,SkyFrame);
  void markerLoadFitsCmd(const char*, const char*, int, const char*);
  void markerMoveCmd(const Vector&);
  void markerMoveCmd(const char*, const Vector&);
  void markerMoveCmd(int id, const Vector&);
  void markerMoveBeginCmd(const Vector&);
  void markerMoveBeginCmd(int, const Vector&);
  void markerMoveMotionCmd(const Vector&);
  void markerMoveMotionCmd(int, const Vector&);
  void markerMoveEndCmd();
  void markerMoveEndCmd(int);
  void markerMoveToCmd(const Vector&, CoordSystem, SkyFrame);
  void markerMoveToCmd(const char*, const Vector&, CoordSystem, SkyFrame);
  void markerMoveToCmd(int, const Vector&, InternalSystem);
  void markerMoveToCmd(int, const Vector&, CoordSystem, SkyFrame);
  void markerPandaCreateAnglesCmd(int, const Vector&);
  void markerPandaCreateRadiusCmd(int, const Vector&);
  void markerPandaDeleteCmd(int, int);
  void markerPandaEditCmd(int, double, double, int, double, double, int,
			  CoordSystem, SkyFormat);
  void markerPandaEditCmd(int, const char*, const char*, CoordSystem,
			  CoordSystem, SkyFormat);
  void markerPolygonCreateVertexCmd(int, int, const Vector&);
  void markerPolygonDeleteVertexCmd(int, int);
  void markerPolygonResetCmd(int, const Vector&, CoordSystem, SkyFormat);
  void markerPreserveCmd(int r) {preserveMarkers = r;}
  void markerPropertyCmd(unsigned short, int);
  void markerPropertyCmd(const char*, unsigned short, int);
  void markerPropertyCmd(int, unsigned short, int);
  void markerPropertyCmd(unsigned short, int, const Vector&);
  void markerProjectionCmd(int, const Vector&, const Vector&, 
			   CoordSystem, SkyFrame, 
			   double, int, CoordSystem, SkyFormat);
  void markerProjectionSystemCmd(int, CoordSystem, SkyFrame);
  void markerRotateBeginCmd(int);
  void markerRotateBeginCmd(const Vector&);
  void markerRotateMotionCmd(const Vector&, int h);
  void markerRotateEndCmd();
  void markerRulerPointCmd(int, const Vector&, const Vector&, 
			   CoordSystem, SkyFrame);
  void markerRulerSystemCmd(int, CoordSystem, SkyFrame, 
			    CoordSystem, SkyFormat);
  void markerSaveCmd(const char*, MarkerFormat, CoordSystem, SkyFrame,
		     SkyFormat, int);
  void markerSelectCmd();
  void markerSelectCmd(const char*);
  void markerSelectCmd(int);
  void markerSelectCmd(const Vector&);
  void markerSelectToggleCmd(const Vector&);
  void markerShowCmd(int);
  void markerTagCmd(const char*);
  void markerTagCmd(int, const char*);
  void markerTagEditCmd(const char*, const char*);
  void markerTagDeleteCmd(const char*);
  void markerTagDeleteAllCmd();
  void markerTagSelectCmd(const char*);
  void markerTagUpdateCmd(const char*);
  void markerTextCmd(int, const char*);
  void markerUnselectCmd(int);
  void markerUnselectAllCmd();

  void markerCopyCmd();
  void markerCopyCmd(const char*);
  void markerCutCmd();
  void markerCutCmd(const char*);
  void markerPasteCmd();
  void markerPasteCmd(CoordSystem, SkyFrame);
  void markerUndoCmd();

  void regionSelectBeginCmd(const Vector&);
  void regionSelectMotionCmd(const Vector&);
  void regionSelectEndCmd();
  void regionSelectShiftEndCmd();

  // Pan Zoom Rotate Orient Commands

  void centerCmd();
  void getOrientCmd();
  void getRotateCmd(Precision);
  void getPanPreserveCmd();
  void getWCSAlignCmd();
  virtual void getWCSZoomCmd(CoordSystem, Precision) =0;
  void getZoomCmd(Precision);
  void orientCmd(Orientation);
  void panCmd(const Vector&, const Vector&);
  void panCmd(const Vector&, InternalSystem);
  void panCmd(const Vector&, CoordSystem, SkyFrame);
  void panBBoxCmd(BBox);
  void panMotionBeginCmd(const Vector&);
  void panMotionCmd(const Vector&);
  void panMotionEndCmd(const Vector&);
  void panToCmd(const Vector&, InternalSystem);
  void panToCmd(const Vector&, CoordSystem, SkyFrame);
  void panPreserveCmd(int r) {preservePan = r;}
  void rotateCmd(double);
  void rotateToCmd(double);
  void rotateMotionBeginCmd();
  void rotateMotionCmd(double);
  void rotateMotionEndCmd();
  void wcsAlignCmd(int,CoordSystem,SkyFrame);
  virtual void wcsZoomCmd(CoordSystem, double) =0;
  void zoomCmd(double);
  void zoomAboutCmd(double, const Vector&, InternalSystem);
  void zoomAboutCmd(double, const Vector&, CoordSystem, SkyFrame);
  void zoomToCmd(double);
  void zoomToAboutCmd(double, const Vector&, InternalSystem);
  void zoomToAboutCmd(double, const Vector&, CoordSystem, SkyFrame);
  virtual void zoomToFitCmd(double =1) =0;

  // Panner/Magnifier Commands

  void magnifierCmd(char*, int, int);
  void magnifierCmd(int);
  void magnifierGraphicsCmd(int);
  void magnifierCursorCmd(int);
  void magnifierZoomCmd(double);

  void pannerCmd(char*, int, int);
  void pannerCmd(int);
  void pannerCmd(CoordSystem, SkyFrame);
  void updatePannerCmd();
  void updateMagnifierCmd();
  void updateMagnifierCmd(const Vector& v);

  // Postscript Commands
       
  void psInterpolateCmd(int w) {psInterpolate = w;}
  void psLevelCmd(int l) {psLevel = (l == 2) ? 2 : 1;}
  void psResolutionCmd(int r) {psResolution = r;}
  void psColorSpaceCmd(PSColorSpace c) {psColorSpace = c;}

  // Bin Commands

  void binCmd(const Vector&, const char*, const char*, const char*, 
	      int, FitsHist::Smooth, int);
  void binCmd(const Vector&, const Vector&, const char*, const char*, 
	      const char*, int, FitsHist::Smooth, int);
  void binCmd(const Vector&, int, const Vector&, 
	      const char*, const char*, const char*, const char*,
	      int, FitsHist::Smooth, int);
  void binCmd(const Vector&, int, const Vector&, const Vector&, 
	      const char*, const char*, const char*, const char*,
	      int, FitsHist::Smooth, int);
  void binAboutCmd(const Vector&);
  void binColsCmd(const char*, const char*, const char*);
  void binDepthCmd(int);
  void binFactorCmd(const Vector&);
  void binFactorAboutCmd(const Vector&, const Vector&);
  void binFactorToCmd(const Vector&);
  void binFactorToAboutCmd(const Vector&, const Vector&);
  void binToFitCmd();
  void binFunctionCmd(FitsHist::Function);
  void binSmoothCmd(int);
  void binSmoothFuncCmd(FitsHist::Smooth);
  void binSmoothRadiusCmd(int);
  void binBufferSizeCmd(int);
  void binFilterCmd(const char*);
  void getBinDepthCmd();
  void getBinFactorCmd();
  void getBinFunctionCmd();
  void getBinSmoothCmd();
  void getBinSmoothFuncCmd();
  void getBinSmoothRadiusCmd();
  void getBinBufferSizeCmd();
  void getBinCursorCmd();
  void getBinFilterCmd();
  void getBinColsCmd();
  void getBinColsMinMaxCmd(const char*);
  void getBinListCmd();
  void hasBinCmd();

  // Contour Commands

  void contourCreateCmd(const char*, int, int, float, int);
  void contourCreateCmd(const char*, int, float, int, const char*);
  void contourLoadCmd(const char*, int, const char*, CoordSystem, SkyFrame);
  void contourCopyCmd(CoordSystem, SkyFrame);
  void contourDeleteCmd();
  void contourDeleteAllCmd();
  void contourPasteCmd(const char*, int, void*, CoordSystem, SkyFrame);
  void contourSaveCmd(const char*, CoordSystem, SkyFrame);
  void contourSetColorCmd(const char*);
  void contourSetLineWidthCmd(int);
  void getContourCmd(CoordSystem sys, SkyFrame sky);
  void getContourLevelCmd();
  void getContourLevelCmd(int, float, float, FrScale::ColorScaleType);
  void getContourMethodCmd();
  void getContourColorNameCmd();
  void getContourLineWidthCmd();
  void hasContourCmd();

  // Grid Commands

  void gridCmd(CoordSystem, SkyFrame, SkyFormat, Grid::GridType, const char*);
  void gridDeleteCmd();
  void hasGridCmd();
  void getGridCmd();
  void getGridOptionCmd();

  // Pixel Table

  void getPixelTableCmd(const Vector&, InternalSystem, int, char*);
  void getPixelTable(const Vector&, Matrix& (FitsImage::*matrixToData)(),
		     int, char*);
  void getPixelTable(FitsImage*, const Vector&, int, char*);

  // Data Values

  void getDataValuesCmd(int, const Vector&, CoordSystem, SkyFrame, 
			const Vector&, char*);
  void getDataValuesCmd(const Vector&, InternalSystem, const Vector&);

  // WCS

  void wcsResetCmd();
  void wcsReplaceCmd(int);
  void wcsReplaceCmd(const char*);
  void wcsAppendCmd(int);
  void wcsAppendCmd(const char*);

  // RGB Commands

  virtual void getRGBChannelCmd() =0;
  virtual void setRGBChannelCmd(const char*) =0;
  virtual void getRGBViewCmd() =0;
  virtual void setRGBViewCmd(int, int, int) =0;
  virtual void getRGBSystemCmd() =0;
  virtual void setRGBSystemCmd(CoordSystem) =0;
};

#endif
