/* 
   sitecopy, for managing remote web sites.
   Copyright (C) 1998-2003, Joe Orton <joe@manyfish.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <config.h>

#include <sys/types.h>

#include <sys/stat.h>

#include <ctype.h>
#include <errno.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include <ne_string.h>
#include <ne_alloc.h>

#include "common.h"
#include "netrc.h"
#include "rcfile.h"
#include "sites.h"

/** Global variables **/
char *copypath;
char *rcfile;
char *netrcfile;
char *home;
int havenetrc;

/* These are used for reporting errors back to the calling procedures. */
int rcfile_linenum; 
char *rcfile_err;

/** Not quite so global variables **/

/* These are appended to $HOME */
#define RCNAME "/.sitecopyrc"
#define COPYNAME "/.sitecopy/"
#define NETRCNAME "/.netrc"

/* Stores the list of entries in the ~/.netrc */
netrc_entry *netrc_list;

const char *rc_get_netrc_password(const char *server, const char *username);

/* The driver definitions */
#ifdef USE_FTP
extern const struct proto_driver ftp_driver;
#endif /* USE_FTP */
#ifdef USE_DAV
extern const struct proto_driver dav_driver;
#endif /* USE_DAV */
#ifdef USE_RSH
extern const struct proto_driver rsh_driver;
#endif /* USE_RSH */

extern const struct proto_driver vfs_driver;

const char *rc_get_netrc_password(const char *server, const char *username) {
    netrc_entry *found;
    found = search_netrc(netrc_list, server);
    if (found == NULL) {
	return NULL;
    }
    if (strcmp(found->account, username) == 0) {
	return found->password;
    } else {
	return NULL;
    }
}

/* Returns zero if site is properly defined, else non-zero */
int rcfile_verify(struct site *any_site) 
{
    struct stat localst;
    char *temp;
    int ret;

    /* Protocol-specific checks first, since if a new protocol driver is used,
     * any of the other checks may be irrelevant. */
    switch (any_site->protocol) {
    case siteproto_ftp:
#ifdef USE_FTP
	any_site->driver = &ftp_driver;
	/* FTP checks */
	if (any_site->symlinks == sitesym_maintain) {
	    return SITE_NOMAINTAIN;
	}
	break;
#else /* !USE_FTP */
	return SITE_UNSUPPORTED;
#endif /* USE_FTP */
    case siteproto_dav:
#ifdef USE_DAV
	any_site->driver = &dav_driver;
	/* HTTP checks */
	if (any_site->remote_isrel) { 
	    return SITE_NOREMOTEREL;
	}
	if (any_site->perms == sitep_all) {
	    return SITE_NOPERMS;
	}
	if (any_site->symlinks == sitesym_maintain) {
	    return SITE_NOMAINTAIN;
	}
	break;
#else /* !USE_DAV */
	return SITE_UNSUPPORTED;
#endif /* USE_DAV */
    case siteproto_rsh:
#ifdef USE_RSH
	any_site->driver = &rsh_driver;
	/* FIXME: rsh checks? */
	break;
#else /* !USE_RSH */
	return SITE_UNSUPPORTED;
#endif /* USE_RSH */
    case siteproto_vfs:
	any_site->driver = &vfs_driver;
	break;
    case siteproto_unknown:
	return SITE_UNSUPPORTED;
    }

    /* Valid options check */
    if (any_site->checkrenames && (any_site->state_method != state_checksum)) {
	return SITE_NORENAMES;
    }

    /* Check they specified everything in the rcfile */
    if (any_site->server.hostname == NULL) {
	return SITE_NOSERVER;
    } 

    if (any_site->server.username != NULL && any_site->server.password == NULL) {
	if (havenetrc) {
	    const char *pass;
	    NE_DEBUG(DEBUG_RCFILE, "Checking netrc for password for %s@%s...",
		   any_site->server.username, any_site->server.hostname);
	    pass = rc_get_netrc_password(any_site->server.hostname, 
					  any_site->server.username);
	    if (pass != NULL) {
		NE_DEBUG(DEBUG_RCFILE, "found!\n");
		any_site->server.password = (char *) pass;
	    } else {
		NE_DEBUG(DEBUG_RCFILE, "none found.\n");
	    }
	}
    }
    /* TODO: lookup proxy username/password in netrc too */

    if (any_site->remote_root_user == NULL) {
	return SITE_NOREMOTEDIR;
    } else if (any_site->local_root_user == NULL) {
	return SITE_NOLOCALDIR;
    }
    
    /* Need a home directory if we're using relative local root */
    if (home == NULL && any_site->local_root)
	return SITE_NOLOCALREL;

    /* Can't use safe mode and nooverwrite mode */
    if (any_site->safemode && any_site->nooverwrite)
	return SITE_NOSAFEOVER;

    if (any_site->safemode && any_site->tempupload)
	return SITE_NOSAFETEMPUP;

    if (any_site->remote_isrel) {
	any_site->remote_root = ne_strdup(any_site->remote_root_user + 2);
    } else {
	any_site->remote_root = ne_strdup(any_site->remote_root_user);
    }
    if (any_site->local_isrel) {
	/* We skip the first char ('~') of l_r_u */
	any_site->local_root = ne_concat(home, any_site->local_root_user + 1,
					 NULL);
    } else {
	any_site->local_root = any_site->local_root_user;
    }

    /* Now check the local directory actually exists.
     * To do this, stat `/the/local/root/.', which will fail if the
     * can't read the directory or if it's a file not a directory */
    temp = ne_concat(any_site->local_root, ".", NULL);
    ret = stat(temp, &localst);
    free(temp);
    if (ret != 0) {
	return SITE_ACCESSLOCALDIR;
    }

    /* Assign default ports if they didn't bother to */
    if (any_site->server.port == 0) {
	NE_DEBUG(DEBUG_RCFILE, "Lookup up default port:\n");
	any_site->server.port = (*any_site->driver->get_server_port)(any_site);
	NE_DEBUG(DEBUG_RCFILE, "Using port: %d\n", any_site->server.port);
    }

    if (any_site->proxy.port == 0) {
	NE_DEBUG(DEBUG_RCFILE, "Lookup default proxy port...\n");
	any_site->proxy.port = (*any_site->driver->get_proxy_port)(any_site);
	NE_DEBUG(DEBUG_RCFILE, "Using port %d\n", any_site->proxy.port);
    }

    /* TODO: ditto for proxy server */
    return 0;
}

int init_netrc() {
    if (!havenetrc) return 0;
    netrc_list = parse_netrc(netrcfile);
    if (netrc_list == NULL) {
	/* Couldn't parse it */
	return 1;
    } else {
	/* Could parse it */
	return 0;
    }
}

/* Checks the perms of the rcfile and site storage directory. */
int init_paths()
{
    struct stat st;
    if (stat(rcfile, &st) < 0) {
	NE_DEBUG(DEBUG_RCFILE, "stat failed on %s: %s\n", 
	       rcfile, strerror(errno));
	return RC_OPENFILE;
    }
#if !defined (__EMX__) && !defined(__CYGWIN__)
    if ((st.st_mode & ~(S_IFREG | S_IREAD | S_IWRITE)) > 0) {
	return RC_PERMS;
    }
#endif
    if ((netrcfile == 0) || (stat(netrcfile, &st) < 0)) {
	havenetrc = false;
#if !defined (__EMX__) && !defined(__CYGWIN__)
    } else if ((st.st_mode & ~(S_IFREG | S_IREAD | S_IWRITE)) > 0) {
	return RC_NETRCPERMS;
#endif
    } else {
	havenetrc = true;
    }
    if (stat(copypath, &st) < 0) {
	NE_DEBUG(DEBUG_RCFILE, "stat failed on %s: %s\n", 
	       copypath, strerror(errno));
	return RC_DIROPEN;
    }
#if !defined (__EMX__) && !defined(__CYGWIN__)
    if (st.st_mode & (S_IRWXG | S_IRWXO)) {
	return RC_DIRPERMS;
    }
#endif
    return 0;
}

int init_env() {
    /* Assign default filenames if they didn't give us any */
    home = getenv("HOME");
    if (home == NULL) {
	if ((rcfile == NULL) || (copypath == NULL)) {
	    /* We need a $HOME or both rcfile and info dir path */
	    return 1;
	} else {
	    /* No $HOME, but we've got the rcfile and info dir path */
	    return 0;
	}
    }
    if (rcfile == NULL) {
	rcfile = ne_concat(home, RCNAME, NULL);
    }
    if (copypath == NULL) {
	copypath = ne_concat(home, COPYNAME, NULL);
    }
    netrcfile = ne_concat(home, NETRCNAME, NULL);
    return 0;
}

