/*  Screem:  plugin.c,
 *  Handles loading of plugins
 *
 *  Copyright (C) 1999 - 2001  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gmodule.h>

#include <glib/gfileutils.h>
#include <glib/gmessages.h>

#include <libgnome/gnome-util.h>
#include <libgnome/gnome-i18n.h>

#include <gtk/gtklabel.h>

#include <string.h>

#include "screem-application.h"

#include "screem-plugin.h"

#include "screem-markup.h"

#include "fileops.h"

#define SPLASH_TEXT( w, t ) gtk_label_set_text( GTK_LABEL( w ),t );

#define REQUIRED_VERSION 4

/*
#define SPLASH_TEXT( w, t ) gtk_label_set_text( GTK_LABEL( w ),t ); \
while( g_main_context_pending(NULL) ) g_main_iteration(FALSE);
*/

typedef void(*Init)( ScreemPlugin *plugin );

typedef struct {
	ScreemPlugin *plugin;
	ScreemWindow *window;
	guint start;
	guint len;
	GSList *attrs;
} PopupData;

static void load_plugins( ScreemApplication *application,
			  GtkWidget *splash_text, 
			  gchar *directory );
static void plugin_popup( GtkWidget *widget, gpointer data );
static void destroy_attr_list( gpointer data, GClosure *closure );



void scan_plugins( ScreemApplication *application, GtkWidget *splash_text )
{
	const gchar *search_paths[ 7 ] = {
		PLUGIN_PATH,
		NULL,
                "/usr/lib/",
                "/usr/X11R6/lib",
		NULL,
                NULL
	};
	gint path = 0;
	gchar *directory = NULL;
	
	/* add these into search_paths here rather than in the declaration
	   as the values would be set at compile rather than runtime
	   otherwise */
	search_paths[ 1 ] = gnome_libdir_file( "" );
	search_paths[ 4 ] = g_getenv( "SCREEM_LIB_PATH" );

	/* global paths to look for the plugins are searched in order,
	   if plugins are found in one of them no more are checked */
	do {
		g_free( directory );
                directory = g_strdup_printf( "%sscreem%cplugins%c",
                                             search_paths[ path ],
                                             G_DIR_SEPARATOR,
                                             G_DIR_SEPARATOR );
                path ++;
        } while( ( search_paths[ path ] ) &&
                 ( ! g_file_test( directory, G_FILE_TEST_IS_DIR ) ) );

	/* now load the global plugins */
	load_plugins( application, splash_text, directory );

	g_free( directory );

	/* now load from the users $HOME/.screem directory */
	directory = screem_get_dot_dir();

	if( g_file_test( directory, G_FILE_TEST_IS_DIR ) )
		load_plugins( application, splash_text, directory );

	g_free( directory );
	g_free( (gchar*)search_paths[ 1 ] );
}

/* load the shared library plugins */
static gint plugin_filter( const GnomeVFSFileInfo *info )
{
	gint ret;
	const gchar *ext;

	ext = g_extension_pointer( info->name );
	ret = strcmp( "so", ext );

	return ret;
}

static void load_plugins( ScreemApplication *application, GtkWidget *splash_text,
			  gchar *directory )
{
	GSList *files;
	GSList *tmp;

	GModule *self;
	GModule *mod;

	ScreemPlugin *plugin;
	Init initfunc;
	gint *plugin_version;

	g_return_if_fail( directory != NULL );

	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	files = screem_vfs_scandir( directory, plugin_filter, 
					(GCompareFunc)g_strcasecmp,
					FALSE );
	for( tmp = files; tmp; tmp = tmp->next ) {
		SPLASH_TEXT( splash_text, tmp->data );

		if( ( mod = g_module_open( tmp->data, 
					G_MODULE_BIND_LAZY ) ) ) {
			Init *ret;
			gint **ver;

			ret = &initfunc;
			ver = &plugin_version;
			if( ! g_module_symbol( mod, "init",
						(gpointer*)ret ) ) {
				g_module_close( mod );
			} else if( g_module_symbol( mod, 
					"screem_plugin_version",
					 (gpointer*)ver ) &&
					   *plugin_version == REQUIRED_VERSION ) {
				plugin = g_new0( ScreemPlugin, 1 );
				plugin->module = mod;
				initfunc( plugin );

				/* now register plugin with the ScreemApplication */
				screem_application_register_plugin( application, plugin );
			}
		} else {
			g_warning( "PLUGIN ERROR: %s\n",
					g_module_error() );
		}
		g_free( tmp->data );
	}
	g_slist_free( files );
}

GModule *screem_plugin_get_sitecopy( ScreemApplication *application )
{
        GList *list;
        ScreemPlugin *plugin;
	GModule *module;

	module = NULL;

        for( list = application->plugins; list && ! module ; list = list->next ) {
                plugin = (ScreemPlugin*)list->data;
                if( plugin->tag && ! strcmp( plugin->tag, "sitecopy_hack" ) )
			module = plugin->module;
        }

        return module;
}

GtkWidget *screem_plugin_tag_wizard( ScreemView *view,
				     const gchar *tag, 
				     const gchar *fulltag,
				     guint start, guint len )
{
	ScreemApplication *application;
	ScreemWindow *window;
	GtkWidget *menu;
	GList *list;
	ScreemPlugin *plugin;

	menu = NULL;

	g_object_get( G_OBJECT( view ),
			"window", &window,
			"app", &application,
			NULL );
	
	plugin = NULL;	
	for( list = application->plugins; list; list = list->next ) {
		plugin = (ScreemPlugin*)list->data;
		if( plugin->tag && ! g_strcasecmp( plugin->tag, tag ) ) {
			break;
		}
	}
	g_object_unref( application );

	if( list && plugin->popup ) {
		PopupData *data;

		data = g_new0( PopupData, 1 );
		data->plugin = plugin;
		data->window = window;
		data->start = start;
		data->len = len;
		data->attrs = screem_markup_build_attributes_list( fulltag,
								   NULL );

		/* we found a wizard for this tag */
		menu = gtk_menu_item_new_with_label( plugin->name );
		gtk_widget_show( menu );
		g_signal_connect_data( G_OBJECT( menu ), "activate",
				       G_CALLBACK( plugin_popup ),
				       data, destroy_attr_list, 0 );
	}

	return menu;
}

static void plugin_popup( GtkWidget *menuitem, gpointer data )
{
	PopupData *popup;

	popup = (PopupData*)data;

	popup->plugin->popup( popup->window, 
			      popup->start, popup->len, popup->attrs );
}

static void destroy_attr_list( gpointer data, GClosure *closure )
{
	PopupData *popup_data;
	GSList *list;

	popup_data = (PopupData*)data;

	list = popup_data->attrs;
	g_slist_foreach( list, (GFunc)g_free, NULL );
	g_slist_free( list );
	g_free( popup_data );
}
