/*  Screem:  screem-mdi.c
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include "config.h"

#include <glib/gi18n.h>
#include <libgnome/gnome-macros.h>

#include <libgnomeui/gnome-popup-menu.h>

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>

#include <gconf/gconf-client.h>

#include <glade/glade.h>

#include <gtk/gtk.h>
#include <gtksourceview/gtksourceview.h>

#include <glib/glist.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "screem-file-browser.h"
#include "screem-mdi.h"
#include "screem-page.h"
#include "screem-window.h"

#include "support.h"

#include "screemmarshal.h"

#include "ephy-event-box.h"

#include "gdl-stock.h"

static void screem_mdi_class_init( ScreemMDIClass *klass );
static void screem_mdi_instance_init( ScreemMDI *mdi );
static void screem_mdi_finalize( GObject *object );
static void screem_mdi_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec );
static void screem_mdi_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec );


static void screem_mdi_window_set( ScreemMDI *mdi );
static void screem_mdi_switch_doc( GtkAction *action, gpointer data );
static GtkWidget *screem_mdi_create_tab( ScreemMDI *mdi, 
					 ScreemPage *page );
static gboolean screem_mdi_notebook_add( ScreemMDI *mdi, 
					 ScreemPage *page );
static void screem_mdi_page_path_change( ScreemPage *page, 
		GParamSpec *spec, gpointer data );
static void screem_mdi_page_change( ScreemPage *page, 
				GParamSpec *spec, gpointer data );
static void screem_mdi_notebook_change( GtkNotebook *notebook, 
					GtkNotebookPage *p,
					guint num,
					ScreemMDI *mdi );
static void screem_mdi_tab_clicked( GtkWidget *label, 
				GdkEventButton *event,
				ScreemMDI *mdi );
static void screem_mdi_tab_popup_switch( GtkWidget *widget, 
					 gpointer data );
static void screem_mdi_real_remove_document( ScreemMDI *mdi,
					     ScreemPage *page,
					     gint index);
static void screem_mdi_document_dialog( GtkAction *action,
					gpointer data );
static void screem_mdi_documents_change( GtkTreeSelection *sel,
					 gpointer data );

static void screem_mdi_can_undo( GtkSourceBuffer *buffer,
				 gboolean can_undo,
				 gpointer data );
static void screem_mdi_can_redo( GtkSourceBuffer *buffer,
				 gboolean can_redo,
				 gpointer data );
static void screem_mdi_close_button( GtkWidget *widget,
				ScreemMDI *mdi );


enum {
	PROP_0,
	PROP_WINDOW
};

enum {
	CHANGED,
	LAST_SIGNAL
};

static guint screem_mdi_signals[ LAST_SIGNAL ] = { 0 };

static ScreemFileBrowser *browser = NULL;

enum {
	MDI_PATHNAME,
	MDI_PIXBUF,
	MDI_MODIFIED,
	MDI_PAGE,
	MDI_MAX_COLS
};

struct ScreemMDIPrivate {

	GList *documents;
	GtkNotebook *notebook;
	GtkTooltips *tip;

	ScreemWindow *window;
	GtkUIManager *merge;
	GtkActionGroup *group;
	
	ScreemPage *current;

	guint uiid;
	guint docs;

	GtkWidget *dialog;

	GtkListStore *store;
};

GNOME_CLASS_BOILERPLATE( ScreemMDI, screem_mdi, GObject, G_TYPE_OBJECT )

static void screem_mdi_class_init( ScreemMDIClass *klass )
{
	GObjectClass *obj_class;
	GParamSpec *spec;
	
	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_mdi_finalize;
	obj_class->get_property = screem_mdi_get_prop;
	obj_class->set_property = screem_mdi_set_prop;


	spec = g_param_spec_object( "window", "Window", "The Window",
				G_TYPE_OBJECT,
				G_PARAM_READWRITE |
				G_PARAM_CONSTRUCT );

	g_object_class_install_property(obj_class, PROP_WINDOW, spec );

	screem_mdi_signals[ CHANGED ] = 
		g_signal_new( "changed",
			      G_OBJECT_CLASS_TYPE( obj_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemMDIClass, 
					       changed),
			      NULL, NULL,
			      screem_marshal_VOID__OBJECT_OBJECT,
			      G_TYPE_NONE, 2,
			      G_TYPE_OBJECT,
			      G_TYPE_OBJECT );
}

static void screem_mdi_instance_init( ScreemMDI *mdi )
{
	ScreemMDIPrivate *priv;
	GtkPositionType position;
	
	priv = mdi->priv = g_new0( ScreemMDIPrivate, 1 );

	priv->tip = gtk_tooltips_new();

	priv->notebook = GTK_NOTEBOOK( gtk_notebook_new() );
	gtk_notebook_set_scrollable( priv->notebook, TRUE );
	position = screem_get_tab_position();
	gtk_notebook_set_tab_pos( priv->notebook, position );
	
	gtk_notebook_set_show_border( priv->notebook, FALSE );
	g_signal_connect( G_OBJECT( priv->notebook ), "switch_page",
			  G_CALLBACK( screem_mdi_notebook_change ),
			  mdi );
	g_object_set_data( G_OBJECT( priv->notebook ), "mdi", mdi );
	gtk_widget_show( GTK_WIDGET( priv->notebook ) );

	priv->group = gtk_action_group_new( "ScreemMDI" );
	gtk_action_group_set_translation_domain( priv->group, 
			GETTEXT_PACKAGE );	

	priv->store = gtk_list_store_new( MDI_MAX_COLS,
					  G_TYPE_STRING,
					  GDK_TYPE_PIXBUF,
					  G_TYPE_STRING,
					  G_TYPE_POINTER );

	if( ! browser ) {
		browser = screem_file_browser_new();
	}
}

static void screem_mdi_finalize( GObject *object )
{
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_MDI( object ) );

	mdi = SCREEM_MDI( object );

	priv = mdi->priv;

/*	we should only be called from the ScreemWindow finalize
 	signal so don't bother doing these
  	
  	gtk_ui_manager_remove_action_group( priv->merge, priv->group );
	gtk_ui_manager_remove_ui( priv->merge, priv->uiid );*/

	g_object_unref( priv->group );
	
	g_list_free( priv->documents );

	if( priv->dialog ) {
		gtk_widget_destroy( priv->dialog );
	}
	g_object_unref( priv->store );
	
	g_free( priv );
	
	GNOME_CALL_PARENT( G_OBJECT_CLASS, finalize, (object) );
}

static void screem_mdi_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec )
{
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	
	mdi = SCREEM_MDI( object );
	priv = mdi->priv;

	switch( prop_id ) {
		case PROP_WINDOW:
			priv->window = g_value_get_object( value );
			screem_mdi_window_set( mdi );
			break;
		default:
			break;
	}
}

static void screem_mdi_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec )
{
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	
	mdi = SCREEM_MDI( object );
	priv = mdi->priv;

	switch( prop_id ) {
		case PROP_WINDOW:
			g_value_set_object( value, priv->window );
			break;
		default:
			break;
	}
}

/* static stuff */
static void screem_mdi_window_set( ScreemMDI *mdi )
{
	ScreemMDIPrivate *priv;
	GConfClient *client;
	GString *str;
	const gchar *start = "<ui><menubar><menu action=\"Documents\"><placeholder name=\"DocumentNav\">";
	const gchar *pattern = "<menuitem action=\"%s\" />";
	const gchar *end = "</placeholder></menu></menubar></ui>";
	gint docs;
	gint i;
	GtkActionEntry *entries;
	GtkAction *action;	
	gchar *name;
	
	priv = mdi->priv;

	priv->merge = GTK_UI_MANAGER( priv->window->merge );
	
	client = gconf_client_get_default();
	
	priv->docs = docs = gconf_client_get_int( client, 
				"/apps/screem/general/doclistsize",
				NULL );

	if( docs < 1 ) {
		priv->docs = docs = 12;
	}
	str = g_string_new( start );

	g_string_append( str, "<menuitem action=\"SwitchDocManage\"/>" );
	g_string_append( str, "</placeholder><placeholder name=\"DocumentsBottom\">" );
	
	/* + 1 so we can have a more... entry to show
	   a dialog with a list of all documents */
	entries = g_new0( GtkActionEntry, docs + 1 );

	entries[ docs ].name = "SwitchDocManage";
	entries[ docs ].label = _( "_Manage Documents" );
	entries[ docs ].tooltip = _( "Show a dialog listing all documents open in this window," );
	entries[ docs ].stock_id = NULL;
	entries[ docs ].callback = G_CALLBACK( screem_mdi_document_dialog );
	for( i = 0; i < docs; ++ i ) {
		name = g_strdup_printf( "SwitchDoc%i", i );
		g_string_append_printf( str, pattern, name );
		entries[ i ].name = name;
		entries[ i ].label = _( "No Document" );
		entries[ i ].tooltip = _( "No Document" );
		entries[ i ].stock_id = NULL;
		if( i < 9 ) {
			entries[ i ].accelerator = g_strdup_printf( "<Alt>%i", i + 1 );
		}
		entries[ i ].callback = G_CALLBACK( screem_mdi_switch_doc );
	}
	
	g_string_append( str, end );

	gtk_action_group_add_actions( priv->group, entries, 
					(guint)docs + 1, mdi );
	for( i = 0; i < docs; ++ i ) {
		name = g_strdup_printf( "SwitchDoc%i", i );
		action = gtk_action_group_get_action( priv->group,
						name );
		g_object_set( G_OBJECT( action ),
				"visible", FALSE,
				NULL );
		g_free( name );

		/* sneak this in here to avoid an extra for loop */
		g_free( (gchar*)entries[ i ].name );
		g_free( (gchar*)entries[ i ].accelerator );
	}
	g_free( entries );
	
	gtk_ui_manager_insert_action_group( priv->merge, priv->group,
					    1 );
	priv->uiid = gtk_ui_manager_add_ui_from_string( priv->merge,
							str->str,
							str->len,
							NULL );

	action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "PreviousDocument" );
	g_object_set( G_OBJECT( action ),
			"sensitive", FALSE, NULL );
	action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "NextDocument" );
	g_object_set( G_OBJECT( action ),
			"sensitive", FALSE, NULL );
	
	g_string_free( str, TRUE );
	
	g_object_unref( client );
}

static void screem_mdi_switch_doc( GtkAction *action, gpointer data )
{
	ScreemMDI *mdi;
	GList *tmp;
	gint i;
	gchar *name;
	
	mdi = SCREEM_MDI( data );

	g_object_get( G_OBJECT( action ), "name", &name, NULL );
	
	sscanf( name, "SwitchDoc%i", &i );
	g_free( name );
	
	tmp = g_list_nth( mdi->priv->documents, i );
	if( tmp ) {
		screem_mdi_set_current( mdi, SCREEM_PAGE( tmp->data ) );
	}
}

static GtkWidget *screem_mdi_create_tab( ScreemMDI *mdi, 
					 ScreemPage *page )
{
	ScreemMDIPrivate *priv;
	GtkWidget *label;
	GtkWidget *clabel;
	GtkWidget *box;
	GtkWidget *event;

	GtkWidget *tbox;
	GtkWidget *close;
	GtkWidget *img;
	gint w;
	gint h;
	
	const gchar *pathname;
	gchar *tiptext;
	
	g_return_val_if_fail( SCREEM_IS_MDI( mdi ), NULL );
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );
	
	priv = mdi->priv;
	
	pathname = screem_page_get_pathname( page );
	
	label = gtk_label_new( "" );
	clabel = gtk_label_new( "" );
		
	g_object_set_data( G_OBJECT( label ), "mdi", mdi );
	g_object_set_data( G_OBJECT( clabel ), "mdi", mdi );
		
	/* we want to know if the page pathname changed */
	g_signal_connect( G_OBJECT( page ), "notify::pathname",
			  G_CALLBACK( screem_mdi_page_path_change ), 
			  label );
	g_signal_connect( G_OBJECT( page ), "notify::changed",
			  G_CALLBACK( screem_mdi_page_change ), 
			  clabel );
	g_signal_connect( G_OBJECT( page ), "notify::mime-type",
			  G_CALLBACK( screem_mdi_page_change ), 
			  clabel );
	
	g_signal_connect( G_OBJECT( page ), "can_undo",
			  G_CALLBACK( screem_mdi_can_undo ),
			  mdi );
	g_signal_connect( G_OBJECT( page ), "can_redo",
			  G_CALLBACK( screem_mdi_can_redo ),
			  mdi );
						
	gtk_widget_show( label );
	gtk_widget_show( clabel );
	event = ephy_event_box_new();
	
	tbox = gtk_hbox_new( 0, FALSE );
	gtk_box_set_spacing( GTK_BOX( tbox ), 6 );
	gtk_box_pack_start( GTK_BOX( tbox ), event, TRUE, TRUE, 0 );
	
	close = gtk_button_new();
	img = gtk_image_new_from_stock( GDL_STOCK_CLOSE,
					GTK_ICON_SIZE_MENU );
	gtk_icon_size_lookup_for_settings( gtk_widget_get_settings( img ),
			GTK_ICON_SIZE_MENU, &w, &h );
	gtk_container_add( GTK_CONTAINER( close ), img );
	gtk_button_set_relief( GTK_BUTTON( close ), GTK_RELIEF_NONE );
	gtk_widget_show_all( close );
	gtk_box_pack_start( GTK_BOX( tbox ), close, FALSE, FALSE, 0 );
	gtk_widget_set_size_request( close, w + 2, h + 2 );
	g_object_set_data( G_OBJECT( close ), "page", page );
	g_signal_connect( G_OBJECT( close ), "clicked",
			G_CALLBACK( screem_mdi_close_button ),
			mdi );
	
	box = gtk_hbox_new( 0, FALSE );
	img = gtk_image_new_from_stock( GTK_STOCK_NEW,
			GTK_ICON_SIZE_MENU );
	g_object_set_data( G_OBJECT( clabel ), "img", img );
	gtk_box_pack_start( GTK_BOX( box ), img, FALSE, FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( box ), label, TRUE, TRUE, 0 );
	gtk_box_pack_start( GTK_BOX( box ), clabel, FALSE, FALSE, 0 );
	gtk_widget_show_all( box );
	gtk_container_add( GTK_CONTAINER( event ), box );
	g_object_set_data( G_OBJECT( event ), "document", 
			   page );
	g_signal_connect( G_OBJECT( event ), 
			  "button_press_event",
			  G_CALLBACK( screem_mdi_tab_clicked ),
			  mdi );
	

	/* set tooltip to full pathname */
	if( ! pathname ) {
		pathname = _( "Unsaved Document" );
	}
	tiptext = gnome_vfs_unescape_string_for_display( pathname );
	gtk_tooltips_set_tip( priv->tip, event, tiptext, 
			      NULL );
	g_free( tiptext );

	gtk_widget_show_all( tbox );
	
	screem_mdi_page_path_change( page, NULL, label );
	screem_mdi_page_change( page, NULL, clabel );
	
	return tbox;
}

static gboolean screem_mdi_notebook_add( ScreemMDI *mdi, 
					 ScreemPage *page )
{
	ScreemApplication *app;
	ScreemMDIPrivate *priv;
	GtkWidget *tab;
	GtkWidget *hack;
	const gchar *pathname;
	gboolean loaded;
	gchar *msg;
	GError *error;

	priv = mdi->priv;
	
	pathname = screem_page_get_pathname( page );
	
	if( priv->window && pathname ) {
		msg = g_strdup_printf( _( "Loading page: %s" ), pathname );
		screem_window_show_message( priv->window, msg, FALSE );
		g_free( msg );
		app = priv->window->application;
		screem_application_set_cursor( app, GDK_WATCH );
	}
	
	g_signal_handlers_block_matched( G_OBJECT( priv->notebook ),
					G_SIGNAL_MATCH_FUNC|
					G_SIGNAL_MATCH_DATA,
					0, 0, NULL, 
					screem_mdi_notebook_change,
					mdi );

	if( ! pathname || screem_page_load( page, &error ) ) {
		tab = screem_mdi_create_tab( mdi, page );
		hack = gtk_hbox_new( 0, FALSE );
		gtk_widget_show( hack );
		gtk_notebook_append_page( GTK_NOTEBOOK( priv->notebook ),
					  hack, tab);
		gtk_widget_set_size_request( hack, 0, 0 );
		
		screem_page_set_open( page, TRUE );
		loaded = TRUE;

		if( priv->window && pathname ) {
			msg = g_strdup_printf( _( "Loaded page: %s" ), pathname );
			screem_window_show_message( priv->window, msg,
					FALSE );
			g_free( msg );
			app = priv->window->application;
			screem_application_set_cursor( app, 
					GDK_LAST_CURSOR );
		}
	} else {
		/* failed to open */
		gchar *temp;
		gchar *primary;
		const gchar *secondary;
			
		if( priv->window && pathname ) {
			msg = g_strdup_printf( _( "Failed to load page: %s" ), pathname );
			screem_window_show_message( priv->window, msg,
					FALSE );
			g_free( msg );
			app = priv->window->application;
			screem_application_set_cursor( app, 
					GDK_LAST_CURSOR );
		}
		
		if( pathname ) {
			temp = gnome_vfs_unescape_string_for_display( pathname );
			secondary = error->message;
		} else {
			temp = g_strdup( "" );
			secondary = "";
		}
		primary = g_strdup_printf( _( "Failed to load %s" ),
				temp );
		screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
				primary, secondary, 
				GTK_WIDGET( priv->window ) );
		g_free( primary );
	
		if( error ) {
			g_error_free( error );
		}
		
		loaded = FALSE;
	}

	g_signal_handlers_unblock_matched( G_OBJECT( priv->notebook ),
					G_SIGNAL_MATCH_FUNC|
					G_SIGNAL_MATCH_DATA,
					0, 0, NULL, 
					screem_mdi_notebook_change,
					mdi );
	
	return loaded;
}

static void screem_mdi_get_page_labels( ScreemPage *page,
					gchar **fname,
					gchar **pname )
{
	const gchar *pathname;
	gchar *filename;
	gchar *temp;
	GnomeVFSURI *uri;

	pathname = screem_page_get_pathname( page );
	
	if( pathname && ( strlen( pathname ) > 0 ) ) {
		uri = gnome_vfs_uri_new( pathname );
		filename = gnome_vfs_uri_extract_short_name( uri );
		gnome_vfs_uri_unref( uri );
		temp = gnome_vfs_unescape_string_for_display( filename );
		g_free( filename );
		filename = temp;
	} else {
		filename = g_strdup( _( "Untitled" ) );
		pathname = _( "Unsaved Document" );
	}
		
	temp = gnome_vfs_unescape_string_for_display( pathname );

	if( fname ) {
		*fname = filename;
	} else {
		g_free( filename );
	}
	if( pname ) {
		*pname = temp;
	} else {
		g_free( temp );
	}
}

static void screem_mdi_page_path_change( ScreemPage *page, 
		GParamSpec *spec, gpointer data )
{
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	GtkWidget *label;
	ScreemWindow *window;
	GtkWidget *event;
	GtkTooltipsData *ttip;
	gchar *filename;
	gchar *title;
	gchar *temp;
	gint i;
	gchar *name;
	GtkAction *action;
	GtkTreeIter it;
	gboolean modified;
	ScreemPage *cpage;
	GdkPixbuf *pixbuf;
	const gchar *pathname;
	
	label = GTK_WIDGET( data );
	
	mdi = SCREEM_MDI( g_object_get_data( G_OBJECT( label ), "mdi" ) );
	priv = mdi->priv;
	window = priv->window;

	event = GTK_WIDGET( data )->parent;

	pathname = screem_page_get_pathname( page );
	modified = screem_page_get_changed( page );
	screem_mdi_get_page_labels( page, &filename, &temp );
	
	gtk_label_set_text( GTK_LABEL( label ), filename );
		
	if( spec ) {
		/* coming from a notify signal, the page is
		   already open */
		i = g_list_index( priv->documents, page );
	} else {
		/* currently adding a new page, so the
		 * index will be the same as the current list length */
		i = g_list_length( priv->documents );
	}
	if( i < priv->docs ) {
		name = g_strdup_printf( "SwitchDoc%i", i );
		action = gtk_action_group_get_action( priv->group,
							name );
		g_free( name );
		name = screem_escape_underlines( filename ); 
		g_object_set( G_OBJECT( action ),
				"label", name,
				"tooltip", temp,
				"visible", TRUE,
				NULL );
		g_free( name );
	}

	if( spec && 
	    gtk_tree_model_get_iter_first( GTK_TREE_MODEL(priv->store),
		    			   &it ) ) {
		gtk_tree_model_get( GTK_TREE_MODEL( priv->store ),
				    &it, MDI_PAGE, &cpage, -1 );
		while( page != cpage &&
			gtk_tree_model_iter_next( GTK_TREE_MODEL( priv->store ), &it ) )  {
			gtk_tree_model_get( GTK_TREE_MODEL( priv->store ),
					    &it, MDI_PAGE, &cpage, -1 );
		}
		g_assert( page == cpage );
	} else {
		gtk_list_store_append( priv->store, &it );	
	}
	pixbuf = NULL;
	if( pathname ) {
		pixbuf = screem_file_browser_get_icon( browser, 
					pathname, 16, 16, FALSE,
					NULL );
	}
	gtk_list_store_set( priv->store, &it,
			MDI_PATHNAME, temp,
			MDI_MODIFIED, modified ? "*" : "",
			MDI_PIXBUF, pixbuf,
			MDI_PAGE, page,
			-1 );
	if( pixbuf ) {
		g_object_unref( G_OBJECT( pixbuf ) );
	}
	
	g_free( filename );
	
	if( event ) {
		event = event->parent;

		ttip = gtk_tooltips_data_get( event );
		gtk_tooltips_set_tip( ttip->tooltips, event,
					temp, "" );
	}

	if( modified ) {
		title = g_strdup_printf( "%s * - Screem", temp );
	} else {
		title = g_strdup_printf( "%s - Screem", temp );
	}

	if( page == priv->current ) {
		gtk_window_set_title( GTK_WINDOW( window ), title );
	}
	g_free( temp );
	g_free( title );
}

static void screem_mdi_page_change( ScreemPage *page, 
				GParamSpec *spec, gpointer data )
{
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	GtkWidget *label;
	GtkWidget *img;
	GdkPixbuf *pixbuf;
	ScreemWindow *window;
	const gchar *pathname;
	gchar *temp;
	gchar *title;
	gboolean modified;
	GtkTreeIter it;
	ScreemPage *cpage;

	const gchar *type;
	const gchar *showtype;
	
	label = GTK_WIDGET( data );
	img = g_object_get_data( G_OBJECT( label ), "img" );
	mdi = SCREEM_MDI( g_object_get_data( G_OBJECT( label ), "mdi" ) );
	priv = mdi->priv;
	window = priv->window;
	
	type = screem_page_get_mime_type( page );
	showtype = gnome_vfs_mime_get_description( type );
	if( ! showtype ) {
		showtype = type;
	}

	pixbuf = screem_file_browser_get_icon( browser,
			NULL, 16, 16, FALSE, (gchar**)&type );
	if( pixbuf ) {
		gtk_image_set_from_pixbuf( GTK_IMAGE( img ), pixbuf );
		g_object_unref( pixbuf );
	}

	pathname = screem_page_get_pathname( page );
	if( ! pathname ) {
		pathname = _( "Untitled" );
		temp = g_strconcat( pathname, " ", showtype, NULL );
	} else {
		temp = gnome_vfs_unescape_string_for_display( pathname );
	}
	
	modified = screem_page_get_changed( page );
		
	if( modified ) {
		gtk_label_set_text( GTK_LABEL( label ), "*" );
		title = g_strdup_printf( "* %s - Screem", temp );
	} else {
		gtk_label_set_text( GTK_LABEL( label ), "" );
		title = g_strdup_printf( "%s - Screem", temp );
	}

	if( page == priv->current ) {
		gtk_window_set_title( GTK_WINDOW( window ), title );
	}

	if( spec && 
	    gtk_tree_model_get_iter_first( GTK_TREE_MODEL(priv->store),
		    			   &it ) ) {
		gtk_tree_model_get( GTK_TREE_MODEL( priv->store ),
				    &it, MDI_PAGE, &cpage, -1 );
		while( page != cpage &&
			gtk_tree_model_iter_next( GTK_TREE_MODEL( priv->store ), &it ) ) {
			gtk_tree_model_get( GTK_TREE_MODEL( priv->store ),
					    &it, MDI_PAGE, &cpage, -1 );
		}
		g_assert( page == cpage );
		gtk_list_store_set( priv->store, &it,
				MDI_MODIFIED, modified ? "*" : "",
				-1 );
	}	
	g_free( temp );
	g_free( title );
}

static void screem_mdi_notebook_change( GtkNotebook *notebook, 
					GtkNotebookPage *p,
					guint num,
					ScreemMDI *mdi )
{
	ScreemMDIPrivate *priv;
	ScreemWindow *window;
        ScreemPage *page;
        GList *list;
	
	priv = mdi->priv;
	window = priv->window;

	list = g_list_nth( priv->documents, num );

	if( list ) {
		page = SCREEM_PAGE( list->data );
		if( page != screem_mdi_get_current( mdi ) ) {
			screem_mdi_set_current( mdi, page );
		}
	} else {
		screem_mdi_set_current( mdi, NULL );
	}
}

static void screem_mdi_tab_clicked( GtkWidget *label, 
				GdkEventButton *event,
				ScreemMDI *mdi )
{
	ScreemMDIPrivate *priv;
        ScreemPage *page;
        ScreemWindow *window;
        gint num;

	priv = mdi->priv;
	window = priv->window;
	
        page = SCREEM_PAGE( g_object_get_data( G_OBJECT( label ),
					       "document" ) );

        num = g_list_index( priv->documents, page );
        if( ( ! event ) || event->button != 3 ) {
                gtk_notebook_set_current_page( priv->notebook, num ); 
	} else {
                /* right click on page tab, show popup */
		GtkWidget *menu;
		GtkWidget *submenu;
		GtkWidget *menuitem;
		GList *docs;
		
		menu = gtk_ui_manager_get_widget( priv->merge,
						"/tabmenu" );

		g_object_set_data( G_OBJECT( window ), 
				"tab_popup", page );
		
		/* add open pages menu */
		menuitem = gtk_menu_item_new_with_label( _( "Opened Pages" ) );
		gtk_widget_show( menuitem );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), menuitem );
		submenu = gtk_menu_new();
		gtk_menu_item_set_submenu( GTK_MENU_ITEM( menuitem ), submenu );
		
		for( docs = priv->documents; docs; docs = docs->next ) {
			GtkWidget *submenuitem;
			const gchar *pathname;
			gchar *base;
			gchar *temp;
			
			page = SCREEM_PAGE( docs->data );
			pathname = screem_page_get_pathname( page );
			
			if( ! pathname ) {
				pathname = _( "Untitled" );
			}
			base = g_path_get_basename( pathname );
			temp = gnome_vfs_unescape_string_for_display( base );
			submenuitem = gtk_menu_item_new_with_label( temp );
			g_free( temp );
			gtk_widget_show( submenuitem );
			g_object_set_data( G_OBJECT( submenuitem ), "document",
					   page );
			g_signal_connect( G_OBJECT( submenuitem ), "activate",
					G_CALLBACK( screem_mdi_tab_popup_switch ),
					mdi );
			gtk_menu_shell_append( GTK_MENU_SHELL( submenu ),
						submenuitem );
			g_free( base );
		}
		
		screem_popup_menu_do_popup_modal( menu, 0, 0, event, 0, label );
		g_object_set_data( G_OBJECT( window ), "tab_popup", NULL );
		
		/* remove menuitem from menu as we don't want the
		   opened pages submenu staying here */
		gtk_container_remove( GTK_CONTAINER( menu ), menuitem );
        }
}

static void screem_mdi_tab_popup_switch( GtkWidget *widget, 
					 gpointer data )
{
	screem_mdi_tab_clicked( widget, NULL, data );	
}

static void screem_mdi_real_remove_document( ScreemMDI *mdi,
					     ScreemPage *page,
					     gint index)
{
	ScreemMDIPrivate *priv;
	
	GList *tmp;
	gchar *name;
	gchar *filename;
	gchar *pathname;
	GtkAction *action;
	GtkTreeIter it;
	ScreemPage *cpage;
	
	g_return_if_fail( SCREEM_IS_MDI( mdi ) );
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	priv = mdi->priv;

	g_signal_handlers_disconnect_matched( G_OBJECT( page ),
					      G_SIGNAL_MATCH_FUNC, 
					      0, 0, NULL, 
					      screem_mdi_page_path_change,
					      NULL );
	g_signal_handlers_disconnect_matched( G_OBJECT( page ),
					      G_SIGNAL_MATCH_FUNC,
					      0, 0, NULL, 
					      screem_mdi_page_change,
					      NULL );
	g_signal_handlers_disconnect_matched( G_OBJECT( page ),
					      G_SIGNAL_MATCH_FUNC,
					      0, 0, NULL, 
					      screem_mdi_can_undo,
					      NULL );
	g_signal_handlers_disconnect_matched( G_OBJECT( page ),
					      G_SIGNAL_MATCH_FUNC,
					      0, 0, NULL, 
					      screem_mdi_can_redo,
					      NULL );
		
	priv->documents = g_list_remove( priv->documents, page );
	if( page == priv->current ) {
		priv->current = NULL;
	}
	gtk_notebook_remove_page( priv->notebook, index );

	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL(priv->store),
		    			   &it ) ) {
		gtk_tree_model_get( GTK_TREE_MODEL( priv->store ),
				    &it, MDI_PAGE, &cpage, -1 );
		while( page != cpage &&
			gtk_tree_model_iter_next( GTK_TREE_MODEL( priv->store ), &it ) )  {
			gtk_tree_model_get( GTK_TREE_MODEL( priv->store ),
					    &it, MDI_PAGE, &cpage, -1 );
		}
		g_assert( page == cpage );
		gtk_list_store_remove( priv->store, &it );
	}

	
	if( index < priv->docs ) {
		/* we need to shuffle the documents menu */
		tmp = g_list_nth( priv->documents, index );
		while( tmp && ( index < priv->docs ) ) {
			page = SCREEM_PAGE( tmp->data );
		
			screem_mdi_get_page_labels( page, 
						&filename,
						&pathname );
			
			name = g_strdup_printf( "SwitchDoc%i", index );
			action = gtk_action_group_get_action( priv->group,
								name );
			g_free( name );
			name = screem_escape_underlines( filename ); 
			g_object_set( G_OBJECT( action ),
					"label", name,
					"tooltip", pathname,
					"visible", TRUE,
					NULL );
			
			g_free( filename );
			g_free( pathname );
			g_free( name );
			tmp = tmp->next;
			++ index;
		}
		/* may have run out of docs before priv->docs was
		   reached */
		while( index < priv->docs ) {
			name = g_strdup_printf( "SwitchDoc%i", index );
			action = gtk_action_group_get_action( priv->group,
								name );
			g_object_set( G_OBJECT( action ),
					"label", _( "No Document" ),
					"tooltip", _( "No Document" ),
					"visible", FALSE,
					NULL );
			g_free( name );
			++ index;
		}
	}
	/* we don't get a changed signal when the last
	   page is closed, so we need handle it here */
	if( ! priv->documents ) {
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "PreviousDocument" );
		g_object_set( G_OBJECT( action ),
				"sensitive", FALSE,
				NULL );
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "NextDocument" );
		g_object_set( G_OBJECT( action ),
				"sensitive", FALSE,
				NULL );
		g_signal_emit( G_OBJECT( mdi ), 
				screem_mdi_signals[ CHANGED ],
				0, NULL, page );
	}
}

static void screem_mdi_document_dialog_response( GtkWidget *dialog,
		gint id, gpointer data )
{
	screem_session_store_dialog( SCREEM_SESSION( data ), dialog );
}

static void screem_mdi_document_dialog( GtkAction *action,
					gpointer data )
{
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	ScreemApplication *app;
	ScreemSession *session;
	gchar *glade_path;
	GladeXML *xml;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkWidget *widget;
	GtkTreeSelection *sel;
	
	mdi = SCREEM_MDI( data );
	priv = mdi->priv;
	
	app = SCREEM_APPLICATION( priv->window->application );
	session = screem_application_get_session( app );
	
	if( ! priv->dialog ) {
		glade_path = screem_get_glade_path();
		xml = glade_xml_new( glade_path, "documents", NULL );
		g_free( glade_path );

		priv->dialog = glade_xml_get_widget( xml, "documents" );
		g_signal_connect( G_OBJECT( priv->dialog ),
				"response",
				G_CALLBACK( screem_mdi_document_dialog_response ),
				session );
		g_signal_connect( G_OBJECT( priv->dialog ),
				"delete_event",
				G_CALLBACK( gtk_widget_destroy ),
				NULL );
		g_signal_connect_swapped( G_OBJECT( priv->dialog ),
				"delete_event",
				G_CALLBACK( g_nullify_pointer ),
				&priv->dialog );
		g_signal_connect_swapped( G_OBJECT( priv->dialog ),
				"destroy",
				G_CALLBACK( g_nullify_pointer ),
				&priv->dialog );
	
		widget = glade_xml_get_widget( xml, "documentlist" );
		sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
		gtk_tree_selection_set_mode( sel, 
					GTK_SELECTION_MULTIPLE );
		g_signal_connect( G_OBJECT( sel ), "changed",
				  G_CALLBACK( screem_mdi_documents_change ),
				  mdi );
		
		col = gtk_tree_view_column_new();
		renderer = gtk_cell_renderer_text_new();
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_column_set_resizable( col, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), 
						col );
		gtk_tree_view_column_set_attributes( col, renderer,
						"text",
						MDI_MODIFIED,
						NULL );
		
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( col, "Filename" );
		
		renderer = gtk_cell_renderer_pixbuf_new();
		gtk_tree_view_column_pack_start( col, renderer, FALSE );
		gtk_tree_view_column_set_attributes( col, renderer,
						"pixbuf",
						MDI_PIXBUF,
						NULL );
	       
		renderer = gtk_cell_renderer_text_new();
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_column_set_resizable( col, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), 
						col );
		gtk_tree_view_column_set_attributes( col, renderer,
						"text",
						MDI_PATHNAME,
						NULL );

		gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
			GTK_TREE_MODEL( priv->store ) );
		g_object_set_data( G_OBJECT( widget ), "mdi", mdi );
		
		glade_xml_signal_autoconnect( xml );
		screem_session_restore_dialog( session, priv->dialog );
	}
	if( ! GTK_WIDGET_VISIBLE( priv->dialog ) ) {
		screem_session_restore_dialog( session, priv->dialog );
	}
	gtk_widget_show( priv->dialog );
}

static void screem_mdi_documents_change( GtkTreeSelection *sel,
					 gpointer data )
{
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	GladeXML *xml;
	GtkWidget *widget;
	gint rows;
	
	mdi = SCREEM_MDI( data );
	priv = mdi->priv;

	if( priv->dialog ) {
		rows = gtk_tree_selection_count_selected_rows( sel );
		
		xml = glade_get_widget_tree( priv->dialog );
		widget = glade_xml_get_widget( xml, "up" );
		gtk_widget_set_sensitive( widget, rows == 1 );
		widget = glade_xml_get_widget( xml, "down" );
		gtk_widget_set_sensitive( widget, rows == 1 );
		widget = glade_xml_get_widget( xml, "close" );
		gtk_widget_set_sensitive( widget, rows != 0 );
		widget = glade_xml_get_widget( xml, "jumpto" );
		gtk_widget_set_sensitive( widget, rows == 1 );
	}
}

void screem_mdi_move_up( GtkWidget *widget )
{
	GladeXML *xml;
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GList *list;
	GtkTreePath *path;
	gchar *str;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "documentlist" );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	
	mdi = g_object_get_data( G_OBJECT( widget ), "mdi" );
	priv = mdi->priv;

	list = gtk_tree_selection_get_selected_rows( sel, &model );
	path = list->data;
	str = gtk_tree_path_to_string( path );
	
	g_list_foreach( list, (GFunc)gtk_tree_path_free, NULL );
	g_list_free( list );

	g_free( str );
}

void screem_mdi_move_down( GtkWidget *widget )
{
	GladeXML *xml;
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GList *list;
	GtkTreePath *path;
	gchar *str;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "documentlist" );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	
	mdi = g_object_get_data( G_OBJECT( widget ), "mdi" );
	priv = mdi->priv;

	list = gtk_tree_selection_get_selected_rows( sel, &model );
	path = list->data;
	str = gtk_tree_path_to_string( path );
	
	g_list_foreach( list, (GFunc)gtk_tree_path_free, NULL );
	g_list_free( list );

	g_free( str );
}

void screem_mdi_close( GtkWidget *widget )
{
	GladeXML *xml;
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GList *list;
	GList *tmp;
	GList *doc;

	GtkTreePath *path;
	gchar *str;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "documentlist" );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	
	mdi = g_object_get_data( G_OBJECT( widget ), "mdi" );
	priv = mdi->priv;

	list = gtk_tree_selection_get_selected_rows( sel, &model );
	for( tmp = list; tmp; tmp = tmp->next ) {
		path = tmp->data;
		str = gtk_tree_path_to_string( path );
		doc = g_list_nth( priv->documents, atoi( str ) );
		tmp->data = doc->data;
		g_free( str );
		gtk_tree_path_free( path );
	}
	/* list is now a list of ScreemPage */
	for( tmp = list; tmp; tmp = tmp->next ) {
		screem_mdi_remove_document( mdi, 
					    SCREEM_PAGE( tmp->data ) );
	}
	g_list_free( list );
}

void screem_mdi_jump_to( GtkWidget *widget )
{
	GladeXML *xml;
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GList *list;
	GtkTreePath *path;
	gchar *str;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "documentlist" );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	
	mdi = g_object_get_data( G_OBJECT( widget ), "mdi" );
	priv = mdi->priv;

	list = gtk_tree_selection_get_selected_rows( sel, &model );
	path = list->data;
	str = gtk_tree_path_to_string( path );
	
	g_list_foreach( list, (GFunc)gtk_tree_path_free, NULL );
	g_list_free( list );

	list = g_list_nth( priv->documents, atoi( str ) );
	screem_mdi_set_current( mdi, SCREEM_PAGE( list->data ) );
	g_free( str );
}

static void screem_mdi_can_undo( GtkSourceBuffer *buffer,
				 gboolean can_undo,
				 gpointer data )
{
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	ScreemPage *page;
	GtkAction *action;
	
	mdi = SCREEM_MDI( data );
	priv = mdi->priv;
	
	page = screem_mdi_get_current( mdi );
	
	if( page && ( GTK_SOURCE_BUFFER( page ) == buffer ) ) {
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "Undo" );
		g_object_set( G_OBJECT( action ), "sensitive", can_undo,
				NULL );
	}
}

static void screem_mdi_can_redo( GtkSourceBuffer *buffer,
				 gboolean can_redo,
				 gpointer data )
{
	ScreemMDI *mdi;
	ScreemMDIPrivate *priv;
	ScreemPage *page;
	GtkAction *action;
	
	mdi = SCREEM_MDI( data );
	priv = mdi->priv;
	
	page = screem_mdi_get_current( mdi );
	if( page && ( GTK_SOURCE_BUFFER( page ) == buffer ) ) {
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "Redo" );
		g_object_set( G_OBJECT( action ), "sensitive", can_redo,
				NULL );
	}
}

static void screem_mdi_close_button( GtkWidget *widget,
				ScreemMDI *mdi )
{
	ScreemPage *page;
	ScreemMDIPrivate *priv;
	GtkActionGroup *group;
	GtkAction *action;
	
	priv = mdi->priv;
	group = priv->window->action_group;
	action = gtk_action_group_get_action( group, "Close Document" );
	
	page = g_object_get_data( G_OBJECT( widget ), "page" );
	g_object_set_data( G_OBJECT( priv->window ), "tab_popup", page );
	gtk_action_activate( action );
	
	g_object_set_data( G_OBJECT( priv->window ), "tab_popup", NULL );
}

/* public stuff */

ScreemMDI *screem_mdi_new( ScreemWindow *window )
{
	return g_object_new( SCREEM_TYPE_MDI, "window", window,
			NULL );
}

GtkNotebook *screem_mdi_get_notebook( ScreemMDI *mdi )
{
	g_return_val_if_fail( SCREEM_IS_MDI( mdi ), NULL );


	return mdi->priv->notebook;
}

GList *screem_mdi_get_document_list( ScreemMDI *mdi )
{
	g_return_val_if_fail( SCREEM_IS_MDI( mdi ), NULL );

	return mdi->priv->documents;
}

ScreemPage *screem_mdi_get_current( ScreemMDI *mdi )
{
	g_return_val_if_fail( SCREEM_IS_MDI( mdi ), NULL );

	return mdi->priv->current;
}

void screem_mdi_remove_document( ScreemMDI *mdi, ScreemPage *page )
{
	ScreemMDIPrivate *priv;
     	gint index;
	GList *tmp;
	
	g_return_if_fail( SCREEM_IS_MDI( mdi ) );
	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	priv = mdi->priv;

	index = g_list_index( priv->documents, page );
	if( index != -1 ) {
		screem_mdi_real_remove_document( mdi, page, index );

		index = gtk_notebook_get_current_page( priv->notebook );
		tmp = g_list_nth( priv->documents, index );
		if( tmp ) {
			screem_mdi_set_current( mdi, tmp->data );
		} else {
			screem_mdi_set_current( mdi, NULL );
		}
	} else {
		g_warning( "Attempt to remove non open document\n" );
	}
}

void screem_mdi_remove_all( ScreemMDI *mdi )
{
	ScreemMDIPrivate *priv;
	ScreemPage *page;
	
	g_return_if_fail( SCREEM_IS_MDI( mdi ) );

	priv = mdi->priv;
	
	g_signal_handlers_block_matched( G_OBJECT( priv->notebook ),
					G_SIGNAL_MATCH_FUNC|
					G_SIGNAL_MATCH_DATA,
					0, 0, NULL, 
					screem_mdi_notebook_change,
					mdi );
	while( priv->documents ) {
		page = SCREEM_PAGE( priv->documents->data );
		screem_mdi_real_remove_document( mdi, page, 0 );
	}
	g_signal_handlers_unblock_matched( G_OBJECT( priv->notebook ),
					G_SIGNAL_MATCH_FUNC|
					G_SIGNAL_MATCH_DATA,
					0, 0, NULL, 
					screem_mdi_notebook_change,
					mdi );
}

gboolean screem_mdi_set_current( ScreemMDI *mdi, ScreemPage *page )
{
	ScreemMDIPrivate *priv;
	ScreemPage *cpage;
	gint len;
	gint index;
	gboolean ret;
	GtkAction *action;
	gboolean can;
	
	g_return_val_if_fail( SCREEM_IS_MDI( mdi ), FALSE );
	g_return_val_if_fail( page == NULL || SCREEM_IS_PAGE( page ), FALSE );

	priv = mdi->priv;

	cpage = screem_mdi_get_current( mdi );
	
	if( page == cpage ) {
		return TRUE;
	}
	
	/* lookup page in list */
	index = g_list_index( priv->documents, page );
	ret = TRUE;

	if( page && index == -1 ) {
		/*g_warning( "Attempt to display non added document,\n" );*/
		ret = screem_mdi_add_document( mdi, page );
		index = g_list_index( priv->documents, page );
	}
	if( ret ) {
		g_signal_handlers_block_matched( G_OBJECT( priv->notebook ),
						G_SIGNAL_MATCH_FUNC|
						G_SIGNAL_MATCH_DATA,
						0, 0, NULL, 
						screem_mdi_notebook_change,
						mdi );

		priv->current = page;
	
		gtk_notebook_set_current_page( priv->notebook, 
						index );
	
		len = g_list_length( priv->documents );

		action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "PreviousDocument" );
		g_object_set( G_OBJECT( action ),
				"sensitive", index > 0, NULL );
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "NextDocument" );
		g_object_set( G_OBJECT( action ),
				"sensitive", index < ( len - 1 ), 
				NULL );
		
		g_signal_handlers_unblock_matched( G_OBJECT( priv->notebook ),
						G_SIGNAL_MATCH_FUNC|
						G_SIGNAL_MATCH_DATA,
						0, 0, NULL, 
						screem_mdi_notebook_change,
						mdi );
		g_signal_emit( G_OBJECT( mdi ),
			       screem_mdi_signals[ CHANGED ],
			       0, page, cpage );
	
		if( page ) {
			action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "Undo" );
			can = gtk_source_buffer_can_undo( GTK_SOURCE_BUFFER( page ) );
			g_object_set( G_OBJECT( action ), "sensitive", can,
				NULL );
			action = gtk_action_group_get_action( GTK_ACTION_GROUP( priv->window->action_group ), "Redo" );
			can = gtk_source_buffer_can_redo( GTK_SOURCE_BUFFER( page ) );
			g_object_set( G_OBJECT( action ), "sensitive", can,
					NULL );
		}
	}

	return ret;
}

gboolean screem_mdi_add_document( ScreemMDI *mdi, ScreemPage *page )
{
	ScreemMDIPrivate *priv;
	gint index;
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_MDI( mdi ), FALSE );
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	priv = mdi->priv;

	/* lookup page in list */
	index = g_list_index( priv->documents, page );
	ret = TRUE;
	
	if( index == -1 ) {
		/* wasn't in the list */
	
		ret = screem_mdi_notebook_add( mdi, page );
		if( ret ) {
			priv->documents = g_list_append( priv->documents,
							page );
		}
	}

	return ret;
}


