/* sndinfo describes sounds */

#include <config.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#if (defined(HAVE_LIBC_H) && (!defined(HAVE_UNISTD_H)))
  #include <libc.h>
#else
  #if (!(defined(_MSC_VER))) && (!(defined(MPW_C)))
    #include <unistd.h>
  #endif
  #include <string.h>
#endif
#include <errno.h>
#include <time.h>

#include "sndlib.h"

#if MACOS
  #include <console.h>
#endif

#if DEBUGGING
void *mem_calloc(size_t len, size_t size, const char *func, const char *file, int line) {return(calloc(len, size));}
void *mem_malloc(size_t len, const char *func, const char *file, int line) {return(malloc(len));}
void mem_free(void *ptr, const char *func, const char *file, int line) {free(ptr);}
void *mem_realloc(void *ptr, size_t size, const char *func, const char *file, int line) {return(realloc(ptr, size));}
#include <fcntl.h>
int io_open(const char *pathname, int flags, mode_t mode, const char *func, const char *file, int line) {return(open(pathname, flags, mode));}
int io_creat(const char *pathname, mode_t mode, const char *func, const char *file, int line) {return(creat(pathname, mode));}
int io_close(int fd, const char *func, const char *file, int line) {return(close(fd));}
FILE *io_fopen(const char *path, const char *mode, const char *func, const char *file, int line) {return(fopen(path, mode));}
int io_fclose(FILE *stream, const char *func, const char *file, int line) {return(fclose(stream));}
char *copy_string(const char *str);
char *copy_string(const char *str) {return(strdup(str));}
#endif

static char *display_maxamps(const char *filename, int chans)
{
  char *ampstr;
  char fstr[16];
  int i;
  mus_sample_t *vals;
  off_t *times;
  ampstr = (char *)CALLOC(chans * 32, sizeof(char));
  vals = (mus_sample_t *)CALLOC(chans, sizeof(mus_sample_t));
  times = (off_t *)CALLOC(chans, sizeof(off_t));
  sprintf(ampstr, "\n  max amp%s: ", (chans > 1) ? "s" : "");
  mus_sound_maxamps(filename, chans, vals, times);
  for (i = 0; i < chans; i++)
    {
      sprintf(fstr, "%.3f ", MUS_SAMPLE_TO_FLOAT(vals[i]));
      strcat(ampstr, fstr);
    }
  FREE(vals);
  FREE(times);
  return(ampstr);
}

int main(int argc, char *argv[])
{
  int chans, srate, format, type, ctr;
  off_t samples;
  float length = 0.0;
  time_t date;
  int *loops = NULL;
  char *comment, *header_name;
  char *format_info = NULL, *format_name, *ampstr = NULL;
  char timestr[64];
#if MACOS
  argc = ccommand(&argv);
#endif
  if (argc == 1) {printf("usage: sndinfo file\n"); exit(0);}
  mus_sound_initialize();
  for (ctr = 1; ctr < argc; ctr++)
    {
      if (mus_file_probe(argv[ctr])) /* see if it exists */
	{
	  date = mus_sound_write_date(argv[ctr]);
	  srate = mus_sound_srate(argv[ctr]);
	  if (srate == MUS_ERROR)
	    {
	      fprintf(stdout, "%s: not a sound file?\n", argv[ctr]);
	      continue;
	    }
	  chans = mus_sound_chans(argv[ctr]);
	  samples = mus_sound_samples(argv[ctr]);
	  comment = mus_sound_comment(argv[ctr]); 
	  if ((chans > 0) && (srate > 0))
	    length = (float)((double)samples / (double)(chans * srate));
	  loops = mus_sound_loop_info(argv[ctr]);
	  type = mus_sound_header_type(argv[ctr]);
	  header_name = (char *)mus_header_type_name(type);
	  format = mus_sound_data_format(argv[ctr]);
	  if (format != MUS_UNKNOWN)
	    format_info = (char *)mus_data_format_name(format);
	  else
	    {
	      if (format_info == NULL) format_info = (char *)calloc(64, sizeof(char));
	      format = mus_sound_original_format(argv[ctr]);
	      format_name = (char *)mus_header_original_format_name(format, type);
	      if (format_name)
		sprintf(format_info, "%d (%s)", format, format_name);
	      else sprintf(format_info, "%d", format);
	    }
	  fprintf(stdout, "%s:\n  srate: %d\n  chans: %d\n  length: %f",
		  argv[ctr], srate, chans, length);
	  if (length < 10.0)
	    {
	      int samps;
	      samps = mus_sound_frames(argv[ctr]);
	      fprintf(stdout, " (%d sample%s)", samps, (samps != 1) ? "s" : "");
	    }
	  fprintf(stdout, "\n");
	  fprintf(stdout, "  type: %s\n  format: %s\n  ",
		  header_name,
		  format_info);
#if HAVE_STRFTIME
	  strftime(timestr, 64, "%a %d-%b-%Y %H:%M %Z", localtime(&date));
	  fprintf(stdout, "written: %s", timestr);
#endif
	  if ((chans > 0) && (mus_sound_maxamp_exists(argv[ctr])))
	    {
	      ampstr = display_maxamps(argv[ctr], chans);
	      if (ampstr) fprintf(stdout, ampstr);
	    }
	  fprintf(stdout, "\n");
	  if (comment) fprintf(stdout, "  comment: %s\n", comment);
	  if (loops)
	    {
	      fprintf(stdout, "  loop: %d to %d\n", loops[0], loops[1]);
	      if (loops[2] != 0)
		fprintf(stdout, "  loop: %d to %d\n", loops[2], loops[3]);
	      if (loops[0] != 0)
		fprintf(stdout, "    base: %d, detune: %d\n", loops[4], loops[5]);
	    }
	}
      else
	fprintf(stderr, "%s: %s\n", argv[ctr], strerror(errno));
      if (ctr < argc - 1) fprintf(stdout, "\n");
    }
  return(0);
}
