/*
 *  p o r t . c			-- ports implementation
 *
 * Copyright  1993-2003 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 *
 *            Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 17-Feb-1993 12:27
 * Last file update: 10-Feb-2003 11:18 (eg)
 *
 */

#include <ctype.h>
#include "stklos.h"
#include "../pcre/pcreposix.h"
#include "vm.h"

#define INITIAL_LINE_SIZE 256		/* Initial size for readline */

static regex_t tilde_regexp;		/* used to detect if an error message 
					 * is formatted or not */
static SCM CrLf;			/* used in read-line only */

static void error_bad_port(SCM p)
{
  STk_error("bad port ~S", p);
}

static void error_closed_port(SCM p)
{
  STk_error("port ~S is closed", p);
}

static void error_bad_name(SCM f)
{
  STk_error("bad file name ~S", f);
}


static SCM verify_port(SCM port, int mode)
{
  if (mode == PORT_WRITE) {
    if (!port) return STk_curr_oport;
    if (!OPORTP(port)) error_bad_port(port);
  } else {
    if (!port) return STk_curr_iport;
    if (!IPORTP(port)) error_bad_port(port);
  }
  if (PORT_IS_CLOSEDP(port)) error_closed_port(port);
  return port;
}


/*
<doc  input-port? output-port?
 * (input-port? obj)
 * (output-port? obj)
 *
 * Returns |#t| if |obj| is an input port or output port respectively, 
 * otherwise returns #f.
doc>
 */
DEFINE_PRIMITIVE("input-port?", input_portp, subr1, (SCM port))
{
  return MAKE_BOOLEAN(IPORTP(port));
}

DEFINE_PRIMITIVE("output-port?", output_portp, subr1, (SCM port))
{
  return MAKE_BOOLEAN(OPORTP(port));
}


/*
<doc ext interactive-port?
 * (interactive-port? port)
 *
 * Returns |#t| if |port| is connected to a terminal and |#f| otherwise.
doc>
 */
DEFINE_PRIMITIVE("interactive-port?", interactive_portp, subr1, (SCM port))
{
  ENTER_PRIMITIVE(interactive_portp);
  if (!PORTP(port)) error_bad_port(port);

  return MAKE_BOOLEAN(PORT_FLAGS(port) & PORT_IS_INTERACTIVE);
}


/*
<doc  current-input-port current-output-port
 * (current-input-port obj)
 * (current-output-port obj)
 *
 * Returns the current default input or output port.
doc>
 */
DEFINE_PRIMITIVE("current-input-port",current_input_port, subr0, (void))
{
  return STk_curr_iport;
}

DEFINE_PRIMITIVE("current-output-port",current_output_port, subr0, (void))
{
  return STk_curr_oport;
}

/*
<doc ext current-error-port
 * (current-error-port obj)
 *
 * Returns the current default error port.
doc>
 */
DEFINE_PRIMITIVE("current-error-port",current_error_port, subr0, (void))
{
  return STk_curr_eport;
}


DEFINE_PRIMITIVE("%set-std-port!", set_std_port, subr2, (SCM index, SCM port))
{
  ENTER_PRIMITIVE(set_std_port);
  
  switch (AS_LONG(index)) {
    case SCM_LONG(0): if (!IPORTP(port)) goto badport; STk_curr_iport = port; break;
    case SCM_LONG(1): if (!OPORTP(port)) goto badport; STk_curr_oport = port; break;
    case SCM_LONG(2): if (!OPORTP(port)) goto badport; STk_curr_eport = port; break;
    default: STk_error("bad code ~S", index);
  }
  return STk_void;
badport:
  error_bad_port(port);
  return STk_void;
}


/*=============================================================================*\
 * 				Read
\*=============================================================================*/

/*
<doc  read
 * (read)
 * (read port)
 *
 * |Read| converts external representations of Scheme objects into the
 * objects themselves. |Read| returns the next object parsable from the given
 * input port, updating port to point to the first character past the end of
 * the external representation of the object.
 *
 * If an end of file is encountered in the input before any characters are found
 * that can begin an object, then an end of file object is returned. The port
 * remains open, and further attempts to read will also return an end of file
 * object. If an end of file is encountered after the beginning of an object's
 * external representation, but the external representation is incomplete 
 * and therefore not parsable, an error is signalled.
 *
 * The port argument may be omitted, in which case it defaults to the value
 * returned by |current-input-port|. It is an error to read from a closed port.
doc>
 */
DEFINE_PRIMITIVE("read", scheme_read, subr01, (SCM port))
{
  ENTER_PRIMITIVE(scheme_read);

  port = verify_port(port, PORT_READ);
  return STk_read(port, FALSE);
}



/* The same one but for reading code => code is really constant */
DEFINE_PRIMITIVE("%read", scheme_read_cst, subr01, (SCM port))
{
  ENTER_PRIMITIVE(scheme_read_cst);

  port = verify_port(port, PORT_READ);
  return STk_read_constant(port, FALSE);
}


/*
<doc  read-char
 * (read-char)
 * (read-char port)
 *
 * Returns the next character available from the input |port|, updating the |port|
 * to point to the following character. If no more characters are available, 
 * an end of file object is returned. |Port| may be omitted, in which case 
 * it defaults to the value returned by |current-input-port|.
doc>
 */
DEFINE_PRIMITIVE("read-char", read_char, subr01, (SCM port))
{
  int c;

  ENTER_PRIMITIVE(read_char);
  port = verify_port(port, PORT_READ);
  c = STk_getc(port);
  return (c == EOF) ? STk_eof : MAKE_CHARACTER(c);
}

/*
<doc  peek-char
 * (peek-char)
 * (peek-char port)
 *
 * Returns the next character available from the input |port|, without updating 
 * the port to point to the following character. If no more characters are
 * available, an end of file object is returned. |Port| may be omitted, in
 * which case it defaults to the value returned by |current-input-port|.
 *
 * @strong{Note}: The value returned by a call to |peek-char| is the same as the
 * value that would have been returned by a call to |read-char| with the same
 * port. The only difference is that the very next call to |read-char| or 
 * |peek-char| on that port will return the value returned by the preceding
 * call to |peek-char|. In particular, a call to |peek-char| on an interactive
 * port will hang waiting for input whenever a call to |read-char| would have
 * hung.
doc>
 */
DEFINE_PRIMITIVE("peek-char", peek_char, subr01, (SCM port))
{
  int c;

  ENTER_PRIMITIVE(peek_char);
  port = verify_port(port, PORT_READ);
  c = STk_getc(port);
  STk_ungetc(c, port);

  return (c == EOF) ? STk_eof : MAKE_CHARACTER(c);
}


/*
<doc  eof-object?
 * (eof-object? obj)
 *
 * Returns |#t| if |obj| is an end of file object, otherwise returns |#f|. 
doc>
 */
DEFINE_PRIMITIVE("eof-object?", eof_objectp, subr1, (SCM obj))
{
  return MAKE_BOOLEAN(obj == STk_eof);
}


/*
<doc  char-ready?
 * (char-ready?)
 * (char-ready? port)
 *
 * Returns |#t| if a character is ready on the input port and returns |#f|
 * otherwise. If char-ready returns |#t| then the next read-char operation on
 * the given port is guaranteed not to hang. If the port is at end of file
 * then |char-ready?| returns |#t|. Port may be omitted, in which case it
 * defaults to the value returned by |current-input-port|.
doc>
 */
DEFINE_PRIMITIVE("char-ready?", char_readyp, subr01, (SCM port))
{
  ENTER_PRIMITIVE(char_readyp);
  port = verify_port(port, PORT_READ);
  return MAKE_BOOLEAN(STk_readyp(port));
}

/*=============================================================================*\
 * 				Write
\*=============================================================================*/


/*
<doc  write
 * (write obj)
 * (write obj port)
 *
 * Writes a written representation of |obj| to the given |port|. Strings that
 * appear in the written representation are enclosed in doublequotes, and 
 * within those strings backslash and doublequote characters are escaped
 * by backslashes. Character objects are written using the |#\| notation. 
 * |Write| returns an unspecified value. The |port| argument may be omitted, in
 * which case it defaults to the value returned by |current-output-port|.
doc>
 */
DEFINE_PRIMITIVE("write", write, subr12, (SCM expr, SCM port))
{
  ENTER_PRIMITIVE(write);
  port = verify_port(port, PORT_WRITE);
  STk_print(expr, port, WRT_MODE);
  return STk_void;
}


/*
<doc EXT write*
 * (write* obj)
 * (write* obj port)
 *
 * Writes a written representation of |obj| to the given port.  The
 * main difference with the |write| procedure is that |write*|
 * handles data structures with cycles. Circular structure written by 
 * this procedure use the ``|#@w{@emph{n}}=|'' and ``|#@w{@emph{n}}#|'' 
 * notations (@pxref{circular structure}).
doc>
*/
DEFINE_PRIMITIVE("write*", write_star, subr12, (SCM expr, SCM port))
{
  port = verify_port(port, PORT_WRITE);
  STk_print_star(expr, port);
  return STk_void;
}

/*
<doc  display
 * (display obj)
 * (display obj port)
 *
 * Writes a representation of |obj| to the given |port|. Strings that
 * appear in the written representation are not enclosed in
 * doublequotes, and no characters are escaped within those
 * strings. Character objects appear in the representation as if
 * written by |write-char| instead of by |write|. |Display| returns an
 * unspecified value. The |port| argument may be omitted, in which
 * case it defaults to the value returned by |current-output-port|.
 *
 * @strong{Rationale}: |Write| is intended for producing machine-readable
 * output and |display| is for producing human-readable output. 
doc>
 */
DEFINE_PRIMITIVE("display", display, subr12, (SCM expr, SCM port))
{
  ENTER_PRIMITIVE(display);
  port = verify_port(port, PORT_WRITE);
  STk_print(expr, port, DSP_MODE);
  return STk_void;
}

/*
<doc  newline
 * (newline)
 * (newline port)
 *
 * Writes an end of line to |port|. Exactly how this is done differs from
 * one operating system to another. Returns an unspecified value. The |port|
 * argument may be omitted, in which case it defaults to the value returned
 * by |current-output-port|.
doc>
 */
DEFINE_PRIMITIVE("newline", newline, subr01, (SCM port))
{
  ENTER_PRIMITIVE(newline);
  port = verify_port(port, PORT_WRITE);
  STk_putc('\n', port);
  return STk_void;
}



/*
<doc  write-char
 * (write-char char)
 * (write-char char port)
 *
 * Writes the character |char| (not an external representation of the
 * character) to the given |port| and returns an unspecified value. 
 * The |port| argument may be omitted, in which case it defaults to the
 * value returned by |current-output-port|.
doc>
 */
DEFINE_PRIMITIVE("write-char", write_char, subr12, (SCM c, SCM port))
{
  ENTER_PRIMITIVE(write_char);

  if (!CHARACTERP(c)) STk_error("bad character ~S", c);
  port = verify_port(port, PORT_WRITE);
  STk_putc(CHARACTER_VAL(c), port);
  return STk_void;
}



/*=============================================================================*\
 * 				Load
\*=============================================================================*/

#define FILE_IS_SOURCE		0
#define FILE_IS_BCODE 		1
#define FILE_IS_OBJECT		2
#define FILE_IS_DIRECTORY	3
#define FILE_DOES_NOT_EXISTS	4

static int find_file_nature(SCM f)
{
  int c;
  SCM tmp;

  c = STk_getc(f); STk_ungetc(c, f);

  if (c != EOF) {
    if ((iscntrl(c) && c!= '\n' && c!= '\t') || !isascii(c))
      return FILE_IS_OBJECT;

    tmp = STk_read(f, TRUE);
    if (tmp == STk_intern("STklos")) {
      /* This is a bytecode file. Skip  the (unused) version number*/
      tmp = STk_read(f, TRUE);
      return FILE_IS_BCODE;
    }
    /* We'll suppose that this is a source file, but we have read the first sexpr */
    STk_rewind_file_port(f);
  }
  return FILE_IS_SOURCE;
}


SCM STk_load_source_file(SCM f)
{
  SCM sexpr;
  SCM eval_symb, eval, ref;
  
  /* //FIXME: eval devrait tre connu sans faire de lookup(i.e. export par la VM)*/
  eval_symb = STk_intern("eval");

  for ( ; ; ) {
    /* We need to lookup for eval symbol for each expr, since it cans
     * change while loading the file (with R5RS macro for instance, it
     * is the case)
     */
    sexpr = STk_read_constant(f, FALSE);
    if (sexpr == STk_eof) break;
    eval  = STk_lookup(eval_symb, STk_current_module, &ref, TRUE);
    STk_C_apply(eval, 1, sexpr);
  }
  STk_close_port(f);
  return STk_true;
}


static SCM load_file(SCM filename)
{
  SCM f;
  char *fname = STRING_CHARS(filename);
  
  /* Verify that file is not a directory */
  if (STk_dirp(fname)) return STk_false;
  
  /* It's Ok, try now to load file */
  f = STk_open_file(fname, "r");
  if (f != STk_false) {
    switch (find_file_nature(f)) {
      case FILE_IS_SOURCE: return STk_load_source_file(f);
      case FILE_IS_BCODE:  return STk_load_bcode_file(f);
      case FILE_IS_OBJECT: return STk_load_object_file(f, fname);
    }
  }
  return STk_false;
}


/*
<doc load
 * (load filename)
 *
 * |Filename| should be a string naming an existing file containing Scheme 
 * expressions. |Load| has been extended in @stklos{} to allow loading of
 * file containing Scheme compiled code as well as object files 
 * (@emph{aka} shared objects). The loading of object files is not available on 
 * all architectures. The value returned by |load| is @emph{void}.
 * 
 * If the file whose name is |filename| cannot be located, |load| will try 
 * to find it in one of the directories given by |(get-load-path)| with 
 * the suffixes given by |(get-load-suffixes)|.
doc>
 */
DEFINE_PRIMITIVE("load", scheme_load, subr1, (SCM filename))
{
  ENTER_PRIMITIVE(scheme_load);
  
  if (!STRINGP(filename)) error_bad_name(filename);
  if (load_file(filename) == STk_false)
    STk_error("cannot load file ~S", filename);
  return STk_void;
}


/*
<doc EXT try-load
 * (try-load filename)
 *
 * |try-load| tries to load the file named |filename|. As |load|, 
 * |try-load| tries to find the file given the current load path 
 * and a set of suffixes if |filename| cannot be loaded. If |try-load|
 * is able to find a readable file, it is loaded, and |try-load| returns 
 * |#t|. Otherwise,  |try-load| retuns |#f|.
doc>
 */
DEFINE_PRIMITIVE("try-load", scheme_try_load, subr1, (SCM filename))
{
  ENTER_PRIMITIVE(scheme_try_load);

  if (!STRINGP(filename)) error_bad_name(filename);
  return load_file(filename);
}

/*===========================================================================*\
 * 
 * 			S T k   b o n u s
 *
\*===========================================================================*/

static SCM internal_format(int argc, SCM *argv, int error)
     /* a very simple and poor format */ 
{
  SCM port, fmt;
  int format_in_string = 0;
  char *p, *start_fmt;
  
  if (error) {
    if (argc < 1) goto Bad_list;
    format_in_string = 1;
    port = STk_open_output_string();
    argc -= 1;
  }
  else {
    if (STRINGP(*argv)) {
      /* This is a SRFI-28 format */
      format_in_string = 1;
      port = STk_open_output_string();
      argc -= 1;
    } else {
      if (argc < 2) goto Bad_list;
      port = *argv--; 
      argc -= 2;
      
      if (BOOLEANP(port)){
	if (port == STk_true) port = STk_curr_oport;
	else {
	  format_in_string = 1;
	  port = STk_open_output_string();
	}
      } else {
	verify_port(port, PORT_WRITE);
      }
    }
  }

  fmt = *argv--;
  if (!STRINGP(fmt)) STk_error("bad format string ~S", fmt);

  /* Parse the format string */
  start_fmt = STRING_CHARS(fmt);
  for(p = start_fmt; *p; p++) {
    if (*p == '~') {
      switch(*(++p)) {
        case 'A':
        case 'a': if (argc-- <= 0) goto TooMuch;
	          STk_print(*argv--, port, DSP_MODE);
		  continue;
        case 'S':
        case 's': if (argc-- <= 0) goto TooMuch;
                  STk_print(*argv--, port, WRT_MODE);
	          continue;        
        case 'W':
        case 'w': if (argc-- <= 0) goto TooMuch;
	  	  STk_print_star(*argv--, port);
	          continue;
        case '%': STk_putc('\n', port);
                  continue;
        case '~': STk_putc('~', port);
                  continue;
        default:  STk_putc('~',  port);
                  /* NO BREAK */
      }
    }
    STk_putc(*p, port);
  }

  /* Verify that it doesn't remain arguments on the list */
  if (argc) 
    STk_error("too few ``~~'' in format string %S", start_fmt);

  return format_in_string ? STk_get_output_string(port) : STk_void;

TooMuch:
  STk_error("too many ``~~'' in format string %S", start_fmt);
Bad_list:
  STk_error("bad list of parameters ~S", *argv);
  return STk_void;
}

/*
<doc ext format 
 * (format port str obj ...)
 * (format str obj)
 *
 * Writes the |obj|s to the given |port|, according to the format
 * string |str|. |Str| is written literally, except for the following 
 * sequences:
 *
 * @itemize --
 * @item |~a| or |~A| is replaced by the printed representation
 * of the next |obj|.
 * 
 * @item |~s| or |~S| is replaced by the ``slashified'' printed
 * representation of the next |obj|.
 *
 * @item |~w| or |~W| is replaced by the printed representation
 * of the next |obj| (circular structures are correctly handled and
 * printed using |writes*|).
 *
 * @item |~~| is replaced by a single tilde character.
 *
 * @item |~%| is replaced by a newline
 * 
 * @end itemize
 *
 * |Port| can be a boolean or a port. If |port| is |#t|, output goes to
 * the current output port; if |port| is |#f|, the output is returned as a
 * string.  Otherwise, the output is printed on the specified port.
 * @lisp
 *    (format #f "A test.")       => "A test."
 *    (format #f "A ~a." "test")  => "A test."
 *    (format #f "A ~s." "test")  => "A \"test\"."
 * @end lisp
 * 
 * The second form of |format| is compliant with SRFI-28. That is, when
 * |port| is omitted, the output is returned as a string as if |port| was 
 * given the value |#f|.
doc>
 */
DEFINE_PRIMITIVE("format", format, vsubr, (int argc, SCM *argv))
{
  ENTER_PRIMITIVE(format);
  return internal_format(argc, argv, FALSE);
}


/*
<doc ext error
 * (error str obj ...)
 * (error name str obj ...)
 *
 * |error| is used to signal an error to the user. The second form 
 * of |error| takes  a symbol as first parameter; it is generally used for the 
 * name of the procedure which raises the error.
 *
 * @b{Note}: The specification string may follow the @emph{``tilde conventions''} 
 * of |format| (@pxref{format}); in this case this procedure builds an 
 * error message according to the specification given in |str|. Otherwise, 
 * this procedure is conform to the |error| procedure defined in SRFI-23 and 
 * |str| is printed with the |display| procedure, whereas the |obj|s are printed 
 * with the |write| procedure. 
 *
 * @noindent
 * Hereafter, are some calls of the |error| procedure using a formatted string
 * @lisp
 * (error "bad integer ~A" "a")
 *                      @print{} bad integer a
 * (error 'vector-ref "bad integer ~S" "a") 
 *                      @print{} vector-ref: bad integer "a"
 * (error 'foo "~A is not between ~A and ~A" "bar" 0 5)
 *                      @print{} foo: bar is not between 0 and 5
 * @end lisp
 *
 * @noindent
 * and some conform to SRFI-23
 * @lisp
 * (error "bad integer" "a")
 *                     @print{} bad integer "a"
 * (error 'vector-ref "bad integer" "a")
 *                    @print{} vector-ref: bad integer "a"
 * (error "bar" "is not between" 0 "and" 5)
 *                    @print{} bar "is not between" 0 "and" 5
 * @end lisp
doc>
 */
static SCM srfi_23_error(int argc, SCM *argv)
{
  SCM port = STk_open_output_string();
  
  STk_print(*argv--, port, DSP_MODE); /* the message (we know that it exists) */
  for (argc--; argc; argc--) {
    STk_putc(' ', port);
    STk_print(*argv--, port, WRT_MODE);
  }
  STk_close_port(port);
  return STk_get_output_string(port);
}

DEFINE_PRIMITIVE("error", scheme_error, vsubr, (int argc, SCM *argv))
{
  regmatch_t pm[10];
  char *p_name = ""; /* use a local cause internal_format can  broke it */

  if (argc > 0) {
    if (SYMBOLP(*argv)) {
      p_name = SYMBOL_PNAME(*argv);
      argc -= 1;
      argv -= 1;
    }
    if (argc > 0) {
      SCM msg;
      
      /* See if we have a formatted message or a plain SRFI-23 call */
      if (STRINGP(*argv) && regexec(&tilde_regexp, STRING_CHARS(*argv), 10, pm, 0))
	msg = srfi_23_error(argc, argv);
      else 
	msg = internal_format(argc, argv, TRUE);
      STk_signal_error(p_name, msg);
    }
  }
  STk_signal_error(p_name, STk_Cstring2string(""));
  return STk_void;
}

/*
<doc close-input-port close-output-port
 * (close-input-port port)
 * (close-output-port port)
 *
 * Closes the port associated with |port|, rendering the port incapable of
 * delivering or accepting characters. These routines have no effect if the
 * port has already been closed. The value returned is @emph{void}.
doc>
 */
DEFINE_PRIMITIVE("close-input-port", close_input_port, subr1, (SCM port))
{
  ENTER_PRIMITIVE(close_input_port);
  if (!IPORTP(port)) STk_error("bad input port ~S", port);
  STk_close(port);
  return STk_void;
}

DEFINE_PRIMITIVE("close-output-port", close_output_port, subr1, (SCM port))
{
  ENTER_PRIMITIVE(close_output_port);
  if (!OPORTP(port)) STk_error("bad output port ~S", port);
  STk_close(port);
  return STk_void;
}


/*
<doc EXT close-port
 * (close-port port)
 *
 * Closes the port associated with |port|.
doc>
 */
DEFINE_PRIMITIVE("close-port", close_port, subr1, (SCM port))
{
  ENTER_PRIMITIVE(close_port);
  if (!PORTP(port))
    error_bad_port(port);
  
  STk_close(port);
  return STk_void;
}

/*
<doc EXT port-closed?
 * (port-closed? port)
 *
 * Returns |#t| if |port| is closed and |#f| otherwise.
doc>
*/
DEFINE_PRIMITIVE("port-closed?", port_closed, subr1, (SCM port))
{
  ENTER_PRIMITIVE(port_closed);
  if (!PORTP(port))
    error_bad_port(port);

  return MAKE_BOOLEAN(PORT_IS_CLOSEDP(port));
}


/*
<doc read-line
 * (read-line)
 * (read-line port)
 *
 * Reads the next line available from the input port |port|. This function
 * returns 2 values: the first one is is the string which contains the line
 * read, and the second one is the end of line delimiter. The end of line
 * delimiter can be an end of file object, a character or a string in case 
 * of a multiple character delimiter. If no more characters are available 
 * on |port|, an end of file object is returned.  |Port| may be omitted, 
 * in which case it defaults to the value returned by |current-input-port|.
 *
 * @strong{Note}: As said in @xref{Multiple values}, if |read-line| is not
 * used in  the context of |call-with-values|, the second value return by 
 * this procedure is ignored.
doc> 
*/
DEFINE_PRIMITIVE("read-line", read_line, subr01, (SCM port))
{
  int prev, c;
  char buffer[INITIAL_LINE_SIZE], *buff;
  size_t i, size = INITIAL_LINE_SIZE;
  SCM res, delim;

  ENTER_PRIMITIVE(read_line);
  port = verify_port(port, PORT_READ);
  buff = buffer;
  prev = ' ';

  for (i = 0; ; i++) {
    if (i == size) {
      /* We must enlarge the buffer */
      size += size / 2;
      if (i == INITIAL_LINE_SIZE) {
	/* This is the first resize. Pass from static to dynamic allocation */
	buff = STk_must_malloc(size);
	strncpy(buff, buffer, INITIAL_LINE_SIZE);
      }
      else 
	buff = STk_must_realloc(buff, size);
    }
    switch (c = STk_getc(port)) {
      case EOF:  res = (i == 0) ? STk_eof : STk_chars2string(buff, i);
		 if (buff != buffer) STk_free(buff);
		 return STk_2_values(res, STk_eof);

      case '\n': if (prev == '\r') 
		   { i -= 1; delim = CrLf; }
      		 else 
		   delim = MAKE_CHARACTER('\n');
	
		 res = STk_chars2string(buff, i);
		 if (buff != buffer) STk_free(buff);
		 return STk_2_values(res, delim);

      default:  buff[i] = prev = c; 
    }
  }
}

#ifdef FIXME
//EG: PRIMITIVE STk_copy_port(SCM in, SCM out)
//EG: {
//EG:   int c;
//EG:   
//EG:   ENTER_PRIMITIVE("copy-port");
//EG: 
//EG:   if (! INP(in))   Serror("bad input port",  in);
//EG:   if (! OUTP(out)) Serror("bad output port", out);
//EG: 
//EG:   while ((c = Getc(in)) != EOF)
//EG:     Putc(c, out);
//EG: 
//EG:   return STk_void;
//EG: }
#endif

/*
<doc EXT flush
 * (flush)
 * (flush port)
 *
 * Flushes the buffer associated with the given output |port|. The
 * |port| argument may be omitted, in which case it defaults to the value
 * returned by |current-output-port|
doc>
 */
DEFINE_PRIMITIVE("flush", port_flush, subr01, (SCM port))
{
  ENTER_PRIMITIVE(port_flush);

  port = verify_port(port, PORT_WRITE);
  if (STk_flush(port))
    STk_error("cannot flush port ~S", port);
  return STk_void;
}



/*
<doc port-current-line
 * (port-current-line)
 * (port-current-line port)
 *
 * Returns the current line number associated to the given input |port| as an
 * integer. The |port| argument may be omitted, in which case it defaults to
 * the value returned by |current-input-port|.
doc>
 */
DEFINE_PRIMITIVE("port-current-line", port_current_line, subr01, (SCM port))
{
  ENTER_PRIMITIVE(port_current_line);

  port = verify_port(port, PORT_READ);
  return MAKE_INT(PORT_LINE(port));
}



/*===========================================================================*\
 * 
 * Initializations
 * 
\*===========================================================================*/

static void print_port(SCM obj, SCM port, int mode)
{
  PORT_PRINT(obj)(obj, port);
}


/* The stucture which describes the port type */
struct extended_type_descr xtype_port = {
  "port",			/* name */
  print_port			/* print function */
};


/*===========================================================================*/

int STk_init_port(void)
{
  /* Initialize tilde_regexp which is used for detecting formatted error
   * messages 
   */
  regcomp(&tilde_regexp, "~(a|A|s|S|w|W|~|%)", 0);

  /* Define a constant for lines terminated by CR/LF to avoid multiple 
   * allocations. Make it constant to avoid the user break it 
   */
  CrLf		       = STk_Cstring2string("\r\n");
  BOXED_INFO(CrLf)    |= STRING_CONST;

  /* Define the port file */
  DEFINE_XTYPE(port, &xtype_port);

  /* and its associated primitives */
  ADD_PRIMITIVE(input_portp);
  ADD_PRIMITIVE(output_portp);
  ADD_PRIMITIVE(interactive_portp);
  ADD_PRIMITIVE(current_input_port);
  ADD_PRIMITIVE(current_output_port);
  ADD_PRIMITIVE(current_error_port);
  ADD_PRIMITIVE(set_std_port);
  ADD_PRIMITIVE(scheme_read);
  ADD_PRIMITIVE(scheme_read_cst);
  ADD_PRIMITIVE(read_char);
  ADD_PRIMITIVE(peek_char);
  ADD_PRIMITIVE(eof_objectp);
  ADD_PRIMITIVE(char_readyp);

  ADD_PRIMITIVE(write);
  ADD_PRIMITIVE(display);
  ADD_PRIMITIVE(newline);
  ADD_PRIMITIVE(write_char);

  ADD_PRIMITIVE(scheme_load);
  ADD_PRIMITIVE(scheme_try_load);

  ADD_PRIMITIVE(write_star);
  ADD_PRIMITIVE(format);
  ADD_PRIMITIVE(scheme_error);

  ADD_PRIMITIVE(close_input_port);
  ADD_PRIMITIVE(close_output_port);
  ADD_PRIMITIVE(close_port);
  ADD_PRIMITIVE(port_closed);

  ADD_PRIMITIVE(read_line);
  ADD_PRIMITIVE(port_flush);
  ADD_PRIMITIVE(port_current_line);

  return STk_init_fport() && 
         STk_init_sport();
}
