#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "wrappers.h"
#include "ux_client_socket.h"
#include "ux_server_socket.h"
#include "network_server_socket.h"
#include "sendfd.h"
#include "recvfd.h"
#include "daemonize.h"
#include "getportnum.h"

struct minion {
  pid_t pid;
  int uxcsd;
  short ready;
};



// we love globals
int MINIONS = 20;
char unix_socket_file[64];
struct minion *minions;
int uxlsd;
int nlsd;
struct sockaddr_un uxaddr;
int uxaddrlen = sizeof(uxaddr);
char *serverprocess[128]; // surely 128 command line args are enough
char DEBUGFILE[64] = "/dev/null"; // writes to /dev/null are fast
FILE *debugmsg = NULL;



// singal handlers
void sigchld(int signo) {
  pid_t pid;
  int stat;
  int counter;

  while ((pid = waitpid(-1, &stat, WNOHANG)) > 0) {
    fprintf(debugmsg, "minion %i terminated\n", (int)pid);
    // find the minion that died and mark it as such
    for(counter = 0; counter < MINIONS; counter++) {
      if(minions[counter].pid == pid) {
	minions[counter].pid = -1;
	close(minions[counter].uxcsd);
      }
    }
      
  }
  return;
}

void sigdie(int signo) {
  int i;

  for(i = 0; i < MINIONS; i++) {
    if(minions[i].pid > -1)
      kill(minions[i].pid, SIGTERM);
  }
  unlink((const char *)unix_socket_file);
  shutdown(nlsd, 2);
  exit(0);
}



void spawnaminion(int whichone){
  pid_t pid;
  int counter;
  int received_fd;
  int uxcsd;

  fprintf(debugmsg, "spawning a minion\n");
  pid = fork();

  if(pid == 0) {
    // we don't want to call the same signal handlers as our parent
    Signal(SIGTERM, SIG_DFL);
    Signal(SIGINT, SIG_DFL);
    // child
    close(uxlsd);
    // close out connections to other minions
    for(counter = 0; counter < MINIONS; counter++) 
      if(minions[counter].pid > -1)
	close(minions[counter].pid);
    uxcsd = ux_client_socket((const char *)unix_socket_file);
    // get the fd from the parent
    if((received_fd = recvfd(uxcsd)) < 0) 
      err_sys("recvfd failed");
    fprintf(debugmsg, "done blocking in child\n");
    dup2(received_fd, 0);
    dup2(received_fd, 1);
    execv(serverprocess[0], serverprocess);
    fprintf(debugmsg, "exec returned badly\n");
    exit(1);
  }

  minions[whichone].pid = pid;
  minions[whichone].ready = 1;

  // wait for the minion to connect. it shouldn't take too long.
  minions[whichone].uxcsd = accept(uxlsd, (struct sockaddr *)&uxaddr, &uxaddrlen);


}


void minioncount() {
  int counter;

  for (counter = 0; counter < MINIONS; counter++) {
    if(minions[counter].pid == -1)
      spawnaminion(counter);
  }
}
    
void printusage(char *argv) {
    fprintf(stderr, "usage: %s [-m numminions] [-l listenq] port absolutepathtobinary [args] ...\n", argv);
    exit(1);
}



int main(int argc, char *argv[]) {
  int counter;
  int pollreturn;
  struct pollfd fdset[1];
  struct sockaddr_in inaddr;
  int inaddrlen = sizeof(inaddr);
  int ncsd;
  int debugmsgfd;
  int i;
  short LISTENQ = 10;
  int listenport;

  if(argc < 3) 
    printusage(argv[0]);

  // daemonize
  daemonize();

  // dynamically set up the minions
  minions = (struct minion *)malloc(sizeof(struct minion) * MINIONS);

  // read in the command line arguements
  while((i = getopt(argc, argv, "m:l:d:")) != EOF) {
    if(i == '?') 
      printusage(argv[0]);
    switch(i) {
    case 'd':
      strcpy(DEBUGFILE, optarg);
      break;
    case 'l':
      LISTENQ = atoi(optarg);
      break;
    case 'm':
      MINIONS = atoi(optarg);
      break;
    }
  }

  // open file for debugging messages
  debugmsgfd = open(DEBUGFILE, O_WRONLY | O_CREAT | O_TRUNC, 0644);
  if(debugmsgfd < 0) {
    perror("open");
    fprintf(stderr, "superd couldn't open debug file.  continuing...\n");
    debugmsgfd = open("/dev/null", O_WRONLY, 0644);
  }

  // attach debug file to a stream and remove buffering
  debugmsg = fdopen(debugmsgfd, "a");
  if(debugmsg == NULL)
    err_quit("couldn't attach debug file to a stream");
  setbuf(debugmsg, NULL);
  fprintf(debugmsg, "start of superd debug\n");

  // check the path to the binary for absoluteness
  if(argv[optind + 1][0] != '/') {
    fprintf(stderr, "the specified binary must have an absolute path\n");
    printusage(argv[0]);
  }

  // set the name of the bin we want to run
  for(counter = 0; argv[counter + optind + 1] != NULL; counter++) 
    serverprocess[counter] = argv[counter + optind + 1];
  serverprocess[counter + optind + 1] = (char *)'\0';
  fprintf(debugmsg, "binary to run: %s\n", serverprocess[0]);

  // name our unix socket file appropriately
  snprintf(unix_socket_file, sizeof(unix_socket_file), "/tmp/%s.%i", argv[0], getpid()); 
  
  // register signal handlers
  Signal(SIGCHLD, sigchld);
  Signal(SIGTERM, sigdie);
  Signal(SIGINT, sigdie);

  // set up unix domain socket
  uxlsd = ux_server_socket((const char *)unix_socket_file, MINIONS);

  // set up network socket
  fprintf(debugmsg, "LISTENQ is %i\n", LISTENQ);
  if((listenport = atoi(argv[optind])) == 0)
    listenport = getportnum(argv[optind], "tcp");
  nlsd = network_server_socket(listenport, LISTENQ);
  fprintf(debugmsg, "looking for port... %s\n", argv[optind]);

  // minion setup
  for(counter = 0; counter < MINIONS; counter++)
    spawnaminion(counter);

  // let the user know what superd's pid is
  fprintf(stderr, "superd's pid is %i\n", (int)getpid());

  // redirect stderr messages
  if((dup2(debugmsgfd, 2)) < 0)
    perror("dup2");

  fdset[0].fd = nlsd;
  fdset[0].events = POLLIN;

  while(1) {
    fprintf(debugmsg, "polling...\n");
    pollreturn = poll(fdset, 1, -1);
    fprintf(debugmsg, "poll returned %i\n", pollreturn);
    if(pollreturn == -1) {
      if(errno == EINTR) {
	fprintf(debugmsg, "poll interrupted.  continuing...\n");
	minioncount();
	continue;
      } else {
	perror("poll");
	exit(1);
      } 
    } // end poll error (if) condition

    // we got a connection from the network
    if(fdset[0].revents & POLLIN) {
      for(counter = 0; counter < MINIONS; counter++) {
	if(minions[counter].ready) {
	  ncsd = accept(nlsd, (struct sockaddr *)&inaddr, &inaddrlen);
	  sendfd(minions[counter].uxcsd, ncsd);
	  close(ncsd);
	  minions[counter].ready = 0;
	  break;
	}
      }
    }

    // let's recheck for dead children in case we're on a system where slow
    // system calls are re-entered when interrupted by signals
    minioncount();

  } // end event (while) loop


}
    
