/*
   SwingWT
   Copyright(c)2003-2004, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

*/

package swingwt.awt;

import org.eclipse.swt.widgets.*;
import org.eclipse.swt.*;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.custom.ExtendedModifyEvent;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.StyledText;

import swingwt.awt.font.TextHitInfo;
import swingwt.awt.image.*;
import swingwt.awt.peer.ComponentPeer;
import swingwt.awt.event.*;
import swingwt.awt.event.EventListener;
import swingwtx.accessibility.*;
import swingwtx.swing.*;

import java.util.*;
import java.text.AttributedCharacterIterator;
import java.text.AttributedString;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 *
 * Tag class for AWT compatibility - subclass JSWTComponent has
 * all the good stuff.
 *
 */
public class Component implements ImageObserver, MenuContainer  {

    /** The SWT peer this component is representing */
    public Control peer = null;
    /** If a default instance of Component is used, then we map a paintable
     *  composite canvas as AWT does */
    protected org.eclipse.swt.widgets.Composite ppeer = null;

    /** The name of this component */
    protected String componentName = this.getClass().getName();

    /** The graphics configuration */
    protected static GraphicsConfiguration graphicsConfig = new GraphicsConfiguration();

    /** The parent container */
    protected Container parent = null;

    /** Used when caching objects with layouts */
    public Object layoutModifier = null;

    /** Cache value */
    protected boolean pVisible = true;
    /** Cache value */
    protected boolean pEnabled = true;
    /** Cache value */
    protected String pToolTipText = "";
    /** Cache value */
    protected swingwt.awt.Color pBackground = null;
    /** Cache value */
    protected swingwt.awt.Color pForeground = null;
    /** Cache value */
    protected Dimension pSize = null;
    /** Cache value */
    protected Dimension pPrefSize = null;
    /** Cache value */
    protected Dimension pMinSize = null;
    /** Cache value */
    protected Dimension pMaxSize = null;
    /** Cache value */
    protected swingwt.awt.Point pLocation = null;
    /** Cache value */
    protected swingwt.awt.Font pFont = null;
    /** Cache value */
    protected Object pLayoutData = null;
    /** Cache value */
    protected swingwt.awt.Cursor pCursor = swingwt.awt.Cursor.getPredefinedCursor(swingwt.awt.Cursor.DEFAULT_CURSOR);
    /** Whether paint events give direct access to component or use an image buffer and render in one go */
    protected boolean pDoubleBuffered = false; 

    /** Action Listener objects */
    protected Vector actionListeners = new Vector();
    /** Mouse Listener objects */
    protected Vector mouseListeners = new Vector();
     /** Mouse Wheel Listener objects */
    protected Vector mouseWheelListeners = new Vector();
   /** Mouse Motion Listener objects */
    protected Vector mouseMotionListeners = new Vector();
    /** Keyboard Listener objects */
    protected Vector keyListeners = new Vector();
    /** Focus Listener objects */
    protected Vector focusListeners = new Vector();
    /** Component Listener objects */
    protected Vector componentListeners = new Vector();
    /** Input Method Listeners objects */
    protected Vector inputMethodListeners = new Vector();
    /** Used with TableLayout to determine whether this component should use up extra width */
    protected boolean consumeAvailableWidth = false;
    /** Used with TableLayout to determine whether this component should use up extra height */
    protected boolean consumeAvailableHeight = false;
    /** Action command */
    protected String actionCommand = "";
    /** Whether a calling program has set the font for this component */
    protected boolean isUserSetFont = false;
    /** Whether this component can get the focus */
    protected boolean focusable = true;
    /** Thread safe object for returning values */
    private Object retval = null;
    private int iretval = 0;
    /** Default adapter for Action Events */
    protected org.eclipse.swt.events.MouseListener swingWTDefaultActionEventAdapter = null;
    /** All-in-one event adapter for components */
    protected Component.SWTEventAdapter swingWTEventAdapter = null;
    /** The accessible context */
    protected AccessibleContext accessibleContext = null;
    /** List of PopupMenus for this Component */
    Vector popupMenus;

    /** Constants */
    public static final float TOP_ALIGNMENT = 0.0f;
    public static final float CENTER_ALIGNMENT = 0.5f;
    public static final float BOTTOM_ALIGNMENT = 1.0f;
    public static final float LEFT_ALIGNMENT = 0.0f;
    public static final float RIGHT_ALIGNMENT = 1.0f;

    /** Override in subclasses to paint on the component */
    public void paint(Graphics g) {
    }

    /** Not normally in Component - member of JComponent, 
     *  implemented here to save a load of class casting
     *  around and to call into paint(Graphics) - this
     *  is the first point of call for repainting..
     */
    protected void paintComponent(Graphics g) {
	paint(g);
    }

    //abstract void paintBackground(int x, int y, int width, int height);

    /**
     * Once a parent component receives an "add" call for a child, this being
     * the child, this should be called to tell us to instantiate the peer
     * and load in any cached properties.
     */
    public void setSwingWTParent(swingwt.awt.Container parent) throws Exception {
        ppeer = new org.eclipse.swt.widgets.Canvas(parent.getComposite(), 0);
        peer = ppeer;
        this.parent = parent;
    }

    /** Returns the SWT peer */
    public Control getSWTPeer() { return peer; }

    public ComponentPeer getPeer() { return null; }

    public void add(PopupMenu popupMenu) {
        if (popupMenus == null) popupMenus = new Vector();
        popupMenus.add(popupMenu);

        // PopupMenus can only belong to one component
        if (popupMenu.parent != null) popupMenu.parent.remove(popupMenu);
        popupMenu.parent = this;
    }

    public void remove(MenuComponent menuComponent) {  if (popupMenus != null) popupMenus.remove(menuComponent); }
    
    /** Makes the component visible */
    public void show() { setVisible(true); }
    /** Makes the component invisible */
    public void hide() { setVisible(false); }

    /** Change the visibility of the component */
    public void setVisible(final boolean b) { 
        pVisible = b;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer))
                    peer.setVisible(b);
            }
        });
    }
    /** Returns true if the component is visible */
    public boolean isVisible() { return pVisible; }
    /** Returns true if the component is visible */
    public boolean isShowing() { return isVisible(); }

    /** Change the enabled state of the component */
    public void setEnabled(final boolean b) { 
        pEnabled = b;
        SwingUtilities.invokeAsync(new Runnable() { 
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer))
                    peer.setEnabled(b);
            } 
        }); 
    }
    /** Returns true if the component is enabled */
    public boolean isEnabled() { return pEnabled; }
    /** Requests the component receives the focus */
    public void requestFocus() { SwingUtilities.invokeSync(new Runnable() { public void run() { if (SwingWTUtils.isSWTControlAvailable(peer)) peer.setFocus();}}); }
    /** Forces keyboard focus to the component */
    public void grabFocus() { SwingUtilities.invokeSync(new Runnable() { public void run() {if (SwingWTUtils.isSWTControlAvailable(peer)) peer.forceFocus();}}); }

    public void repaint(int x, int y, int width, int height) {
        repaint(0, x, y, width, height);
    }

    public void repaint(long tm, int x, int y, int width, int height) {
        // TODO
        repaint();
    }

    /** Redraws the component */
    public void repaint() {
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer)) {
                    // If this component is drawable peer (ie. has
                    // no descendant overriding it), then we assume the developer
                    // wanted to paint on it, so we call down to paint() again
                    if (ppeer != null) {
                        Graphics g = getGraphics();
			paintComponent(g);
                        g.dispose();
                    }
                    else {
                        // Otherwise, tell the SWT peer widget to redraw
                        peer.redraw();
		    }
                }
            }
        });
    }

    /** Returns the background colour of the component */
    public swingwt.awt.Color getBackground() {
        if (!SwingWTUtils.isSWTControlAvailable(peer))
            return pBackground;
        else {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    retval = new swingwt.awt.Color(peer.getBackground());
                }
            });
            return (Color) retval;
        }
    }
    /** Sets the background colour of the component */
    public void setBackground(final swingwt.awt.Color c) {
        if (!SwingWTUtils.isSWTControlAvailable(peer))
            pBackground = c;
        else
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    peer.setBackground(c.getSWTColor());
                }
            });
    }
    /** Returns the foreground colour of the component */
    public swingwt.awt.Color getForeground() {
        if (!SwingWTUtils.isSWTControlAvailable(peer))
            return pForeground;
        else {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                   retval = new swingwt.awt.Color(peer.getForeground());
                }
            });
            return (Color) retval;
        }
    }
    /** Sets the foreground colour of the component */
    public void setForeground(final swingwt.awt.Color c) {
        if (!SwingWTUtils.isSWTControlAvailable(peer))
            pForeground = c;
        else
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    peer.setForeground(c.getSWTColor());
                }
            });
    }

    /** Returns the width of the component */
    public int getWidth() {
        return getSize().width;
    }

    /** Returns the height of the component */
    public int getHeight() {
        return getSize().height;
    }

    /** Sets the width of the component. */
    public void setWidth(final int width) {
        if (!SwingWTUtils.isSWTControlAvailable(peer)) return;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                org.eclipse.swt.graphics.Point p = peer.getSize();
                p.x = width; peer.setSize(p);
            }
        });
    }

    /** Sets the height of the component. */
    public void setHeight(final int height) {
        if (!SwingWTUtils.isSWTControlAvailable(peer)) return;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                org.eclipse.swt.graphics.Point p = peer.getSize();
                p.y = height; peer.setSize(p);
            }
        });
    }

    /** Return a graphics context for drawing on this object. Note that only instances of JComponent
     *  are allowed */
    public Graphics getGraphics() {
        if (!SwingWTUtils.isSWTControlAvailable(peer))
            return null;
        else {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    graphics = new SWTGraphics2DRenderer(new org.eclipse.swt.graphics.GC(peer), true);
                }
            });
            return (Graphics) graphics;
        }
    }
    private Object graphics = null;

   /** Returns the preferred size the component would like to be displayed at -
     * very important for proper layout managers.
     * @author Robin Rawson-Tetley
     */
    public Dimension getPreferredSize() {
	    
        // Since preferred size is fulfilling it's "real" role now, it needs to be
        // the height the component would prefer to be displayed at. This is either
        // what the user set, or (if not set), a computed estimate
	// 
        if (pPrefSize != null)
            return pPrefSize;
	
        else if (SwingWTUtils.isSWTControlAvailable(peer)) {
            // Toolbars and CoolBars don't use AWT layouts
            if (this instanceof JToolBar || this instanceof swingwtx.custom.JCoolBar) {
                org.eclipse.swt.graphics.Point p = peer.computeSize(-1, -1);
                return new Dimension(p.x, p.y);
            }
            // Container classes
            else if (this instanceof Container) {

                // If there's no composite peer (ie. It's really a component),
		// use the component size
		if (((Container) this).getComposite() == null) {
		    return computePreferredSize();
		}
		    
		// If there's no layout, get the peer
		// to compute their size
                if (((Container) this).getLayout() == null) 
		    return computePreferredSize();
		
		// Otherwise, use the layout manager's preferred
		// size.
                return ((Container) this).getLayout().preferredLayoutSize((Container) this);
            }
            else
            {
                // Since we corrected the hierarchy, all components are containers now
                // since they descend JComponent like Swing. This should never get
                // called, but I leave it here just in case someone subclasses Component
                // or something.
                return computePreferredSize();
            }
        }
        else {
            // The component has not yet been realised, so we are going to
            // have to make some guesses based on the type
            return calculatePreferredSize();
        } 
    }
    
    /** This is called by getPreferredSize() when we want the peer
     *  to compute it's own size. This base method delegates to the
     *  peer, but some components may want to override this value
     *  hence the reason we have a separate method here.
     *
     *  NOTE THAT THIS ROUTINE IS DIFFERENT FROM calculatePreferredSize() -
     *  this routine is called by getPreferredSize() when none has been set
     *  by the user and allows the peer to compute it's size, 
     *  calculatePreferredSize() is used when no peer has been created and
     *  we need to guess an appropriate size.
     */
    protected Dimension computePreferredSize() {
       org.eclipse.swt.graphics.Point p = peer.computeSize(-1, -1); 
       return new Dimension(p.x, p.y);
    }
    
    /** Sets the preferred size of the object for layout purposes */
    public void setPreferredSize(Dimension d) { pPrefSize = d; }
    /** 
     *  Override in subclasses to return the preferred size for
     *  a given (non-realised) component.
     *
     *  Note that you should use setSize() to whatever this routine
     *  returns once called to prevent inconsistency with null
     *  layout managers.
     *
     *  Also note that this is different from the computePreferredSize()
     *  method, which is used for calculating the preferredSize of a
     *  realised component (this one is for unrealised components)
     */
    protected Dimension calculatePreferredSize() {
        Dimension size = new Dimension(150, 50);
        setSize(size);
        return size;
    }
    
    /** Returns the maximum size of the component */
    public Dimension getMaximumSize() { if (pMaxSize == null) pMaxSize = new Dimension(1024, 1024); return pMaxSize; }
    /** Sets the maximum size of the component */
    public void setMaximumSize(Dimension d) { pMaxSize = d; }
    /** Returns the minimum size of the component */
    public Dimension getMinimumSize() { if (pMinSize == null) pMinSize = new Dimension(1, 1); return pMinSize; }
    /** Sets the minimum size of the component */
    public void setMinimumSize(Dimension d) { pMinSize = d; }
    public boolean isMinimumSizeSet() { return pMinSize!=null; }

    /** Sets the peer's actual size */
    public void setSize(final int width, final int height) {
        setSize(new Dimension(width, height));
    }
    /** Sets the peer's actual size */
    public void setSize(final Dimension d) {
        
        pSize = d;
        if (!SwingWTUtils.isSWTControlAvailable(peer)) return;
        
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                peer.setBounds(peer.getLocation().x, peer.getLocation().y, d.width, d.height);
            }
        });
    }
    
    public boolean contains(int x, int y) {
        return (x >= 0 && y >= 0) && 
        	   (x < getWidth() && y < getHeight());
    }
    
    public boolean contains(Point point) { return contains(point.x, point.y); }
    public boolean inside(int x, int y) { return contains(x, y); }

    /**
     * Returns the peers actual size. Or, if the peer is embedded inside
     * a ScrollPane component, the larger of the preferredSize/peers
     * actual size.
     * This works rather nicely for handling layouts inside a scrollbar as
     * we can simply offset a large composite according to the scrollbar
     * positions inside JScrollPane.
     *
     * One note - if the component is JTree, JTable, JTextArea or JEditorPane, then
     * we return the size as usual since these components handle their
     * own scrolling and we don't want the JScrollPane behaving weirdly.
     */
    public Dimension getSize() {
        if (parent instanceof JScrollPane &&
           (!(this instanceof JTable)) &&
           (!(this instanceof JTextArea)) &&
           (!(this instanceof JTree)) &&
           (!(this instanceof JEditorPane))
           ) {
            Dimension returnDim = getPreferredSize();
            Dimension peers = getPeerSize();
            if (returnDim.width < peers.width) returnDim.width = peers.width;
            if (returnDim.height < peers.height) returnDim.height = peers.height;
            return returnDim;
        }
        else
            return getPeerSize();
    }
    
    /**
     * Overloaded version of getSize() that uses the passed
     * in dimension to put the size in to prevent allocation of
     * another on the heap.
     *
     * NB: Still creates a new dimension due to how SwingWT works
     *     so there is no benefit to using this method.
     */
    public Dimension getSize(Dimension rv) {
        return getSize();
    }

    /** Returns the peers actual size, or 0,0 if it hasn't been created yet/is invisible */
    public Dimension getPeerSize() {
        if (!SwingWTUtils.isSWTControlAvailable(peer))
            return pSize == null ? new Dimension(0,0) : pSize;
        final Dimension d = new Dimension();
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                d.height = peer.getSize().y;
                d.width = peer.getSize().x;
            }
        });
        return d;
    }
    
    /** FIXME: NOT IMPLEMENTED */
    public void setFocusTraversalKeysEnabled(boolean b) {}

    /** Returns the component's location on screen */
    public Point getLocation() {
        if (!SwingWTUtils.isSWTControlAvailable(peer))
            return pLocation == null ? new Point(0,0) : pLocation;
        final Point p = new Point();
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                p.x = peer.getLocation().x;
                p.y = peer.getLocation().y;
            }
        });
        return p;
    }

    /** FIXME: Bad algorithm here, but can't find appropriate routine. Display.map() looks
     *  interesting, but it is undocumented.
     */
    public Point getLocationOnScreen() {
        final Point p = new Point(0, 0);
        if (!SwingWTUtils.isSWTControlAvailable(peer)) return p;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                p.x = peer.getLocation().x + peer.getShell().getLocation().x;
                p.y = peer.getLocation().y + peer.getShell().getLocation().y;
            }
        });
        return p;
    }

    /** Creates and returns an image */
    public Image createImage(int width, int height) {
        swingwt.awt.image.BufferedImage i = new swingwt.awt.image.BufferedImage(width, height, swingwt.awt.image.BufferedImage.TYPE_INT_RGB);
        return i;
    }

    /** Creates and returns a volatile image */
    public swingwt.awt.image.VolatileImage createVolatileImage(int width, int height) {
        swingwt.awt.image.VolatileImage i = new swingwt.awt.image.VolatileImage(width, height, swingwt.awt.image.BufferedImage.TYPE_INT_RGB);
        return i;
    }

    /** Returns the font metrics for the given font */
    public swingwt.awt.FontMetrics getFontMetrics(swingwt.awt.Font f) {
        return Toolkit.getDefaultToolkit().getFontMetrics(f);
    }
    

    /** Returns the Font used on the component */
    public swingwt.awt.Font getFont() {
        if (!SwingWTUtils.isSWTControlAvailable(peer))
            return pFont;
        else {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    retval = new swingwt.awt.Font(peer.getFont());
                }
            });
            return (swingwt.awt.Font) retval;
        }
    }
    /** Sets the font used by the component */
    public void setFont(final swingwt.awt.Font f) {
        if (f!=null) {
            isUserSetFont = true;
            if (!SwingWTUtils.isSWTControlAvailable(peer))
                pFont = f;
            else
                SwingUtilities.invokeSync(new Runnable() {
                        public void run() {
                            peer.setFont(f.getSWTFont());
                        }
                    });
        } else {
            isUserSetFont = false;
        }
    }

    /**
     * Returns true if a calling process has set the font on
     * this component. So other parts of SwingWT know when to
     * override the standard font as part of your system theme.
     */
    public boolean hasSetFont() {
        return isUserSetFont;
    }

    /** Returns the tooltip text used on the component */
    public String getToolTipText() { return pToolTipText; }
    /** Sets the tooltip text used on the component */
    public void setToolTipText(final String text) { pToolTipText = text; SwingUtilities.invokeSync(new Runnable() { public void run() {if (SwingWTUtils.isSWTControlAvailable(peer)) peer.setToolTipText(text);}}); }
    /** Sets the physical bounds of the component - used by layout managers */
    public void setBounds(final int x, final int y, final int width, final int height) {
        
        pSize = new Dimension(width, height);
        pLocation = new Point(x, y);
        
        if (!SwingWTUtils.isSWTControlAvailable(peer)) return;
        
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                peer.setBounds(x, y, width, height);
            }
        });
    }
    /** Sets the physical bounds of the component */
    public void setBounds(Rectangle r) {
        setBounds(r.x, r.y, r.width, r.height);    
    }
    /** Returns the actual bounds of the component (or 0,0,0,0 if it isn't realised yet) */
    public Rectangle getBounds() {
        if (!SwingWTUtils.isSWTControlAvailable(peer)) return new Rectangle(0, 0, 0, 0);
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                retval = new Rectangle(peer.getLocation().x, peer.getLocation().y, peer.getSize().x, peer.getSize().y);
            }
        });
        return (Rectangle) retval;
    }
    /** NOT IMPLEMENTED */
    public float getAlignmentX() { return 0; }
    /** NOT IMPLEMENTED */
    public float getAlignmentY() { return 0; }
    /** NOT IMPLEMENTED */
    public void setAlignmentX(float val) {}
    /** NOT IMPLEMENTED */
    public void setAlignmentY(float val) {}
    
    /** Sets the focus to the next component as if tab was pressed */
    public void transferFocus() {
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                peer.traverse(SWT.TRAVERSE_TAB_NEXT);    
            }
        });
    }
    
    /** Sets the focus to the next component as if shift+tab was pressed */
    public void transferFocusBackward() {
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                peer.traverse(SWT.TRAVERSE_TAB_PREVIOUS);    
            }
        });
    }
    
    /** NOT IMPLEMENTED: FIXME - focusCycleRoot not done */
    public void transferFocusUpCycle() {
    }

    /** Override in container classes to relayout child
      * components - calls repaint here */
    public void invalidate() { repaint(); }

    /** NOT IMPLEMENTED */
    public void validate() {}
    
    /** Returns the action command for this component */
    public String getActionCommand() { return actionCommand; }
    /** Sets the action command for this component */
    public void setActionCommand(String command) { actionCommand = command; }
    /** Sets the component's location in it's container */
    public void setLocation(final Point p) {
        pLocation = p;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer))
                    peer.setBounds(p.x, p.y, peer.getSize().x, peer.getSize().y);
            }
        });
    }
    /** Sets the component's location in it's container */
    public void setLocation(int x, int y) {
        setLocation(new Point(x, y));
    }
    /** Returns the component's X position in it's parent container */
    public int getX() {
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer))
                    iretval = peer.getLocation().x;
                else
                    iretval = 1;
            }
        });
        return iretval;
    }
    /** Returns the component's Y position in it's parent container */
    public int getY() {
            SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(peer))
                    iretval = peer.getLocation().y;
                else
                    iretval = 1;
            }
        });
        return iretval;
    }
    /** Returns the component's parent */
    public Container getParent() { return parent; }

    /** Returns the components name */
    public String getName() { return componentName; }
    /** Sets the components name */
    public void setName(String newName) { componentName = newName; }
    /** Sets the components cursor */
    public void setCursor(final swingwt.awt.Cursor c) {
        pCursor = c;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
		try {
                    if (SwingWTUtils.isSWTControlAvailable(peer))
			peer.setCursor(c.getSWTCursor());
		}
                catch (IllegalArgumentException e) {}
		catch (Exception e) {
		    e.printStackTrace();
		}
            }
        });
    }
    /** Returns true if the component can receive the focus */
    public boolean isFocusable() { return focusable; }
    /** Set whether the component can be focused */
    public void setFocusable(boolean b) { focusable = b; }
    /** Returns the components cursor */
    public swingwt.awt.Cursor getCursor() { return pCursor; }
    /** Gets the Toolkit for this component */
    public Toolkit getToolkit() { return Toolkit.getDefaultToolkit(); }
    /** Gets the GraphicsConfiguration for this component */
    public GraphicsConfiguration getGraphicsConfiguration() { return graphicsConfig; }

    public ComponentOrientation getComponentOrientation() { return ComponentOrientation.LEFT_TO_RIGHT; }
    public void setComponentOrientation(ComponentOrientation o) {  }

    /** Destroys the component and stops all listeners */
    public void dispose() {
        componentOnlyDispose();
    }

    /**
     *  Because dispose() can be overridden to do more, calling this routine
     *  guarantees that destruction of the component occurs (and nothing else).
     */
    public void componentOnlyDispose() {

        if (SwingWTUtils.isSWTControlAvailable(peer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    peer.dispose();
                }
            });
        }

        peer = null;
        ppeer = null;

        mouseListeners.removeAllElements();
	mouseWheelListeners.removeAllElements();
        mouseMotionListeners.removeAllElements();
        actionListeners.removeAllElements();
        focusListeners.removeAllElements();
        keyListeners.removeAllElements();
        componentListeners.removeAllElements();
        inputMethodListeners.removeAllElements();
    }

    /** Add an <code>ActionListener</code> to this component */
    public void addActionListener(ActionListener l) {
        actionListeners.add(l);
    }

    /** Remove an <code>ActionListener</code> from the component */
    public void removeActionListener(ActionListener l) {
        actionListeners.remove(l);
    }

    /** Add an <code>ComponentListener</code> to this component */
    public void addComponentListener(ComponentListener l) {
        componentListeners.add(l);
    }

    /** Remove an <code>ComponentListener</code> from the component */
    public void removeComponentListener(ComponentListener l) {
        componentListeners.remove(l);
    }

    /** Add a <code>MouseListener</code> to this component */
    public void addMouseListener(MouseListener l) {
        mouseListeners.add(l);
    }

    /** Remove an <code>MouseListener</code> from the component */
    public void removeMouseListener(MouseListener l) {
        mouseListeners.remove(l);
    }

    /** Add a <code>MouseWheelListener</code> to this component */
    public void addMouseWheelListener(MouseWheelListener l) {
        mouseWheelListeners.add(l);
    }

    /** Remove a <code>MouseWheelListener</code> to this component */
    public void removeMouseWheelListener(MouseWheelListener l) {
        mouseWheelListeners.remove(l);
    }

    /** Add a <code>MouseMotionListener</code> to this component */
    public void addMouseMotionListener(MouseMotionListener l) {
        mouseMotionListeners.add(l);
    }

    /** Remove an <code>MouseMotionListener</code> from the component */
    public void removeMouseMotionListener(MouseMotionListener l) {
        mouseMotionListeners.remove(l);
    }

    /** Add a <code>KeyListener</code> to this component */
    public void addKeyListener(KeyListener l) {
        keyListeners.add(l);
    }

    /** Remove an <code>KeyListener</code> from the component */
    public void removeKeyListener(KeyListener l) {
        keyListeners.remove(l);
    }

    /** Add a <code>KeyListener</code> to this component */
    public void addFocusListener(FocusListener l) {
        focusListeners.add(l);
    }

    /** Remove an <code>KeyListener</code> from the component */
    public void removeFocusListener(FocusListener l) {
        focusListeners.remove(l);
    }

    /** Add an <code>InputMethodListener</code> to this component */
    public void addInputMethodListener(InputMethodListener l) {
        inputMethodListeners.add(l);
    }

    /** Remove an <code>InputMethodListener</code> to this component */
    public void removeInputMethodListener(InputMethodListener l) {
        inputMethodListeners.remove(l);
    }

    // TODO: handle event dispatching here?
    public void dispatchEvent(AWTEvent awtEvent) {}
    
    // TODO: do something here?
    public void addNotify() {}
    
    // TODO: do something here?
    public void removeNotify() {}
    
    public EventListener[] getListeners(Class listenerType) {
    	if (listenerType.equals(ActionListener.class)) {
    		return createListenerArrayFromVector(actionListeners);
    	}
    	if (listenerType.equals(MouseListener.class)) {
    		return createListenerArrayFromVector(mouseListeners);
    	}
    	if (listenerType.equals(KeyListener.class)) {
    		return createListenerArrayFromVector(keyListeners);
    	}
    	if (listenerType.equals(FocusListener.class)) {
    		return createListenerArrayFromVector(focusListeners);
    	}
    	return null;
    }
    
    protected EventListener[] createListenerArrayFromVector(Vector listeners) {
    	EventListener[] arr = new EventListener[listeners.size()]; 
    	for (int i = 0; i < listeners.size(); i++) {
    		arr[i] = (EventListener) listeners.get(i);
    	}
    	return arr;
    }
    
    /**
     *  Sends action events to listeners
     */
    public void processActionEvent(int id) {
        Iterator i = actionListeners.iterator();
        ActionEvent ae = new ActionEvent(this, id, this.getActionCommand());
        while (i.hasNext()) {
            ActionListener al = (ActionListener) i.next();
            al.actionPerformed(ae);
        }
    }

    public void processComponentEvent(ComponentEvent e) {
        for (int i = 0; i < componentListeners.size(); i++) {
            if (e.getID() == ComponentEvent.COMPONENT_HIDDEN)
                ((ComponentListener) componentListeners.get(i)).componentHidden(e);
            if (e.getID() == ComponentEvent.COMPONENT_MOVED)
                ((ComponentListener) componentListeners.get(i)).componentMoved(e);
            if (e.getID() == ComponentEvent.COMPONENT_RESIZED)
                ((ComponentListener) componentListeners.get(i)).componentResized(e);
            if (e.getID() == ComponentEvent.COMPONENT_SHOWN)
                ((ComponentListener) componentListeners.get(i)).componentShown(e);
        }
    }

    /**
     *  Sends mouse events to component listeners
     */
    public void processMouseEvent(MouseEvent e) {
        Iterator i = mouseListeners.iterator();
        while (i.hasNext()) {
            MouseListener ml = (MouseListener) i.next();
            if (e.eventID == MouseEvent.CLICKED) ml.mouseClicked(e);
            else if (e.eventID == MouseEvent.ENTERED) ml.mouseEntered(e);
            else if (e.eventID == MouseEvent.EXITED) ml.mouseExited(e);
            else if (e.eventID == MouseEvent.PRESSED) ml.mousePressed(e);
            else if (e.eventID == MouseEvent.RELEASED) ml.mouseReleased(e);
        }
    }

    /**
     *  Sends mouse motion events to component listeners
     */
    public void processMouseMotionEvent(MouseEvent e) {
        for (int i = 0; i < mouseMotionListeners.size(); i++) {
            MouseMotionListener ml = (MouseMotionListener) mouseMotionListeners.get(i);
            if (e.getButton() != 0) {
                ml.mouseDragged(e);
            }
            else
                ml.mouseMoved(e);
        }
    }

    /**
     * Sends mouse wheel events to component listeners
     **/
   protected void processMouseWheelEvent(MouseWheelEvent e) {
        for (int i = 0; i < mouseWheelListeners.size(); i++) {
            MouseWheelListener mwl = (MouseWheelListener) mouseWheelListeners.get(i);
            mwl.mouseWheelMoved(e);
        }
   }

    /**
     * Sends KeyEvents to component listeners
     */
    public void processKeyEvent(KeyEvent e) {
        for (int i = 0; i < keyListeners.size(); i++) {
            KeyListener ml = (KeyListener) keyListeners.get(i);
            if (e.getID() == KeyEvent.TYPED) ml.keyTyped(e);
            if (e.getID() == KeyEvent.RELEASED) ml.keyReleased(e);
            if (e.getID() == KeyEvent.PRESSED) ml.keyPressed(e);
        }
    }

    /**
     * Sends Focus Events to component listeners
     */
    public void processFocusEvent(FocusEvent e) {
        for (int i = 0; i < focusListeners.size(); i++) {
            FocusListener fl = (FocusListener) focusListeners.get(i);
            if (e.getID() == FocusEvent.FOCUS_GAINED) fl.focusGained(e);
            if (e.getID() == FocusEvent.FOCUS_LOST) fl.focusLost(e);
        }
    }

     /**
      * Sends Input Method Events to component listeners
      */
     public void processInputMethodEvent(InputMethodEvent e) {
         for (int i = 0; i < inputMethodListeners.size(); i++) {
             InputMethodListener iml = (InputMethodListener) inputMethodListeners.get(i);
             if (e.getID() == InputMethodEvent.CARET_POSITION_CHANGED) iml.caretPositionChanged(e);
             if (e.getID() == InputMethodEvent.INPUT_METHOD_TEXT_CHANGED) iml.inputMethodTextChanged(e);
         }
     }

    /**
     * If the component has just been added to a container, but
     * properties were set in the meantime, this routine gets
     * called by JSWTComposite to set the cached properties.
     */
    public void setCachedProperties() {
        if (peer == null) return;
        peer.setToolTipText(pToolTipText);
        if (pBackground != null) peer.setBackground(pBackground.getSWTColor());
        if (pForeground != null) peer.setForeground(pForeground.getSWTColor());
        if (pFont != null) peer.setFont(pFont.getSWTFont());
        if (pSize != null && pLocation != null) {
            peer.setBounds(pLocation.x, pLocation.y, pSize.width, pSize.height);
        }
        else if (pSize != null)
            peer.setBounds(peer.getLocation().x, peer.getLocation().y, pSize.width, pSize.height);
        else if (pLocation != null)
            peer.setBounds(pLocation.x, pLocation.y, peer.getSize().x, peer.getSize().y);
        if (pCursor != null && !pCursor.equals(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR))) peer.setCursor(pCursor.getSWTCursor());
        if (pLayoutData != null) peer.setLayoutData(pLayoutData);
        peer.setVisible(pVisible);
        peer.setEnabled(pEnabled);
    }

    /**
     * When the component has been added to a container, this
     * routine gets called to tell it to set up SWT listeners
     * for all its events. We can map these then to the
     * AWT-style events
     */
    public void registerEvents() {

        if (!SwingWTUtils.isSWTControlAvailable(peer)) return;

        if (swingWTEventAdapter == null)
            swingWTEventAdapter = new Component.SWTEventAdapter(this);

        registerActionEvents();
        registerComponentEvents();
        registerMouseEvents();
        registerKeyEvents();
        registerFocusEvents();
        registerTraversalEvents();
        registerPaintEvents();
        registerModifyEvents();
        registerExtendedModifyEvents();
    }

    /**
     * Adapter class to map SWT events to SwingWT ones. We use this for 2 reasons:
     * 1. Less anonymous inner classes - only needs one to handle all events.
     * 2. We can guarantee there will only be one SwingWT adapter
     */
    private class SWTEventAdapter implements org.eclipse.swt.events.PaintListener,
                                             org.eclipse.swt.events.TraverseListener,
                                             org.eclipse.swt.events.FocusListener,
                                             org.eclipse.swt.events.MouseListener,
                                             org.eclipse.swt.events.MouseTrackListener,
                                             org.eclipse.swt.events.MouseMoveListener,
                                             org.eclipse.swt.events.KeyListener,
                                             org.eclipse.swt.events.ControlListener,
                                             org.eclipse.swt.custom.ExtendedModifyListener,
                                             org.eclipse.swt.events.ModifyListener {

        /** The component this class is acting as an event adapter for */
        private Component parent = null;

        public SWTEventAdapter(Component parent) {
            this.parent = parent;
        }
        public void paintControl(org.eclipse.swt.events.PaintEvent e) {
            
            // If the double-buffered flag is set, give the component an image to paint
            // on and we'll render that onto the GC instead.
            if (pDoubleBuffered) {
                org.eclipse.swt.graphics.Image img = 
                    new org.eclipse.swt.graphics.Image(SwingWTUtils.getDisplay(), e.width, e.height);
                org.eclipse.swt.graphics.GC gc = 
                    new org.eclipse.swt.graphics.GC(img);
                // Call out to the paint method
		paintComponent(new SWTGraphics2DRenderer(gc));
                // Copy the buffered image onto the real gc
                e.gc.drawImage(img, 0, 0);
                // Destroy our temporary gc and image
                gc.dispose(); img.dispose();
            }
            else 
	         paintComponent(new SWTGraphics2DRenderer(e.gc));
            
        }
        public void keyTraversed(org.eclipse.swt.events.TraverseEvent e) {
            processTraverseEvent(e);
        }
        public void focusGained(org.eclipse.swt.events.FocusEvent e) {
            FocusEvent ev = new FocusEvent(parent, FocusEvent.FOCUS_GAINED);
            processFocusEvent(ev);
        }
        public void focusLost(org.eclipse.swt.events.FocusEvent e) {
            FocusEvent ev = new FocusEvent(parent, FocusEvent.FOCUS_LOST);
            processFocusEvent(ev);
        }
        public void mouseDown(org.eclipse.swt.events.MouseEvent e) {
            MouseEvent ev = new MouseEvent(parent);
            ev.clickCount = 1;
            ev.eventID = MouseEvent.PRESSED;
            ev.setX(e.x);
            ev.setY(e.y);
            ev.setButton(e.button);
            processMouseEvent(ev);
        }
        public void mouseDoubleClick(org.eclipse.swt.events.MouseEvent e) {
            MouseEvent ev = new MouseEvent(parent);
            ev.clickCount = 2;
            ev.eventID = MouseEvent.CLICKED;
            ev.setX(e.x);
            ev.setY(e.y);
            ev.setButton(e.button);
            processMouseEvent(ev);
        }
        public void mouseUp(org.eclipse.swt.events.MouseEvent e) {
            MouseEvent ev = new MouseEvent(parent);
            ev.clickCount = 1;
            ev.eventID = MouseEvent.RELEASED;
            ev.setX(e.x);
            ev.setY(e.y);
            ev.setButton(e.button);
            processMouseEvent(ev);
            processMouseMotionEvent(ev);  // Released should also fire a drag event
            ev.eventID = MouseEvent.CLICKED;
            processMouseEvent(ev);
        }
        public void mouseEnter(org.eclipse.swt.events.MouseEvent e) {
            MouseEvent ev = new MouseEvent(parent);
            ev.clickCount = 0;
            ev.eventID = MouseEvent.ENTERED;
            ev.setX(e.x);
            ev.setY(e.y);
            ev.setButton(e.button);
            processMouseEvent(ev);
        }
        public void mouseExit(org.eclipse.swt.events.MouseEvent e) {
            MouseEvent ev = new MouseEvent(parent);
            ev.clickCount = 0;
            ev.eventID = MouseEvent.EXITED;
            ev.setX(e.x);
            ev.setY(e.y);
            ev.setButton(e.button);
            processMouseEvent(ev);
        }
        public void mouseHover(org.eclipse.swt.events.MouseEvent e) {
            MouseEvent ev = new MouseEvent(parent);
            ev.eventID = MouseEvent.ENTERED;
            ev.setX(e.x);
            ev.setY(e.y);
            ev.setButton(e.button);
            processMouseMotionEvent(ev);
        }
        public void mouseMove(org.eclipse.swt.events.MouseEvent e) {
            MouseEvent ev = new MouseEvent(parent);
            ev.eventID = MouseEvent.ENTERED;
            ev.setX(e.x);
            ev.setY(e.y);
            ev.setButton(e.button);
            processMouseMotionEvent(ev);
        }
        public void keyPressed(org.eclipse.swt.events.KeyEvent e) {
            KeyEvent ev = new KeyEvent(parent, KeyEvent.PRESSED, e.time, KeyEvent.translateSWTModifiers(e.stateMask), KeyEvent.translateSWTKey(e.keyCode), e.character);
            processKeyEvent(ev);

            // consumption
            if (e.doit) e.doit = !ev.isConsumed();
        }
        public void keyReleased(org.eclipse.swt.events.KeyEvent e) {
            KeyEvent ev = new KeyEvent(parent, KeyEvent.RELEASED, e.time, KeyEvent.translateSWTModifiers(e.stateMask), KeyEvent.translateSWTKey(e.keyCode), e.character);
            processKeyEvent(ev);

            // consumption
            if (e.doit) e.doit = !ev.isConsumed();
        }

        public void controlMoved(org.eclipse.swt.events.ControlEvent controlEvent) {
            processComponentEvent(new ComponentEvent(parent, ComponentEvent.COMPONENT_MOVED));
        }

        public void controlResized(org.eclipse.swt.events.ControlEvent controlEvent) {
            processComponentEvent(new ComponentEvent(parent, ComponentEvent.COMPONENT_RESIZED));
        }

        public void modifyText(ExtendedModifyEvent modifyEvent) {

            // Modified text uses the INPUT_METHOD_TEXT_CHANGED event.  caret change?  not sure yet
            int id = InputMethodEvent.INPUT_METHOD_TEXT_CHANGED;

            if (id > 0) {

                String fullText = "";
                if ( peer instanceof StyledText ) fullText = ((StyledText)peer).getText();
                if ( peer instanceof Text ) fullText = ((Text)peer).getText();
                if ( peer instanceof CCombo ) fullText = ((CCombo)peer).getText();
                if ( peer instanceof Combo ) fullText = ((Combo)peer).getText();

                int caretPosition = modifyEvent.start + modifyEvent.length;

                AttributedString attributedString = new AttributedString(fullText);
                AttributedCharacterIterator text = attributedString.
                        getIterator(null, modifyEvent.start, caretPosition);
                TextHitInfo textHitInfo = TextHitInfo.trailing(caretPosition);
                InputMethodEvent inputMethodEvent = new InputMethodEvent( parent, id, text, modifyEvent.length,
                                                                          textHitInfo, textHitInfo );
                processInputMethodEvent(inputMethodEvent);

            }
            else {
                throw new IllegalComponentStateException("Unknown ExtendedModifyState error");
            }
        }

        public void modifyText(ModifyEvent modifyEvent) {
            // HTF do you use this event?
        }

        
    }


    protected void registerPaintEvents() {
        peer.removePaintListener(swingWTEventAdapter);
        peer.addPaintListener(swingWTEventAdapter);
    }

    protected void registerTraversalEvents() {
        peer.removeTraverseListener(swingWTEventAdapter);
        peer.addTraverseListener(swingWTEventAdapter);
    }

    protected void registerFocusEvents() {
        peer.removeFocusListener(swingWTEventAdapter);
        peer.addFocusListener(swingWTEventAdapter);
    }

    protected void registerComponentEvents() {
        peer.removeControlListener(swingWTEventAdapter);
        peer.addControlListener(swingWTEventAdapter);
    }

    protected void registerMouseEvents() {
        peer.removeMouseListener(swingWTEventAdapter);
        peer.removeMouseTrackListener(swingWTEventAdapter);
        peer.addMouseListener(swingWTEventAdapter);
        peer.addMouseTrackListener(swingWTEventAdapter);
    }

    protected void registerKeyEvents() {
        peer.removeKeyListener(swingWTEventAdapter);
        peer.addKeyListener(swingWTEventAdapter);
    }

    protected void registerActionEvents() {
        /* Should this be here?  It triggers action events on mouse clicks
        if (swingWTDefaultActionEventAdapter == null)
            swingWTDefaultActionEventAdapter = new org.eclipse.swt.events.MouseListener() {
                public void mouseDown(org.eclipse.swt.events.MouseEvent e) {
                    processActionEvent(0);
                }
                public void mouseDoubleClick(org.eclipse.swt.events.MouseEvent e) {
                    processActionEvent(0);
                }
                public void mouseUp(org.eclipse.swt.events.MouseEvent e) {
                }
            };
            */

        if (swingWTDefaultActionEventAdapter != null) {
	        peer.removeMouseListener(swingWTDefaultActionEventAdapter);
	        peer.addMouseListener(swingWTDefaultActionEventAdapter);
        }
    }


    /**
     * Handle ModifyEvents (maps to ImputMethodEvents) by checking the final derivation
     * of this Component.  Unusual way to do it, but since InputMethodEvents are handled in the base Component,
     * unlike SWT, it's the most effective way to maintain backwards compatibility.
     *
     * This method is being used for speed - the alternatives are to use peer.getClass().getMethod(...) or
     * peer.getClass().getMethods() and search for required methods.  getMethod() has the problem of throwing
     * an exception, which will eat up resources given how often this code will be called.  getMethods(), likewise,
     * will usually return a very large list, taking a long time to traverse.
     *
     * This means if SWT is updated to have new components return ModifyEvents, it will need to be added here.
     */
    protected void registerModifyEvents() {
        if ( peer instanceof StyledText || peer instanceof Text ||
             peer instanceof CCombo || peer instanceof Combo )
        {
            boolean success = false;
            try
            {
                Method addMethod = peer.getClass().getMethod("addModifyListener", new Class[] {ModifyListener.class});
                Method removeMethod = peer.getClass().getMethod("removeModifyListener", new Class[] {ModifyListener.class});
                removeMethod.invoke(peer, new Object[] {swingWTEventAdapter});
                addMethod.invoke(peer, new Object[] {swingWTEventAdapter});

                success = true;
            }
            catch (NoSuchMethodException e) {}
            catch (SecurityException e) {}
            catch (IllegalAccessException e) {}
            catch (InvocationTargetException e) {}

            if (!success)
                throw new IllegalComponentStateException("SWT peer missing required methods! (?)");
        }
    }

    /**
     * Same as above, but for ExtendedModifyEvents
     */
    protected void registerExtendedModifyEvents() {
        if ( peer instanceof StyledText  )
        {
            boolean success = false;
            try
            {
                Method addMethod = peer.getClass().getMethod("addExtendedModifyListener",
                                                             new Class[] {ModifyListener.class});
                Method removeMethod = peer.getClass().getMethod("removeExtendedModifyListener",
                                                                new Class[] {ModifyListener.class});
                removeMethod.invoke(peer, new Object[] {swingWTEventAdapter});
                addMethod.invoke(peer, new Object[] {swingWTEventAdapter});

                success = true;
            }
            catch (NoSuchMethodException e) {}
            catch (SecurityException e) {}
            catch (IllegalAccessException e) {}
            catch (InvocationTargetException e) {}

            if (!success)
                throw new IllegalComponentStateException("SWT peer missing required methods! (?)");
        }
    }

    /**
     * Called when a traversal key is pressed by the user. This routine
     * determines whether we should use the custom focus manager if one is
     * avaliable, or just let it go.
     */
    protected void processTraverseEvent(org.eclipse.swt.events.TraverseEvent e) {

        // If it is ENTER that's pressed, then that is actually an ActionEvent
        // in the Swing world!
        if (e.keyCode == SWT.CR) {
            processActionEvent(0);
        }

        // If there is no focus manager - forget it - use whatever the
        // hell SWT uses (creation order I think)
        if (swingwtx.swing.FocusManager.getCurrentManager() == null) { return; }

        // Focus previous component
        if (((e.keyCode == SWT.TAB) && ((e.stateMask & SWT.SHIFT)) != 0) ||
           (e.detail == SWT.TRAVERSE_TAB_PREVIOUS)) {
            e.doit = false;
            swingwtx.swing.FocusManager.getCurrentManager().focusPreviousComponent(this);
        }

        // Focus next component
        if (((e.keyCode == SWT.TAB) && ((e.stateMask & SWT.SHIFT)) == 0) ||
           (e.detail == SWT.TRAVERSE_TAB_NEXT)) {
            e.doit = false;
            swingwtx.swing.FocusManager.getCurrentManager().focusNextComponent(this);
        }

    }

    /**
     * When a container removes a component, it calls this method. You can override
     * this in your subclass to perform any specific action you need to take
     * when that component is removed.
     */
    protected void setComponentRemoved() {
    }

    /** Here for compatibility - I'm not sure what it's for. Doesn't do anything anyway. :-) */
    public boolean imageUpdate(
            Image img,
            int infoflags,
            int x,
            int y,
            int width,
            int height) {
            //Graphics g = img.getGraphics();
            return false;
    }
    /** Returns true if this component has the focus */
    public boolean hasFocus() {
	final boolean[] ret = new boolean[1];
	SwingUtilities.invokeSync(
	    new Runnable() {
		public void run() {
                    if (SwingWTUtils.isSWTControlAvailable(peer))
			ret[0] = peer.isFocusControl();
		    else
		        ret[0] = false;
		}
	    });
	return ret[0];
    }

    /** For compatibility with 3rd party layout managers - 
     *  SwingWT doesn't need this as all laying out is
     *  marshalled onto a queued single thread
     */
    protected Object LOCK = new Object();
    /** For compatibility with 3rd party layout managers - 
     *  SwingWT doesn't need this as all laying out is
     *  marshalled onto a queued single thread
     */
    public Object getTreeLock() { return LOCK; }

    /** Returns the component's accessible context */
    public AccessibleContext getAccessibleContext() { return accessibleContext; }
}

/*
   $Log: Component.java,v $
   Revision 1.97  2004/11/02 11:07:29  bobintetley
   Fixed a number of small compatibility bugs

   Revision 1.96  2004/10/30 20:11:54  bobintetley
   Code cleanup

   Revision 1.95  2004/09/24 10:56:55  bobintetley
   Mousewheel/printing stubs (David Jung)

   Revision 1.94  2004/09/08 04:09:28  dannaab
   Don't register a default action event handler for mouse clicks on all components

   Revision 1.93  2004/06/11 10:33:02  bobintetley
   Fixed KeyEvent to use Event.getID()

   Revision 1.92  2004/06/11 10:13:24  bobintetley
   Fix to allow unrealised peers to be enabled/invisible

   Revision 1.91  2004/06/11 06:34:43  bobintetley
   PreferredSize fix for containers with no layout manager (eg: JTabbedPane)

   Revision 1.90  2004/06/11 03:29:26  dannaab
   AWT improvements: implement Button over JButton; Menu fixes; add missing geom me
   thods; add some missing awt event methods/features

   Revision 1.89  2004/06/10 07:57:57  dannaab
   Added AWTSwingWrapper to map AWT components to the corresponding Swing version.  Implemted using it: TextComponent, TextArea, TextField; used similar technique for Menu components

   Revision 1.87  2004/06/08 09:24:21  dannaab
   Rename Component.getPeer() -> getSWTPeer().  added ComponentPeer and stubbed out support classes.

   Revision 1.86  2004/06/07 14:01:27  bobintetley
   JTable fixes for Win32, and ability to turn off SWT VIRTUAL tables. Off by
   default due to Win32 bugs in 3.0M8

   Revision 1.85  2004/06/04 14:36:19  bobintetley
   Fix to key events to use new translation routines correctly

   Revision 1.84  2004/06/04 14:21:09  bobintetley
   JTextArea default size fixes and new computePreferredSize() method that
      delegates to peer to size a component

   Revision 1.83  2004/05/27 03:04:50  dannaab
   Add a couple methods from Component.java to Window.java so the Shell peer is used instead of the generic composite peer

   Revision 1.82  2004/05/26 04:46:06  dannaab
   Changed how Window/Dialog/Frame are initialized.
   Now, the SWT peer's instantiation is delayed until setVisible(true).  This
   has the benefit of allowing Swing's setter methods that can only be set via
   the SWT Shell peer's construction (ie, setResizable(), setModel())

   Revision 1.81  2004/05/25 01:04:11  dannaab
   Misc bugfixes, ActionMap.java added, added swt source to lib dir (for debugging
   purposes), misc import optimization

   Revision 1.80  2004/05/07 12:11:15  bobintetley
   Default layout fixes and correct behaviour for null layout

   Revision 1.79  2004/05/06 12:35:21  bobintetley
   Parity with Swing constants for Binary Compatibility + fixes to JDesktopPane

   Revision 1.78  2004/05/05 23:39:39  laurentmartelli
    - Added more repaint() methods
    - Added isMinimumSizeSet()
    - setFont(Font): do not set isUserSetFont is font is null

   Revision 1.77  2004/05/05 19:29:35  bobintetley
   JComponent.paintComponent() is now called correctly

   Revision 1.76  2004/05/05 12:43:19  bobintetley
   Patches/new files from Laurent Martell

   Revision 1.75  2004/05/04 00:57:02  dannaab
   Sun java and javax package naming improvements.  Fix ambiguous class names by hardcoding SWT full class name

   Revision 1.74  2004/05/03 20:53:42  dannaab
   small tweak to layoutmanager instance objects to make easier to read

   Revision 1.73  2004/04/30 23:18:24  dannaab
   List selection support, misc bug fixes

   Revision 1.72  2004/04/30 21:54:21  bobintetley
   Moved log to the end of commonly changed files

   Revision 1.71  2004/04/30 13:20:40  bobintetley
   Fix to unrealised peer preferred sizes, forwarding window events to
   content panes and fix for mouse drag events.

   Revision 1.70  2004/04/28 08:38:05  bobintetley
   Hierarchy fixes, code cleanup for base classes, additional javadocs and use of flag to identify JComponent descendants with peers

   Revision 1.69  2004/04/27 06:37:42  bobintetley
   Corrected hierarchy with JComponent descending Container

   Revision 1.68  2004/04/23 10:02:47  bobintetley
   MouseEvent BUTTON_MASK constants, JSlider.createStandardLabels(), FontMetrics
   thread safety, Component/Toolkit.getFontMetrics() and MouseMotionAdapter

   Revision 1.67  2004/04/23 00:52:31  dannaab
   Handle borders in a Swing-like way. Implement EmptyBorder & TitledBorder

   Revision 1.66  2004/04/21 07:10:44  bobintetley
   Fixes to native build script and a few code tweaks. Removal of AWT
   native lib for compilation purposes.

   Revision 1.65  2004/04/20 19:05:38  bobintetley
   Code cleanup/refactoring

   Revision 1.64  2004/04/20 14:34:15  bobintetley
   Double buffering support for component paint()

   Revision 1.63  2004/04/19 15:03:25  bobintetley
   Missing Component/JComponent focus methods

   Revision 1.62  2004/04/16 18:29:24  bobintetley
   Fixes to JTable to only use TableCursors if necessary (at least one row of data and at least one editable column)

   Revision 1.61  2004/04/16 10:19:05  dannaab
   Misc bug fixes, InputMap implementation, preliminary undo support

   Revision 1.60  2004/04/15 12:28:00  bobintetley
   Fix to cache location/dimension info

   Revision 1.59  2004/04/15 11:24:32  bobintetley
   (Dan Naab) ComponentUI, UIDefaults/UIManager and Accessibility support.
   (Antonio Weber) TableColumnModelListener implementation and support

   Revision 1.58  2004/03/31 21:26:11  bobintetley
   Component.hasFocus() support

   Revision 1.57  2004/03/31 08:26:41  bobintetley
   MouseMotionListener support

   Revision 1.56  2004/03/31 08:07:34  bobintetley
   JTable bug fixed that prevented it redrawing after being dropped from
   a container. Protected component method allows catching of when a component
   is removed from a container.

   Revision 1.55  2004/03/30 14:22:27  bobintetley
   Fix to Component min/max sizing code, fix to JPanel insets with titled
   borders (all of which indirectly fix problems with BoxLayout). Addition
   of ComponentAdapter

   Revision 1.54  2004/03/12 11:05:23  bobintetley
   Fixed memory leak in container destruction

   Revision 1.53  2004/03/02 08:39:53  bobintetley
   Less resource intensive event pump and JTable defaults gridlines now

   Revision 1.52  2004/03/01 15:58:28  bobintetley
   Various little bug fixes

   Revision 1.51  2004/03/01 12:25:46  bobintetley
   Better HTML conversion, custom JFileChooser support, JLabel, Window and
   Image fixes to improve compatibility

   Revision 1.50  2004/02/24 09:36:39  bobintetley
   Compatibility methods

   Revision 1.49  2004/02/23 12:11:21  bobintetley
   JScrollPane bug fixed, tabbing in JTextArea fixed

   Revision 1.48  2004/02/23 10:07:09  bobintetley
   Cell render font support

   Revision 1.47  2004/02/02 14:11:50  bobintetley
   Tweaks and refining of JScrollPane implementation

   Revision 1.46  2004/02/02 12:36:36  bobintetley
   Proper JScrollPane/ScrollBar implementation

   Revision 1.45  2004/01/27 11:06:09  bobintetley
   Fixed bugs in dropping/re-adding the same components

   Revision 1.44  2004/01/26 14:26:29  bobintetley
   Container/Component Listener/Event support

   Revision 1.43  2004/01/26 12:02:49  bobintetley
   JPanel titled border support

   Revision 1.42  2004/01/26 10:37:09  bobintetley
   Better painting support

   Revision 1.41  2004/01/26 08:10:59  bobintetley
   Many bugfixes and addition of SwingSet

   Revision 1.40  2004/01/20 07:38:05  bobintetley
   Bug fixes and compatibility methods

   Revision 1.39  2004/01/16 15:53:32  bobintetley
   Many compatibility methods added to Container, Component, JInternalFrame,
      UIManager, SwingUtilities, JTabbedPane, JPasswordField, JCheckBox
      and JRadioButton.

   Revision 1.38  2004/01/15 15:58:41  bobintetley
   Extra thread safety

   Revision 1.37  2004/01/15 15:20:29  bobintetley
   Java2D work

   Revision 1.36  2004/01/15 10:11:14  bobintetley
   Fixed AWT constructors/hierarchy

   Revision 1.35  2004/01/13 11:14:25  bobintetley
   ErrorProvider and validation implementation

   Revision 1.34  2004/01/09 10:33:56  bobintetley
   Changes for JToolBar to allow platform ToolBars, mixed with other components

   Revision 1.33  2004/01/08 12:56:30  bobintetley
   Experiment with hotkeys - unworkable in current SWT

   Revision 1.32  2004/01/05 15:29:35  bobintetley
   TableLayout fixes

   Revision 1.31  2004/01/05 11:41:34  bobintetley
   Closer to working...

   Revision 1.30  2004/01/05 11:31:46  bobintetley
   More layout fixes

   Revision 1.29  2004/01/05 10:09:52  bobintetley
   Various fixes for new layouts

   Revision 1.28  2004/01/05 08:36:42  bobintetley
   Had to merge Daniel's changes in by hand (thanks)

   Revision 1.26  2003/12/22 15:24:17  bobintetley
   Additional methods to allow components using TableLayout to consume
   available space

   Revision 1.25  2003/12/17 16:30:35  bobintetley
   Flowlayout fix, vertical toolbar support and cleaned up text alignment
   hierarchy.

   Revision 1.24  2003/12/17 15:24:33  bobintetley
   Threading fixes

   Revision 1.23  2003/12/17 11:37:46  bobintetley
   Better GridBag support

   Revision 1.22  2003/12/17 10:57:35  bobintetley
   JTableHeader implementation plus Table event/model fixes

   Revision 1.21  2003/12/16 15:47:44  bobintetley
   Thread safety added to common methods

   Revision 1.20  2003/12/16 14:08:05  bobintetley
   Corrected event hierarchy for Button ActionEvents

   Revision 1.19  2003/12/16 13:48:27  bobintetley
   ActionEvent maps to CR press

   Revision 1.18  2003/12/16 13:44:04  bobintetley
   Fix to broken focus management

   Revision 1.17  2003/12/16 13:14:33  bobintetley
   Use of SwingWTUtils.isSWTControlAvailable instead of null test

   Revision 1.16  2003/12/16 12:23:31  bobintetley
   Corrected handling of table selection + keyboard action events

   Revision 1.15  2003/12/16 11:43:57  bobintetley
   Fixed focus traversal for use by FocusManager

   Revision 1.14  2003/12/16 11:40:37  bobintetley
   JTable refreshes now when model updates

   Revision 1.13  2003/12/16 09:20:07  bobintetley
   KeyEvent consumption logic wrong way round

   Revision 1.12  2003/12/16 09:19:02  bobintetley
   Various small fixes to match Swing more closely

   Revision 1.11  2003/12/15 18:29:56  bobintetley
   Changed setParent() method to setSwingWTParent() to avoid conflicts with applications

   Revision 1.10  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/
