/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

/*
    NOTES:

      Since GDBM allocates datum.dptr space via smalloc(), we use smalloc()/
      sfree() whenever dealing with datum structures for the sake of
      consistency and avoiding errors.
*/

#include <algorithm>
#include <climits>
#include <cctype>
#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <iterator>

#include <gdbm.h>

#include "thoughttracker.h"
#include "database.h"
#include "util.h"

#ifdef ENABLE_DEBUG
#undef DMSG 
#define DMSG cerr << "TTDatabase::" << __FUNCTION__ << "(): "
#endif  /* ENABLE_DEBUG */

const dbid_t TTDatabase::max_id_key = TTDatabase::encode(-1);  // value: dbid_t
const dbid_t TTDatabase::ver_key    = TTDatabase::encode(-2);  // value: dbid_t

const datum TTDatabase::max_id_key_datum = {
  (char*) &TTDatabase::max_id_key, sizeof(dbid_t)
};

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
TTDatabase::TTDatabase(const string &path, void (*callback)(const string&))
  : dbf_data(0), dbf_links(0), critical_callback(callback), critical(false)
{
  static const dbid_t ver_id = encode(2);  // current DB layout revision
  static const datum ver_key_datum = { (char*) &ver_key, sizeof(dbid_t) };
  datum ver_data = { (char*) &ver_id, sizeof(dbid_t) };
  datum data;

#ifdef ENABLE_DEBUG
  DMSG << "opening databases in " << path << endl;
#endif
  // open data DB...
  string datafile = path + "/data";
  dbf_data = gdbm_open((char*) datafile.c_str(), 0, GDBM_WRCREAT, 0600,
    0);  // FIXME
  if (!dbf_data)
    set_critical(_("Failed to open database file ") + datafile + ".\n" +
      _("Is another instance running?"));
  if (!gdbm_exists(dbf_data, max_id_key_datum)) {  // new database
    dbid_t val = -1;
#ifdef ENABLE_DEBUG
    DMSG << "new DB--initializing MAX_ID with " << val << endl;
#endif
    dbid_t val_enc = encode(val);
    data.dptr = (char*) &val_enc;
    data.dsize = sizeof(dbid_t);
    gdbm_store(dbf_data, max_id_key_datum, data, GDBM_REPLACE);
    gdbm_store(dbf_data, ver_key_datum, ver_data, GDBM_REPLACE);
  } else {  // existing database; check version
    data = gdbm_fetch(dbf_data, ver_key_datum);
    if (!data.dptr)
      set_critical(_("Database corrupted: couldn't read version information."));
    else {
      dbid_t my_ver = decode(ver_id);
      dbid_t db_ver = decode(*(dbid_t*) data.dptr);

      if (db_ver > my_ver) {
        sfree(data.dptr);
        set_critical(_("Database was last written by a version of "
		  "ThoughtTracker newer than this\none.  Install the latest release "
          "and try again."));
      }
      sfree(data.dptr);

      if (db_ver < my_ver) upgrade_db(datafile, db_ver, decode(ver_id));
    }
  }

  // open links DB
  string linksfile = path + "/links";
  dbf_links = gdbm_open((char*) linksfile.c_str(), 0, GDBM_WRCREAT, 0600,
    0);  // FIXME
  if (!dbf_links)
    set_critical(_("Failed to open database file ") + linksfile + ".\n" +
      _("Is another instance running?"));
}

//.............................................................................
TTDatabase::~TTDatabase()
{
#ifdef ENABLE_DEBUG
  DMSG << "closing databases" << endl;
#endif
  if (dbf_data) {
    gdbm_close(dbf_data);
    dbf_data = 0;
  }
  if (dbf_links) {
    gdbm_close(dbf_links);
    dbf_links = 0;
  }
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
dbid_t
TTDatabase::store(dbid_t req_id, const dbf_t flags, const string &summary,
  const string &text)
{
  if (critical) return DB_FAILURE;

  // check MAX_ID
  datum data = gdbm_fetch(dbf_data, max_id_key_datum);
  if (!data.dptr) return DB_FAILURE;
  dbid_t max_id = decode(*(dbid_t*) data.dptr);
  sfree(data.dptr);

  dbid_t id;
  if (req_id < 0) {  // assign new ID as (MAX_ID + 1)
    if (max_id >= INT_MAX) return DB_FAILURE;
    id = max_id + 1;
  } else  // use specified ID
    id = req_id;

#ifdef ENABLE_DEBUG
  DMSG << "saving record #" << id << " (" << req_id << " in request)" << endl;
#endif
  dbid_t id_enc = encode(id);
  datum key;
  key.dptr = (char*) &id_enc;
  key.dsize = sizeof(dbid_t);
  data = construct_record(flags, summary, text);
  dbid_t retval = gdbm_store(dbf_data, key, data, GDBM_REPLACE);
  sfree(data.dptr);
  if (retval) return DB_FAILURE;

  if (id > max_id) {  // save new MAX_ID
    data.dptr = (char*) &id_enc;
    data.dsize = sizeof(dbid_t);
#ifdef ENABLE_DEBUG
  DMSG << "increasing MAX_ID to " << id << endl;
#endif
    retval = gdbm_store(dbf_data, max_id_key_datum, data, GDBM_REPLACE);
  }
  return retval ? DB_FAILURE : id;
}

//.............................................................................
int
TTDatabase::read(dbid_t id, dbf_t &flags, string &summary, string &text)
{
  if (critical || id < 0) return DB_FAILURE;

#ifdef ENABLE_DEBUG
  DMSG << "reading record #" << id << endl;
#endif
  datum key, data;
  dbid_t id_enc = encode(id);
  key.dptr = (char*) &id_enc;
  key.dsize = sizeof(dbid_t);
  data = gdbm_fetch(dbf_data, key);
  if (!data.dptr || !evaluate_record(data, flags, summary, text))
    return DB_FAILURE;
  return DB_SUCCESS;
}

//.............................................................................
bool
TTDatabase::exists(dbid_t id)
{
  if (critical || id < 0) return false;

  dbid_t id_enc = encode(id);
  datum key = { (char*) &id_enc, sizeof(dbid_t) };
  return gdbm_exists(dbf_data, key);
}

//.............................................................................
int
TTDatabase::del(dbid_t id)
{
  datum key;

  if (critical || id < 0) return DB_FAILURE;

#ifdef ENABLE_DEBUG
  DMSG << "deleting record #" << id << endl;
#endif
  // remove all backward links
  if (unlink_all(id) == DB_FAILURE) return DB_FAILURE;
  dbid_t id_enc = encode(id);
  key.dptr = (char*) &id_enc;
  key.dsize = sizeof(dbid_t);
  // del link record
  if (linked(id) && gdbm_delete(dbf_links, key)) return DB_FAILURE;
  return gdbm_delete(dbf_data, key) ? DB_FAILURE : DB_SUCCESS;  // del entry
}

//.............................................................................
int
TTDatabase::link(dbid_t id1, dbid_t id2)
{
  if (critical || id1 < 0 || id2 < 0) return DB_FAILURE;
  if (id1 == id2) return DB_SUCCESS;  // pointless operation

#ifdef ENABLE_DEBUG
  DMSG << "linking records #" << id1 << " and #" << id2 << endl;
#endif
  if (add_link_to_list(id1, id2) == DB_FAILURE) return DB_FAILURE;
  return add_link_to_list(id2, id1);
}

//.............................................................................
bool
TTDatabase::linked(dbid_t list_of, dbid_t link)
{
  if (critical || list_of < 0 || link < 0) return false;
  if (list_of == link) return true;  // per definition

  int retval = false;
  dbid_t list_of_enc = encode(list_of);
  datum key = { (char*) &list_of_enc, sizeof(dbid_t) };
  datum data = gdbm_fetch(dbf_links, key);
  if (data.dptr) {  // record exists
    if (data.dsize % sizeof(dbid_t)) {
      // inconsistency in DB, but since this function reports only true or
      // false, let's try to make the best of it
      data.dsize -= data.dsize % sizeof(dbid_t);
    }
    for (unsigned int n = 0; n < data.dsize/sizeof(dbid_t); n++)
      if (decode(((dbid_t*) data.dptr)[n]) == link) {
        retval = true;
        break;
      }
    sfree(data.dptr);
  }
#ifdef ENABLE_DEBUG
  DMSG << "records #" << list_of << " and #" << link << " are " <<
	(retval ? "linked" : "NOT linked") << endl;
#endif
  return retval;
}

//.............................................................................
int
TTDatabase::linked(dbid_t id)
{
  if (critical || id < 0) return false;

  dbid_t id_enc = encode(id);
  datum key = { (char*) &id_enc, sizeof(dbid_t) };
  // gdbm_exists would be sufficient, but lets be tolerant
  datum data = gdbm_fetch(dbf_links, key);
  int retval = data.dptr ? data.dsize/sizeof(dbid_t) : 0;
  sfree(data.dptr);
#ifdef ENABLE_DEBUG
  DMSG << "record #" << id << " is " << (retval ? "linked" : "isolated") <<
    endl;
#endif
  return retval;
}

//.............................................................................
int
TTDatabase::links(dbid_t id, list<dbid_t> &links)
{
  if (critical || id < 0) return DB_FAILURE;

  links.clear();
#ifdef ENABLE_DEBUG
  DMSG << "listing links for #" << id << "...";
#endif
  dbid_t id_enc = encode(id);
  datum key = { (char*) &id_enc, sizeof(id) };
  datum data = gdbm_fetch(dbf_links, key);
  if (data.dptr) {
    if (data.dsize % sizeof(dbid_t)) {
      sfree(data.dptr);
      return DB_FAILURE;  // Huh?
    }
    for (unsigned int n = 0; n < data.dsize/sizeof(dbid_t); n++)
      links.push_back(decode(((dbid_t*) data.dptr)[n]));
    sfree(data.dptr);
  }
#ifdef ENABLE_DEBUG
  cerr << ' ' << links.size() << " found" << endl;
#endif
  return DB_SUCCESS;
}

//.............................................................................
int
TTDatabase::unlink(dbid_t id1, dbid_t id2)
{
  if (critical || id1 < 0 || id2 < 0) return DB_FAILURE;
  if (id1 == id2) return DB_SUCCESS;  // pointless operation

#ifdef ENABLE_DEBUG
  DMSG << "unlinking records #" << id1 << " and #" << id2 << endl;
#endif
  if (del_link_from_list(id1, id2) == DB_FAILURE) return DB_FAILURE;
  return del_link_from_list(id2, id1);
}

//.............................................................................
int
TTDatabase::unlink_all(dbid_t id)
{
  if (critical) return DB_FAILURE;

  list<dbid_t> l;
  if (links(id, l) == DB_FAILURE) return DB_FAILURE;
#ifdef ENABLE_DEBUG
  DMSG << "unlinking all links involving record #" << id << " (" <<
    l.size() << ')' << endl;
#endif
  // remove all backward links
  for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++)
    if (del_link_from_list(*i, id) == DB_FAILURE) return DB_FAILURE;
  // delete own link record
  dbid_t id_enc = encode(id);
  datum key = { (char*) &id_enc, sizeof(dbid_t) };
  if (linked(id))
    if (gdbm_delete(dbf_links, key)) return DB_FAILURE;
  return DB_SUCCESS;
}

//.............................................................................
int
TTDatabase::search(const string &s, unsigned params, list<dbid_t> &list,
  unsigned int max)
{
  if (critical) return DB_FAILURE;

  list.clear();
  datum key = gdbm_firstkey(dbf_data);
  while (key.dptr && (!max || list.size() < max)) {
    dbid_t id = decode(*(dbid_t*) key.dptr);
    if (id >= 0) {
      datum data = gdbm_fetch(dbf_data, key);
      if (data.dptr) {
        dbf_t flags;
        string summary, text;
        bool is_linked = linked(id);
        if (evaluate_record(data, flags, summary, text) &&
            ((!(flags & DB_FLAG_HUB) && params & DB_SEARCH_ENT) ||
             (flags & DB_FLAG_HUB && params & DB_SEARCH_HUB)) &&
            ((is_linked && params & DB_SEARCH_LNK) ||
             (!is_linked && params & DB_SEARCH_ISO)))
        {
          string buf = summary;
          if (!(params & DB_SEARCH_SUMMARIES)) buf += text;
          if (match(buf, s, !(params & DB_SEARCH_OR))) list.push_back(id);
        }
        sfree(data.dptr);
      }
    }
    datum nextkey = gdbm_nextkey(dbf_data, key);
    sfree(key.dptr);
    key = nextkey;
  }
  if (key.dptr) sfree(key.dptr);

#ifdef ENABLE_DEBUG
  DMSG << "returning " << list.size() << " result" <<
    (list.size() == 1 ? "" : "s") << endl;
#endif
  return DB_SUCCESS;
}

//.............................................................................
int
TTDatabase::check_administrative_information(bool repair)
{
  if (critical) return DB_FAILURE;

#ifdef ENABLE_DEBUG
  DMSG << "checking MAX_ID record... ";
#endif
  // read MAX_ID record
  datum data = gdbm_fetch(dbf_data, max_id_key_datum);
  // assigning -2 to max_id_orig here causes the MAX_ID record to be written
  // if it didn't exist in the first place
  dbid_t max_id_orig = -2, max_id_new = -1;
  if (data.dptr) {
    max_id_orig = decode(*(dbid_t*) data.dptr);
    sfree(data.dptr);
  }

  // traverse through all records, remembering the highest ID encountered
  datum key = gdbm_firstkey(dbf_data);
  while (key.dptr) {
    dbid_t id = decode(*(dbid_t*) key.dptr);
    if (id > max_id_new) max_id_new = id; 
    datum nextkey = gdbm_nextkey(dbf_data, key);
    sfree(key.dptr);
    key = nextkey;
  }

  // evaluate, repair if requested
  if (max_id_new > max_id_orig) {  // oops
#ifdef ENABLE_DEBUG
    cerr << "error: record = ";
    if (max_id_orig < -1)
      cerr << "(none)";
    else
      cerr << max_id_orig;
    cerr << ", max used = " << max_id_new << endl;
#endif
    // repair
    if (repair) {
#ifdef ENABLE_DEBUG
      DMSG << "writing valid MAX_ID record... ";
#endif
      dbid_t id = encode(max_id_new);
      data.dptr = (char*) &id;
      data.dsize = sizeof(dbid_t);
      if (gdbm_store(dbf_data, max_id_key_datum, data, GDBM_REPLACE))
        return DB_FAILURE;
#ifdef ENABLE_DEBUG
      cerr << "succeeded" << endl;
#endif
    }  // repair
    return 1;
  } else {  // OK
#ifdef ENABLE_DEBUG
    cerr << "OK (" << max_id_orig << ')' << endl; 
#endif
    return DB_SUCCESS;
  }
}

//.............................................................................
int
TTDatabase::check_data_structures(bool repair)
{
  if (critical) return DB_FAILURE;

#ifdef ENABLE_DEBUG
  DMSG << "bad records: ";
#endif
  datum key = gdbm_firstkey(dbf_data);
  list<dbid_t> l;
  // as goes for all these function: it should be OK to store seemingly
  // unlimited numbers of keys in a list, as dbid_t's size is only 4 bytes,
  // and so even 25,000 records (which is certainly a much larger number than
  // will ever be achieved by a user) require a mere 100K of memory
  while (key.dptr) {
    dbid_t id = decode(*(dbid_t*) key.dptr);
    if (id >= 0) {
      datum data = gdbm_fetch(dbf_data, key);
      if (data.dptr) {  // read successful
        dbf_t flags;
        string summary, text;
        if (!evaluate_record(data, flags, summary, text)) {
#ifdef ENABLE_DEBUG
          cerr << (l.size() ? "#" : ", #") << id;
#endif
          l.push_back(id);
        }
      }
    }
    datum nextkey = gdbm_nextkey(dbf_data, key);
    sfree(key.dptr);
    key = nextkey;
  }
  if (!l.size()) {
#ifdef ENABLE_DEBUG
    cerr << "none" << endl;
#endif
    return DB_SUCCESS;
#ifdef ENABLE_DEBUG
  } else {
    DMSG << endl;
#endif
  }

  if (repair) {  // truncate records
    for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++) {
#ifdef ENABLE_DEBUG
      DMSG << "repairing #" << *i << "..." << endl;
#endif
      dbid_t id_enc = encode(*i);
      datum key = { (char*) &id_enc, sizeof(dbid_t) };
      datum data = gdbm_fetch(dbf_data, key);
      if (!data.dptr) return DB_FAILURE;
      dbf_t flags;
      string summary, text;
      evaluate_record(data, flags, summary, text);
      datum newdata = construct_record(flags, summary, text);
      sfree(data.dptr);
      int retval = gdbm_store(dbf_data, key, newdata, GDBM_REPLACE);
      sfree(newdata.dptr);
      if (retval) return DB_FAILURE;
    }
  }
#ifdef ENABLE_DEBUG
  DMSG << "successfully repaired " << l.size() << " record" <<
    (l.size() == 1 ? "" : "s") << endl;
#endif
  return l.size();
}

//.............................................................................
int
TTDatabase::check_stray_linklists(bool repair)
{
  if (critical) return DB_FAILURE;

#ifdef ENABLE_DEBUG
  DMSG << "stray link lists: ";
#endif
  datum key = gdbm_firstkey(dbf_links);
  list<dbid_t> l;
  while (key.dptr) {
    dbid_t id = decode(*(dbid_t*) key.dptr);
    if (id >= 0 && !exists(id)) {
#ifdef ENABLE_DEBUG
      cerr << (l.size() ? "#" : ", #") << id;
#endif
      l.push_back(id);
    }
    datum nextkey = gdbm_nextkey(dbf_links, key);
    sfree(key.dptr);
    key = nextkey;
  }
  if (!l.size()) {
#ifdef ENABLE_DEBUG
    cerr << "none" << endl;
#endif
    return DB_SUCCESS;
#ifdef ENABLE_DEBUG
  } else {
    DMSG << endl;
#endif
  }

  if (repair) {  // delete respective link lists
    for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++) {
#ifdef ENABLE_DEBUG
      DMSG << "deleting stray link list for #" << *i << "..." << endl;
#endif
      dbid_t id_enc = encode(*i);
      datum key = { (char*) &id_enc, sizeof(dbid_t) };
      gdbm_delete(dbf_links, key);
    }
  }
#ifdef ENABLE_DEBUG
  DMSG << "successfully repaired " << l.size() << " record" <<
    (l.size() == 1 ? "" : "s") << endl;
#endif
  return l.size();
}

//.............................................................................
int
TTDatabase::check_linklist_structures(bool repair)
{
  if (critical) return DB_FAILURE;

#ifdef ENABLE_DEBUG
  DMSG << "bad link list structures: ";
#endif
  datum key = gdbm_firstkey(dbf_links);
  list<dbid_t> l;
  while (key.dptr) {
    dbid_t id = decode(*(dbid_t*) key.dptr);
    if (id >= 0) {
      datum data = gdbm_fetch(dbf_links, key);
      if (data.dptr) {  // read successful
        if (data.dsize % sizeof(dbid_t)) {
#ifdef ENABLE_DEBUG
          cerr << (l.size() ? "#" : ", #") << id;
#endif
          l.push_back(id);
        }
      }
    }
    datum nextkey = gdbm_nextkey(dbf_links, key);
    sfree(key.dptr);
    key = nextkey;
  }
  if (!l.size()) {
#ifdef ENABLE_DEBUG
    cerr << "none" << endl;
#endif
    return DB_SUCCESS;
#ifdef ENABLE_DEBUG
  } else {
    DMSG << endl;
#endif
  }

  if (repair) {  // truncate link lists
    for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++) {
#ifdef ENABLE_DEBUG
      DMSG << "repairing #" << *i << "..." << endl;
#endif
      dbid_t id_enc = encode(*i);
      datum key = { (char*) &id_enc, sizeof(dbid_t) };
      datum data = gdbm_fetch(dbf_links, key);
      if (!data.dptr) return DB_FAILURE;
      data.dsize -= data.dsize % sizeof(dbid_t);
      int res = gdbm_store(dbf_links, key, data, GDBM_REPLACE);
      sfree(data.dptr);
      if (res) return DB_FAILURE;
    }
  }
#ifdef ENABLE_DEBUG
  DMSG << "successfully repaired " << l.size() << " record" <<
    (l.size() == 1 ? "" : "s") << endl;
#endif
  return l.size();
}

//.............................................................................
int
TTDatabase::check_links_consistency(bool repair)
{
  if (critical) return DB_FAILURE;

#ifdef ENABLE_DEBUG
  // we handle this differently here because links() will output stuff of its
  // own
  DMSG << "inconsistent links?  checking..." << endl;
#endif
  datum key = gdbm_firstkey(dbf_links);
  list<list<dbid_t> > l;
  while (key.dptr) {
    dbid_t id = decode(*(dbid_t*) key.dptr);
    if (id >= 0) {
      list<dbid_t> ll, bad;
      links(id, ll);
      for (list<dbid_t>::iterator i = ll.begin(); i != ll.end(); i++)
        if (!linked(*i, id) || *i == id)  // invalid links in list
          bad.push_back(*i);
      if (bad.size()) {
        bad.push_front(id);  // first element becomes list_of
        l.push_back(bad);
      }
    }
    datum nextkey = gdbm_nextkey(dbf_links, key);
    sfree(key.dptr);
    key = nextkey;
  }
  if (!l.size()) {
#ifdef ENABLE_DEBUG
    DMSG << "no inconsistencies found" << endl;
#endif
    return DB_SUCCESS;
#ifdef ENABLE_DEBUG
  } else {
    DMSG << "inconsistent links: ";
    bool first = true;
    for (list<list<dbid_t> >::iterator i = l.begin(); i != l.end(); i++) {
      cerr << (first ? "#" : ", #") << i->front();
      first = false;
    }
    cerr << endl;
#endif
  }

  if (repair) {  // remove invalid links from lists
    for (list<list<dbid_t> >::iterator i = l.begin(); i != l.end(); i++) {
      list<dbid_t> cur = *i;
      dbid_t id = cur.front();
      cur.pop_front();  // remove list_of (first element)
#ifdef ENABLE_DEBUG
      DMSG << "pruning link list of #" << id << "..." << endl;
#endif
      // prune list
      for (list<dbid_t>::iterator j = cur.begin(); j != cur.end(); j++)
        if (del_link_from_list(id, *j) == DB_FAILURE) return DB_FAILURE;
    }
  }
#ifdef ENABLE_DEBUG
  DMSG << "successfully repaired " << l.size() << " record" <<
    (l.size() == 1 ? "" : "s") << endl;
#endif
  return l.size();
}

//.............................................................................
bool
TTDatabase::optimize()
{
  if (critical) return false;
  if (gdbm_reorganize(dbf_data)) return false;
  return !gdbm_reorganize(dbf_links);
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTDatabase::set_critical(const string msg)
{
  critical_msg = msg;
  critical = true;
  if (critical_callback) critical_callback(msg);
};

//.............................................................................
int
TTDatabase::add_link_to_list(dbid_t list_of, dbid_t link)
{
  if (critical || list_of < 0 || link < 0) return DB_FAILURE;

#ifdef ENABLE_DEBUG
  DMSG << "adding entry #" << link << " to #" << list_of <<
    "'s list of links" << endl;
#endif
  // get the current list of links
  list<dbid_t> l;
  if (links(list_of, l) == DB_FAILURE) return DB_FAILURE;

  // check whether the list already contains that link
  for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++)
    if (*i == link) return DB_SUCCESS;  // nothing to be done

  // add link to list
  l.push_back(link);
  dbid_t *new_list = (dbid_t*) smalloc(l.size() * sizeof(dbid_t));
  int n = 0;
  for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++)
    new_list[n++] = encode(*i);

  dbid_t list_of_enc = encode(list_of);
  datum key = { (char*) &list_of_enc, sizeof(dbid_t) };
  datum data = { (char*) new_list, n * sizeof(dbid_t) };
  int res = gdbm_store(dbf_links, key, data, GDBM_REPLACE);
  sfree(new_list);
  return res ? DB_FAILURE : DB_SUCCESS;
}

//.............................................................................
int
TTDatabase::del_link_from_list(dbid_t list_of, dbid_t link)
{
  if (critical || list_of < 0 || link < 0) return DB_FAILURE;

#ifdef ENABLE_DEBUG
  DMSG << "deleting entry #" << link << " from #" << list_of <<
    "'s list of links" << endl;
#endif
  // get the current list of links
  list<dbid_t> l;
  if (links(list_of, l) == DB_FAILURE) return DB_FAILURE;
  if (!l.size()) return DB_SUCCESS;  // nothing to be done

  // build new list
  dbid_t *new_list = (dbid_t*) smalloc(l.size() * sizeof(dbid_t));
  int n = 0;
  for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++)
    if (*i != link) new_list[n++] = encode(*i);

  // write new link list
  int res;
  dbid_t list_of_enc = encode(list_of);
  datum key = { (char*) &list_of_enc, sizeof(dbid_t) };
  if (n) {  // links remaining: write list record
    datum data = { (char*) new_list, n * sizeof(dbid_t) };
    res = gdbm_store(dbf_links, key, data, GDBM_REPLACE);
  } else  // no links remaining: delete record
    res = gdbm_delete(dbf_links, key);
    
  sfree(new_list);
  return res ? DB_FAILURE : DB_SUCCESS;
}

//.............................................................................
bool
TTDatabase::match(const string &string, const string &search_string,
  bool mode)
{
  bool res = mode ? true : false;
  bool no_keyword = true;

  char *s = new char[string.length() + 1];
  strcpy(s, string.c_str());
  for (char *t = s; *t; t++) *t = (char) tolower(*t);
  char *ss = new char[search_string.length() + 1];
  strcpy(ss, search_string.c_str());
  for (char *t = ss; *t; t++) *t = (char) tolower(*t);

  for (char *t = strtok(ss, " \t\n"); t; t = strtok(0, " \t\n")) {
    no_keyword = false;
    if (mode) {  // AND
      if (!strstr(s, t)) {
        res = false;
        break;
      }
    } else {  // OR
      if (strstr(s, t)) {
        res = true;
        break;
      }
    }
  }

  delete[] ss;
  delete[] s;

  return no_keyword || res;
}

//.............................................................................
void
TTDatabase::upgrade_db(const string &filename, dbid_t from, dbid_t to)
{
  if (to <= from) return;
  cerr << _("Upgrading database to a new revision...") << endl;

  // create temporary DB file
  string datafile = filename;
  string tempfile = filename + ".tmp";
  GDBM_FILE dbf_temp = gdbm_open((char*) tempfile.c_str(), 0,
    GDBM_WRCREAT | GDBM_NEWDB, 0600, 0);  // FIXME
  if (!dbf_temp)
    set_critical(string(_("Failed to open temporary file ")) + tempfile + '!');

  // copy existing DB, making necessary adaptations
  datum key = gdbm_firstkey(dbf_data);
  while (key.dptr) {
    dbid_t id = decode(*(dbid_t*) key.dptr);
    datum data = gdbm_fetch(dbf_data, key);
    if (!data.dptr) set_critical(_("Read error!"));
    datum newdata = data;
    if (id >= 0) {
      switch (from) {
        case 0:
          newdata.dsize = data.dsize + sizeof(dbf_t);
          newdata.dptr = (char*) smalloc(newdata.dsize);
          *(dbf_t*) newdata.dptr = 0;  // default flags
          memcpy(newdata.dptr + sizeof(dbf_t), data.dptr, data.dsize);
        case 1:
          if (!newdata.dptr[sizeof(dbf_t) +
                            strlen(newdata.dptr + sizeof(dbf_t)) + 1])
          {
            (*(dbf_t*) newdata.dptr) |= DB_FLAG_HUB;
          }
      }
    }
    if (gdbm_store(dbf_temp, key, newdata, GDBM_REPLACE))
      set_critical(_("Write error!"));
    if (id >= 0 && from == 0)
      sfree(newdata.dptr);
    sfree(data.dptr);
    datum nextkey = gdbm_nextkey(dbf_data, key);
    sfree(key.dptr);
    key = nextkey;
  }

  // close databases
  gdbm_close(dbf_temp);
  gdbm_close(dbf_data);

  // rename files
  if (rename(datafile.c_str(), (datafile + ".old").c_str()) == -1 ||
      rename(tempfile.c_str(), datafile.c_str()) == -1)
  {
    set_critical(_("Error renaming database files after version upgrade!"));
  }

  // open upgraded DB
  dbf_data = gdbm_open((char*) datafile.c_str(), 0, GDBM_WRITER, 0600, 0);
  if (!dbf_data)
    set_critical(_("Failed to open database file after version upgrade!"));

  // save new version
  dbid_t to_enc = encode(to);
  key.dptr = (char*) &ver_key;
  key.dsize = sizeof(dbid_t);
  datum data = { (char*) &to_enc, sizeof(dbid_t) };
  if (gdbm_store(dbf_data, key, data, GDBM_REPLACE)) {
    set_critical(_("Failed to write new database version record after "
      "upgrade!"));
  }
}

//.............................................................................
datum
TTDatabase::construct_record(const dbf_t flags, const string summary,
  const string text)
{
  datum data;
  data.dsize = sizeof(dbf_t) +
              summary.length() + 1 +
              text.length() + 1;
  data.dptr = (char*) smalloc(data.dsize);
  char *ptr = data.dptr;

  *(dbf_t*) ptr = flags;
  ptr += sizeof(flags);

  strcpy(ptr, summary.c_str());
  ptr += summary.length()+1;

  strcpy(ptr, text.c_str());

  return data;
}

//.............................................................................
bool
TTDatabase::evaluate_record(const datum data, dbf_t &flags, string &summary,
  string &text)
{
  // sane start values in case of an early failure
  flags = 0;
  summary = "";
  text = "";

  gint size = data.dsize;
  char *ptr = data.dptr;

  // check & copy flags field
  size -= sizeof(dbf_t);
  if (size < 0) return false;
  flags = *(dbf_t*) ptr;
  ptr += sizeof(dbf_t);

  // check & copy summary field
  char *buf = new char[size+1];
  buf[size] = '\0';
  strncpy(buf, ptr, size);
  summary = ptr;
  size -= summary.length()+1;
  if (size < 1) {
    delete[] buf;
    return false;
  }
  ptr += summary.length()+1;

  // check & copy text field
  buf[size] = '\0';
  strncpy(buf, ptr, size);
  text = ptr;
  size -= text.length()+1;
  delete[] buf;
  return size ? false : true;
}

