/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#include <algorithm>
#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <list>
#include <strstream>

#include <gtk--/main.h>
#include <gtk--/menu.h>

#include "thoughttracker.h"
#include "app.h"
#include "menubar.h"
#include "optionsdlg.h"
#include "querydlg.h"
#include "xpm.h"

#ifdef ENABLE_DEBUG
#undef DMSG 
#define DMSG cerr << "TTMenuBar::" << __FUNCTION__ << "(): "
#endif  /* ENABLE_DEBUG */

using SigC::bind;
using SigC::slot;

using namespace Gtk;

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
TTMenuBar::TTMenuBar(TTApplication *application)
  : Gtk::MenuBar(), app(application)
{
  {
    using namespace Menu_Helpers;

    // menu "File"
    Menu *menu = manage(new Menu);
    menu->items().push_back(MenuElem(_("_Options..."),
      slot(this, &TTMenuBar::menu_options)));
    menu->items().push_back(SeparatorElem());
    menu->items().push_back(MenuElem(_("_Import details..."), 0));
    w.im = menu->items().back();
    menu->items().push_back(MenuElem(_("_Export details..."), 0));
    w.ex = menu->items().back();
    menu->items().push_back(SeparatorElem());
    menu->items().push_back(MenuElem(_("_Quit"), _("<control>q"),
      slot(this, &TTMenuBar::menu_quit)));
    items().push_back(MenuElem(_("_File"), _("<alt>f"), *menu));

    // menu "Edit"
    menu = manage(new Menu);
    menu->items().push_back(MenuElem(_("_Copy"), _("<control>c"),
      slot(this, &TTMenuBar::menu_copy)));
    w.copy = menu->items().back();
    menu->items().push_back(MenuElem(_("_Paste"), _("<control>v"),
      slot(this, &TTMenuBar::menu_paste)));
    w.paste = menu->items().back();
    menu->items().push_back(SeparatorElem());
    menu->items().push_back(MenuElem(_("_Revert"), 0));
    w.revert = menu->items().back();
    items().push_back(MenuElem(_("_Edit"), _("<alt>e"), *menu));

    // menu "Bookmarks"
    w.bm = manage(new Menu);
    w.bm->items().push_back(MenuElem(_("_Add bookmark for active entry"),
      slot(this, &TTMenuBar::menu_add_bookmark)));
    w.add = w.bm->items().back();
    w.bm->items().push_back(MenuElem(_("_Delete bookmark for active entry"),
      bind(slot(this, &TTMenuBar::menu_del_bookmark), true)));
    w.del = w.bm->items().back();
    w.bm->items().push_back(SeparatorElem());
    items().push_back(MenuElem(_("_Bookmarks"), _("<alt>b"), *w.bm));

    // menu "Help"
    menu = manage(new Menu);
    menu->items().push_back(MenuElem(_("_About..."),
      slot(this, &TTMenuBar::menu_about)));
    items().push_back(MenuElem(_("_Help"), _("<alt>h"), *menu));
  }

  // init some stuff
  set_cb_widget(0, 0);
  bm_build_menu();

  show_all();
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTMenuBar::menu_options()
{
  TTOptionsDialog *dlg = new TTOptionsDialog(app);
  dlg->run();
  delete dlg;
}

//.............................................................................
void
TTMenuBar::menu_quit()
{
  if (!(app->w.entry->save_record(false))) return;
  app->save_config();  // savesize requires us to do that here, too
  Main::quit();
}

//.............................................................................
void
TTMenuBar::menu_copy()
{
  gint a = 0, b = -1;
  if (cb.widget->has_selection()) {
    a = cb.widget->get_selection_start_pos();
    b = cb.widget->get_selection_end_pos();
    if (a > b) {  // swap if start > end
      gint c = a;
      a = b;
      b = c;
    }
  }
  cb.buffer = cb.widget->get_chars(a, b);
  w.paste->set_sensitive(cb.widget && cb.buffer.length());
}

//.............................................................................
void
TTMenuBar::menu_paste()
{
  if (cb.widget->has_selection()) cb.widget->delete_selection();
  if (cb.buffer.length()) {
    gint pos = cb.widget->get_position();
    // for some reason, these functions ain't part of the Gtk::Editable class
    gtk_editable_insert_text(GTK_EDITABLE(cb.widget->gtkobj()),
	  cb.buffer.c_str(), cb.buffer.length(), &pos);
    gtk_editable_set_position(GTK_EDITABLE(cb.widget->gtkobj()), pos);
  }
}

//.............................................................................
void
TTMenuBar::menu_add_bookmark()
{
  dbid_t active = app->get_active();

  // sanity checks
  if (active < 0) return;
  for (vector<ttbm_t>::iterator i = bm.begin(); i != bm.end(); i++)
    if (i->id == active) {
      app->sbar_msg(_("The active entry is already bookmarked."));
      return;
    }

  // add active entry as bookmark
  if (bm_add(active)) {
    bm_build_menu();
    app->save_config();
    app->sbar_msg(_("The active entry has been bookmarked."));
  } else
    internal_error_popup(app, TTQD_ERROR_DB_READ);
}

//.............................................................................
void
TTMenuBar::menu_del_bookmark(bool complain)
{
  dbid_t active = app->get_active();

  // sanity checks
  if (active < 0) return;

  // delete the entry
  if (bm_del(active)) {
    bm_build_menu(false);
    app->save_config();
    app->sbar_msg(_("The active entry has been removed from the list of "
      "bookmarks."));
  } else
    app->sbar_msg(_("The active entry isn't bookmarked."));
}

//.............................................................................
void
TTMenuBar::menu_about()
{
  TTQueryDialog dlg(app, logo_xpm, _("About ThoughtTracker..."),
    string(COPYRIGHT) + ", " + MAILADDR + ".\n" +
    _("This program is distributed under the terms of the GNU General Public "
      "License 2 (or any later revision thereof), which is included with the "
      "application's source package."));
  dlg.set_position(GTK_WIN_POS_CENTER);
  dlg.run();
}

//.............................................................................
void
TTMenuBar::update_bookmark(dbid_t id)
{
  // determine IDs to check
  list<dbid_t> l; 
  if (id >= 0)  // specific id
    l.push_back(id);
  else  // check all IDs
    for (vector<ttbm_t>::iterator i = bm.begin(); i != bm.end(); i++)
      l.push_back(i->id);
  
  // update menu for requested IDs
  for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++)
    if (bm_del(id)) bm_add(id);
  bm_build_menu();
}

//.............................................................................
void
TTMenuBar::update()
{
  if (app->get_active() < 0) {  // no active entry
    w.add->set_sensitive(false);
    w.del->set_sensitive(false);
  } else {  // active entry
    w.add->set_sensitive(true);
    w.del->set_sensitive(true);
  }
  if (app->w.entry->is_visible()) {
    w.im->set_sensitive(true);
    w.ex->set_sensitive(true);
  } else {
    w.im->set_sensitive(false);
    w.ex->set_sensitive(false);
  }
  w.revert->set_sensitive(false);  // enabled by the TTEntryWidget
}

//.............................................................................
int
TTMenuBar::set_cb_widget(GdkEventFocus *dummy, Editable *widget)
{
  cb.widget = widget;
  w.copy->set_sensitive(widget);
  w.paste->set_sensitive(widget && cb.buffer.length());
  return 0;
}

//.............................................................................
void
TTMenuBar::clear_bookmarks()
{
  bm.clear();
  app->save_config();
  bm_build_menu(false);
}

//.............................................................................
string
TTMenuBar::get_bm_string()
{
  if (!bm.size()) return string();

  strstream s;
  for (vector<ttbm_t>::iterator i = bm.begin(); i != bm.end(); i++) {
    if (s.pcount()) s << ',';
    s << i->id;
  }
  string t = s.str();  // segfaults if nothing has been written
  t.resize(s.pcount());
  return string(t);
}

//.............................................................................
int
TTMenuBar::parse_bm_string(const string &s)
{
  bm.clear();  // clear current list of bookmarks

  // parse string
  const char *p = s.c_str();
  for (;;) {
    char *q;
    dbid_t id = strtol(p, &q, 10);
    if (p == q) break;  // parse error or end of string
    if (id >= 0) bm_add(id);
    for (++q; *q == ',' || isspace(*q); q++);
    p = (const char*) q;
  }
#ifdef ENABLE_DEBUG
  DMSG << "read " << bm.size() << " boomark" << (bm.size() == 1 ? "" : "s") <<
    endl;
#endif
  bm_build_menu();
  return bm.size();
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTMenuBar::bm_recall(dbid_t id)
{
  dbid_t active = app->get_active();
  if (active != id) {
    if (!app->w.entry->save_record(false)) return;
    app->w.entry->load_record(id);
  }
  // we don't complain if active == id because we can't determine whether
  // the entry widget is running--if that isn't the case, the request is
  // justified
  app->run_entry();
}

//.............................................................................
void
TTMenuBar::bm_clear_menu()
{
  if (w.bm->items().size() > 3)
    // awkward, I know, but this iterator doesn't support operator+(int)
    w.bm->items().erase(++(++(++(w.bm->items().begin()))), w.bm->items().end());
}

//.............................................................................
void
TTMenuBar::bm_build_menu(bool sort_first)
{
  bm_clear_menu();  // remove existing bookmark items from the menu

  // sort, if necessary
  if (sort_first) sort(bm.begin(), bm.end(), bm_cmp);

  // add the entries to the Bookmarks menu
  if (bm.size()) {  // bookmarks
    for (vector<ttbm_t>::iterator i = bm.begin(); i != bm.end(); i++) {
      w.bm->items().push_back(Menu_Helpers::MenuElem(i->text,
        bind(slot(this, &TTMenuBar::bm_recall), i->id)));
    }
  } else {  // no bookmarks
    w.bm->items().push_back(Menu_Helpers::MenuElem(_("(no bookmarks)"), 0));
    w.bm->items().back()->set_sensitive(0);
  }
}

//.............................................................................
bool
TTMenuBar::bm_add(dbid_t id)
{
  dbf_t flags;
  string summary, text;
  if (app->db.read(id, flags, summary, text) == DB_SUCCESS) {
    if (summary.length() > 80) {  // truncate text if too long
      summary.resize(77);
      summary += "...";
    }
    ttbm_t x = { id, summary };
    bm.push_back(x);
    return true;
  } else
    return false;
}

//.............................................................................
bool
TTMenuBar::bm_del(dbid_t id)
{
  for (vector<ttbm_t>::iterator i = bm.begin(); i != bm.end(); i++)
    if (i->id == id) {
      bm.erase(i);
      return true;
    }
  return false;
}

