/* ==================================================== ======== ======= *
 *
 *  uevent.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _uevent_hpp_
#define	_uevent_hpp_
//pragma ident	"@(#)uevent.hpp	ubit:03.05.03"
#include <ubit/udefs.hpp>

/** X Event hidden type.
 */
typedef union _XEvent* UX_Event; 

/* ==================================================== ======== ======= */

/** Rectangular Region.
 */
struct URegion {
  u_pos x, y;
  u_dim width, height;
 
  URegion() {};
  URegion(const URegion& r) {*this = r;}
  URegion(u_pos rx, u_pos ry, u_dim rwidth, u_dim rheight) {
    x = rx; y = ry; width = rwidth; height = rheight;
  }

  void set(const URegion& r) {*this = r;}
  void set(u_pos rx, u_pos ry, u_dim rwidth, u_dim rheight) {
    x = rx; y = ry; width = rwidth; height = rheight;
  }
  
  // effectue l'intersection de this avec clip
  // this est mis a jour (contient l'intersection), clip ne change pas.
  // renvoie:
  // - false si pas d'intersection (region inchangee)
  // - true sinon si region est incluse dans clip (ie. clipping inutile)
  //
  bool setInter(const URegion &clip2);
  bool setInter(const URegion *clip2);
  bool setInter(u_pos x, u_pos y, u_dim w, u_dim h);

  //Note: les regions vides ne sont pas prises en compte!
  void setClosure(const URegion &clip2);
  void setClosure(const URegion *clip2);
  void setClosure(u_pos x, u_pos y, u_dim w, u_dim h);
};

/* ==================================================== ======== ======= */

/** Event Properties of the Source & Source View.
 */
struct USourceProps {
  unsigned char enabled;
  ///< is the source enabled ?.

  unsigned char inMenu;
  ///< is the source directly or indirectly included in a menu?.
  
  unsigned char autoCloseMenu;
  ///< does the source close menus when clicked?.

  unsigned char redrawStatus;
  URegion redrawClip;
  ///< redrawClip is the region that must be redrawn (defined only if redrawStatus is true).

  class UView* opaqueView;
  ///< the last opaque view beneath the sourceView.

  class UView* layoutView;
  ///< layout should take place from this view.

  class UView* incrustView;

  const UCursor* cursor;
  ///< the cursor of the sourceView.

  class UGroup   *browsingGroup, *parentBrowsingGroup;  ///< [impl].
  class UContext *parentContext;  ///< [impl].

  USourceProps(UView* hardwin_view) {reset(hardwin_view);}

  void reset(UView* hardwin_view);
  void set(UView* view, UGroup*, const UContext& curp);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Ubit Event class.
 */
class UEvent {
public:
  /// standard Event IDs (correspond to UOn conditions: see class UOn for details).
  enum ID {
    rawEvent  	= 0,
    mpress	= 1<<0, 
    mrelease	= 1<<1, 
    mmove 	= 1<<2, 
    mdrag 	= 1<<3, 
    kpress 	= 1<<4, 
    krelease 	= 1<<5, 
    enter 	= 1<<6, 
    leave 	= 1<<7, 
    change 	= 1<<8,
    select 	= 1<<9,
    unselect 	= 1<<10,
    action      = 1<<11,  // pas utilise correctement!
    add		= 1<<15,
    remove	= 1<<16,
    close       = 1<<17,
    destruct    = 1<<18,
    viewPaint 	= 1<<20, 
    viewResize 	= 1<<21, 
    viewMove 	= 1<<22, 
    search	= 1<<23, 
    //input	= 1<<24, 
    //timeout	= 1<<25, 
    umessage 	= 1<<26,
    notifyEvent = 1<<27 //!BEWARE: no comma after last item!
  };

  static const u_id MButton1, MButton2, MButton3, MButton4, MButton5;
  /**< mouse button IDs for UEvent::getButtons().
   * note: same values as X defs: Button1Mask etc.
   */

  static const u_id MShift, MLock, MControl, MOption; //MAlt, MNumLock, MAltGr;
  /**< key modifiers IDs for UEvent::getMods().
   * note: same values as X defs: ShiftMask etc.
   */

  static const u_id UbitEventFlow;
  /**< state mask raised for non standard event flows.
   * one of the possible ORed values of getState().
   * if true, getFlow() will return an non standard event flow.
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // basics

  int getID() const {return id;}
  ///< returns the event id (see: UEvent::ID).

  const class UCond* getCond() const {return cond;}
  /**< returns the xxx condition of the:  UOn::xxx / ucall( ).
   * Note: as the same UEvent can be catched by several ucall() conditions
   * this value should be checked immediately in the callback function
   * as it can be changed at a later time
   */
  
  UView* getView() const {return sourceView;}
  /**< returns the source VIEW that received this event.
    * Note: a box can control several Views.
    */
  
  UGroup* getSource() const;
  UBox* getBoxSource() const;
  /**< returns the object that produced this event.
    * - getBoxSource() returns null if the source is not a UBox.
    * - getSource() returns null if the source is not a UGroup.
    */
  
  UGroup* getTarget() const;
  UBox*   getBoxTarget() const;
  UBrick* getBrickTarget() const;
  /**< returns the object that was modified by this event (when available).
    * used for UOn::childSelect/childUnselect/add/remove/DND callbacks to indicate
    * the object that was selected, unselected, added, removed, etc.
    */

  class UAppli* getAppli() const;
  ///< returns the Application Context.

  class UDisp* getDisp() const;
  int getDispID() const;
  /**< returns the Display or Display ID where this event was produced.
    * the ID of the default Display is 0.
    */
  class UFlow* getFlow() const {return eflow;}
  int getFlowID() const;
  /**< returns the Event Flow that produced this event.
    * the ID of the default event flow (= the standard X event flow) is 0.
    */

  u_time getTime() const;
  ///< returns the time when this event was produced (when available).
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // mouse events

  u_pos getX() const;
  u_pos getY() const;
  /**< (mouse events) return the location of the mouse in the Source VIEW .
   * returns -1 if source view wasn't found (= when getView() returns null)
    */

  u_pos getXscreen() const;
  u_pos getYscreen() const;
  ///< (mouse events) return the location of the mouse on the SCREEN.

  u_pos getXwin()  const {return xmouse;}
  u_pos getYwin()  const {return ymouse;}
  /**< (mouse events) return the location of the mouse in the WINDOW that caught this event.
   *   this location is relative to the window returned by getHardWin())
   */

  int getClickCount() const;
  ///< (mouse events) returns the number of mouse clicks.

  u_id getButtons() const;
  /**< (mouse events) returns the button(s) that are being pressed.
   * details:
   * - for press and release events:
   *   value is ONE of UEvent::MButton1, UEvent::MButton2, etc.
   *   exple:   if (e.getButtons() == UEvent::MButton1)
   *
   * - for move or drag events:
   *   value is an ORed COMBINATION of these values (as several buttons can be
   *   pressed simultaneously).
   *   exple:   if (e.getButtons() & (UEvent::MButton1 | Event::MButton3))
   *
   * - returns 0 if no button is pressed.
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // mouse and keyboard events

  u_id getMods() const;
  /**< (mouse & keyboard events) returns the key modifiers that are being pressed.
    * returns an ORed combination of these modifiers
    * exple:   if (e.getMods() & (UEvent::KShift | UEvent::KControl))
    */

  //u_id getState() const;
  // (mouse & keyboard events) returns a combination of key and mouse button modifiers.
    
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // keyboard events

  int getKeyChar() const;
  ///< (key events) returns the typed character (in ASCII encoding).

  void setKeyChar(int);
  ///< (key events) changes the typed character (in ASCII encoding).

  u_id getKeySym() const;
  /**< (key events) returns the X KeySym.
    * Note: key modifiers (Shift, Alt..) are not taken into into account.
    */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // change events.
  
  UProp* getChangedProp() const;
  UElem* getChangedElem() const;
  UStr*  getChangedStr() const;
  ///< (UOn::change, propChange, elemChange, strChange callbacks) returns the object whose value was changed.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // umessage events.
  
  bool getMessage(UStr&) const;
  ///< (UOn::umessage callbacks) returns the data sent through an Ubit message.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // element retrieval

  class UElem* getElem();
  ///< returns the element that was under the mouse (null if none found).

  class UStr*  getStr();
  /**< returns the UStr that was under the mouse (null if no str found).
   * Note: same function as getElem() but discards elements that 
   * are not strings.
   */

  class UElem* getElem(class UElemProps&);
  class UStr*  getStr(class UElemProps&);
  /**<
   * variants of getElem() and getStr() that return useful information.
   * argument UElemProps (see this class) is filled with useful data such
   * as the elem location, its graphical properties, etc.
   *
   * Note on getStr() : the region returned in UElemProps does not contain
   * the entire string but just the line that is beneath the mouse 
   * (in case of UFlowView rendering)
   */

  class UElem* searchElem(class UElemProps&, const UElem* searched_elem,
			  int strpos1 = -1, int strpos2 = -1);
  /**<
   * searches the element that is given as an argument.
   * returns useful information about this element in UElemProps
   * notes:
   * - args. strpos1, strpos2 are used when searching for UStr(ings)
   * - the region returned in UElemProps does not contain the entire string
   *   but just its 'strpos1' to 'strpos2' subpart
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // flags and event management for transparent tools

  void goThrough(UBox*);
  ///< events will go through this object.

  void setMouse(u_pos x, u_pos y);
  ///< changes mouse coordinates (x,y are in window coordinates).

  void addFlagdef(const class UFlagdef*);
  ///< adds a Flag to the Event (for transparent tools, etc); see: UFlag.

  const class UFlagdef* getFlagdef(const class UFlag&) const;
  const class UPropdef* getPropdef(const class UFlag&) const;

  const class UFlagdef** getFlagdefs() const {return flagdefs;}
  u_count getFlagdefCount() const {return flagdefCount;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  class UWin*  getHardWin() const;
  class UView* getHardWinView() const;
  ///< [impl] returns the HARD Window that received this event.

  class USourceProps& getSourceProps() {return sp;}
  ///< [impl] gets the source properties.

  UX_Event getXEvent() const {return xev;}
  ///< [impl] returns the X Event (NB: only valid at reception time).
  
  UEvent(int event_id, class UFlow*, class UView* hardwin_view, UX_Event);
  ///< [impl] 'win_view' must be a valid UWin view or null.

  ~UEvent(); 

#ifndef NO_DOC
  USourceProps   sp;		 // Source properties
private:
  friend class UFlow;
  friend class UCtrl;
  friend class UView;

  class UGroup*  source;	 // the box or group that produced this event
  class UView*   sourceView;     // the view that contains the mouse
  class UView*   hardWinView;    // the view of the containing HARD window
  class UBrick*  aux;            // target or non group source
  class UFlow*   eflow;          // input flow that produced this event
  const class UCond* cond;	 // UOn condition of the ucall()
  const class UFlagdef**  flagdefs;    // event flags
  unsigned short flagdefCount;
  unsigned char preChildEventNotify, postChildEventNotify, resendEvent;
  int    id;	                 // event id
  int    detail;                 // additional data
  u_time time;	                 // event time stamp
  u_pos  xmouse, ymouse;         // current mouse position in the HARD window
  u_pos  xdrag_ref, ydrag_ref;   // screen coords of the View that got the mpress
  UX_Event xev;	                 // pointer to X event (hidden type)

public:
  void setID(int);               // sets Event ID
  void setSource(UView*);        // sets the View & the Box that produced this event
  void setSource(UGroup*);       // sets the Group (when no View is available)
  void setAux(UBrick*);
  void setCond(const UCond*);
  void setTime(u_time t)               {time = t;}
  void setPostChildEventNotify(bool s) {postChildEventNotify = bool(s);}
  bool isPostChildEventNotify()        {return bool(postChildEventNotify);}

  UView* locateSource(UView* searched_view);
  UView* locateSource(u_pos x_in_hardwin, u_pos y_in_hardwin);
  ///< [impl] returns UView if found and null otherwise.

  void actualize(int event_id, u_pos x_in_hardwin, u_pos y_in_hardwin, UX_Event ev)
  {id = event_id; xev = ev; xmouse = x_in_hardwin; ymouse = y_in_hardwin;}
  ///< [impl] actualize events with new Event ID and new coordinates.

  void copy(const UEvent&);
  ///< [impl] copies UEvent content AND the content of all pointed fields.
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Useful information on UElem objects that are retrieved 
 *  by UEvent::getElem().
 */
class UElemProps {
public:
  class UElem* elem;
  class ULink* elemLink;
  /**< points to the element that is located under the mouse when calling 
   *  UEvent::getElem()/getStr(). */

  URegion region;
  /**< the region that encloses this element. 
   * NOTE that these coordinates are relative to the enclosing WINDOW 
   * (not the elem parent!). use methods getX(), getY() for obtaining 
   * local coordinates. */

  bool exactMatch;
  /**< true if this element is exactly located under the Mouse.
   * false if this elem is the last element before the Mouse position */

  int strpos, strpos2;
  ///< strpos = position of the mouse in the string if elem is an UStr.

  class UWinContext* winContext;
  class UContext*    elemContext;
  /**< graphical contexts of the enclosing window and this element.
   * Note: this data is automatically deleted when UElemProps is deleted */

public:
  UElemProps(); 
  ~UElemProps();

  u_pos getX(class UEvent*);
  u_pos getY(class UEvent*);
  /**< return the location of the element relatively to the source VIEW.
   * (ie. the view returned by e.getView()) */

  u_dim getWidth();
  u_dim getHeight();
  ///< return the width and the height of the element.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

#ifndef NO_DOC
  friend class UEvent;
  friend class UView;

  void set(UContext&, UElem*, class ULink*, const URegion&, 
	   int _strpos, bool exact_match);
  void merge(UContext&, UElem*, class ULink*, const URegion&,
	     bool exact_match);
#endif
};


#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */

