/* @(#) Maps a one band uchar in image to a 3 band uchar out image. The
 * @(#) colours are chosen to give an image of uniform luminance, but 
 * @(#) with a wide range of hues.
 * @(#)
 * @(#) Input should be either memory mapped or in buffer, out should have
 * @(#) been set by a call to im_openout() or im_setbuf().
 * @(#)
 * @(#) Usage: 
 * @(#) 
 * @(#) 	int 
 * @(#) 	im_falsecolour( IMAGE *in, IMAGE *out )
 * @(#)
 * @(#) Returns 0 on success and -1 on error
 * @(#)
 *
 * 23/6/95 JC
 *	- rewritten for PIO
 *	- now walks edges of colour cube to get more saturated appearance
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#include <stdio.h>
#include <stdlib.h>

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* IM_TYPE_RGB walk around the edge of a colour cube in 48 perceptually equal steps.
 * From Niblack, Intro. to digital image processing, p 50.
 */
static const double table[48][3] = {
	{ 1.000, 0.000, 0.000 },	/* 1 */
	{ 1.000, 0.064, 0.000 },
	{ 1.000, 0.193, 0.000 },
	{ 1.000, 0.258, 0.000 },
	{ 1.000, 0.323, 0.000 },

	{ 1.000, 0.387, 0.000 },	/* 6 */
	{ 1.000, 0.452, 0.000 },
	{ 1.000, 0.516, 0.000 },
	{ 1.000, 0.548, 0.000 },
	{ 1.000, 0.613, 0.000 },

	{ 1.000, 0.677, 0.000 },	/* 11 */
	{ 1.000, 0.742, 0.000 },
	{ 1.000, 0.806, 0.000 },
	{ 1.000, 0.871, 0.000 },
	{ 1.000, 0.936, 0.000 },

	{ 1.000, 1.000, 0.000 },	/* 16 */
	{ 0.903, 1.000, 0.000 },
	{ 0.806, 1.000, 0.000 },
	{ 0.677, 1.000, 0.000 },
	{ 0.516, 1.000, 0.000 },

	{ 0.000, 1.000, 0.452 },	/* 21 */
	{ 0.000, 1.000, 0.710 },
	{ 0.000, 1.000, 1.000 },
	{ 0.000, 0.871, 1.000 },
	{ 0.000, 0.677, 1.000 },

	{ 0.000, 0.613, 1.000 },	/* 26 */
	{ 0.000, 0.548, 1.000 },
	{ 0.000, 0.484, 1.000 },
	{ 0.000, 0.419, 1.000 },
	{ 0.000, 0.355, 1.000 },

	{ 0.000, 0.290, 1.000 },	/* 31 */
	{ 0.000, 0.226, 1.000 },
	{ 0.000, 0.161, 1.000 },
	{ 0.000, 0.000, 1.000 },
	{ 0.516, 0.000, 1.000 },

	{ 0.581, 0.000, 1.000 },	/* 36 */
	{ 0.645, 0.000, 1.000 },
	{ 0.710, 0.000, 1.000 },
	{ 0.774, 0.000, 1.000 },
	{ 0.839, 0.000, 1.000 },

	{ 0.903, 0.000, 1.000 },	/* 41 */
	{ 0.968, 0.000, 1.000 },
	{ 1.000, 0.000, 0.935 },
	{ 1.000, 0.000, 0.839 },
	{ 1.000, 0.000, 0.742 },

	{ 1.000, 0.000, 0.613 },	/* 46 */
	{ 1.000, 0.000, 0.484 },
	{ 1.000, 0.000, 0.097 }
};

/* False colour. One band uchar images only.
 */
int
im_falsecolour( IMAGE *in, IMAGE *out )
{
	IMAGE *lut = im_open_local( out, "im_falsecolour:1", "t" );
	PEL *p;
	int i;

	/* Check our args. 
	 */
	if( !lut || im_piocheck( in, out ) ) 
		return( -1 );
	if( in->Bands != 1 || in->Coding != IM_CODING_NONE ||
		in->BandFmt != IM_BANDFMT_UCHAR ) {
		im_errormsg( "im_falsecolour: input image not one band "
			"uchar uncoded" );
		return( -1 );
	}

	/* Set up the LUT. 
	 */
	if( im_cp_desc( lut, in ) ) 
		return( -1 );
	lut->Bands = 3;
	lut->Xsize = 256;
	lut->Ysize = 1;
	if( im_setupout( lut ) ) 
		return( -1 );

	/* Build the LUT. At each point, map to our 48 steps, and interpolate
	 * linearly.
	 */
	for( p = (PEL *) lut->data, i = 0; i < 256; i++ ) {
		double pos = i * (47.0 / 256.0);
		int index = pos;
		double err = pos - index;

#define interpolate( N ) \
	((table[index+1][N] - table[index][N]) * err + table[index][N])

		p[0] = 255 * interpolate( 0 );
		p[1] = 255 * interpolate( 1 );
		p[2] = 255 * interpolate( 2 );
		p += 3;
	}

	/* Map the image through the lut above.
	 */
	if( im_maplut( in, out, lut ) )
		return( -1 );

	return( 0 );
}
