//<copyright>
//
// Copyright (c) 1994,95
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>
//
// Note: this implementation is based on similar code for IrisGL,
// see copyright notice of original version below

//<file>
//
// File:        oglcontext.C - glyph for 3D drawing
//              implementation of glcontext for the
//              MESA (OpenGL workalike) graphics library
//              see xmesa.h how things work here
//
// Created:     10 Aug 94   Michael Pichler
//
// Changed:     17 Oct 96   Michael Pichler
//
// $Id: mesacontext.C,v 1.14 1997/02/25 17:03:58 mpichler Exp $
//
//</file>

/*
 * Copyright (c) 1991 Stanford University
 * Copyright (c) 1991 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Stanford and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Stanford and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */


#include "gecontext.h"
#include "wtable.h"

#ifdef GE3DTEXT
#include <ge3d/ge3d.h>
#include <InterViews/font.h>
#include <IV-X11/xfont.h>
#endif

#include <InterViews/canvas.h>
#include <InterViews/display.h>
#include <InterViews/session.h>
#include <InterViews/style.h>
#include <InterViews/window.h>

#include <hyperg/utils/verbose.h>
#include <hyperg/utils/str.h>

#include <IV-X11/Xlib.h>
#include <IV-X11/xcanvas.h>
#include <IV-X11/xdisplay.h>
#include <IV-X11/xwindow.h>

#include <IV-X11/Xdefs.h>      /* Display, Font of X11 */
/* include <mesa/GL/xmesa.h> */
#include <mesa/GL/glx.h>
#include <mesa/GL/glu.h>
#include <IV-X11/Xundefs.h>  /* get back to IV's names */

#include <iostream.h>
#include <stdlib.h>
#include <string.h>


// class GLWindow for Implementation of class GEContext


#ifdef XMESA_MAJOR_VERSION
// XMesaCreateContext and XMesaDestroyContext prototypes changed in Mesa 2.0;
// XMesaVisual, XMesaBuffer newly introduced (heavy changes)
#define MESA_2_0
#endif


class GLWindow: public Window
{
  public:
    GLWindow (Glyph*, Window* parent, int overlay);
    ~GLWindow ();

    virtual void place (Coord x, Coord y);
    virtual void bind ();
    virtual void repair ();

    void makeMesaContext (XDisplay* xdpy,
      XMesaContext firstmesacontext, int imgcontext);

    virtual void setattributes ()
    { set_attributes (); }

    void activateOverlay (int clear);
    void deactivateOverlay ();

    void shading (int flag)
    { shading_ = flag; }

  private:
    Window* parent_;
    XWindow overlay_xwin_;
    int double_buffered_;
    int requestoverlay_;
    int shading_;
    XMesaContext pixcon_, imgcon_;      // mesa context with X-pixmap/X-image
    XVisualInfo xvisinfo_;              // Mesa 2.0 keeps a pointer to this info
#ifdef MESA_2_0
    XMesaVisual pixvis_, imgvis_;       // mesa visual for X-pixmap/X-image
    XMesaBuffer pixbuf_, imgbuf_;       // mesa buffer with X-pixmap/X-image
#endif
};



GLWindow::GLWindow (Glyph* g, Window* parent, int overlay)
: Window(g)
{
  parent_ = parent;
  requestoverlay_ = overlay;
  shading_ = 1;  // first image will be shaded (otherwise noticeable color difference)
  overlay_xwin_ = 0;
  pixcon_ = 0;
  imgcon_ = 0;
#ifdef MESA_2_0
  pixbuf_ = 0;
  imgbuf_ = 0;
  pixvis_ = 0;
  imgvis_ = 0;
#endif

  Display* disp = parent->display ();
  display (disp);
  // xvisinfo_ set in bind

  Style* s = new Style (disp->style ());
  s->alias ("GLWindow");
  // use double buffering for GL window (unless single buffered required explicitly)
  double_buffered_ = !s->value_is_on ("single_buffered");  // default on
  s->attribute ("double_buffered", "off");  // otherwise InterViews causes errors!
  // Mesa: always indirect rendering
  style (s);
}


GLWindow::~GLWindow()
{
#ifdef MESA_2_0
  if (pixbuf_)
    XMesaDestroyBuffer (pixbuf_);
  if (imgbuf_)
    XMesaDestroyBuffer (imgbuf_);
  if (pixvis_)
    XMesaDestroyVisual (pixvis_);
  if (imgvis_)
    XMesaDestroyVisual (imgvis_);
#endif
  if (pixcon_)
    XMesaDestroyContext (pixcon_);
  if (imgcon_)
    XMesaDestroyContext (imgcon_);
}


// no GLWindow::draw


void GLWindow::place (Coord x, Coord y)
{
  DEBUGNL ("GLWindow::place");

  Window::place (x, y);
  WindowRep* wrep = rep ();
  Display* dis = wrep->display_;
  wrep->xpos_ = dis->to_pixels (wrep->left_);
  wrep->ypos_ = parent_->canvas ()->pheight () - dis->to_pixels (wrep->bottom_) 
                - canvas ()->pheight ();
  if (wrep->xwindow_ != WindowRep::unbound)
    XMoveWindow (dis->rep()->display_, wrep->xwindow_, wrep->xpos_, wrep->ypos_);

  DEBUGNL ("finished GLWindow::place");
}


// helper function for making a mesa context
// with various settings/flags

void GLWindow::makeMesaContext (
  XDisplay* xdpy, XMesaContext firstmesacontext, int imgcontext
)
{
  XMesaContext context;

#ifdef MESA_2_0

  XMesaVisual mvis = XMesaCreateVisual (
    xdpy, &xvisinfo_, 1,  // rgb mode
    0,  // no alpha planes (Mesa 2.0)
    double_buffered_,
    imgcontext,  // ximage or pixmap
    1, 0, 0,  // depth buffer, no stencil, no accum (Mesa 2.0)
    0  // level (Mesa 2.0)
  );

  if (!mvis)
  { cerr << "3D context. error: XMesaCreateVisual failed." << endl;
    context = 0;
  }
  else
    context = XMesaCreateContext (mvis, firstmesacontext);

#else
  // XMesaCreateContext interface prior to 2.0
  context = XMesaCreateContext (
    xdpy, &xvisinfo_, 1,  // rgb mode
    double_buffered_,
    imgcontext,  // ximage or pixmap
    firstmesacontext  // for sharing display lists
  );
#endif

  if (imgcontext)
    imgcon_ = context;
  else
    pixcon_ = context;

#ifdef MESA_2_0
  if (imgcontext)
    imgvis_ = mvis;
  else
    pixvis_ = mvis;
#endif

} // makeMesaContext


// bind

void GLWindow::bind ()
{
  DEBUGNL ("beginning GLWindow::bind (Mesa)");

  DisplayRep* drep = display ()->rep ();
  XDisplay* xdpy = drep->display_;

  // create a window

  WindowRep* w = rep ();
  Canvas* c = w->canvas_;
  WindowTable* t = drep->wtable_;
  XWindow xw = w->xwindow_;

  WindowVisual* wvis = drep->default_visual_;
  Visual* vi_sual = wvis->visual ();
  int dep_th = wvis->depth ();

  if (xw != WindowRep::unbound)
    t->remove (xw);

  Window::set_attributes ();
  w->xattrmask_ &= ~CWDontPropagate;     // IV 3.1
  w->xattrs_.do_not_propagate_mask = 0;  // IV 3.1
//   w->xattrmask_ |= CWColormap;
//   w->xattrs_.colormap = color_map;

  // interested in structure and pointer events
  w->xattrs_.event_mask = ExposureMask | StructureNotifyMask |
    ButtonPressMask | ButtonReleaseMask |
    PointerMotionMask | PointerMotionHintMask;

  DEBUGNL ("creating X window ...");

  xw = XCreateWindow (
    xdpy, parent_->rep ()->xwindow_,
    w->xpos_, w->ypos_, c->pwidth (), c->pheight (), 0, // border width
    dep_th, w->xclass_, vi_sual, w->xattrmask_, &w->xattrs_
  );

  DEBUGNL ("... X window created");

  c->rep ()->xdrawable_ = xw;
  t->insert (xw, this);
  w->xwindow_ = xw;  // this X window will be used for GL
  w->xtoplevel_ = w->toplevel_->rep ()->xwindow_;

  static XMesaContext firstmesacontext = 0;  // to share display lists for texturing

  // optionally pixmap or ximage context only (for performance tests)
  int backbuffers = 3;  // both (auto)
  RString backbuftype;
  if (style ()->find_attribute ("mesabackbuf", backbuftype))
  { if (strstr (backbuftype.string (), "pixmap"))
      backbuffers = 1;  // Xpixmap only
    else if (strstr (backbuftype.string (), "image"))
      backbuffers = 2;  // Ximage only
    else if (strstr (backbuftype.string (), "auto"))
      backbuffers = 3;
  }

  DEBUGNL ("Mesa backbuffer (1 = pixmap, 2 = ximage, 3 = both): " << backbuffers);

#if 0
  // old MesaCreateContext (up to 1.1.2(?)): XWindow argument
  xmscon_ = XMesaCreateContext (
    xdpy, xw, 1 /* rgb */,
    double_buffered_,
    backimage_ /* 1: XImage, 0: pixmap */
  );
  if (!xmscon_)
    cerr << "3D context. fatal: XMesaCreateContext failed." << endl;
#else

  // new MesaCreateContext (from 1.1.5(?)): XVisualInfo argument, new XMesaBindWindow
  // assert: XMesaCreateContext only access only certain visinfo fields (visual, depth)
  xvisinfo_.visual = vi_sual;
  xvisinfo_.depth = dep_th;
  // Mesa 2.0 accesses some more fields from visinfo (most of them accessible via Visual anyway) ...
  xvisinfo_.visualid = vi_sual->visualid;
  xvisinfo_.screen = wvis->screen ();
  xvisinfo_.c_class = vi_sual->c_class;
  xvisinfo_.red_mask = vi_sual->red_mask;
  xvisinfo_.green_mask = vi_sual->green_mask;
  xvisinfo_.blue_mask = vi_sual->blue_mask;
  xvisinfo_.colormap_size = vi_sual->map_entries;
  xvisinfo_.bits_per_rgb = vi_sual->bits_per_rgb;

  DEBUGNL ("making context. visual ID: 0x" << hex << xvisinfo_.visualid << dec << ", depth: " << xvisinfo_.depth);

  if (backbuffers & 1)  // pixmap
  {
    makeMesaContext (xdpy, firstmesacontext, 0);  // pixmap
    // sets pixcon_ and pixvis_ (Mesa 2.0)
    if (!firstmesacontext)
      firstmesacontext = pixcon_;
  }

  if (backbuffers & 2)  // ximage
  {
    makeMesaContext (xdpy, firstmesacontext, 1);  // ximage
    // sets imgcon_ and imgvis_ (Mesa 2.0)
    if (!firstmesacontext)
      firstmesacontext = imgcon_;
  }

  if (!pixcon_ && !imgcon_)
    cerr << "3D context. fatal: XMesaCreateContext failed." << endl;

  DEBUGNL ("creating mesa window buffer");

#ifdef MESA_2_0
  if (pixcon_ && !(pixbuf_ = XMesaCreateWindowBuffer (pixvis_, xw)))
    cerr << "3D context. fatal: XMesaCreateWindowBuffer failed for pixmap context" << endl;
  if (imgcon_ && !(imgbuf_ = XMesaCreateWindowBuffer (imgvis_, xw)))
    cerr << "3D context. fatal: XMesaCreateWindowBuffer failed for ximage context" << endl;
#else
  if ((pixcon_ && !XMesaBindWindow (pixcon_, xw)) || (imgcon_ && !XMesaBindWindow (imgcon_, xw)))
    cerr << "3D context. fatal: XMesaBindWindow failed." << endl;
#endif

#endif

  DEBUGNL ("mapping window");
  XMapRaised (drep->display_, xw);

  DEBUGNL ("finished GLWindow::bind");

} // GLWindow::bind (MESA)



void GLWindow::repair ()
{
//cerr << "GLWindow::repair () entered" << endl;

  WindowRep* wrep = rep ();
  Canvas* c = wrep->canvas_;
  CanvasRep* can = c->rep();

  if (can->damaged_)
  {
    // XDisplay* xdpy = wrep->display_->rep ()->display_;
    XMesaContext xmcon;
#ifdef MESA_2_0
    XMesaBuffer xmbuf;
#endif

    if (shading_)
    {
      xmcon = imgcon_ ? imgcon_ : pixcon_;
#ifdef MESA_2_0
      xmbuf = imgbuf_ ? imgbuf_ : pixbuf_;
#endif
      // cerr << "using imgcon (shading)" << endl;
    }
    else
    {
      xmcon = pixcon_ ? pixcon_ : imgcon_;
#ifdef MESA_2_0
      xmbuf = pixbuf_ ? pixbuf_ : imgbuf_;
#endif
      // cerr << "using pixcon (wire)" << endl;
    }
#ifdef MESA_2_0
//  cerr << "XMesaMakeCurrent (" << (void*) xmcon << ", " << (void*) xmbuf << ")" << endl;
    XMesaMakeCurrent (xmcon, xmbuf);
#else
    XMesaMakeCurrent (xmcon);
#endif

    // cerr << "nodither" << endl;  glDisable (GL_DITHER);  // test

    // some OpenGL libraries automatically reshape the viewport, others don't
    // glViewport (0, 0, c->pwidth () - 1 , c->pheight () - 1);  // -1 for Mesa 1.2.1 (bugfix)!!
//  cerr << "glViewport (0, 0, " << c->pwidth () << ", " << c->pheight () << ")" << endl;
    glViewport (0, 0, c->pwidth (), c->pheight ());
    // cerr << "test: smaller viewport!!!" << endl;
    // glViewport (10, 10, 500, 300);

/*
    // GL: in no double buffering is available, redraw only the damaged region
    if (!double_buffered_)
    { CanvasDamage& dam = can->damage_;
      Display* d = can->display_;
      scrmask (d->to_pixels (dam.left), d->to_pixels (dam.right),
               d->to_pixels (dam.bottom), d->to_pixels (dam.top));
    }
*/
//  cerr << "drawing body glyph" << endl;
    wrep->glyph_->draw (wrep->canvas_, wrep->allocation_);

    if (double_buffered_)
    {
#ifdef MESA_2_0
//    cerr << "XMesaSwapBuffers (" << (void*) xmbuf << ")" << endl;
      XMesaSwapBuffers (xmbuf);  // Mesa 2.0
#else
      XMesaSwapBuffers ();  // up to Mesa 1.2.x
#endif
    }
    else
      glFlush ();

//  cerr << "can->clear_damage ();" << endl;
    can->clear_damage ();
  }

//cerr << "GLWindow::repair () done" << endl;

} // repair


void GLWindow::activateOverlay (int /*clearing*/)
{
}


void GLWindow::deactivateOverlay ()
{
}


#ifdef GE3DTEXT
// ge3d_text

void ge3d_text (float x, float y, float z, const char* str)
{
  static int fontlistbase = -1;

//cerr << "writing '" << str << "' at position (" << x << ", " << y << ", " << z << ")." << endl;

  if (fontlistbase < 0)
  {
#ifdef OLDFONTIMPL
    const Font* font = Font::lookup ("fixed");
    if (!font)
    { fontlistbase = 0;
      return;
    }
    Resource::ref (font);
    FontRep* rep = font->rep (Session::instance ()->default_display ());
    // Interviews change: FontRep now XFontSet instead of XFontStruct

    long fid = rep->font_->fid;
#else
    XDisplay* dpy = Session::instance ()->default_display ()->rep()->display_;
    XFontStruct* xf = XLoadQueryFont (dpy, "fixed");

    if (!xf)
    { cerr << "VRweb: unable to load fixed font" << endl;
      fontlistbase = 0;
      return;
    }

    long fid = xf->fid;
#endif

    // supported characters: ' ' (0x20) to DEL (0x7f), i.e. 96 chars
    fontlistbase = glGenLists (96);
    if (fontlistbase)
      glXUseXFont (fid, 0x20, 96, fontlistbase);

#ifdef OLDFONTIMPL
    Resource::unref (font);
#endif
  }

  if (fontlistbase)  // assuming correct characters
  {
    const char* chr = str;
    int base2 = fontlistbase - 0x20;

    glRasterPos3f (x, y, z);
    while (*chr)
      glCallList (*chr++ + base2);  // *chr++ - 0x20 + fontlistbase
  }
}


// ge3dText

void ge3dText (const point3D* p, const char* str)
{
  ge3d_text (p->x, p->y, p->z, str);
}
#endif


// Implementation of class GEContext


GEContext::GEContext (Glyph* g, int requests)
: MonoGlyph (g)
{
  glwin_ = 0;  // created in allocate
  origcur_ = 0;
  requestoverlay_ = requests & RequestOverlay;
}


GEContext::~GEContext()
{
  delete glwin_;
}


Window* GEContext::window ()
{ 
  return glwin_;
}


void GEContext::setCursor (Cursor* csr)
{
  if (glwin_)
  { if (!origcur_)
      origcur_ = glwin_->cursor ();
    glwin_->cursor (csr);
  }
}


void GEContext::resetCursor ()
{
  if (glwin_ && origcur_)
    glwin_->cursor (origcur_);
}


void GEContext::pushCursor (Cursor* csr)
{
  if (glwin_)
  { if (!origcur_)
      origcur_ = glwin_->cursor ();
    glwin_->push_cursor ();
    glwin_->cursor (csr);
  }
}


void GEContext::popCursor ()
{
  if (glwin_)
    glwin_->pop_cursor ();
}


int GEContext::overlaySupport ()
{
  return 0;  // not yet ported; will depend on available visuals
}


void GEContext::activateOverlay (int clear)
{
  if (glwin_ && requestoverlay_)
    glwin_->activateOverlay (clear);
}


void GEContext::deactivateOverlay ()
{
  if (glwin_)
    glwin_->deactivateOverlay ();
}


void GEContext::allocate (Canvas* c, const Allocation& a, Extension&)
// told my actual allocation
{
  DEBUGNL ("GEContext::allocate");

  const Allotment& ax = a.x_allotment ();
  const Allotment& ay = a.y_allotment ();


  if (!glwin_)  // if first allocation, create a new child window for GL
  {
    glwin_ = new GLWindow (body (), c->window (), requestoverlay_);

    glwin_->canvas()->size (ax.span (), ay.span ());
    glwin_->place (ax.origin (), ay.origin ());

    glwin_->bind ();
  }
  else  // otherwise resize existing child window
  {
    glwin_->canvas ()->size (ax.span (), ay.span ());
    glwin_->place (ax.origin (), ay.origin ());

    glwin_->resize ();
  }

  DEBUGNL ("end of GEContext::allocate");

} // GEContext::allocate



// unbind
// delete glx window managed by gecontext
// when unmapping a window containing a gecontext do (in order):
// win->unmap ();  gecontext->unbind ();  win->unbind ();

void GEContext::unbind ()
{
  // not necessary with modified unbind of xwindow.c
  delete glwin_;  // is created again in next allocate
  glwin_ = 0;
  origcur_ = 0;
}



void GEContext::redraw (int shading)
{
  // user request to redraw whole window
  if (glwin_)
  { glwin_->shading (shading);
    glwin_->rep ()->canvas_->damage_all ();
  }
  // damage whole canvas of the glwindow
}


const char* GEContext::implementation ()
{
  return ("MESA");
}


int GEContext::implementationHints ()
{
  return impl_slow;  // no HW acceleration
}


int GEContext::graphicsSupport (Display*)
{
  return 1;  // every Xserver allows MESA output!
}
