// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: TimeWarpSimulationManager.cpp
// 
//---------------------------------------------------------------------------

#include "StopWatch.h"
#include "ObjectStub.h"
#include "SimulationObjectProxy.h"
#include "TimeWarpSimulationManager.h"
#include "TimeWarpSimulationStream.h"
#include "SchedulingManager.h"
#include "CommunicationManager.h"
#include "TerminationManager.h"
#include "AggressiveOutputManager.h"
#include "Application.h"
#include "SchedulingData.h"
#include "PartitionInfo.h"
#include "TokenPassingTerminationManager.h"
#include "EventMessage.h"
#include "InitializationMessage.h"
#include "NegativeEventMessage.h"
#include "StartMessage.h"
#include <clutils/Debug.h>
#include <algorithm>
#include <sstream>
using std::istringstream;


TimeWarpSimulationManager::TimeWarpSimulationManager( unsigned int numProcessors,
						      Application *initApplication ) :
  SimulationManagerImplementationBase(numProcessors), 
  mySimulationManagerID( 0 ),
  simulationCompleteFlag( false ),
  suppressMessage( NO ),   // initially, we do not suppress any messages
  coastForwardTime(0),
  messageAggregation( false ),   // message aggregation is turned off by default
  simulationTime(initApplication->getZero().clone()),
  myStateManager( 0 ),
  myGVTManager( 0 ),
  myCommunicationManager( 0 ),
  mySchedulingManager( 0 ),
  myOutputManager( 0 ),
  myEventSet( 0 ),
  mySchedulingData( new SchedulingData() ),
  myTerminationManager( 0 ),
  myApplication( initApplication ){
}

TimeWarpSimulationManager::~TimeWarpSimulationManager(){
  garbageCollect( getPositiveInfinity() );

  vector<SimulationObject *> *objects = globalArrayOfSimObjPtrs.getElementVector();
  for( unsigned int i = 0; i < objects->size(); i++ ){
    delete (*objects)[i];
  }
  delete objects;

  // This simulation manager is responsible for the deletion of all
  // its components.
  delete myEventSet;
  delete myStateManager;
  delete mySchedulingManager;
  delete myCommunicationManager;
  delete myOutputManager;
  delete myGVTManager;
  delete mySchedulingData;
  delete localArrayOfSimObjPtrs;
  delete simulationTime;
  delete coastForwardTime;
}

const VTime &
TimeWarpSimulationManager::getSimulationTime() const {
  ASSERT( simulationTime != 0 );
  return *simulationTime;
}

const VTime &
TimeWarpSimulationManager::getCommittedTime(){
  assert( myGVTManager != 0 );
  return myGVTManager->getGVT();
}

const VTime &
TimeWarpSimulationManager::getNextEventTime(){
  const Event *nextEvent = myEventSet->peekEvent( 0 );
  if( nextEvent != 0 ){
    return nextEvent->getReceiveTime();
  }
  else{
    return getPositiveInfinity();
  }
}

const VTime &
TimeWarpSimulationManager::getCoastForwardTime() const {
  ASSERT( coastForwardTime != 0 );
  return *coastForwardTime;
}

bool
TimeWarpSimulationManager::checkSuppressMessageFlag(){
  return ((suppressMessage != NO) ? true : false);
}

MsgSuppression
TimeWarpSimulationManager::getSuppressMessageFlag(){
  return suppressMessage;
}

void
TimeWarpSimulationManager::setSuppressMessageFlag(MsgSuppression flag){
  suppressMessage = flag;
}

bool
TimeWarpSimulationManager::checkMessageAggregationFlag(){
  return messageAggregation;
}

void
TimeWarpSimulationManager::setMessageAggregationFlag(bool flag){
  messageAggregation = flag;
}

// simulation time = minimum of all object LVTs
void
TimeWarpSimulationManager::calculateSimulationTime(){
  for(unsigned int i = 0; i < numberOfObjects; i++){
    const VTime &localVirtualTime =
      localArrayOfSimObjIDs[i]->getSimulationTime();
    if( simulationTime == 0 || localVirtualTime < getSimulationTime() ){
      setSimulationTime( localVirtualTime );
    }
  }
}

void 
TimeWarpSimulationManager::registerSimulationObjects (){
  // allocate memory for our reverse map
  localArrayOfSimObjIDs.resize(numberOfObjects);

  // allocate memory for our global reverse map - first dimension
  globalArrayOfSimObjIDs.resize(numberOfSimulationManagers);
   
  vector<SimulationObject*> *simulationObjectsArray = new vector<SimulationObject*>;

  // allocate memory for the second dimension of our 2-D array
  globalArrayOfSimObjIDs[mySimulationManagerID].resize(numberOfObjects);

  vector<SimulationObject *> *objects = localArrayOfSimObjPtrs->getElementVector();
  vector<string> *keys = localArrayOfSimObjPtrs->getKeyVector();
  for( unsigned int i = 0; i < objects->size(); i++ ){
    // create and store in the map a relation between ids and object names
    OBJECT_ID *id = new OBJECT_ID( i, mySimulationManagerID);

    SimulationObject *object = (*objects)[i];
      
    // store in the global map
    globalArrayOfSimObjPtrs.insert( (*keys)[i], object );
      
    // store this objects id for future reference
    object->setObjectID(id);

    // store a handle to our simulation manager in the object
    object->setSimulationManager(this);

    // lets allocate the initial state here
    object->setInitialState(object->allocateState());

    // save map of ids to ptrs
    localArrayOfSimObjIDs[i] = object;
    globalArrayOfSimObjIDs[mySimulationManagerID][i] = object;

    // store the scheduler's list of simulation object ptrs
    simulationObjectsArray->push_back(object);
  }
  delete objects;
  delete keys;
}

// this function constructs the map of simulation object names versus
// simulation object pointers by interacting with the application.
// It also performs random partitioning of objects.
StringHashMap<SimulationObject *, true> *
TimeWarpSimulationManager::createMapOfObjects(){
  StringHashMap<SimulationObject *, true> *retval = 0;

  const PartitionInfo *appPartitionInfo =
    myApplication->getPartitionInfo( numberOfSimulationManagers );

  vector<SimulationObject *> *localObjects = 
    appPartitionInfo->getObjectSet( getSimulationManagerID() );
  for( vector<SimulationObject *>::iterator i = localObjects->begin();
       i < localObjects->end();
       i++ ){
    (*i)->setSimulationManager( this );
  }
  retval = partitionVectorToHashMap( localObjects );

  delete appPartitionInfo;

  return retval;
}


// this function is used by the communication manager to send out the
// names of objects on this simulation manager in the form of an init
// message during the start-up of a distributed simulation. 
vector<string>*
TimeWarpSimulationManager::getSimulationObjectNames(){
  return localArrayOfSimObjPtrs->getKeyVector();
}

// this function is called by the communication manager to make the
// simulation manager add the set of simulation object proxies (a list
// of which is passed in as a parameter for a specific simulation
// manager)

void
TimeWarpSimulationManager::registerSimulationObjectProxies( const vector<string> *arrayOfObjectProxies, 
							    unsigned int sourceSimulationManagerID,
							    unsigned int destSimulationManagerID){
  // allocate space of the second dimension of this 2-D array.
  globalArrayOfSimObjIDs[destSimulationManagerID].resize(arrayOfObjectProxies->size());

  unsigned int count = 0;
  // iterate through the vector ...
  for( vector<string>::const_iterator iter =  (*arrayOfObjectProxies).begin();
       iter < (*arrayOfObjectProxies).end();
       iter++ ){
    // create and store in the map
    OBJECT_ID *id = new OBJECT_ID(count, destSimulationManagerID);

    ASSERT(myCommunicationManager != NULL);
      
    // create the proxy simulation object. 
    SimulationObjectProxy *object =
      new SimulationObjectProxy( *iter,
				 sourceSimulationManagerID,
				 destSimulationManagerID,
				 myCommunicationManager );

    // store this proxy object's id for future reference
    object->setObjectID(id);

    // store a handle to our simulation manager in this proxy object
    object->setSimulationManager(this);
    
    // save in global map of names to ptrs
    globalArrayOfSimObjPtrs.insert( (*iter), object );
      
    // save in global map of ids to ptrs
    globalArrayOfSimObjIDs[destSimulationManagerID][count] = object;

    count++;
  }
} 

// the TimeWarpSimulationManager must register the following message
// types with the communication manager:

// InitializationMessage, StartMessage, EventMessage, NegativeEventMessage,
// CheckIdleMessage, AbortSimulationMessage
void
TimeWarpSimulationManager::registerWithCommunicationManager(){

  const int numberOfMessageTypes = 6;
  string messageType[numberOfMessageTypes] = { "InitializationMessage",
					       "StartMessage",
					       "EventMessage",
					       "NegativeEventMessage",
					       "CheckIdleMessage",
					       "AbortSimulationMessage" };
  ASSERT(myCommunicationManager != NULL);
  for(int count = 0; count < numberOfMessageTypes; count++){
    myCommunicationManager->registerMessageType( messageType[count], this );
  }
}

bool 
TimeWarpSimulationManager::simulationComplete( const VTime &simulateUntil ){
  bool retval = false;
  if( myGVTManager->getGVT() >= simulateUntil ){
    clutils::debug << "(" << getSimulationManagerID() << ") GVT = " << 
      myGVTManager->getGVT() << ", >= " << simulateUntil << endl;
    retval = true;
  }
  else if( myTerminationManager->terminateSimulation() ){
    clutils::debug << "(" << getSimulationManagerID() << 
      ") Termination manager says we're complete." << endl;
    simulationCompleteFlag = true;
    retval = true;
  }
  return retval;
}

void
TimeWarpSimulationManager::getMessages(){
  if(numberOfSimulationManagers > 1){
    myCommunicationManager->checkPhysicalLayerForMessages(1000);
  }
}

bool
TimeWarpSimulationManager::executeObjects(){
  const VTime *lastGVT = myApplication->getZero().clone();
  bool hadAtLeastOne = false;
  const Event *nextEvent = mySchedulingManager->peekNextEvent();
  while( nextEvent != 0 ){
    hadAtLeastOne = true;
    setSimulationTime( nextEvent->getReceiveTime() );
    SimulationObject *nextObject = getObjectHandle( nextEvent->getReceiver() );
    ASSERT( nextObject != 0 );
    ASSERT( dynamic_cast<SimulationObjectProxy *>( nextObject ) == 0 );
    nextObject->setSimulationTime( nextEvent->getReceiveTime() );
    nextObject->executeProcess();
    // lets save this state
    myStateManager->saveState( getSimulationTime(), nextObject );
    if(myGVTManager->checkGVTPeriod()){
      myGVTManager->calculateGVT();
      if( myGVTManager->getGVT() > *lastGVT ){
	garbageCollect(myGVTManager->getGVT());
	delete lastGVT;
	lastGVT = myGVTManager->getGVT().clone();
      }
    }
    nextEvent = mySchedulingManager->peekNextEvent();
  }
  return hadAtLeastOne;
}

void
TimeWarpSimulationManager::simulate( const VTime& simulateUntil ){
  StopWatch stopwatch;
  stopwatch.start();

  cerr << "SimulationManager(" << mySimulationManagerID
       << "): Starting simulation - End time: " << simulateUntil << ")\n";

  while( !simulationComplete( simulateUntil ) ){
    getMessages();
    ASSERT(mySchedulingManager != NULL);
    if( executeObjects() ){
      myTerminationManager->setStatusActive();
    }
  }
  stopwatch.stop();

  cerr << "Simulation complete (" << stopwatch.elapsed() << " secs)" << endl;
}


const Event *
TimeWarpSimulationManager::getEvent(SimulationObject *object){
  ASSERT(myEventSet != NULL);
  const Event *retval;

  if (suppressMessage == COASTFORWARD) {
    retval = myEventSet->getEvent( object, getCoastForwardTime() );
  }
  else {
    retval = myEventSet->getEvent(object);
  }

  return retval;
}

const Event *
TimeWarpSimulationManager::peekEvent(SimulationObject *object){
  ASSERT(myEventSet != NULL);
  const Event *retval;

  if( suppressMessage == COASTFORWARD ){
    retval = myEventSet->peekEvent( object, getCoastForwardTime() );
  }
  else {
    retval = myEventSet->peekEvent(object);
  }

  return retval;
}

void
TimeWarpSimulationManager::handleEvent( const Event *event ){
  // first set the send time of the event
  ASSERT(event != NULL);
  // Calculate gvt manager info.
  getGVTManager()->calculateGVTInfo();
   
  if( contains( event->getReceiver() ) == true ){
    handleLocalEvent( event );
  }
  else {
    handleRemoteEvent( event );
  }
}
 
void
TimeWarpSimulationManager::handleLocalEvent( const Event *event ){
  ASSERT(myOutputManager != NULL);
  if(suppressMessage == NO){
    // insert a copy in the output queue
    myOutputManager->insert(event);
  }
  else if(suppressMessage == LAZYCANCEL){
    abort();
  }

  SimulationObject *receiver = getObjectHandle( event->getReceiver() );
  clutils::debug << "(" << getSimulationManagerID() << ") - " << 
    receiver->getName() << " receiving event at time " << event->getReceiveTime() <<
    ", simulation time = " << receiver->getSimulationTime() << endl;
  if( event->getReceiveTime() >= receiver->getSimulationTime() ){
    acceptNewEvent( event );
  }
  else{
    rollback( receiver, event->getReceiveTime() );
    acceptNewEvent( event );
  }
}

void
TimeWarpSimulationManager::handleRemoteEvent( const Event *event ){
  if( getSuppressMessageFlag() != COASTFORWARD ){
    SimulationObjectProxy *proxyObj;
    proxyObj = dynamic_cast<SimulationObjectProxy *>( getObjectHandle(event->getReceiver()));
    ASSERT( proxyObj != 0 );

    unsigned int destSimMgrId = proxyObj->getDestId();

    const string gVTInfo =
      getGVTManager()->getGVTInfo( getSimulationManagerID(),
				   destSimMgrId,
				   event->getSendTime() );
    
    KernelMessage *msg = new EventMessage( getSimulationManagerID(),
					   destSimMgrId,
					   event,
					   gVTInfo );
    
    myCommunicationManager->sendMessage( msg,
					 destSimMgrId );
  }
}

void
TimeWarpSimulationManager::cancelLocalEvents( const vector<const Event *> &eventsToCancel ){
  for( vector<const Event *>::const_iterator i = eventsToCancel.begin();
       i < eventsToCancel.end();
       i++ ){
    SimulationObject *object = getObjectHandle( (*i)->getReceiver() );
    ASSERT( object != 0 );
    myEventSet->handleAntiMessage( object, (*i)->getEventId() );
  }
}

void
TimeWarpSimulationManager::cancelRemoteEvents( const vector<const Event *> &eventsToCancel ){
  ASSERT( eventsToCancel.size() > 0 );
  const string &receiverName = eventsToCancel[0]->getReceiver();
  const SimulationObject *forObject = getObjectHandle( receiverName );
  unsigned int destId = forObject->getObjectID()->getSimulationManagerID();
  vector<EventId> eventIds;
  for( vector<const Event *>::const_iterator i = eventsToCancel.begin();
       i < eventsToCancel.end();
       i++ ){
    ASSERT( (*i)->getReceiver() == receiverName );
    eventIds.push_back( (*i)->getEventId() );
  }

  NegativeEventMessage *newMessage = new NegativeEventMessage( getSimulationManagerID(),
							       destId,
							       eventIds );
  myCommunicationManager->sendMessage( newMessage, destId );
}

void
TimeWarpSimulationManager::handleNegativeEvents( const vector<EventId> &negativeEvents ){
  abort();
}

void 
TimeWarpSimulationManager::cancelEvents( const vector<const Event *> &eventsToCancel ){
  ASSERT( eventsToCancel.size() > 0 );
  const Event *firstEvent = eventsToCancel[0];
  ASSERT( firstEvent != 0 );
  if( contains( firstEvent->getReceiver() ) ){
    cancelLocalEvents( eventsToCancel );
  }
  else{
    cancelRemoteEvents( eventsToCancel );
  }
}

void
TimeWarpSimulationManager::acceptNewEvent( const Event *newEvent ){
  if(suppressMessage != COASTFORWARD){
    myEventSet->insert( newEvent );
  }
}

void
TimeWarpSimulationManager::rollback( SimulationObject *object, const VTime &rollbackTime ){
  ASSERT( object != 0 );
  ASSERT( dynamic_cast<SimulationObjectProxy *>(object) == 0 );
  clutils::debug << object->getName() << " rollback from " << object->getSimulationTime()
		 << " to " << rollbackTime << endl;
   
  if(rollbackTime < myGVTManager->getGVT()){
    cerr << "Rollback beyond the Global Virtual Time" << endl;
    cerr << "rollbackTime = " << rollbackTime
	 << " GVT = " << myGVTManager->getGVT()
	 << " lastScheduledEvent = " << mySchedulingManager->getLastEventScheduledTime() 
	 << endl;
      
    abort();
  }
   
  // the object's local virtual time is the timestamp of the restored
  // state.  after insert/handleAntiMessage, the next event to
  // execute will posses a timestamp greater than the object's lVT.
  const VTime &restoredTime = myStateManager->restoreState( rollbackTime, object );
  setSimulationTime( rollbackTime );
  clutils::debug << object->getName() << " rolled state back to " << restoredTime << endl;
  ASSERT( restoredTime <= getSimulationTime() );

  const OBJECT_ID *objID = object->getObjectID();
  // now rollback the file queues
  for( vector<TimeWarpSimulationStream*>::iterator i = mapOfInFileQueues[objID].begin();
       i != mapOfInFileQueues[objID].end();
       i++ ){
    (*i)->rollbackTo(rollbackTime);
  }
   
  for( vector<TimeWarpSimulationStream*>::iterator i = mapOfOutFileQueues[objID].begin();
       i != mapOfOutFileQueues[objID].end();
       i++ ){
    (*i)->rollbackTo(rollbackTime);
  }

  // send out the cancel messages
  myOutputManager->rollback( object, rollbackTime );
  myEventSet->rollback( object, rollbackTime );

  // if we are using an infequent state saving policy then we must
  // regenerate the current state since an old state will have been
  // restored.
  if(myStateManager->getStatePeriod() != 0){
    // Note: If an infrequent state saving scheme has been utilized,
    // then the currentPos pointer in the input queue will have been
    // repositioned to the first event that needs to be re-executed
    // during the coastforward phase by the earlier call to insert
    // or handleAntiMessage. If a frequent state saving scheme is
    // utilized, then the currentPos pointer will point to the next
    // event to execute (which will be the straggler event).
    coastForward( restoredTime, rollbackTime, object );
  }
}

void
TimeWarpSimulationManager::coastForward(const VTime &coastForwardFromTime,
                                        const VTime &moveUpToTime,
                                        SimulationObject *object){
  ASSERT( moveUpToTime >= coastForwardFromTime );
  clutils::debug << "(" << getSimulationManagerID() << ") " << object->getName() << 
    " coasting forward from " << coastForwardFromTime << " to " << 
    moveUpToTime << endl;

  // first turn off message transmission
  suppressMessage = COASTFORWARD;

  // Needed when the application calls 
  // TimeWarpSimulationManager::getEvent()
  setCoastForwardTime( moveUpToTime );

  // go to the first event to coastforward from and call the object's
  // execute process until the rollbackToTime is reached.
  const Event *findEvent = myEventSet->peekEvent(object, moveUpToTime);
  while(findEvent != NULL && findEvent->getReceiveTime() < moveUpToTime){
    setSimulationTime( findEvent->getReceiveTime() );
    SimulationObject *object = getObjectHandle( findEvent->getReceiver() );
    clutils::debug << "(" << getSimulationManagerID() << ") - coasting forward, skipping " <<
      "event at time << " << findEvent->getReceiveTime() << endl;
    object->executeProcess();
    findEvent = myEventSet->peekEvent(object, moveUpToTime);
  }

  // finally turn on message transmission
  suppressMessage = NO;
}

void
TimeWarpSimulationManager::receiveKernelMessage(KernelMessage *msg){
  ASSERT(msg != NULL);
   
  if( dynamic_cast<InitializationMessage *>(msg) != 0 ){
    InitializationMessage *initMsg = dynamic_cast<InitializationMessage *>(msg);
    registerSimulationObjectProxies( &initMsg->getObjectNames(),
				     initMsg->getReceiver(),
				     initMsg->getSender());
  }
  else if( dynamic_cast<StartMessage *>(msg) != 0 ){
    clutils::debug << "SimulationManager(" << mySimulationManagerID
		   << "): Starting Simulation" << endl;
  }
  else if( dynamic_cast<EventMessage *>(msg) != 0 ){
    EventMessage *eventMsg = dynamic_cast<EventMessage *>(msg);
    const Event *event = eventMsg->getEvent();

    // have to take care of some gvt specific actions here
    myGVTManager->updateEventRecord( eventMsg->getGVTInfo(),
				     mySimulationManagerID );
    handleEvent( event ); //, POSITIVE );
  }
  else if( dynamic_cast<NegativeEventMessage *>(msg) ){
    NegativeEventMessage *negEventMsg = dynamic_cast<NegativeEventMessage *>(msg);
    const vector<EventId> events = negEventMsg->getEvents();
    handleNegativeEvents( events );
  }
  else if( msg->getDataType() == "AbortSimulationMessage" ){
    cerr << "TimeWarpSimulationManager is going to abort simulation"
	 << endl;
  }
  else {
    cerr << "TimeWarpSimulationManager::receiveKernelMessage() received"
	 << " unknown (" << msg->getDataType() << ") message type" << endl;
    cerr << "Aborting simulation ..." << endl;
    abort();
  }
  // we are done with this kernel message
  delete msg;
}

bool
TimeWarpSimulationManager::checkIdleStatus(){
  ASSERT(mySchedulingManager != NULL);
  bool retval = !mySchedulingManager->peekNextEvent();
  if( retval == true ){
    // this means that all objects on this simulation manager are idle
    // so lets set simulation time to PINIFINITY
    setSimulationTime( myApplication->getPositiveInfinity() );
  }

  return retval;
}

void
TimeWarpSimulationManager::garbageCollect(const VTime& garbageCollectTime){
  if( localArrayOfSimObjPtrs != 0 ){
    vector<SimulationObject *> *objects = localArrayOfSimObjPtrs->getElementVector();
    for( unsigned int i = 0; i < objects->size(); i++ ){
      myStateManager->garbageCollect(garbageCollectTime, (*objects)[i]);
      myOutputManager->garbageCollect(garbageCollectTime, (*objects)[i]);
      myEventSet->garbageCollect( (*objects)[i], garbageCollectTime );
      // call garbage collect on the file queues
      if( garbageCollectTime != getPositiveInfinity() ){
	const OBJECT_ID *objID = (*objects)[i]->getObjectID();
	if(!mapOfInFileQueues[objID].empty()){
	  vector<TimeWarpSimulationStream*>::iterator iter =
	    mapOfInFileQueues[objID].begin();
	  vector<TimeWarpSimulationStream*>::iterator iter_end =
	    mapOfInFileQueues[objID].end();
	  while(iter != iter_end){
	    (*iter)->garbageCollect(garbageCollectTime);
	    ++iter;
	  }
	}
	
	if(!mapOfOutFileQueues[objID].empty()){
	  vector<TimeWarpSimulationStream*>::iterator iter =
	    mapOfOutFileQueues[objID].begin();
	  vector<TimeWarpSimulationStream*>::iterator iter_end =
	    mapOfOutFileQueues[objID].end();
	  while(iter != iter_end){
	    (*iter)->garbageCollect(garbageCollectTime);
	    ++iter;
	  }
	}
      }
    }
    delete objects;
  }
}

void
TimeWarpSimulationManager::initialize(){
  cerr << "SimulationManager(" << mySimulationManagerID
       << "): Initializing Objects" << endl;

  vector<SimulationObject *> *objects = localArrayOfSimObjPtrs->getElementVector();
  for( unsigned int i = 0; i < objects->size(); i++ ){
    // call initialize on the object
    (*objects)[i]->initialize();
    // save the initial state in the state queue
    ASSERT(myStateManager != NULL);
    myStateManager->saveState( getSimulationTime(), (*objects)[i] );
  }
  delete objects;
  
  if( numberOfSimulationManagers > 1 ){
    if(mySimulationManagerID == 0){
      // send out simulation start messages
      myCommunicationManager->sendStartMessage(mySimulationManagerID);
    }
    else {
      // else wait for the start message
      myCommunicationManager->waitForStart();
    }
  }
}

void
TimeWarpSimulationManager::finalize(){
  const VTime& currentTime = getSimulationTime();
  vector<SimulationObject *> *objects = localArrayOfSimObjPtrs->getElementVector();
  for( unsigned int i = 0; i < objects->size(); i++ ){
    SimulationObject *object = (*objects)[i];
    // call finalize on the object
    object->finalize();
  }
  delete objects;

  myCommunicationManager->finalize();

  //  garbageCollect(currentTime);
}

SimulationStream*
TimeWarpSimulationManager::getIFStream(const string &fileName,
                                       SimulationObject *object){
  TimeWarpSimulationStream *simStream =
    new TimeWarpSimulationStream(fileName, ios::in, this);
  // store for later reference
  const OBJECT_ID* objID = object->getObjectID();
  mapOfInFileQueues[objID].push_back(simStream);
  return simStream;
}

SimulationStream*
TimeWarpSimulationManager::getOFStream(const string &fileName,
                                       SimulationObject *object,
                                       ios::openmode mode){
  TimeWarpSimulationStream *simStream =
    new TimeWarpSimulationStream(fileName, mode, this);
  // store for later reference
  const OBJECT_ID* objID = object->getObjectID();
  mapOfOutFileQueues[objID].push_back(simStream);
  return simStream;
}

SimulationStream*
TimeWarpSimulationManager::getIOFStream(const string &fileName,
                                        SimulationObject *object){
  // this function is currently not implemented fully as I dont
  // see the utility of this function -- ramanan [05/12/00]
  TimeWarpSimulationStream *simStream =
    new TimeWarpSimulationStream(fileName, ios::in|ios::app, this);
  return simStream;
}
      
// print out the name to simulation object ptr map
void
TimeWarpSimulationManager::displayGlobalObjectMap(ostream& out){
  if(!globalArrayOfSimObjPtrs.empty()){    
    vector<string> *keys = localArrayOfSimObjPtrs->getKeyVector();
    vector<SimulationObject *> *objects = localArrayOfSimObjPtrs->getElementVector();
    for( unsigned int i = 0; i < objects->size(); i++ ){
      out << (*keys)[i] << ": " << (*objects)[i]->getObjectID();
    }
    delete objects;
    delete keys;
  }
  else {
    out << "Object Names to Object Pointers Map is empty" << endl;
  }
}

void
TimeWarpSimulationManager::configure( SimulationConfiguration &configuration ){
  const CommunicationManagerFactory *myCommFactory = 
    CommunicationManagerFactory::instance();
  
  myCommunicationManager = 
    dynamic_cast<CommunicationManager *>( myCommFactory->allocate( configuration,
								   this ));
  ASSERT( myCommunicationManager != 0 );
  
  myCommunicationManager->configure( configuration );
  mySimulationManagerID = myCommunicationManager->getId();

  ASSERT( myApplication != 0 );
  setNumberOfObjects( myApplication->getNumberOfSimulationObjects(mySimulationManagerID) );

  // register simulation manager's message types with the comm. manager
  registerWithCommunicationManager();  

  /*
    Currently the only termination manager available is the
    TokenPassingTerminationManager.
  */
  myTerminationManager = new TokenPassingTerminationManager(this);

  // the createMapOfObjects function does the following:
  // a. registers the correct set of objects for this simulation
  //    manager
  // b. resets the value of numberOfObjects to the number of objects
  //    actually resident on this simulation manager.
  localArrayOfSimObjPtrs = createMapOfObjects();
  
  // configure the event set manager
  const TimeWarpEventSetFactory *myEventSetFactory = TimeWarpEventSetFactory::instance();
  Configurable *eventSet = myEventSetFactory->allocate( configuration, this );
  myEventSet = dynamic_cast<TimeWarpEventSet *>( eventSet );
  ASSERT( myEventSet != 0 );
  myEventSet->configure( configuration );

  // lets now set up and configure the state manager
  const StateManagerFactory *myStateFactory = StateManagerFactory::instance();
  myStateManager = dynamic_cast<StateManager *>(myStateFactory->allocate( configuration, this ));
  ASSERT( myStateManager != 0 );
  myStateManager->configure( configuration );

  // lets now set up and configure the scheduler
  const SchedulingManagerFactory *mySchedulingFactory = SchedulingManagerFactory::instance();
  mySchedulingManager = 
    dynamic_cast<SchedulingManager *>(mySchedulingFactory->allocate( configuration, this ));
  ASSERT( mySchedulingManager != 0 );
  mySchedulingManager->configure( configuration );

  // lets now set up and configure the output manager
  const OutputManagerFactory *myOutputManagerFactory = OutputManagerFactory::instance();
  myOutputManager = 
    dynamic_cast<OutputManager *>(myOutputManagerFactory->allocate( configuration, this ));
  ASSERT( myOutputManager != 0 );
  myOutputManager->configure( configuration );
  
  // lets now set up and configure the gvt manager
  const GVTManagerFactory *myGVTFactory = GVTManagerFactory::instance();
  myGVTManager = dynamic_cast<GVTManager *>(myGVTFactory->allocate( configuration, this ));
  ASSERT( myGVTFactory != 0 );
  myGVTManager->configure( configuration );

  registerSimulationObjects();

  ASSERT(myCommunicationManager != NULL);
  // call the communication manager to send out initialization
  // messages and wait for all initialization messages to arrive
  // note: we dont need an initialization message from ourself; and
  // hence the n - 1.
  myCommunicationManager->waitForInitialization(numberOfSimulationManagers-1);

}

bool 
TimeWarpSimulationManager::contains( const string &object ) const {
  SimulationObject *simObject = getObjectHandle( object );
  if( dynamic_cast<SimulationObjectProxy *>( simObject ) == 0 ){
    // It's not a proxy, we hold this object...
    return true;
  }
  else{
    // It's a proxy; this object is not ours.
    return false;
  }
}

const VTime &
TimeWarpSimulationManager::getPositiveInfinity() const {
  return myApplication->getPositiveInfinity();
}

const VTime &
TimeWarpSimulationManager::getZero() const {
  return myApplication->getZero();
}

void
TimeWarpSimulationManager::shutdown( const string &errorMessage ){
  // We SHOULD send out a message and shut things down nicely - for the moment we're
  // essentially going to crash.
  cerr << errorMessage << endl;
  exit( -1 );
}
