#ifndef MPI_PHYSICAL_COMMUNICATION_LAYER_H
#define MPI_PHYSICAL_COMMUNICATION_LAYER_H

// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

#include "warped/warped.h"
#include "eclmplCommonInclude.h"
#include "PhysicalCommunicationLayer.h"
#include "MPIMessage.h"

/** The MPIPhysicalCommunicationLayer class.

*/
class MPIPhysicalCommunicationLayer : public PhysicalCommunicationLayer {
public:
  /**@name Public Class Methods of MPIPhysicalCommunicationLayer. */
  //@{

  /// Default Constructor.
  MPIPhysicalCommunicationLayer();

  /// Destructor.
  ~MPIPhysicalCommunicationLayer();
  
  /** Init physical layer.

      MPI_init(argc, argv) is called from this method, as need to be
      done before any other MPI call can be done. Thus, argc and argv
      needs to be passed in to this method.
      
      @param configuration The simulation configuration being used.      
  */
  virtual void physicalInit( SimulationConfiguration &configuration );

  /** Get the Id of the simulation manager.

      @return Id of the simulation manager.
  */
  int physicalGetId() const;

  /** Send buffer.

      @param buffer Char buffer to send.
      @param size Size of the buffer to send.
  */
  virtual void physicalSend( const SerializedInstance *toSend, unsigned int dest );

  /** Check the MPI-probe to see if there are message to retrieve.

      @return The retrieved message (NULL if no message).
  */
  virtual SerializedInstance *physicalProbeRecv();

  /** Retrieve message into a buffer.

      @param buffer Buffer to which we save the message.
      @param size Size of the buffer.
      @param sizeStatus Was the size of retr. msg > size?
      @return True/False, Was any message retrieved?
  */
  virtual bool physicalProbeRecvBuffer(char *buffer, int size, bool& sizeStatus);

  /// Clean up and call MPI_Finalize.
  void physicalFinalize();
  
  /** Return how many processes are involved in the communicator.

      @return The number of processes involved in the communicator.
  */
  int physicalGetSize() const;
  
  //@} // End of Public Class Methods of MPIPhysicalCommunicationLayer.

protected:
  /**@name Protected Class Attributes of MPIPhysicalCommunicationLayer. */
  //@{

  /// Id of the simulation manager.
  int mySimulationManagerID;

  //@} // End of Protected Class Attributes of MPIPhysicalCommunicationLayer.

private:
  /**
     Checks the status of pending sends.  Clears completed sends out of the
     pendingSend vector, and deallocates the resources associated with
     them.
  */
  void checkPendingSends();

  /**
     Takes an argument vector and turns it back into C-style arguments required
     by MPI.
  */
  void getCStyleArguments( int &argc, 
			   char ** &argv,
			   const vector<string> &arguments );

  /**
     Writes out a procgroup file based on our Warped configuration.
  */
  void writeProcGroupFile( SimulationConfiguration &configuration );

  /**
     Does the dirty work of actually starting MPI.
  */
  void startMPI( const vector<string> &arguments );

  
  vector<MPIMessage> pendingSends;
  vector<MPIMessage> pendingReceives;
};

#endif
