/*\ Equalizer FIR filter by Willem Monsuwe (willem@stack.nl) \*/

#include "cdread.h"
#include <stdio.h>
#include <string.h>
#include <math.h>

typedef double real;

/*\ Number of filter coefficients.  Directly affects computation time \*/
#define ORDER 32

/*\ Multiplication factor from dB to whatever \*/
#define DB_MULT_FACT 0.04

/*\ Frequency bands \*/
static struct eq_band {
	real frq_lo, frq_hi;	/*\ Frequency band (in Hz) \*/
	real tap[ORDER+1];	/*\ Tap coefficients for this band \*/
} *bands = 0;

static int num_bands = 0;
static int cd_eq_on = 0;

typedef struct {
	short l, r;
} sample;

static real fc[ORDER+1];	/*\ Filter coefficients \*/
static sample prev[ORDER];	/*\ Previous samples \*/
static int prev_idx;

/*\ Filter a block of 16-bit signed, native endian audio data \*/
void
cd_filter(void *buf, int len)
{
	int i, j, k;
	sample *data = buf;
	real ol, or;

	if (!cd_eq_on) return;

	for (i = 0; i < len; i++) {
		ol = fc[0] * (real)data[i].l;
		or = fc[0] * (real)data[i].r;
		j = 1;
		while (1) {
			ol += (real)prev[prev_idx].l * fc[j];
			or += (real)prev[prev_idx].r * fc[j];
			if (++j > ORDER) break;
			prev_idx = (prev_idx + 1) % ORDER;
		}
		prev[prev_idx] = data[i];
		if (ol > 32767) ol = 32767;
		else if (ol < -32768) ol = -32768;
		if (or > 32767) or = 32767;
		else if (or < -32768) or = -32768;
		data[i].l = (short)ol;
		data[i].r = (short)or;
	}
}

#define PI 3.1415926535897932384626433832795029L

/*\ Calculate the tap coefficients for the filter bands.
|*|  Assumes ORDER is even.
\*/
static void
calc_tap(struct eq_band *band)
{
	real bnd, ctr;
	int i;
	real *tap;
	real b[ORDER / 2 + 1];
	real rl, im, gn;

	tap = band->tap;
	bnd = (band->frq_hi - band->frq_lo) / 2;
	ctr = (band->frq_hi + band->frq_lo) / 2;

	tap[ORDER/2] = 2 * bnd;
	for (i = 1; i <= (ORDER / 2); i++) {
		tap[ORDER/2 - i] = 2 * sin(PI * i * bnd)
				     * cos(PI * i * ctr) / (PI * i);
	}

	for (i = 0; i < ORDER/2; i++) {
		tap[i] *= (0.54 - 0.46 * cos((2 * PI * i) / ORDER));
		tap[ORDER - i] = tap[i];
	}

	rl = 0;
	im = 0;
	for (i = 0; i <= ORDER; i++) {
		rl += tap[i] * cos((ORDER - i) * PI * ctr);
		im += tap[i] * sin((ORDER - i) * PI * ctr);
	}

	gn = 1.0 / sqrt(rl * rl + im * im);
	for (i = 0; i <= ORDER; i++)
		tap[i] *= gn;
}

/*\ Recalculate the filter coefficients \*/
void
cd_set_eq(int on, float total_gain, float *gain)
{
	int i, j;
	real t;

	for (i = 0; i <= ORDER; i++) {
		t = 0.0;
		for (j = 0; j < num_bands; j++)
			t += bands[j].tap[i] * gain[j] * DB_MULT_FACT;
		fc[i] = t;
	}
	fc[0] += 1.0 + (total_gain * DB_MULT_FACT);
	cd_eq_on = on;
}

static void
cd_set_eq_bands(int eq_bands[][2], int num, int srate)
{
	int i;

	num_bands = num;
	RENEW(bands, num);

	for (i = 0; i < num; i++) {
		bands[i].frq_lo = (real)eq_bands[i][0] / srate;
		bands[i].frq_hi = (real)eq_bands[i][1] / srate;
		calc_tap(&bands[i]);
	}
}

void
cd_init_eq(void)
{
	static int eq_bands[][2] = {
		{ 0, 120 },
		{ 60, 280 },
		{ 170, 450 },
		{ 310, 890 },
		{ 600, 1400 },
		{ 1000, 5000 },
		{ 3000, 9000 },
		{ 6000, 14000 },
		{ 12000, 16000 },
		{ 14000, 20000 } };
	cd_set_eq_bands(eq_bands, NELEM(eq_bands), 44100);
	cd_eq_on = 0;
}
