
#include "WhiteCap.h"

#include "WhiteCap_Proj.h"

#include "EgOSUtils.h"

#define WC_NO_MORPH		0
#define WC_NORMAL_MORPH		-1
#define WC_QUICK_MORPH  	1.4

static PluginInfo info = {

	PLUGIN_ID,
	PLUGIN_AUTHOR,
	_PLUGIN_NAME,
	_PLUGIN_VERS,
	LONG_VERS_STR,
	MAIN_CONFIGS_FOLDER,
	PREFS_NAME,
	PREFS_COMPAT_VERSION
};

WhiteCap::WhiteCap( long inHostVers, void* inRefCon ) :
	PluginGlue( inHostVers, inRefCon, info ) {

	// Init runtime vars...
	mSlideShowOn		= true;
	mNewConfigNotify	= false;
	mNumWorlds		= 0;
	mCurConfigNum		= -1;
	mNextShapeChange	= 0x7FFFFFFF;

	Init();

	CacheFolder( "WhiteCap", mConfigs, &mConfigPlayList, 3 * mCacheDepth );

	if ( !mConfigPlayList.Count() ) {

		Println( "No WhiteCap files found.  The \"WhiteCap\" folder must" );
		Println( "   be in the same folder as SingIt." );
		Println( "" );
	}

	// Since we're starting, do the first slide quickly
	LoadConfig( mConfigPlayList.Fetch( 1 ), WC_NO_MORPH );
	mNextShapeChange = mT + 6.0;
}

WhiteCap::~WhiteCap() {

	// Delete all the existing worlds
	for ( int i = 0; i < mNumWorlds; i++ )
		delete mWorld[ i ];
}

void WhiteCap::SetPrefsFactory() {

	static WhiteCap::FactoryPrefPresets presets = {

		PREFS_COMPAT_VERSION,

		FACTORY_FFT_BIN_START,
		FACTORY_FFT_STEPS_PER_BIN,
		FACTORY_FFT_SMOOTH,
		FACTORY_FFT_NUM_BINS,
		FACTORY_FFT_FADEAWAY,
		FACTORY_FFT_TRANSFORM,

		FACTORY_DEPTH,
		{
			FACTORY_MAX_X,
			FACTORY_MAX_Y
		},
		FACTORY_SAMPLE_SMOOTH,
		FACTORY_SAMPLE_NUMBINS,
		FACTORY_FPS,
		FACTORY_FPS
	};

	PluginGlue::SetPrefsFactory(presets);

	mKeyMap.Assign( "THRU~OGF,.\\[]{}L()<>M`" );
}

#define __assertPref( field,arg )								if ( ! inPrefs.PrefExists( field ) )		\
	inPrefs.SetPref( field, arg )

#define __compilePref( field, expr ) \
do { \
	inPrefs.GetPref( field, str );	\
	expr.Compile( str, mDict ); \
} while (0)

void WhiteCap::LoadPrefs( Prefs& inPrefs ) {

	PluginGlue::LoadPrefs( inPrefs );

	mTrackTextDur = mPrefs.GetPref( MCC4_TO_INT("TDur") );

	UtilStr str;
	__compilePref( MCC4_TO_INT("Slde"), mMorphDurationExpr );
	__compilePref( MCC4_TO_INT("MDur"), mSlideShowIntervalExpr );
	__compilePref( MCC4_TO_INT("MTrs"), mMorphTransExpr );

	if ( mFullscreenDepth < 16 )
		mFullscreenDepth = 16;
}

void WhiteCap::SavePrefs( Prefs& inPrefs ) {

	PluginGlue::SavePrefs( inPrefs );

	__assertPref( MCC4_TO_INT("TDur"), 7 );
	__assertPref( MCC4_TO_INT("Slde"), "15 + rnd( 10 )" );
	__assertPref( MCC4_TO_INT("MDur"), "5 + rnd( 17 )" );
	__assertPref( MCC4_TO_INT("MTrs"), "i^1.6" );
}


#define SUBMENU_SIZE 17

void WhiteCap::CommandClick() {

	long sel;
	Point inPt;
	UtilStr name;

	EgOSUtils::GetMouse( inPt );

#if EG_MAC
	MenuHandle	curSubMenu = ::NewMenu( 55, "\pBlah" ), menu;
	long groupNum, num = mConfigs.Count();


	// If we found a world, show a popup menu to select from the configs
	menu = ::NewMenu( 55, "\p " );

	// Add the start slideshow item
	::AppendMenu( menu, "\p " );
	::SetMenuItemText( menu, 1, "\p(Start Slide Show)" );
	if ( mSlideShowOn )
#if TARGET_API_MAC_CARBON
		::DisableMenuItem( menu, 1 );
#else
		::DisableItem( menu, 1 );
#endif

	for ( i = 1; i <= num; i++ ) {

		groupNum = 1 + i / SUBMENU_SIZE;

		// Split up the config list into groups of submenus
		if ( i % SUBMENU_SIZE == 1 ) {

			// Make the submenu
			curSubMenu = ::NewMenu( 55 + groupNum, "\pBlah" );
			::InsertMenu( curSubMenu, hierMenu );

			// Contruct and rename the title string
			name.Assign( "Group " );
			name.Append( groupNum );
			::AppendMenu( menu, "\p " );
			::SetMenuItemText( menu, 1 + groupNum, name.getPasStr() );

			// Tell MacOS to make it a heir menu
			::SetItemCmd( menu, 1 + groupNum, hMenuCmd );
			::SetItemMark( menu, 1 + groupNum, 55 + groupNum );
		}

		// Get the name for the config number and set the menu item to that name
		name.Assign( mConfigs.Fetch( i ) );
		::AppendMenu( curSubMenu, "\p " );
		sel = 1 + ( i - 1 ) % SUBMENU_SIZE;
		::SetMenuItemText( curSubMenu, sel, name.getPasStr() );

		if ( mCurConfigNum == i )
			::CheckMenuItem( curSubMenu, sel, true );
	}

	::InsertMenu( menu, -1 );
	sel = ::PopUpMenuSelect( menu, inPt.v - 6, inPt.h, 1 );

	// Figure out what config number was selected
	i = sel >> 16;
	if ( i == 55 && ( ( sel & 0xFFFF ) == 1 ) )
		sel = 1;
	else if ( i )
		sel = ( i - 56 ) * SUBMENU_SIZE + (sel & 0xFFFF) + 1;
	else
		sel = 0;

	//  Cleanup
	i = 0;
	do {
		menu = ::GetMenuHandle( 55 + i );
		if ( menu ) {
			::DeleteMenu( 55 + i );
			::DisposeMenu( menu );
		}
		i++;
	} while ( menu );
#endif


#if EG_WIN
	long flags, numSubMenus = 0;
	long num = mConfigs.Count();

	HMENU curSubMenu = nil, myMenu = ::CreatePopupMenu();

	// Add item to start slideshow
	flags = MF_STRING;
	if ( mSlideShowOn )
		flags |= MF_GRAYED;
	else
		flags |= MF_ENABLED;
	::AppendMenu( myMenu, flags, 1, "(Start Slide Show)" );

	for ( i = 1; i <= num; i++ ) {

		// Split up the config list into groups of submenus
		if ( i % SUBMENU_SIZE == 1 ) {
			curSubMenu = ::CreatePopupMenu();

			// Contruct the submenu title string
			name.Assign( "Group " );
			name.Append( 1 + i / SUBMENU_SIZE );
			::InsertMenu( myMenu, 0xFFFFFFFF, MF_POPUP | MF_STRING | MF_BYPOSITION, (long) curSubMenu, name.getCStr() );
		}

		// Set flags for the item
		flags = MF_ENABLED | MF_STRING;
		if ( mCurConfigNum == i )
			flags |= MF_CHECKED;

		// Get the name for the config number and set the menu item to that name
		name.Assign( mConfigs.Fetch( i ) );
		::AppendMenu( curSubMenu, flags, i + 1, name.getCStr() );
	}


	// Track the mouse
	sel = ::TrackPopupMenu( myMenu, TPM_NONOTIFY | TPM_RETURNCMD, inPt.h, inPt.v, 0, GetWindow(), nil );
	::DestroyMenu( myMenu );
#endif

	if ( sel > 0 ) {
		if ( sel == 1 )
			EnableSlideShow();
		else if ( sel >= 1 ) {
			LoadConfig( sel - 1, WC_NO_MORPH );
			if ( mSlideShowOn ) {
				Println( "Slideshow OFF" );
				mSlideShowOn = false;
			}
		}
	}
}

void WhiteCap::EnableSlideShow() {

	mSlideShowOn = true;
	mNextShapeChange = mT;
	mConfigPlayList.Randomize();
	LoadConfig( mConfigPlayList.Fetch( 1 ), WC_NORMAL_MORPH );
}

void WhiteCap::MakeStateCmdLine( UtilStr& outCmdList ) {

	// Turn the all the running config into a command line
	outCmdList.Assign( mCurConfigName );
}

bool WhiteCap::ExecuteCmdCode( long inCode, bool inShiftKey ) {

	bool handled = true;
	long n;

	switch ( inCode ) {
	case cStartSlideshow:
		if ( ! mSlideShowOn ) {
			EnableSlideShow();
			Println( "Slideshow ON" );
		}
		break;

	case cStopSlideshow:
		if ( mSlideShowOn ) {
			mSlideShowOn = false;
			Println( "Slideshow OFF" );
		}
		break;

	case cReloadConfig: {
		FileObj item = mConfigs.FetchID( mCurConfigNum );
		mMainFolder.PurgeCached( item, 1 );
		LoadConfig( mCurConfigNum, WC_NO_MORPH );
		Println( "Config reloaded." );  }
		break;

	case cToggleConfigName:
		mNewConfigNotify = ! mNewConfigNotify;
		if ( mNewConfigNotify )
			Println( "Show names ON" );
		else
			Println( "Show names OFF" );
		break;

	case cCurrentConfigName:
		Print( "Current config: " );
		Println( &mCurConfigName );
		break;

	case cPrevConfig:
	case cNextConfig:
		n = mConfigPlayList.FindIndexOf( mCurConfigNum ) + 1;
		if ( inCode == cPrevConfig )
			n -= 2;
		LoadConfig( mConfigPlayList.FetchWrapped( n ), WC_QUICK_MORPH );

		if ( mSlideShowOn ) {
			mSlideShowOn = false;
			Println( "Slideshow OFF" );
		}
		break;

	default:
		handled = PluginGlue::ExecuteCmdCode( inCode, inShiftKey );
	}

	return handled;
}


#define __setChar( n, ID )		s.setChar( n, mKeyMap.getChar( ID ) )

void WhiteCap::ShowHelp() {

	UtilStr s;

	Println(  "RET  Fullscreen ON/OFF" );
	s.Assign( "X    Switch fullscreen mode/resolution" );		__setChar( 1, cNextFullscreenMode );	Println( &s );
	s.Assign( "X    Take snapshot" );				__setChar( 1, cTakeSnapshot );		Println( &s );
	s.Assign( "X    Track title" );					__setChar( 1, cDispTrackTitle );	Println( &s );
	s.Assign( "X    Frame rate" );					__setChar( 1, cFrameRate );		Println( &s );
	s.Assign( "X    Config names ON/OFF" );				__setChar( 1, cToggleConfigName );	Println( &s );
	s.Assign( "X    Show name of current config" );			__setChar( 1, cCurrentConfigName );	Println( &s );
	s.Assign( "X    Reload current config" );			__setChar( 1, cReloadConfig );		Println( &s );
	s.Assign( "X    Console text ON/OFF" );				__setChar( 1, cToggleConsole );		Println( &s );
	s.Assign( "X X  Slideshow ON/OFF" );				__setChar( 1, cStartSlideshow );	__setChar( 3, cStopSlideshow );		Println( &s );
	s.Assign( "X X  -/+ sample scale" );				__setChar( 1, cDecSampScale );		__setChar( 3, cIncSampScale );		Println( &s );
	s.Assign( "X X  -/+ num bins per sample" );			__setChar( 1, cDecNumFFTBins );		__setChar( 3, cIncNumFFTBins );		Println( &s );
	s.Assign( "X X  -/+ bin range" );				__setChar( 1, cDecFFT_BinRange );	__setChar( 3, cIncFFT_BinRange );	Println( &s );
	s.Assign( "X X  -/+ sample smoothing" );			__setChar( 1, cDecFFT_Smooth );		__setChar( 3, cIncFFT_Smooth );		Println( &s );
	s.Assign( "X X  Prev/Next config" );				__setChar( 1, cPrevConfig );		__setChar( 3, cNextConfig );		Println( &s );
	Println( "" );
	Println( "CTRL+X        Run 'X Key' script" );
	Println( "CTRL+SHIFT+X  Store configs as 'X Key' script" );
	Println( "" );
	s.Assign(  "SHIFT+X       Show HTML Help" );			__setChar( 7, cShowHelp );				Println( &s );
	Println( "" );

#if EG_MAC
	Println( "Command-Click to choose a config." );
#else
	Println( "Right-click to choose a config." );
#endif
}

void WhiteCap::DoFrame() {

	Rect	dirtyRect, r;
	long	i;

	// Load a random waveshape every so often, and randomize things
	if ( mT > mNextShapeChange && mSlideShowOn )
		LoadNextConfig();

	// We're about to draw, so record some sound
	RecordSample( true, false );

	// Prepare to tell DrawFrame() what's dirty
	dirtyRect.top = dirtyRect.left = 32000;
	dirtyRect.bottom = dirtyRect.right = -32000;

	// Loop thru each pane in the wind (ie, each WC world) and draw to the offscreen port
	for ( i = 0; i < mNumWorlds; i++ ) {

		// Tell the world to draw/update itself to the offscreen draw port and update dirty rect
		mWorld[ i ] -> RecordSample( mT, mFFT_Fcn, mFFT_Bass );
		mWorld[ i ] -> Render( mPort, r );
		::UnionRect( &dirtyRect, &r, &dirtyRect );
	}

	// Draw any track text that's active...
	ManageTrackText();

	// Draw the console (if its being drawn)
	if ( mConsoleLines.Count() > 0 ) {

		mPort.SetTextColor( *mWorld[ 0 ] -> GetForeColor() );

		// Draw the console to the offscreen port and know the rect needs updating
		DrawConsole( &mPort );
		if ( IsValidRect( mConsoleDirtyRgn ) )
			::UnionRect( &dirtyRect, &mConsoleDirtyRgn, &dirtyRect );

		mVideoOutput -> AcceptFrame( mPort, &dirtyRect, mT_MS ); }
	else
		mVideoOutput -> AcceptFrame( mPort, &dirtyRect, mT_MS );
}

void WhiteCap::ManageTrackText() {

	RGBColor textColor;
	float i, t;

	// Is there any track text active?
	if ( mTrackText.length() > 0 ) {

		// Have the text fade in, then fade out...
		t = 1.0 - ( mTrackTextEndT - mT ) / ( (float) mTrackTextDur );
		i = 1.6 * sin( PI * t );

		if ( mT <= mTrackTextEndT ) {

			// Calc and set the port text color then draw it
			mWorld[ 0 ] -> CalcForeground( i, textColor );
			mPort.SetTextColor( textColor );

			mPort.SetClipRect();
			mPort.DrawText( mTrackTextRect.left, mTrackTextRect.top, mTrackText );

			// We need to redraw that area in the next frame
			RefreshRect( &mTrackTextRect ); }

		// Signal no more track text if it's expired
		else
			mTrackText.Wipe();
	}

}

void WhiteCap::PortResized( long inX, long inY ) {

	long depth;

	if ( mVideoOutput -> GetDepth() == 16 )
		depth = 16;
	else
		depth = 32;
	mPort.Init( inX, inY, depth );

	// Use the font and size in the prefs...
	mPort.SetDefaultFont( mTextFont, mTextSize );

	for ( int i = 0; i < mNumWorlds; i++ ) {
		mWorld[ i ] -> ExpireSamples();
	}

	ResizeWorlds();
}


#define ___maxSide( r ) (_MAX( r.right - r.left, r.bottom - r.top ))

void WhiteCap::ResizeWorlds() {

	Rect r;
	long i, j, bestLen, cutLen, best;

	if ( mNumWorlds == 0 )
		return;

	// Resize the panes in this window:  We made our PixPort the size of the rect we're drawing in, so all
	// WhiteCap cords within worlds are relative to the corner of where the PixPort origin is.  That is,
	// the port cords of point (3,4) in a WhiteCap world are (mPaneRect.left+3, mPaneRect.top+4).
	::SetRect( &r, 0, 0, mPort.GetX(), mPort.GetY() );

	mWorld[ 0 ] -> SetPaneRect( r );
	for ( i = 1; i < mNumWorlds; i++ ) {

		// Find the best world to cut in half
		for ( bestLen = -1, j = 0; j < i; j++ ) {
			cutLen = ___maxSide( (*mWorld[ j ] -> PaneRect()) );
			if ( cutLen >= bestLen ) {
				bestLen = cutLen;
				best = j;
			}
		}

		// Change the pane rect of both the unplaced world and the world that's about to get cut in half
		r = *(mWorld[ best ] -> PaneRect());
		if ( r.right - r.left == bestLen ) {
			r.right -= bestLen / 2;
			mWorld[ best ] -> SetPaneRect( r );
			OffsetRect( &r,  bestLen / 2,  0 );
			mWorld[ i ] -> SetPaneRect( r ); }
		else {
			r.bottom -= bestLen / 2;
			mWorld[ best ] -> SetPaneRect( r );
			OffsetRect( &r,  0,  bestLen / 2 );
			mWorld[ i ] -> SetPaneRect( r );
		}
	}
}

void WhiteCap::RefreshRect( Rect* inRect ) {

	for ( int i = 0; i < mNumWorlds; i++ )
		mWorld[ i ] -> Refresh( inRect );
}

void WhiteCap::LoadNextConfig() {

	int i;

	// Load the next config in the (randomized) config list...
	i = mConfigPlayList.FindIndexOf( mCurConfigNum );

	// Make a new play list if we've reached the end of the list...
	if ( i >= mConfigPlayList.Count() ) {
		mConfigPlayList.Randomize();
		i = 0;
	}
	LoadConfig( mConfigPlayList.Fetch( i + 1 ), WC_NORMAL_MORPH );
}

void WhiteCap::LoadConfig( int inConfigNum, float inMorphDur ) {

	int i;
	long flags, oldNumWorlds = mNumWorlds;
	float transitionTime;
	FileObj itemID;
	XLongList catalog;
 	const UtilStr *str, *name;

	// Fetch the spec for our config file or folder
	itemID = mConfigs.FetchID( inConfigNum );
	flags = mMainFolder.GetInfo( itemID, MCC4_TO_INT("flag") );
	name = mConfigs.Fetch( inConfigNum );
	mCurConfigName.Assign( name );

	if ( mNewConfigNotify && name ) {
		Print( "Loaded: " );
		Println( name );
	}

	if ( flags && name ) {

	 	// See how long we're gonna let the existing config transform into the oncoming config
	 	if ( inMorphDur == 0 )
	 		transitionTime = 0;
		else if ( inMorphDur < 0 )
			transitionTime = mMorphDurationExpr.Evaluate();
		else
			transitionTime = inMorphDur;

 		// Make a list of the configs we need to load...
		if ( flags & FILE_SYS_FOLDER )
			mMainFolder.Catalog( itemID, catalog );
		else if ( flags & FILE_SYS_DATA )
			catalog.Add( itemID );

		// Load each config...
		mNumWorlds = 0;
		for ( i = 1; i <= catalog.Count(); i++ ) {
			itemID = catalog.Fetch( i );
			str = mMainFolder.Read( itemID );
			if ( mNumWorlds >= oldNumWorlds )
				mWorld[ mNumWorlds ] = new WhiteCapWorld( &mTransition_I, &mMorphTransExpr );
			mWorld[ mNumWorlds ] -> Init( str, mT, mFFT_NumBins, transitionTime );
			mNumWorlds++;
		}
	}
	else {
		if ( oldNumWorlds == 0 )
			mWorld[ 0 ] = new WhiteCapWorld( &mTransition_I, &mMorphTransExpr );

		mWorld[ 0 ] -> Init( nil, mT, mFFT_NumBins, 0 );
		mNumWorlds = 1;
	}

	// Know what to put a check mark next to in the popup menu
	mCurConfigNum = inConfigNum;

	for ( i = mNumWorlds; i < oldNumWorlds; i++ )
		delete mWorld[ i ];

	// Set the time for when we load a random config automatically
	mNextShapeChange = mT + mSlideShowIntervalExpr.Evaluate() + transitionTime;

	// Make sure all WhiteCap's configs live happily inside the allowed draw area
	ResizeWorlds();
	Refresh( NULL );
}

void WhiteCap::StartTrackText() {

	long x, y;

	mTrackText.Wipe();

	if ( mArtist.length() > 0 ) {
		mTrackText.Append( mArtist );
		mTrackText.Append( '\r' );
	}

	if ( mSongTitle.length() > 0 ) {
		mTrackText.Append( mSongTitle );
		mTrackText.Append( '\r' );
	}

	if ( mAlbum.length() > 0 ) {
		mTrackText.Append( mAlbum );
		mTrackText.Append( '\r' );
	}

	if ( mTrackText.getChar( mTrackText.length() ) == '\r' )
		mTrackText.Trunc( 1 );

	if ( mTrackText.length() > 0 ) {
		mPort.SetDefaultFont();
		mPort.TextRect( mTrackText.getCStr(), x, y );

		// Will the track text fit?  If so, put it in the bottom left corner
		if ( y < mPort.GetY() && mTrackTextDur > 0 ) {
			mTrackTextRect.left		= 7;
			mTrackTextRect.top 		= mPort.GetY() - y;
			mTrackTextRect.right 	= mTrackTextRect.left + x;
			mTrackTextRect.bottom 	= mTrackTextRect.top  + y;

			mTrackTextEndT = mT + mTrackTextDur; }

		// If mTrackDesc is empty, that means no track tect is active
		else
			mTrackText.Wipe();
	}
}

bool WhiteCap::ExecuteCmd( UtilStr& inCmd ) {

	long i;

	// see if we're commanded to do a transtion or not
	bool hasAsterisk = false;
	if ( inCmd.getChar( inCmd.length() ) == '*' ) {
		hasAsterisk = true;
		inCmd.Trunc( 1 );
	}

	//  Look for a matching WaveShape...
	i = mConfigs.FindString( &inCmd, false );
	if ( i > 0 ) {
		LoadConfig( i, hasAsterisk ? WC_NORMAL_MORPH : WC_QUICK_MORPH );
		mSlideShowOn = false;
		return true;
	}

	// If still no luck, can we find a script w/ the name?
	return PluginGlue::ExecuteCmd( inCmd );
}

void WhiteCap::TimeIndexHasChanged( long ) {

	// Chuck all the samples we had
	for ( int i = 0; i < mNumWorlds; i++ )
		mWorld[ i ] -> ExpireSamples();

	// Reinit the config to be safe
	LoadConfig( mCurConfigNum, WC_NO_MORPH );
}
