/*
 * gdd.c
 */
/*
 * yank  -  yet another NoteKeeper
 * Copyright (C) 1999, 2000, 2001 Michael Humann <m.hussmann@home.ins.de>
 *
 * This program is free software; you  can redistribute it and/or modify it
 * under the terms of  the GNU General  Public License as published  by the
 * Free Software Foundation;  either version 2 of  the License, or (at your
 * option) any later version.
 *
 * This program  is  distributed in the  hope  that it will  be useful, but
 * WITHOUT  ANY   WARRANTY;  without   even   the  implied    warranty   of
 * MERCHANTABILITY or    FITNESS FOR A PARTICULAR   PURPOSE.   See  the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the  GNU General Public License along
 * with this program; if not, write to the  Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include "gdd.h"
#include <glade/glade-widget-tree.h>

#define FIXFUNC(msg) printf("FIXME :" __FUNCTION__ " %s\n", msg);

/*
 * private data
 */

typedef struct _GddWidgetData GddWidgetData;

struct _GddPrivate
{
    GladeWidgetTree *tree;
    GList           *wlist;                   /* list of GddWidgetData */
    gboolean        auto_disconnect_cmonitor; /* automatic disconnection
                                                 of cmonitor - default
                                                 is off */
    gboolean        signals_are_connected;
    gboolean        signals_are_blocked;
};

struct _GddWidgetData
{
    GladeWidgetInfo *gwi;
    GtkObject       *connected_object;    /* needed to handle GtkRange */
    guint           connection_id;
};

/*
 * list of widget-names we're handling
 */

static const gchar *gdd_typelist[] =
{
    "GtkEntry",            /* editable */
    "GtkText",    
    "GtkToggleButton",     /* toggle-button */
    "GtkRadioButton",
    "GtkCheckButton",
    "GtkHScale",           /* range */ 
    "GtkVScale"
};
static const gint  gdd_types = sizeof (gdd_typelist) /
                               sizeof (gdd_typelist[0]);

/*
 * constants
 */

static const gchar tb_toggled_str[] = "on";
static const gchar tb_utoggled_str[] = "off";

/*
 * signals
 */

enum signals
{
    CHANGED_SIGNAL,
    LAST_SIGNAL
};
static gint gdd_signals[LAST_SIGNAL] = {0};

/*
 * parent class pointer
 */

static GtkObjectClass *parent_class = NULL;

/*
 * private function prototypes
 */

static void       gdd_init(Gdd *gdd);
static void       gdd_class_init(GddClass *class);
static void       gdd_destroy(GtkObject *gdd);
static void       gdd_scan_tree(const Gdd *gdd, const gchar *root);
static GtkWidget* gdd_widget_from_wlist_by_name(const Gdd *gdd,
                                                const gchar *name);
static void       gdd_cb_object_changed(GtkObject *o, gpointer gdd);
static void       gdd_connect_widget_list_element(gpointer lelem, gpointer gdd);

/* ------------------------------------------------------ */
/*
 * return a new object
 */

Gdd*
gdd_new(const gchar *fname,
        const gchar *root,
        const gchar *domain)
{
    Gdd *gdd;

    gdd = gtk_type_new(TYPE_GDD);
    if (gdd_construct(gdd, fname, root, domain) == FALSE)
    {
        gtk_object_unref(GTK_OBJECT(gdd));
        return (NULL);
        /* notreached */
    }

    return (gdd);
}

/* ------------------------------------------------------ */
/*
 * construction of the object
 */

gboolean
gdd_construct(const Gdd *gdd,
              const gchar *fname,
              const gchar *root,
              const gchar *domain)
{
    gboolean constructed;
    
    g_return_val_if_fail(gdd != NULL, FALSE);
    g_return_val_if_fail(IS_GDD(gdd), FALSE);

    constructed = glade_xml_construct(GLADE_XML(gdd), fname, root, domain);
    if (constructed == FALSE)
    {
        return (FALSE);
        /* notreached */
    }

    /* steal the tree */
    gdd->priv->tree = glade_widget_tree_parse_file(GLADE_XML(gdd)->filename);
    
    /* build list of widgets */
    gdd_scan_tree(gdd, root);
    
    return (TRUE);
}

/* ------------------------------------------------------ */
/*
 * init the gdd class
 */

static void
gdd_class_init(GddClass *class)
{
    GtkObjectClass *object_class;

    parent_class = gtk_type_class(glade_xml_get_type());

    object_class = GTK_OBJECT_CLASS(class);
    object_class->destroy = gdd_destroy;    
    
    gdd_signals[CHANGED_SIGNAL] =
        gtk_signal_new ("changed", GTK_RUN_FIRST, object_class->type,
                        GTK_SIGNAL_OFFSET(GddClass, changed),
                        gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);
    gtk_object_class_add_signals(object_class, gdd_signals, LAST_SIGNAL);
    class->changed = NULL;
}

/* ------------------------------------------------------ */
/*
 * init the gdd object
 */

static void
gdd_init(Gdd *gdd)
{
    g_return_if_fail(gdd != NULL);
    g_return_if_fail(IS_GDD(gdd));

    gdd->priv = g_new(GddPrivate, 1);
    if (gdd->priv == NULL)
    {
        g_warning(__FUNCTION__ ": can't alloc GddPrivate\n");
        return;
        /* notreached */
        /* terminate prg here? */
    }
    gdd->priv->tree = NULL;
    gdd->priv->wlist = NULL;
    gdd->priv->auto_disconnect_cmonitor = FALSE;
    gdd->priv->signals_are_connected = FALSE;
    gdd->priv->signals_are_blocked = FALSE;
}

/* ------------------------------------------------------ */
/*
 * destroy handler
 */

static void
gdd_destroy(GtkObject *object)
{
    Gdd        *gdd;
    GddPrivate *priv;
    
    g_return_if_fail(object != NULL);
    g_return_if_fail(IS_GDD(object));

    gdd = GDD(object);
    priv = gdd->priv;
    
    if (priv != NULL)
    {
        if (priv->tree != NULL)
        {
            glade_widget_tree_unref(priv->tree);
        }

        if (priv->wlist != NULL)
        {
            if (priv->signals_are_connected == TRUE)
            {
                GList         *list;
                GddWidgetData *wdata;
            
                /*
                 * disconnect all signals in wlist
                 * (not sure if needed)
                 */
                
                for (list = g_list_first(priv->wlist);
                     list != NULL;
                     list = g_list_next(list))
                {
                    wdata = list->data;
                    gtk_signal_disconnect(wdata->connected_object,
                                          wdata->connection_id);
                }
            }

            g_list_free(priv->wlist);
            priv->wlist = NULL;
        }
        
        g_free(priv);
        gdd->priv = NULL;
    }

    if (parent_class->destroy)
    {
        (* parent_class->destroy)(object);
    }
}

/* ------------------------------------------------------ */
/*
 * type function
 */

GtkType
gdd_get_type(void)
{
    static GtkType gdd_type = 0;

    if (gdd_type == 0)
    {
        static const GtkTypeInfo gdd_info =
        {
            "Gdd",
            sizeof (Gdd),
            sizeof (GddClass),
            (GtkClassInitFunc) gdd_class_init,
            (GtkObjectInitFunc) gdd_init,
            /* reserved_1 */ NULL,
            /* reserved_2 */ NULL,
            (GtkClassInitFunc) NULL,
        };
        gdd_type = gtk_type_unique(glade_xml_get_type(), &gdd_info);
    }
    
    return (gdd_type);
}

/* ------------------------------------------------------ */
/*
 * scan tree for widgets we care about and collect them in
 * gdd->priv->wlist
 */

static void
gdd_scan_tree(const Gdd *gdd,
              const gchar *root)
{
    GladeWidgetInfo *gwi;
    GList           *wlist;
    GddPrivate      *priv;
    GddWidgetData   *gddwd;
    gboolean        found;
    gint            tnum;
    
    g_return_if_fail(gdd != NULL);
    g_return_if_fail(IS_GDD(gdd));

    priv = gdd->priv;
    if (root != NULL)
    {
        gwi = g_hash_table_lookup(priv->tree->names, root);
        if (gwi == NULL)
        {
            return;
            /* notreached */
        }
        
        found = FALSE;
        for (tnum = 0; (tnum < gdd_types && found == FALSE); tnum++)
        {
            if (gdd_typelist[tnum] == NULL || gwi->class == NULL)
            {
                continue;
                /* notreached */
            }
            
            if (strcmp(gdd_typelist[tnum], gwi->class) != 0)
            {
                continue;
                /* notreached */
            }
            
            found = TRUE;
            
            gddwd = g_new(GddWidgetData, 1);
            gddwd->gwi = gwi;
            gddwd->connected_object = NULL;
            gddwd->connection_id = 0;
            
            priv->wlist = g_list_append(priv->wlist, gddwd);
        }
        
        /* continue with children */
        wlist = gwi->children;
    }
    else
    {
        /* scan toplevel widgets */
        wlist = priv->tree->widgets;
    }

    for (;
         wlist != NULL;
         wlist = g_list_next(wlist))
    {
        gwi = wlist->data;
        gdd_scan_tree(gdd, gwi->name);
    }    
}

/* ------------------------------------------------------ */
/*
 * turn on change-monitor for all scanned widgets
 * return: 1 = error
 *         0 = ok
 */
 
gint
gdd_cmonitor_on(Gdd *gdd,
                const gboolean auto_disconnect)
{
    GddPrivate *priv;
    
    g_return_val_if_fail(gdd != NULL, 1);
    g_return_val_if_fail(IS_GDD(gdd), 1);
    g_return_val_if_fail(gdd->priv->wlist != NULL, 1);

    priv = gdd->priv;
    
    if (priv->signals_are_connected == FALSE)
    {
        g_list_foreach(priv->wlist, (*gdd_connect_widget_list_element), gdd);
        priv->signals_are_connected = TRUE;
    }
    else
    {
        if (priv->signals_are_blocked == TRUE)
        {            
            GList         *wlist;
            GddWidgetData *wdata;
            
            for (wlist = g_list_first(priv->wlist);
                 wlist != NULL;
                 wlist = g_list_next(wlist))
            {
                wdata = wlist->data;
                gtk_signal_handler_unblock(wdata->connected_object,
                                           wdata->connection_id);
            }
        }
    }
    
    priv->signals_are_blocked = FALSE;
    priv->auto_disconnect_cmonitor = auto_disconnect;

    return (0);
}

/* ------------------------------------------------------ */
/*
 * turn off change-monitor
 * return: 1 = error
 *         0 = ok
 */

gint
gdd_cmonitor_off(Gdd *gdd)
{
    GList         *wlist;
    GddWidgetData *wdata;

    g_return_val_if_fail(gdd != NULL, 1);
    g_return_val_if_fail(IS_GDD(gdd), 1);
    g_return_val_if_fail(gdd->priv->wlist != NULL, 1);

    for (wlist = g_list_first(gdd->priv->wlist);
         wlist != NULL;
         wlist = g_list_next(wlist))
    {
        wdata = wlist->data;
        gtk_signal_handler_block(wdata->connected_object, wdata->connection_id);
    }

    gdd->priv->signals_are_blocked = TRUE;

    return (0);
}

/* ------------------------------------------------------ */
/*
 * guess what it does
 */

gboolean
gdd_cmonitor_is_running(const Gdd *gdd)
{
    g_return_val_if_fail(gdd != NULL, FALSE);
    g_return_val_if_fail(IS_GDD(gdd), FALSE);

    return (gdd->priv->signals_are_connected &&
            (! gdd->priv->signals_are_blocked) );
}

/* ------------------------------------------------------ */
/*
 * try to set values 
 * return: 1 = error
 *         0 = ok
 */

gint
gdd_set_values(const Gdd *gdd,
               const xmlNodePtr node)
{
    xmlNodePtr nlist;
    xmlChar    *val;
    GtkWidget  *widget;
    
    g_return_val_if_fail(gdd != NULL, 1);
    g_return_val_if_fail(IS_GDD(gdd), 1);
    g_return_val_if_fail(node != NULL, 1);

    for (nlist = node->childs;
         nlist != NULL;
         nlist = nlist->next)
    {
        val = xmlNodeListGetString(nlist->doc, nlist->childs, 1);
        widget = gdd_widget_from_wlist_by_name(gdd, nlist->name);
        
        if (GTK_IS_EDITABLE(widget) == TRUE)
        {
            gint pos;
            
            gtk_editable_delete_text(GTK_EDITABLE(widget), 0, -1);
            pos = 0;
            if (val != NULL)
            {                
                gtk_editable_insert_text(GTK_EDITABLE(widget), val, strlen(val),
                                         &pos);
            }

            continue;
            /* notreached */
        }

        if (GTK_IS_TOGGLE_BUTTON(widget) == TRUE)
        {
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget),
                                         !strcmp(val, tb_toggled_str));

            continue;
            /* notreached */
        }

        if (GTK_IS_RANGE(widget) == TRUE)
        {
            GtkAdjustment *adj;
            
            adj = gtk_range_get_adjustment(GTK_RANGE(widget));
            if (adj != NULL)
            {
                gtk_adjustment_set_value(adj, g_strtod(val, NULL));
            }
            else
            {
                g_warning(__FUNCTION__ ": can't find adjustment for range.");
            }
            
            continue;
            /* notreached */
        }        
    }
    
/*     FIXFUNC("set/ clear values for other widget types"); */
    return(0);
}

/* ------------------------------------------------------ */
/*
 * create xml from names& values  <GtkEntry>some text</GtkEntry>
 * return: NULL  = error
 *         !NULL = xml-node
 */

xmlNodePtr
gdd_get_values(const Gdd *gdd,
               const xmlDocPtr doc,
               const gchar *name)
{
    xmlNodePtr    node;
    GList         *list;
    GtkWidget     *widget;
    GddWidgetData *wdata;
    gchar         *buf;
    
    g_return_val_if_fail(gdd != NULL, NULL);
    g_return_val_if_fail(IS_GDD(gdd), NULL);

    node = xmlNewDocNode(doc, NULL, g_strdup(name), NULL);

    for (list = g_list_first(gdd->priv->wlist);
         list != NULL;
         list = g_list_next(list))
    {
        buf = NULL;
        wdata = list->data;
        if (wdata == NULL)
        {
            g_warning(__FUNCTION__ ": got empty wdata!\n");
            continue;
            /* notreached */
        }
        widget = glade_xml_get_widget(GLADE_XML(gdd), wdata->gwi->name);
        
        if (GTK_IS_EDITABLE(widget) == TRUE) 
        {
            buf = gtk_editable_get_chars(GTK_EDITABLE(widget), 0, -1);
            xmlNewChild(node, NULL, wdata->gwi->name, buf);

            continue;
            /* notreached */
        }

        if (GTK_IS_TOGGLE_BUTTON(widget) == TRUE)
        {
            buf = g_strdup(
                gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))
                ? tb_toggled_str : tb_utoggled_str);
            xmlNewChild(node, NULL, wdata->gwi->name, buf);        

            continue;
            /* notreached */
        }

        if (GTK_IS_RANGE(widget) == TRUE)
        {
            GtkAdjustment *adj;
            
            adj = gtk_range_get_adjustment(GTK_RANGE(widget));
            if (adj != NULL)
            {
                buf = g_strdup_printf("%f", adj->value);
                xmlNewChild(node, NULL, wdata->gwi->name, buf);        
            }
            else
            {
                g_warning(__FUNCTION__": Cant find adjustment for range.");
            }

            continue;
            /* notreached */
        }

        g_warning(__FUNCTION__ ": got unknown widget-type");
    }
    
/*     FIXFUNC("get values from other widget types"); */
    
    return (node);
}

/* ------------------------------------------------------ */
/*
 * reset widget values
 * return: 1 = error
 *         0 = ok
 */

gint
gdd_clear_values(const Gdd *gdd)
{
    GddWidgetData *wdata;
    GtkWidget     *widget;
    GList         *list;
    
    g_return_val_if_fail(gdd != NULL, 1);
    g_return_val_if_fail(IS_GDD(gdd), 1);

    for (list = g_list_first(gdd->priv->wlist);
         list != NULL;
         list = g_list_next(list))
    {
        wdata = list->data;
        if (wdata == NULL)
        {
            g_warning(__FUNCTION__ ": got empty wdata!\n");
            continue;
            /* notreached */
        }

        widget = glade_xml_get_widget(GLADE_XML(gdd), wdata->gwi->name);
        
        if (GTK_IS_EDITABLE(widget) == TRUE) 
        {
            gtk_editable_delete_text(GTK_EDITABLE(widget), 0, -1);
            continue;
            /* notreached */
        }
        
        if (GTK_IS_TOGGLE_BUTTON(widget) == TRUE)
        {
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), FALSE);
            continue;
            /* notreached */
        }

        if (GTK_IS_RANGE(widget) == TRUE)
        {
            GtkAdjustment *adj;
            
            adj = gtk_range_get_adjustment(GTK_RANGE(widget));
            if (adj != NULL)
            {
                gtk_adjustment_set_value(adj, 0);
            }
            else
            {
                g_warning(__FUNCTION__ ": Cant find adjustment for range.");
            }            
            continue;
            /* notreached */
        }        

        g_warning(__FUNCTION__ ": got unknown widget-type");
    }
    
    return (0);
}

/* ------------------------------------------------------ */
/*
 * return: NULL  = not found || error
 *         !NULL = widget from list of scanned widgets
 */

static GtkWidget*
gdd_widget_from_wlist_by_name(const Gdd *gdd,
                              const gchar *name)
{
    GList         *list;
    gboolean      found;
    GtkWidget     *ret;
    GddWidgetData *wdata;
    
    g_return_val_if_fail(gdd != NULL, NULL);
    g_return_val_if_fail(IS_GDD(gdd), NULL);
    g_return_val_if_fail(name != NULL, NULL);
    g_return_val_if_fail((strlen(name) > 0), NULL);

    ret = NULL;
    found = FALSE;    

    for (list = g_list_first(gdd->priv->wlist);
         (list != NULL && found == FALSE);
         list = g_list_next(list))
    {
        wdata = list->data;
        if (wdata == NULL)
        {
            g_warning(__FUNCTION__ ": got empty wdata!\n");
            continue;
            /* notreached */
        }
        if (strcmp(wdata->gwi->name, name) == 0)
        {
            ret = glade_xml_get_widget(GLADE_XML(gdd), wdata->gwi->name);
            found = TRUE;
        }
    }
    
    return (ret);
}

/* ------------------------------------------------------ */
/*
 * notify listeners that something has changed
 */

static void
gdd_cb_object_changed(GtkObject *o,
                      gpointer gdd)
{
    g_return_if_fail(o != NULL);
    g_return_if_fail(GTK_IS_OBJECT(o));
    g_return_if_fail(gdd != NULL);
    g_return_if_fail(IS_GDD(gdd));
    
    gtk_signal_emit(GTK_OBJECT(gdd), gdd_signals[CHANGED_SIGNAL]);

    if (gdd_cmonitor_get_auto_disconnect(gdd) == TRUE)
    {
        gdd_cmonitor_off(GDD(gdd));
    }
}

/* ------------------------------------------------------ */

gint
gdd_cmonitor_set_auto_disconnect(const Gdd *gdd,
                                 const gboolean auto_disconnect)
{
    g_return_val_if_fail(gdd != NULL, 1);
    g_return_val_if_fail(IS_GDD(gdd), 1);

    gdd->priv->auto_disconnect_cmonitor = auto_disconnect;

    return (0);
}

/* ------------------------------------------------------ */

gboolean
gdd_cmonitor_get_auto_disconnect(const Gdd *gdd)
{
    g_return_val_if_fail(gdd != NULL, FALSE);
    g_return_val_if_fail(IS_GDD(gdd), FALSE);

    return (gdd->priv->auto_disconnect_cmonitor);
}

/* ------------------------------------------------------ */
/*
 * print name and class of scanned widgets
 */

void
gdd_print_scanned_widgets(const Gdd *gdd)
{
    GList         *list;
    GddWidgetData *wdata;
    
    g_return_if_fail(gdd != NULL);
    g_return_if_fail(IS_GDD(gdd));

    for (list = g_list_first(gdd->priv->wlist);
         list != NULL;
         list = g_list_next(list))
    {
        wdata = list->data;
        printf(__FUNCTION__ ": name: %s\tclass:%s\n", 
               wdata->gwi->name, wdata->gwi->class);
    }
}

/* ------------------------------------------------------ */
/*
 * print the names of all widget from the glade-file
 */

void
gdd_print_glade_widgets(const Gdd *gdd,
                        const gchar *root)
{
    GladeWidgetInfo *gwi;
    GList           *wlist;
    
    g_return_if_fail(gdd != NULL);
    g_return_if_fail(IS_GDD(gdd));

    if (root != NULL)
    {
        gwi = g_hash_table_lookup(gdd->priv->tree->names, root);
        printf(__FUNCTION__ ": name: %s\tclass: %s\n", gwi->name, gwi->class);

        /* continue with children */
        wlist = gwi->children;
    }
    else
    {
        /* start from toplevel widgets */
        wlist = gdd->priv->tree->widgets;
    }

    for (;
         wlist != NULL;
         wlist = g_list_next(wlist))
    {
        gwi = wlist->data;
        gdd_print_glade_widgets(gdd, gwi->name);
    }
}

/* ------------------------------------------------------ */
/*
 * connect scanned widgets
 * called from gdd_cmonitor_on()
 */
 
static void
gdd_connect_widget_list_element(gpointer lelem,
                                gpointer gdd)
{
    GddWidgetData *wdata;
    GtkWidget     *widget;

    g_return_if_fail(lelem != NULL);
    g_return_if_fail(gdd != NULL);
    g_return_if_fail(IS_GDD(gdd));
    
    wdata = lelem;
    widget = glade_xml_get_widget(GLADE_XML(gdd), wdata->gwi->name);

    if (GTK_IS_EDITABLE(widget) == TRUE)
    {
        wdata->connection_id =
            gtk_signal_connect(GTK_OBJECT(widget), "changed",
                               GTK_SIGNAL_FUNC(gdd_cb_object_changed), gdd);
        wdata->connected_object = GTK_OBJECT(widget);

        return;
        /* notreached */
    }
    
    if (GTK_IS_TOGGLE_BUTTON(widget) == TRUE)
    {
        wdata->connection_id =
            gtk_signal_connect(GTK_OBJECT(widget), "toggled",
                                   GTK_SIGNAL_FUNC(gdd_cb_object_changed), gdd);
        wdata->connected_object = GTK_OBJECT(widget);

        return;
        /* notreached */
    }

    if (GTK_IS_RANGE(widget) == TRUE)
    {
        GtkAdjustment *adj;
        
        adj = gtk_range_get_adjustment(GTK_RANGE(widget));
        if (adj != NULL)
        {
            wdata->connection_id =
                gtk_signal_connect(GTK_OBJECT(adj), "value-changed",
                                   GTK_SIGNAL_FUNC(gdd_cb_object_changed), gdd);
            wdata->connected_object = GTK_OBJECT(adj);
        }
        else
        {
            g_warning(__FUNCTION__ ": Cant find adjustment for range.");
        }

        return;
        /* notreached */
    }
    g_warning(__FUNCTION__ ": got unknown widget-type");
}

/* ------------------------------------------------------ */
