/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <glib.h>

#include <yehia/plugin.h>

#include "python-generic.h"

namespace Yehia
{

namespace Script
{

namespace
{

GMemChunk *slot_memchunk = NULL;

void 		generic_function_free(PyObject *self);
PyObject *	generic_function_call(PyObject *self, 
                                      PyObject *args, 
                                      PyObject *kw);

}

PyTypeObject pythonGenericFunctionType =
{
  PyObject_HEAD_INIT(NULL)
  0,
  "GenericFunction",
  sizeof(pythonGenericFunction),
  0,
  generic_function_free, /*tp_dealloc*/
  0,          /*tp_print*/
  0,          /*tp_getattr*/
  0,          /*tp_setattr*/
  0,          /*tp_compare*/
  0,          /*tp_repr*/
  0,          /*tp_as_number*/
  0,          /*tp_as_sequence*/
  0,          /*tp_as_mapping*/
  0,          /*tp_hash */
  generic_function_call,
  0
};

PyObject *pythonGenericFunction_New()
{
  pythonGenericFunction *pyfunc;
  
  if (slot_memchunk == NULL)
  {
    slot_memchunk = g_mem_chunk_new("python uCXX slot memchunk",
                                    sizeof(ObjectSlot),
                                    64,
                                    G_ALLOC_AND_FREE);
  }
  
  pyfunc = PyObject_New(pythonGenericFunction, &pythonGenericFunctionType);
  pyfunc->slots = NULL;
  
  return (PyObject *)pyfunc;
}

void pythonGenericFunction_AddSlot(PyObject *self, const ObjectSlot& slot)
{
  pythonGenericFunction *pyfunc = (pythonGenericFunction *)self;
  
  g_return_if_fail(self != NULL);
  g_return_if_fail(slot_memchunk);
  
  gpointer mem = g_mem_chunk_alloc(slot_memchunk);
  new (mem) ObjectSlot(slot);
  //cout << "py-gen-add-slot: slot.node_=" << ((SlotBase *)mem)->node_ << endl;
  pyfunc->slots = g_slist_append(pyfunc->slots, mem);
}

namespace
{

void generic_function_free(PyObject *self)
{
  pythonGenericFunction *pyfunc = (pythonGenericFunction *)self;

  g_assert(slot_memchunk != NULL);
  
  for (GSList *node = pyfunc->slots; node; node = g_slist_next(node))
    g_chunk_free(node->data, slot_memchunk);
  g_slist_free(pyfunc->slots);
  
  PyObject_Del(self);
}

PyObject *generic_function_call(PyObject *self, PyObject *args, PyObject *kw)
{
  pythonGenericFunction *pyfunc = (pythonGenericFunction *)self;
  Language *lang = LanguageManager::instance().language("python");
  pythonObjectFactory *factory = lang ? 
    &dynamic_cast<pythonObjectFactory&>(lang->factory()) : 0;
  Object *result = 0;
  ParamList params;
  
  g_return_val_if_fail(factory, NULL);
  
  if (!PyTuple_Check(args))
    return NULL;
  
  if (PyInstance_Check(self))
    params.push_back(&factory->create_object(self));
  
  for (int i = 0; i < PyTuple_Size(args); i++)
  {
    PyObject *item = PyTuple_GET_ITEM(args, i);
    Py_INCREF(item); // it's decref'd in the Object destructor.
    params.push_back(&factory->create_object(item));
  }
  
  bool match = false, exception = false;
  for (GSList *node = pyfunc->slots; node; node = g_slist_next(node))
  {
    try
    {
      result = (*((ObjectSlot *)node->data))(params);
      match = true;
      break; // no exception -> we have called a func with matching
             // signature
    }
    // BadParam and BadAnyCast mean that there was no match, we simply
    // ignore them and reiterate
    catch (const BadParam& e) { }
    catch (const BadAnyCast& e) { }
    
    // Other exceptions are forwarded to Python
    catch (const std::runtime_error& e)
    {
      exception = true;
      PyErr_SetString(PyExc_RuntimeError, e.what());
      break;
    }
    catch (...)
    {
      exception = true;
      PyErr_SetString(PyExc_RuntimeError, "unknown C++ exception");
      break;
    }
  }
  
  for (ParamList::iterator it = params.begin(); it != params.end(); ++it)
    (*it)->unreference();
  
  if (exception)
    return NULL;
  
  if (!match)
  {
    std::string msg = "No applicable method for call to ";
    
    PyObject *strobj = PyObject_Str(self);
    msg += PyString_AsString(strobj);
    Py_DECREF(strobj);
    msg += " with ";
    strobj = PyObject_Str(args);
    msg += PyString_AsString(strobj);
    Py_DECREF(strobj);
    
    PyErr_SetString(PyExc_LookupError, msg.c_str());
    return NULL;
  }

  pythonObject *pyoresult = dynamic_cast<pythonObject *>(result);
  PyObject *pyresult = pyoresult ? pyoresult->pyobj() : NULL;
  
  if (pyresult == NULL)
    pyresult = Py_None;
  
  Py_INCREF(pyresult);

  if (result)
    result->unreference();
  
  return pyresult;
}


}

}

}
